// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifdef __GNUG__
#pragma implementation
#endif
#include "Entity.H"
#include "ParserState.H"
#include "macros.H"
#include "InternalInputSource.H"
#include "MessageArg.H"
#include "EntityManager.H"

Entity::Entity(const CString &name, DeclType declType, DataType dataType,
	       const Location &defLocation)
: NamedResource(name),
  declType_(declType),
  dataType_(dataType),
  defLocation_(defLocation)
{
}

const SubstTable<Char> *Entity::substTable(const Syntax &syntax) const
{
  switch (declType()) {
  case generalEntity:
  case parameterEntity:
    return syntax.namecaseEntity() ? syntax.entitySubstTable() : 0;
  case doctype:
  case linktype:
    return syntax.namecaseGeneral() ? syntax.generalSubstTable() : 0;
  default:
    CANNOT_HAPPEN();
  }
  return 0;			// not reached
}

InternalEntity::InternalEntity(const CString &name,
			       DeclType declType,
			       DataType dataType,
			       const Location &defLocation,
			       Text &text)
: Entity(name, declType, dataType, defLocation)
{
  text.moveTo(text_);
}

PiEntity::PiEntity(const CString &name, DeclType declType,
		   const Location &defLocation, Text &text)
: InternalEntity(name, declType, pi, defLocation, text)
{
}

Entity *PiEntity::copy() const
{
  return new PiEntity(*this);
}

InternalDataEntity::InternalDataEntity(const CString &name, DataType dataType,
				       const Location &defLocation, Text &text)
: InternalEntity(name, generalEntity, dataType, defLocation, text)
{
}


InternalCdataEntity::InternalCdataEntity(const CString &name,
					 const Location &defLocation,
					 Text &text)
: InternalDataEntity(name, cdata, defLocation, text)
{
}

Entity *InternalCdataEntity::copy() const
{
  return new InternalCdataEntity(*this);
}

InternalSdataEntity::InternalSdataEntity(const CString &name,
					 const Location &defLocation,
					 Text &text)
: InternalDataEntity(name, sdata, defLocation, text)
{
}

Entity *InternalSdataEntity::copy() const
{
  return new InternalSdataEntity(*this);
}

InternalTextEntity::InternalTextEntity(const CString &name, DeclType declType,
				       const Location &defLocation, Text &text,
				       Bracketed bracketed)
: InternalEntity(name, declType, sgmlText, defLocation, text),
  bracketed_(bracketed)
{
}

Entity *InternalTextEntity::copy() const
{
  return new InternalTextEntity(*this);
}


ExternalEntity::ExternalEntity(const CString &name,
			       DeclType declType,
			       DataType dataType,
			       const Location &defLocation,
			       ExternalId &id)
: Entity(name, declType, dataType, defLocation)
{
  id.moveTo(externalId_);
}

const ExternalEntity *ExternalEntity::asExternalEntity() const
{
  return this;
}

ExternalTextEntity::ExternalTextEntity(const CString &name,
				       DeclType declType,
				       const Location &defLocation,
				       ExternalId &id)
: ExternalEntity(name, declType, sgmlText, defLocation, id)
{
}

Entity *ExternalTextEntity::copy() const
{
  return new ExternalTextEntity(*this);
}

ExternalNonTextEntity::ExternalNonTextEntity(const CString &name,
					     DataType dataType,
					     const Location &defLocation,
					     ExternalId &id)
: ExternalEntity(name, generalEntity, dataType, defLocation, id)
{
}

ExternalDataEntity::ExternalDataEntity(const CString &name,
				       DataType dataType,
				       const Location &defLocation,
				       ExternalId &id,
				       const ConstResourcePointer<Notation> &nt,
				       
				       AttributeList &attributes)
: ExternalNonTextEntity(name, dataType, defLocation, id),
  notation_(nt)
{
  attributes.moveTo(attributes_);
}

void ExternalDataEntity::setAttributes(AttributeList &attributes)
{
  attributes.moveTo(attributes_);
}

Entity *ExternalDataEntity::copy() const
{
  return new ExternalDataEntity(*this);
}

SubdocEntity::SubdocEntity(const CString &name,
			   const Location &defLocation,
			   ExternalId &id)
: ExternalNonTextEntity(name, subdoc, defLocation, id)
{
}

Entity *SubdocEntity::copy() const
{
  return new SubdocEntity(*this);
}

Boolean Entity::isDataOrSubdoc() const
{
  return 0;
}

Boolean Entity::isCharacterData() const
{
  return 0;
}

const ExternalEntity *Entity::asExternalEntity() const
{
  return 0;
}

const ExternalDataEntity *Entity::asExternalDataEntity() const
{
  return 0;
}

const SubdocEntity *Entity::asSubdocEntity() const
{
  return 0;
}

const InternalEntity *Entity::asInternalEntity() const
{
  return 0;
}

void Entity::dsReference(ParserState &parser,
			 const ResourcePointer<EntityOrigin> &origin)
     const
{
  normalReference(parser, origin);
}

void Entity::declReference(ParserState &parser,
			   const ResourcePointer<EntityOrigin> &origin)
     const
{
  normalReference(parser, origin);
}

void Entity::contentReference(ParserState &parser,
			      const ResourcePointer<EntityOrigin> &origin)
     const
{
  normalReference(parser, origin);
}

void Entity::rcdataReference(ParserState &parser,
			   const ResourcePointer<EntityOrigin> &origin)
     const
{
  normalReference(parser, origin);
}

void Entity::litReference(Text &, ParserState &parser,
			  const ResourcePointer<EntityOrigin> &origin,
			  Boolean)
     const
{
  normalReference(parser, origin);
}

const InternalEntity *InternalEntity::asInternalEntity() const
{
  return this;
}

void PiEntity::litReference(Text &, ParserState &parser,
			    const ResourcePointer<EntityOrigin> &,
			    Boolean) const
{
  parser.message(Messages::piEntityReference);
}

void PiEntity::normalReference(ParserState &parser,
			       const ResourcePointer<EntityOrigin> &origin) const
{
  parser.noteMarkup();
  parser.eventHandler().pi(new (parser.eventAllocator())
			   PiEntityEvent(this, origin.pointer()));
}

void PiEntity::declReference(ParserState &parser,
			     const ResourcePointer<EntityOrigin> &) const
{
  parser.message(Messages::piEntityReference);
}

void PiEntity::rcdataReference(ParserState &parser,
			       const ResourcePointer<EntityOrigin> &) const
{
  parser.message(Messages::piEntityRcdata);
}

void InternalDataEntity::declReference(ParserState &parser,
				       const ResourcePointer<EntityOrigin> &) const
{
  parser.message(Messages::internalDataEntityReference);
}

Boolean InternalDataEntity::isDataOrSubdoc() const
{
  return 1;
}

void InternalDataEntity::addChars(Text &text,
				  Location &loc,
				  Boolean squeeze,
				  Char space) const
{
  const CString &str(text_.string());
  if (squeeze) {
    for (size_t i = 0; i < str.length(); loc += 1, i++) {
      if (str[i] == space && (text.length() == 0 || text.lastChar() == space))
	text.ignoreChar(str[i], loc);
      else
	text.addChar(str[i], loc);
    }
  }
  else {
    text.addChars(str, loc);
    loc += str.length();
  }
}

void InternalCdataEntity::normalReference(ParserState &parser,
					  const ResourcePointer<EntityOrigin> &origin) const
{
  checkEntlvl(parser);
  // FIXME emit entity start and entity end events in esis+ mode
  if (string().length() > 0) {
    parser.noteData();
    parser.eventHandler().data(new (parser.eventAllocator())
			       CdataEntityEvent(this, origin.pointer()));
  }
}

Boolean InternalCdataEntity::isCharacterData() const
{
  return string().length() > 0;
}

void InternalCdataEntity::litReference(Text &text,
				       ParserState &parser,
				       const ResourcePointer<EntityOrigin> &origin,
				       Boolean squeeze) const
{
  checkEntlvl(parser);
  Location loc(origin.pointer(), 0);
  text.addCdataEntityStart(loc);
  addChars(text, loc, squeeze, parser.syntax().space());
  text.addCdataEntityEnd(loc);
}


void InternalSdataEntity::normalReference(ParserState &parser,
					  const ResourcePointer<EntityOrigin> &origin) const
{
  checkEntlvl(parser);
  parser.noteData();
  parser.eventHandler().sdataEntity(new (parser.eventAllocator())
				    SdataEntityEvent(this,
						     origin.pointer()));
}

Boolean InternalSdataEntity::isCharacterData() const
{
  return 1;
}

void InternalSdataEntity::litReference(Text &text,
				       ParserState &parser,
				       const ResourcePointer<EntityOrigin> &origin,
				       Boolean squeeze) const
{
  checkEntlvl(parser);
  Location loc(origin.pointer(), 0);
  text.addSdataEntityStart(loc);
  addChars(text, loc, squeeze, parser.syntax().space());
  text.addSdataEntityEnd(loc);
}

void InternalTextEntity::normalReference(ParserState &parser,
					 const ResourcePointer<EntityOrigin> &origin) const
{
  checkEntlvl(parser);
  if (checkNotOpen(parser))
    parser.pushInput(new (parser.internalAllocator())
		       InternalInputSource(text_.string(), origin.pointer()));
}

void InternalTextEntity::litReference(Text &text,
				      ParserState &parser,
				      const ResourcePointer<EntityOrigin> &origin,
				      Boolean) const
{
  text.addEntityStart(Location(origin.pointer(), 0));
  normalReference(parser, origin);
}

void ExternalTextEntity::normalReference(ParserState &parser,
					 const ResourcePointer<EntityOrigin> &origin) const
{
  checkEntlvl(parser);
  if (checkNotOpen(parser))
    parser.pushInput(parser.entityManager()
	     ->openExternal(*this,
			    origin.pointer(),
			    parser.sd().docCharset(),
			    substTable(parser.syntax()),
			    parser.syntax().delimGeneral(Syntax::dPERO),
			    0,
			    parser.inputContext()));
}

const ExternalDataEntity *ExternalDataEntity::asExternalDataEntity() const
{
  return this;
}

void ExternalDataEntity::contentReference(ParserState &parser,
					  const ResourcePointer<EntityOrigin> &origin) const
{
  checkEntlvl(parser);
  parser.noteData();
  parser.eventHandler().externalDataEntity(new (parser.eventAllocator())
					   ExternalDataEntityEvent(this, origin.pointer()));
}

Boolean ExternalNonTextEntity::isDataOrSubdoc() const
{
  return 1;
}

Boolean ExternalNonTextEntity::isCharacterData() const
{
  return 1;
}


void ExternalNonTextEntity::normalReference(ParserState &parser,
					    const ResourcePointer<EntityOrigin> &) const
{
  parser.message(Messages::externalNonTextEntityReference);
}

void ExternalNonTextEntity::litReference(Text &,
					 ParserState &parser,
					 const ResourcePointer<EntityOrigin> &,
					 Boolean) const
{
  parser.message(Messages::externalNonTextEntityRcdata);
}

void ExternalNonTextEntity::rcdataReference(ParserState &parser,
					    const ResourcePointer<EntityOrigin> &) const
{
  parser.message(Messages::externalNonTextEntityRcdata);
}

void SubdocEntity::contentReference(ParserState &parser,
				    const ResourcePointer<EntityOrigin> &origin) const
{
  checkEntlvl(parser);
  parser.noteData();
  parser.eventHandler().subdocEntity(new (parser.eventAllocator())
				     SubdocEntityEvent(this, origin.pointer()));
}

const SubdocEntity *SubdocEntity::asSubdocEntity() const
{
  return this;
}

void Entity::checkEntlvl(ParserState &parser)
{
  // -1 because document entity isn't counted
  if (parser.inputLevel() - 1 == parser.syntax().entlvl())
    parser.message(Messages::entlvl);
}

Boolean Entity::checkNotOpen(ParserState &parser) const
{
  if (parser.entityIsOpen(this)) {
    parser.message(Messages::recursiveEntityReference,
		   StringMessageArg(name()));
    return 0;
  }
  return 1;
}
