// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifndef Entity_INCLUDED
#define Entity_INCLUDED 1
#ifdef __GNUG__
#pragma interface
#endif

#include "types.H"
#include "CString.H"
#include "NamedResource.H"
#include "Location.H"
#include "Owner.H"
#include "Attribute.H"
#include "ExternalId.H"
#include "Text.H"
#include "SubstTable.H"

class EntityManager;
class InputContext;
class InputSource;
class EntityOrigin;
class ParserState;
class ExternalEntity;
class ExternalDataEntity;
class SubdocEntity;
class InternalEntity;
class Notation;
class Syntax;

class Entity : public NamedResource {
public:
  enum DeclType { generalEntity, parameterEntity, doctype, linktype };
  enum { nDeclType = 4 };
  enum DataType { sgmlText, pi, cdata, sdata, ndata, subdoc };
  Entity(const CString &name, DeclType declType, DataType dataType,
	 const Location &defLocation);
  // reference in a literal
  virtual void litReference(Text &, ParserState &,
			    const ResourcePointer<EntityOrigin> &,
			    Boolean squeezeSpaces)
    const;
  // reference in a declaration
  virtual void declReference(ParserState &,
			     const ResourcePointer<EntityOrigin> &)
    const;
  // reference in a declaration subset
  virtual void dsReference(ParserState &,
			   const ResourcePointer<EntityOrigin> &)
    const;
  // reference in content
  virtual void contentReference(ParserState &,
				const ResourcePointer<EntityOrigin> &)
    const;
  // reference in rcdata
  virtual void rcdataReference(ParserState &,
			       const ResourcePointer<EntityOrigin> &)
    const;
  // for entity name attribute checking
  virtual Boolean isDataOrSubdoc() const;
  // for determining whether we need to validate as character data
  virtual Boolean isCharacterData() const;
  virtual const ExternalDataEntity *asExternalDataEntity() const;
  virtual const SubdocEntity *asSubdocEntity() const;
  virtual const InternalEntity *asInternalEntity() const;
  virtual const ExternalEntity *asExternalEntity() const;
  // needed for default entity
  virtual Entity *copy() const = 0;
  DeclType declType() const;
  DataType dataType() const;
  const Location &defLocation() const;
  // return 0 if no substitution necessary
  const SubstTable<Char> *substTable(const Syntax &) const;
protected:
  static void checkEntlvl(ParserState &);
  Boolean checkNotOpen(ParserState &) const;
private:
  virtual void normalReference(ParserState &,
			       const ResourcePointer<EntityOrigin> &)
    const = 0;
  Location defLocation_;
  DeclType declType_;
  DataType dataType_;
};

class InternalEntity : public Entity {
public:
  InternalEntity(const CString &, DeclType declType, DataType dataType,
		 const Location &, Text &);
  const CString &string() const;
  const Text &text() const;
  const InternalEntity *asInternalEntity() const;
protected:
  Text text_;
};

class PiEntity : public InternalEntity {
public:
  PiEntity(const CString &, DeclType, const Location &, Text &);
  void litReference(Text &, ParserState &,
		    const ResourcePointer<EntityOrigin> &,
		    Boolean) const;
  void normalReference(ParserState &,
		       const ResourcePointer<EntityOrigin> &) const;
  void declReference(ParserState &,
		     const ResourcePointer<EntityOrigin> &) const;
  void rcdataReference(ParserState &,
		       const ResourcePointer<EntityOrigin> &) const;
  Entity *copy() const;
};

class InternalDataEntity : public InternalEntity {
public:
  InternalDataEntity(const CString &, DataType, const Location &, Text &);
  void declReference(ParserState &,
		     const ResourcePointer<EntityOrigin> &) const;
  Boolean isDataOrSubdoc() const;
protected:
  void addChars(Text &text, Location &loc, Boolean squeeze, Char space) const;
};

class InternalCdataEntity : public InternalDataEntity {
public:
  InternalCdataEntity(const CString &, const Location &, Text &);
  void normalReference(ParserState &,
		    const ResourcePointer<EntityOrigin> &) const;
  void litReference(Text &, ParserState &,
		    const ResourcePointer<EntityOrigin> &,
		    Boolean) const;
  Entity *copy() const;
  Boolean isCharacterData() const;
};

class InternalSdataEntity : public InternalDataEntity {
public:
  InternalSdataEntity(const CString &, const Location &, Text &);
  void normalReference(ParserState &,
		       const ResourcePointer<EntityOrigin> &) const;
  void litReference(Text &, ParserState &,
		    const ResourcePointer<EntityOrigin> &,
		    Boolean) const;
  Entity *copy() const;
  Boolean isCharacterData() const;
};

class InternalTextEntity : public InternalEntity {
public:
  enum Bracketed {
    none,
    starttag,
    endtag,
    ms,
    md
    };
  InternalTextEntity(const CString &, DeclType, const Location &, Text &,
		     Bracketed);
  Entity *copy() const;
private:
  void normalReference(ParserState &,
		    const ResourcePointer<EntityOrigin> &) const;
  void litReference(Text &, ParserState &,
		    const ResourcePointer<EntityOrigin> &,
		    Boolean) const;
  Bracketed bracketed_;
};

class ExternalEntity : public Entity {
public:
  ExternalEntity(const CString &, DeclType, DataType, const Location &,
		 ExternalId &);
  const ExternalId &externalId() const;
  const ExternalEntity *asExternalEntity() const;
private:
  ExternalId externalId_;
};

class ExternalTextEntity : public ExternalEntity {
public:
  ExternalTextEntity(const CString &, DeclType, const Location &,
		     ExternalId &);
  Entity *copy() const;
private:
  void normalReference(ParserState &,
		       const ResourcePointer<EntityOrigin> &) const;
};

class ExternalNonTextEntity : public ExternalEntity {
public:
  ExternalNonTextEntity(const CString &, DataType,
			const Location &, ExternalId &);
  Boolean isDataOrSubdoc() const;
  void litReference(Text &, ParserState &,
		    const ResourcePointer<EntityOrigin> &,
		    Boolean) const;
  void rcdataReference(ParserState &,
		       const ResourcePointer<EntityOrigin> &) const;
  void normalReference(ParserState &,
		       const ResourcePointer<EntityOrigin> &) const;
  Boolean isCharacterData() const;
};

class ExternalDataEntity : public ExternalNonTextEntity {
public:
  ExternalDataEntity(const CString &, DataType, const Location &,
		     ExternalId &, const ConstResourcePointer<Notation> &,
		     AttributeList &);
  const AttributeList &attributes() const;
  const Notation *notation() const;
  const ExternalDataEntity *asExternalDataEntity() const;
  Entity *copy() const;
  void contentReference(ParserState &,
			const ResourcePointer<EntityOrigin> &) const;
  void setAttributes(AttributeList &);
private:
  ConstResourcePointer<Notation> notation_;
  AttributeList attributes_;
};

class SubdocEntity : public ExternalNonTextEntity {
public:
  SubdocEntity(const CString &, const Location &, ExternalId &);
  const SubdocEntity *asSubdocEntity() const;
  Entity *copy() const;
  void contentReference(ParserState &,
			const ResourcePointer<EntityOrigin> &) const;
private:
};

inline
Entity::DeclType Entity::declType() const
{
  return declType_;
}

inline
Entity::DataType Entity::dataType() const
{
  return dataType_;
}

inline
const Location &Entity::defLocation() const
{
  return defLocation_;
}

inline
const CString &InternalEntity::string() const
{
  return text_.string();
}

inline
const Text &InternalEntity::text() const
{
  return text_;
}

inline
const ExternalId &ExternalEntity::externalId() const
{
  return externalId_;
}

inline
const AttributeList &ExternalDataEntity::attributes() const
{
  return attributes_;
}

inline
const Notation *ExternalDataEntity::notation() const
{
  return notation_.pointer();
}

#endif /* not Entity_INCLUDED */
