// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifndef Event_INCLUDED
#define Event_INCLUDED 1
#ifdef __GNUG__
#pragma interface
#endif

#include "Link.H"
#include "Allocator.H"
#include "Location.H"
#include "Vector.H"
#include "Owner.H"
#include "Boolean.H"
#include "types.H"
#include "ResourcePointer.H"
#include "CString.H"
#include "Notation.H"
#include "Sd.H"
#include "Syntax.H"
#include "Dtd.H"
#include "ElementType.H"
#include "Text.H"
#include "Lpd.H"

class EventHandler;

class Event : public Link {
public:
  enum Type {
    message,
    characterData,
    ignoredChars,
    startElement,
    endElement,
    pi,
    sdataEntity,
    externalDataEntity,
    subdocEntity,
    reOrigin,
    appinfo,
    startDtd,
    endDtd,
    endProlog,
    sgmlDecl,
    simpleLink,
    complexLink,
    uselink,
    // These are for SDIF and such like.
    notationDecl,
    externalEntityDecl
    };
  Event(Type);
  virtual ~Event();
  virtual void handle(EventHandler &) = 0;
  virtual void copyData();
  void *operator new(size_t sz, Allocator &alloc) { return alloc.alloc(sz); }
  void *operator new(size_t sz) { return Allocator::allocSimple(sz); }
  void operator delete(void *p) { Allocator::free(p); }
  Type type() const;
private:
  Event(const Event &);		// undefined
  void operator=(const Event &); // undefined
  Type type_;
};

class LocatedEvent : public Event {
public:
  LocatedEvent(Type type, const Location &);
  const Location &location() const;
private:
  LocatedEvent(const LocatedEvent &); // undefined
  void operator=(const LocatedEvent &);	// undefined
  Location location_;
};

class OpenElementInfo {
public:
  OpenElementInfo();
  PackedBoolean included;
  CString gi;
  CString matchType;
  unsigned matchIndex;
private:
  OpenElementInfo(const OpenElementInfo &); // undefined
  void operator=(const OpenElementInfo &);
};

class MessageArg;

class MessageEvent : public LocatedEvent {
public:
  typedef const char *Source;
  // Must match flags in InputContext.
  enum Type {
    info,
    warning,
    quantityError,
    error
    };
  MessageEvent(Source, Type, const Location &, int number,
	       Vector<Owner<MessageArg> > &args,
	       const Location &auxLoc);
  const MessageArg *arg(int i) const { return args_[i].pointer(); }
  int nArgs() const { return args_.length(); }
  int number() const { return number_; }
  Source source() const { return source_; }
  Type type() const { return type_; }
  const Location &auxLocation() const;
  void handle(EventHandler &);
  Boolean isError() const;
  void setOpenElementInfo(Vector<OpenElementInfo> &);
  const OpenElementInfo &openElementInfo(unsigned) const;
  unsigned nOpenElements() const;
private:
  MessageEvent(const MessageEvent &); // undefined
  void operator=(const MessageEvent &);	// undefined
  Source source_;
  Type type_;
  int number_;
  Vector<Owner<MessageArg> > args_;
  Location auxLocation_;
  Vector<OpenElementInfo> openElements_;
};

class ElementEvent : public LocatedEvent {
public:
  enum Minimization {
    complete,
    unclosed,
    empty,
    net,
    omittedOmittag,
    omittedEmpty,
    omittedConref
    };
  ElementEvent(Type type,
	       const ElementType *,
	       const ConstResourcePointer<Dtd> &,
	       const Location &,
	       Index nextIndex,
	       Minimization, CString &);
  ElementEvent(Type type,
	       const ElementType *,
	       const ConstResourcePointer<Dtd> &,
	       const Location &,
	       Index nextIndex,
	       Minimization);
  Index nextIndex() const;	// index in entity of next character after tag
  void setIncluded();
  Boolean included() const;
  Minimization minimization() const;
  const CString &name() const;
  const ElementType *elementType() const;
private:
  ElementEvent(const ElementEvent &); // undefined
  void operator=(const ElementEvent &);	// undefined
  const ElementType *elementType_;
  ConstResourcePointer<Dtd> dtd_;
  Minimization minimization_;
  PackedBoolean included_;
  PackedBoolean copied_;	// has name_ been copied?
  Index nextIndex_;
  CString origName_;
};

class AttributeList;

class StartElementEvent : public ElementEvent {
public:
  StartElementEvent(const ElementType *,
		    const ConstResourcePointer<Dtd> &,
		    const Location &,
		    Index nextIndex,
		    Minimization,
		    AttributeList *,
		    CString &);
  StartElementEvent(const ElementType *,
		    const ConstResourcePointer<Dtd> &,
		    const Location &,
		    Index nextIndex,
		    Minimization,
		    AttributeList *);
  ~StartElementEvent();
  void handle(EventHandler &);
  Boolean netEnabling() const;
  const AttributeList &attributes() const;
  void copyData();
private:
  StartElementEvent(const StartElementEvent &);	// undefined
  void operator=(const StartElementEvent &);	// undefined
  AttributeList *attributes_;
  PackedBoolean copied_;
};

class EndElementEvent : public ElementEvent {
public:
  EndElementEvent(const ElementType *,
		  const ConstResourcePointer<Dtd> &,
		  const Location &, Index nextIndex,
		  Minimization, CString &);
  EndElementEvent(const ElementType *,
		  const ConstResourcePointer<Dtd> &,
		  const Location &, Index nextIndex,
		  Minimization);

  void handle(EventHandler &);
private:
  EndElementEvent(const EndElementEvent &); // undefined
  void operator=(const EndElementEvent &);  // undefined
};

class DataEvent : public LocatedEvent {
public:
  DataEvent(Type, const Char *, size_t, const Location &);
  void handle(EventHandler &);
  const Char *data() const;
  size_t dataLength() const;
protected:
  const Char *p_;
  size_t length_;
private:
  DataEvent(const DataEvent &);	// undefined
  void operator=(const DataEvent &); // undefined
};

class ImmediateDataEvent : public DataEvent {
public:
  ImmediateDataEvent(Type type, const Char *, size_t, const Location &,
		     Boolean copy);
  ~ImmediateDataEvent();
  void copyData();
private:
  ImmediateDataEvent(const ImmediateDataEvent &); // undefined
  void operator=(const ImmediateDataEvent &);	  // undefined
  Char *alloc_;
};

class InternalDataEntity;

class DataEntityEvent : public DataEvent {
public:
  DataEntityEvent(Type type, const InternalDataEntity *,
		  const ConstResourcePointer<Origin> &);
private:
  DataEntityEvent(const DataEntityEvent &); // undefined
  void operator=(const DataEntityEvent &);  // undefined
  ConstResourcePointer<Origin> origin_;
};

class InternalCdataEntity;

class CdataEntityEvent : public DataEntityEvent {
public:
  CdataEntityEvent(const InternalCdataEntity *,
		   const ConstResourcePointer<Origin> &);
private:
  CdataEntityEvent(const CdataEntityEvent &); // undefined
  void operator=(const CdataEntityEvent &);   // undefined
};

class InternalSdataEntity;

class SdataEntityEvent : public DataEntityEvent {
public:
  SdataEntityEvent(const InternalSdataEntity *,
		   const ConstResourcePointer<Origin> &);
  void handle(EventHandler &);
private:
  SdataEntityEvent(const SdataEntityEvent &); // undefined
  void operator=(const SdataEntityEvent &);   // undefined
};

class ReOriginEvent : public Event {
public:
  ReOriginEvent(const Location &, unsigned long serial);
  void handle(EventHandler &);
private:
  ReOriginEvent(const ReOriginEvent &);	// undefined
  void operator=(const ReOriginEvent &); // undefined
  Location location_;
  unsigned long serial_;
};

class ReEvent : public ImmediateDataEvent {
public:
  ReEvent(const Char *, const Location &, unsigned long serial);
private:
  ReEvent(const ReEvent &);	// undefined
  void operator=(const ReEvent &); // undefined
  unsigned long serial_;
};

class IgnoredCharsEvent : public ImmediateDataEvent {
public:
  IgnoredCharsEvent(const Char *, size_t, const Location &, Boolean);
  void handle(EventHandler &);
private:
  IgnoredCharsEvent(const IgnoredCharsEvent &);	// undefined
  void operator=(const IgnoredCharsEvent &);	// undefined
};

class RsEvent : public IgnoredCharsEvent {
public:
  RsEvent(const Char *, const Location &, Boolean);
private:
  RsEvent(const RsEvent &);	// undefined
  void operator=(const RsEvent &); // undefined
};

class IgnoredReEvent : public IgnoredCharsEvent {
public:
  IgnoredReEvent(const Char *, const Location &, unsigned long serial);
private:
  IgnoredReEvent(const IgnoredReEvent &); // undefined
  void operator=(const IgnoredReEvent &); // undefined
  unsigned long serial_;
};

class PiEntity;

class PiEvent : public LocatedEvent {
public:
  PiEvent(const Char *, size_t, const Location &);
  const Char *data() const;
  size_t dataLength() const;
  void handle(EventHandler &);
private:
  PiEvent(const PiEvent &);	// undefined
  void operator=(const PiEvent &); // undefined
  const Char *data_;
  size_t dataLength_;
};

class ImmediatePiEvent : public PiEvent {
public:
  ImmediatePiEvent(CString &, const Location &);
private:
  ImmediatePiEvent(const ImmediatePiEvent &); // undefined
  void operator=(const ImmediatePiEvent &);   // undefined
  CString string_;
};

class PiEntityEvent : public PiEvent {
public:
  PiEntityEvent(const PiEntity *entity,
		const ConstResourcePointer<Origin> &origin);
private:
  PiEntityEvent(const PiEntityEvent &);	// undefined
  void operator=(const PiEntityEvent &); // undefined
  ConstResourcePointer<Origin> origin_;
};

class ExternalNonTextEntity;
class ExternalDataEntity;
class SubdocEntity;

class ExternalEntityEvent : public Event {
public:
  ExternalEntityEvent(Type type,
		      const ExternalNonTextEntity *,
		      const ResourcePointer<EntityOrigin> &);
  const ResourcePointer<EntityOrigin> &entityOrigin() const;
private:
  ExternalEntityEvent(const ExternalEntityEvent &); // undefined
  void operator=(const ExternalEntityEvent &);	    // undefined
  const ExternalNonTextEntity *entity_;
  ResourcePointer<EntityOrigin> origin_;
};

class ExternalDataEntityEvent : public ExternalEntityEvent {
public:
  ExternalDataEntityEvent(const ExternalDataEntity *,
			  const ResourcePointer<EntityOrigin> &);
  void handle(EventHandler &);
  const ExternalDataEntity *entity() const;
private:
  ExternalDataEntityEvent(const ExternalDataEntityEvent &); // undefined
  void operator=(const ExternalDataEntityEvent &);	    // undefined
  const ExternalDataEntity *dataEntity_;
};

class SubdocEntityEvent : public ExternalEntityEvent {
public:
  SubdocEntityEvent(const SubdocEntity *,
		    const ResourcePointer<EntityOrigin> &);
  void handle(EventHandler &);
  const SubdocEntity *entity() const;
private:
  SubdocEntityEvent(const SubdocEntityEvent &);	// undefined
  void operator=(const SubdocEntityEvent &);	// undefined
  const SubdocEntity *subdocEntity_;
};

class AppinfoEvent : public LocatedEvent {
public:
  AppinfoEvent(const Location &);
  AppinfoEvent(const Text &, const Location &);
  void handle(EventHandler &);
  Boolean literal(const CString *&) const;
private:
  AppinfoEvent(const AppinfoEvent &); // undefined
  void operator=(const AppinfoEvent &);	// undefined
  Boolean appinfoNone_;
  Text appinfo_;
};

class SimpleLinkEvent : public LocatedEvent {
public:
  SimpleLinkEvent(const CString &,
		  AttributeList &,
		  const Location &);
  void handle(EventHandler &);
  const AttributeList &attributes() const;
  const CString &name() const;
private:
  SimpleLinkEvent(const SimpleLinkEvent &); // undefined
  void operator=(const SimpleLinkEvent &);
  CString name_;
  AttributeList attributes_;
};

class ComplexLinkEvent : public Event {
public:
  ComplexLinkEvent(const ConstResourcePointer<ComplexLpd> &,
		   const VectorBase<ConstResourcePointer<Lpd> > &activeLpds);
  void handle(EventHandler &);
  const CString &name() const;
  const ConstResourcePointer<ComplexLpd> &lpd() const;
  const VectorBase<ConstResourcePointer<Lpd> > &activeLpds() const;
private:
  ComplexLinkEvent(const ComplexLinkEvent &); // undefined
  void operator=(const ComplexLinkEvent &);
  ConstResourcePointer<ComplexLpd> lpd_;
  Vector<ConstResourcePointer<Lpd> > activeLpds_;
};

class UselinkEvent : public LocatedEvent {
public:
  UselinkEvent(const ConstResourcePointer<Lpd> &,
	       const LinkSet *,
	       Boolean restore,
	       const Location &);
  void handle(EventHandler &);
  const ConstResourcePointer<Lpd> &lpd() const;
  const LinkSet *linkSet() const;
  Boolean restore() const;
private:
  UselinkEvent(const UselinkEvent &); // undefined
  void operator=(const UselinkEvent &);	// undefined
  ConstResourcePointer<Lpd> lpd_;
  const LinkSet *linkSet_;
  Boolean restore_;
};

class StartDtdEvent : public LocatedEvent {
public:
  StartDtdEvent(const CString &, const Location &);
  void handle(EventHandler &);
  const CString &name() const;
private:
  StartDtdEvent(const StartDtdEvent &);	// undefined
  void operator=(const StartDtdEvent &); // undefined
  CString name_;
};

class EndDtdEvent : public LocatedEvent {
public:
  EndDtdEvent(const ConstResourcePointer<Dtd> &, const Location &);
  void handle(EventHandler &);
  const Dtd &dtd() const;
private:
  EndDtdEvent(const EndDtdEvent &); // undefined
  void operator=(const EndDtdEvent &); // undefined
  ConstResourcePointer<Dtd> dtd_;
};

class EndPrologEvent : public LocatedEvent {
public:
  EndPrologEvent(const Location &);
  void handle(EventHandler &);
private:
  EndPrologEvent(const EndPrologEvent &); // undefined
  void operator=(const EndPrologEvent &); // undefined
};

class SgmlDeclEvent : public LocatedEvent {
public:
  // for an implied SGML declaration
  SgmlDeclEvent(const ConstResourcePointer<Sd> &,
		const ConstResourcePointer<Syntax> &syntax);
  // for an explicit SGML declaration
  SgmlDeclEvent(const ConstResourcePointer<Sd> &,
		const ConstResourcePointer<Syntax> &syntax,
		const ConstResourcePointer<Syntax> &instanceSyntax,
		const Location &startLocation,
		Index nextIndex);
  void handle(EventHandler &);
  const Sd &sd() const;
  const ConstResourcePointer<Sd> &sdPointer() const;
  const Syntax &prologSyntax() const;
  const ConstResourcePointer<Syntax> &prologSyntaxPointer() const;
  const Syntax &instanceSyntax() const;
  const ConstResourcePointer<Syntax> &instanceSyntaxPointer() const;
private:
  SgmlDeclEvent(const SgmlDeclEvent &);	// undefined
  void operator=(const SgmlDeclEvent &); // undefined
  ConstResourcePointer<Sd> sd_;
  ConstResourcePointer<Syntax> prologSyntax_;
  ConstResourcePointer<Syntax> instanceSyntax_;
  Index nextIndex_;
};

class ExternalEntity;

class ExternalEntityDeclEvent : public Event {
public:
  ExternalEntityDeclEvent(const ConstResourcePointer<Entity> &,
			  Boolean duplicate);
  void handle(EventHandler &);
  const ExternalEntity &externalEntity() const;
  const ConstResourcePointer<Entity> &entityPointer() const;
  Boolean duplicate() const;
  // The name of the entity will be empty if this is the default entity.
private:
  Boolean duplicate_;
  // This will actually point to an external entity.
  ConstResourcePointer<Entity> entity_;
};

class NotationDeclEvent : public Event {
public:
  NotationDeclEvent(const ConstResourcePointer<Notation> &);
  void handle(EventHandler &);
  const Notation &notation() const;
  const ConstResourcePointer<Notation> &notationPointer() const;
private:
  NotationDeclEvent(const NotationDeclEvent &);	// undefined
  void operator=(const NotationDeclEvent &);	// undefined
  ConstResourcePointer<Notation> notation_;
};

class EventHandler {
public:
  virtual void message(MessageEvent *) = 0;
  virtual void data(DataEvent *);
  virtual void ignoredChars(IgnoredCharsEvent *);
  virtual void startElement(StartElementEvent *);
  virtual void endElement(EndElementEvent *);
  virtual void pi(PiEvent *);
  virtual void sdataEntity(SdataEntityEvent *);
  virtual void externalDataEntity(ExternalDataEntityEvent *);
  virtual void subdocEntity(SubdocEntityEvent *);
  virtual void reOrigin(ReOriginEvent *);
  virtual void appinfo(AppinfoEvent *);
  virtual void simpleLink(SimpleLinkEvent *);
  virtual void complexLink(ComplexLinkEvent *);
  virtual void uselink(UselinkEvent *);
  virtual void startDtd(StartDtdEvent *);
  virtual void endDtd(EndDtdEvent *);
  virtual void endProlog(EndPrologEvent *);
  virtual void sgmlDecl(SgmlDeclEvent *);
  virtual void notationDecl(NotationDeclEvent *);
  virtual void externalEntityDecl(ExternalEntityDeclEvent *);
};

inline
Event::Event(Type type)
: type_(type)
{
}

inline
Event::Type Event::type() const
{
  return type_;
}

inline
const Location &LocatedEvent::location() const
{
  return location_;
}

inline
const Location &MessageEvent::auxLocation() const
{
  return auxLocation_;
}

inline
Boolean MessageEvent::isError() const
{
  return type_ != info && type_ != warning;
}

inline
const OpenElementInfo &MessageEvent::openElementInfo(unsigned i) const
{
  return openElements_[i];
}

inline
unsigned MessageEvent::nOpenElements() const
{
  return openElements_.length();
}

inline
const AttributeList &StartElementEvent::attributes() const
{
  return *attributes_;
}

inline
const ElementType *ElementEvent::elementType() const
{
  return elementType_;
}

inline
const CString &ElementEvent::name() const
{
  return elementType_->name();

}

inline
void ElementEvent::setIncluded()
{
  included_ = 1;
}

inline
Index ElementEvent::nextIndex() const
{
  return nextIndex_;
}

inline
Boolean ElementEvent::included() const
{
  return included_;
}

inline
ElementEvent::Minimization ElementEvent::minimization() const
{
  return minimization_;
}

inline
Boolean StartElementEvent::netEnabling() const
{
  return minimization() == net;
}

inline
const Char *DataEvent::data() const
{
  return p_;
}

inline
size_t DataEvent::dataLength() const
{
  return length_;
}

inline
const Char *PiEvent::data() const
{
  return data_;
}

inline
size_t PiEvent::dataLength() const
{
  return dataLength_;
}

inline
const ResourcePointer<EntityOrigin> &ExternalEntityEvent::entityOrigin() const
{
  return origin_;
}

inline
const ExternalDataEntity *ExternalDataEntityEvent::entity() const
{
  return dataEntity_;
}

inline
const SubdocEntity *SubdocEntityEvent::entity() const
{
  return subdocEntity_;
}

inline
const AttributeList &SimpleLinkEvent::attributes() const
{
  return attributes_;
}

inline
const CString &SimpleLinkEvent::name() const
{
  return name_;
}

inline
const CString &ComplexLinkEvent::name() const
{
  return lpd_->name();
}

inline
const ConstResourcePointer<ComplexLpd> &ComplexLinkEvent::lpd() const
{
  return lpd_;
}

inline
const VectorBase<ConstResourcePointer<Lpd> > &
ComplexLinkEvent::activeLpds() const
{
  return activeLpds_;
}

inline
const ConstResourcePointer<Lpd> &UselinkEvent::lpd() const
{
  return lpd_;
}

inline
const LinkSet *UselinkEvent::linkSet() const
{
  return linkSet_;
}

inline
Boolean UselinkEvent::restore() const
{
  return restore_;
}

inline
const CString &StartDtdEvent::name() const
{
  return name_;
}

inline
const Dtd &EndDtdEvent::dtd() const
{
  return *dtd_;
}

inline
const Sd &SgmlDeclEvent::sd() const
{
  return *sd_;
}

inline
const ConstResourcePointer<Sd> &SgmlDeclEvent::sdPointer() const
{
  return sd_;
}

inline
const Syntax &SgmlDeclEvent::prologSyntax() const
{
  return *prologSyntax_;
}

inline
const ConstResourcePointer<Syntax> &SgmlDeclEvent::prologSyntaxPointer() const
{
  return prologSyntax_;
}

inline
const Syntax &SgmlDeclEvent::instanceSyntax() const
{
  return *instanceSyntax_;
}

inline
const ConstResourcePointer<Syntax> &SgmlDeclEvent::instanceSyntaxPointer() const
{
  return instanceSyntax_;
}

inline
const ConstResourcePointer<Entity> &ExternalEntityDeclEvent::entityPointer() const
{
  return entity_;
}

inline
const ExternalEntity &ExternalEntityDeclEvent::externalEntity() const
{
  return *(const ExternalEntity *)entity_.pointer();
}

inline
Boolean ExternalEntityDeclEvent::duplicate() const
{
  return duplicate_;
}

inline
const Notation &NotationDeclEvent::notation() const
{
  return *notation_;
}

inline
const ConstResourcePointer<Notation> &NotationDeclEvent::notationPointer() const
{
  return notation_;
}

#endif /* not Event_INCLUDED */
