// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifdef __GNUG__
#pragma implementation
#endif
#include "ExternalId.H"
#include "CharsetInfo.H"
#include "macros.H"

ExternalId::ExternalId()
: haveSystem_(0), havePublic_(0)
{
}

void ExternalId::setSystem(Text &text)
{
  text.moveTo(system_);
  haveSystem_ = 1;
}

void ExternalId::moveTo(ExternalId &to)
{
  system_.moveTo(to.system_);
  to.haveSystem_ = haveSystem_;
  to.havePublic_ = havePublic_;
  if (havePublic_)
    public_.moveTo(to.public_);
}

Boolean ExternalId::setPublic(Text &text, const CharsetInfo &charset,
			      Char space, PublicId::FormalError &error)
{
  havePublic_ = 1;
  return public_.init(text, charset, space, error);
}

PublicId::PublicId()
{
}

void PublicId::moveTo(PublicId &to)
{
  to.formal_ = formal_;
  if (formal_) {
    to.ownerType_ = ownerType_;
    to.textClass_ = textClass_;
    to.unavailable_ = unavailable_;
    to.haveDisplayVersion_ = haveDisplayVersion_;
  }
  owner_.moveTo(to.owner_);
  description_.moveTo(to.description_);
  languageOrDesignatingSequence_.moveTo(to.languageOrDesignatingSequence_);
  displayVersion_.moveTo(to.displayVersion_);
  text_.moveTo(to.text_);
}

Boolean PublicId::init(Text &text, const CharsetInfo &charset,
		       Char space, FormalError &error)
{
  text.moveTo(text_);
  const CString &str(text_.string());
  formal_ = 0;
  const Char *next = str.pointer();
  const Char *lim = str.pointer() + str.length();
  Char solidus = charset.execToDesc('/');
  Char minus = charset.execToDesc('-');
  Char plus = charset.execToDesc('+');
  const Char *fieldStart;
  size_t fieldLength;
  if (!nextField(solidus, next, lim, fieldStart, fieldLength)) {
    error = missingField;
    return 0;
  }
  if (fieldLength == 1 && (*fieldStart == minus || *fieldStart == plus)) {
    ownerType_ = (*fieldStart == plus ? registered : unregistered);
    if (!nextField(solidus, next, lim, fieldStart, fieldLength)) {
      error = missingField;
      return 0;
    }
  }
  else
    ownerType_ = ISO;
  owner_.set(fieldStart, fieldLength);
  if (!nextField(solidus, next, lim, fieldStart, fieldLength)) {
    error = missingField;
    return 0;
  }
  int i;
  for (i = 0; i < fieldLength; i++)
    if (fieldStart[i] == space)
      break;
  if (i >= fieldLength) {
    error = missingTextClassSpace;
    return 0;
  }
  CString textClassString(fieldStart, i);
  if (!lookupTextClass(textClassString, charset, textClass_)) {
    error = invalidTextClass;
    return 0;
  }
  i++;				// skip the space
  fieldStart += i;
  fieldLength -= i;
  if (fieldLength  == 1 && *fieldStart == minus) {
    unavailable_ = 1;
    if (!nextField(solidus, next, lim, fieldStart, fieldLength)) {
      error = missingField;
      return 0;
    }
  }
  else
    unavailable_ = 0;
  description_.set(fieldStart, fieldLength);
  if (!nextField(solidus, next, lim, fieldStart, fieldLength)) {
    error = missingField;
    return 0;
  }
  if (textClass_ != CHARSET) {
    for (i = 0; i < fieldLength; i++) {
      UnivChar c;
      if (!charset.descToUniv(fieldStart[i], c)
	  || c < UnivCharsetDesc::A || c >= UnivCharsetDesc::A + 26) {
	error = invalidLanguage;
	return 0;
      }
    }
    // The public text language must be a name.
    // Names cannot be empty.
    if (fieldLength == 0) {
      error = invalidLanguage;
      return 0;
    }
  }
  languageOrDesignatingSequence_.set(fieldStart, fieldLength);
  if (nextField(solidus, next, lim, fieldStart, fieldLength)) {
    switch (textClass_) {
    case CAPACITY:
    case CHARSET:
    case NOTATION:
    case SYNTAX:
      error = illegalDisplayVersion;
      return 0;
    default:
      break;
    }
    haveDisplayVersion_ = 1;
    displayVersion_.set(fieldStart, fieldLength);
  }
  else
    haveDisplayVersion_ = 0;
  if (next != 0) {
    error = extraField;
    return 0;
  }
  formal_ = 1;
  return 1;
}

Boolean PublicId::nextField(Char solidus,
				  const Char *&next,
				  const Char *lim,
				  const Char *&fieldStart,
				  size_t &fieldLength)

{
  if (next == 0)
    return 0;
  fieldStart = next;
  for (; next < lim; next++) {
    if (next[0] == solidus && next + 1 < lim && next[1] == solidus) {
      fieldLength = next - fieldStart;
      next += 2;
      return 1;
    }
  }
  fieldLength = lim - fieldStart;
  next = 0;
  return 1;
}

const char *const PublicId::textClasses[] = {
  "CAPACITY",
  "CHARSET",
  "DOCUMENT",
  "DTD",
  "ELEMENTS",
  "ENTITIES",
  "LPD",
  "NONSGML",
  "NOTATION",
  "SHORTREF",
  "SUBDOC",
  "SYNTAX",
  "TEXT",
};

Boolean PublicId::lookupTextClass(const CString &str,
					const CharsetInfo &charset,
					TextClass &textClass)
{
  for (size_t i = 0; i < SIZEOF(textClasses); i++)
    if (str == charset.execToDesc(textClasses[i])) {
      textClass = TextClass(i);
      return 1;
    }
  return 0;
}

Boolean PublicId::getOwnerType(OwnerType &result) const
{
  if (!formal_)
    return 0;
  result = ownerType_;
  return 1;
}

Boolean PublicId::getOwner(CString &result) const
{
  if (!formal_)
    return 0;
  result = owner_;
  return 1;
}

Boolean PublicId::getTextClass(TextClass &result) const
{
  if (!formal_)
    return 0;
  result = textClass_;
  return 1;
}

Boolean PublicId::getUnavailable(Boolean &result) const
{
  if (!formal_)
    return 0;
  result = unavailable_;
  return 1;
}

Boolean PublicId::getDescription(CString &result) const
{
  if (!formal_)
    return 0;
  result = description_;
  return 1;
}

Boolean PublicId::getLanguage(CString &result) const
{
  if (!formal_ || textClass_ == CHARSET)
    return 0;
  result = languageOrDesignatingSequence_;
  return 1;
}

Boolean PublicId::getDesignatingSequence(CString &result) const
{
  if (!formal_ || textClass_ != CHARSET)
    return 0;
  result = languageOrDesignatingSequence_;
  return 1;
}

Boolean PublicId::getDisplayVersion(CString &result) const
{
  if (!formal_)
    return 0;
  if (haveDisplayVersion_)
    result = displayVersion_;
  return 1;
}
