// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifndef Lpd_INCLUDED
#define Lpd_INCLUDED 1
#ifdef __GNUG__
#pragma interface
#endif

#include "Attribute.H"
#include "CString.H"
#include "ResourcePointer.H"
#include "Resource.H"
#include "Boolean.H"
#include "Named.H"
#include "Dtd.H"
#include "NamedTable.H"
#include "Syntax.H"
#include "Location.H"

class ElementType;

struct ResultElementSpec {
  ResultElementSpec();
  const ElementType *elementType;
  AttributeList attributeList;
  void moveTo(ResultElementSpec &);
  void destruct() { this->ResultElementSpec::~ResultElementSpec(); }
};

class Lpd : public EntitySet, public Named {
public:
  enum Type { simple, implicit, explicit };
  Lpd(const CString &, Type, const Location &);
  Type type() const;
  const Location &location() const;
  Boolean active() const;
  void activate();
private:
  Lpd(const Lpd &);		// undefined
  void operator=(const Lpd &);	// undefined
  Type type_;
  Location location_;
  Boolean active_;
};

class SimpleLpd : public Lpd, public Attributed {
public:
  SimpleLpd(const CString &, const Location &);
private:
  SimpleLpd(const SimpleLpd &);	// undefined
  void operator=(const SimpleLpd &); // undefined
};

class LinkSet;

// A link rule whose source element specification is not implied.

class SourceLinkRule {
public:
  SourceLinkRule();
  void setLinkAttributes(AttributeList &);
  void setResult(const ElementType *, AttributeList &);
  void setUselink(const LinkSet *);
  void setPostlink(const LinkSet *);
  void setPostlinkRestore();
  void moveTo(SourceLinkRule &);
  const AttributeList &attributes() const;
  const ResultElementSpec &resultElementSpec() const;
  const LinkSet *uselink() const;
  const LinkSet *postlink() const;
  Boolean postlinkRestore() const;
private:
  SourceLinkRule(const SourceLinkRule &); // undefined
  void operator=(const SourceLinkRule &); // undefined
  const LinkSet *uselink_;
  const LinkSet *postlink_;
  Boolean postlinkRestore_;
  AttributeList linkAttributes_;
  ResultElementSpec resultElementSpec_;
};

class SourceLinkRuleResource : public Resource, public SourceLinkRule {
public:
  SourceLinkRuleResource();
private:
  SourceLinkRuleResource(const SourceLinkRuleResource &); // undefined
  void operator=(const SourceLinkRuleResource &);	  // undefined
};

class LinkSet : public Named {
public:
  LinkSet(const CString &, const Dtd *);
  void setDefined();
  Boolean defined() const;
  void addImplied(const ElementType *, AttributeList &);
  size_t nLinkRules(const ElementType *) const;
  const SourceLinkRule &linkRule(const ElementType *, size_t) const;
  void addLinkRule(const ElementType *,
		   const ConstResourcePointer<SourceLinkRuleResource> &);
  size_t nImpliedLinkRules() const;
  const ResultElementSpec &impliedLinkRule(size_t) const;
  Boolean impliedResultAttributes(const ElementType *,
				  const AttributeList *&);
private:
  LinkSet(const LinkSet &);	// undefined
  void operator=(const LinkSet &); // undefined
  Boolean defined_;
  // indexed by typeIndex of source elements
  Vector<GrowableVectorD<ConstResourcePointer<SourceLinkRuleResource> > > 
    linkRules_;
  GrowableVectorD<ResultElementSpec> impliedSourceLinkRules_;
};

class IdLinkRule : public SourceLinkRule {
public:
  IdLinkRule();
  Boolean isAssociatedWith(const ElementType *) const;
  void setAssocElementTypes(Vector<const ElementType *> &);
  void moveTo(IdLinkRule &);
  void destruct() { this->IdLinkRule::~IdLinkRule(); }
private:
  IdLinkRule(const IdLinkRule &); // undefined
  void operator=(const IdLinkRule &); // undefined
  Vector<const ElementType *> assocElementTypes_;
};

// A collection of link rules in a ID link set that are
// assocated with the same name (unique identifier).

class IdLinkRuleGroup : public Named {
public:
  IdLinkRuleGroup(const CString &);
  size_t nLinkRules() const;
  const IdLinkRule &linkRule(size_t) const;
  void addLinkRule(IdLinkRule &);
private:
  IdLinkRuleGroup(const IdLinkRuleGroup &); // undefined
  void operator=(const IdLinkRuleGroup &);  // undefined
  GrowableVectorD<IdLinkRule> linkRules_;
};

// An implicit or explicit LPD.

class ComplexLpd : public Lpd {
public:
  typedef ConstNamedTableIter<LinkSet> ConstLinkSetIter;
  ComplexLpd(const CString &, Type,
	     const Location &,
	     const Syntax &syntax,
	     const ConstResourcePointer<Dtd> &sourceDtd,
	     const ConstResourcePointer<Dtd> &resultDtd);
  size_t allocAttributeDefinitionListIndex();
  size_t nAttributeDefinitionList() const;
  LinkSet *initialLinkSet();
  const LinkSet *initialLinkSet() const;
  const LinkSet *emptyLinkSet() const;
  const LinkSet *lookupLinkSet(const CString &) const;
  const IdLinkRuleGroup *lookupIdLink(const CString &) const;
  IdLinkRuleGroup *lookupCreateIdLink(const CString &);
  void insertIdLink(IdLinkRuleGroup *);
  ConstLinkSetIter linkSetIter() const;
  Boolean hadIdLinkSet() const;
  void setHadIdLinkSet();

  LinkSet *lookupLinkSet(const CString &);
  LinkSet *insertLinkSet(LinkSet *);
  const ConstResourcePointer<Dtd> &sourceDtd() const;
  const ConstResourcePointer<Dtd> &resultDtd() const;
  const ConstResourcePointer<AttributeDefinitionList>
    attributeDef(const ElementType *) const;
  void setAttributeDef(const ElementType *,
		       const ConstResourcePointer<AttributeDefinitionList> &);
private:
  ComplexLpd(const ComplexLpd &); // undefined
  void operator=(const ComplexLpd &); // undefined
  ConstResourcePointer<Dtd> sourceDtd_;
  ConstResourcePointer<Dtd> resultDtd_;
  Vector<ConstResourcePointer<AttributeDefinitionList> > linkAttributeDefs_;
  NamedTable<LinkSet> linkSetTable_;
  LinkSet initialLinkSet_;
  LinkSet emptyLinkSet_;
  Boolean hadIdLinkSet_;
  NamedTable<IdLinkRuleGroup> idLinkTable_;
  size_t nAttributeDefinitionList_;
};

inline
Lpd::Type Lpd::type() const
{
  return type_;
}

inline
const Location &Lpd::location() const
{
  return location_;
}

inline
Boolean Lpd::active() const
{
  return active_;
}

inline
void Lpd::activate()
{
  active_ = 1;
}

inline
void SourceLinkRule::setLinkAttributes(AttributeList &attributes)
{
  attributes.moveTo(linkAttributes_);
}

inline
const AttributeList &SourceLinkRule::attributes() const
{
  return linkAttributes_;
}

inline
void SourceLinkRule::setResult(const ElementType *element,
			       AttributeList &attributes)
{
  resultElementSpec_.elementType = element;
  attributes.moveTo(resultElementSpec_.attributeList);
}

inline
const ResultElementSpec &SourceLinkRule::resultElementSpec() const
{
  return resultElementSpec_;
}

inline
void SourceLinkRule::setUselink(const LinkSet *linkSet)
{
  uselink_ = linkSet;
}

inline
void SourceLinkRule::setPostlink(const LinkSet *linkSet)
{
  postlink_ = linkSet;
}

inline
void SourceLinkRule::setPostlinkRestore()
{
  postlinkRestore_ = 1;
}

inline
const LinkSet *SourceLinkRule::uselink() const
{
  return uselink_;
}

inline
const LinkSet *SourceLinkRule::postlink() const
{
  return postlink_;
}

inline
Boolean SourceLinkRule::postlinkRestore() const
{
  return postlinkRestore_;
}

inline
Boolean LinkSet::defined() const
{
  return defined_;
}

inline
void LinkSet::setDefined()
{
  defined_ = 1;
}

inline
const SourceLinkRule &LinkSet::linkRule(const ElementType *e, size_t i) const
{
  return *linkRules_[e->index()][i];
}

inline
size_t LinkSet::nImpliedLinkRules() const
{
  return impliedSourceLinkRules_.length();
}

inline
const ResultElementSpec &LinkSet::impliedLinkRule(size_t i) const
{
  return impliedSourceLinkRules_[i];
}

inline
const ConstResourcePointer<Dtd> &ComplexLpd::sourceDtd() const
{
  return sourceDtd_;
}

inline
const ConstResourcePointer<Dtd> &ComplexLpd::resultDtd() const
{
  return resultDtd_;
}

inline
LinkSet *ComplexLpd::initialLinkSet()
{
  return &initialLinkSet_;
}

inline
const LinkSet *ComplexLpd::initialLinkSet() const
{
  return &initialLinkSet_;
}

inline
const LinkSet *ComplexLpd::emptyLinkSet() const
{
  return &emptyLinkSet_;
}

inline
const LinkSet *ComplexLpd::lookupLinkSet(const CString &name) const
{
  return linkSetTable_.lookup(name);
}

inline
LinkSet *ComplexLpd::lookupLinkSet(const CString &name)
{
  return linkSetTable_.lookup(name);
}

inline
LinkSet *ComplexLpd::insertLinkSet(LinkSet *e)
{
  return linkSetTable_.insert(e);
}

inline
size_t ComplexLpd::nAttributeDefinitionList() const
{
  return nAttributeDefinitionList_;
}

inline
size_t ComplexLpd::allocAttributeDefinitionListIndex()
{
  return nAttributeDefinitionList_++;
}

inline
ComplexLpd::ConstLinkSetIter ComplexLpd::linkSetIter() const
{
  return ConstLinkSetIter(linkSetTable_);
}

inline
const ConstResourcePointer<AttributeDefinitionList>
ComplexLpd::attributeDef(const ElementType *e) const
{
  return linkAttributeDefs_[e->index()];
}

inline
void ComplexLpd::setAttributeDef(const ElementType *e,
				 const ConstResourcePointer<AttributeDefinitionList> &attdef)
{
  linkAttributeDefs_[e->index()] = attdef;
}

inline
Boolean ComplexLpd::hadIdLinkSet() const
{
  return hadIdLinkSet_;
}

inline
void ComplexLpd::setHadIdLinkSet()
{
  hadIdLinkSet_ = 1;
}

inline
const IdLinkRuleGroup *ComplexLpd::lookupIdLink(const CString &id) const
{
  return idLinkTable_.lookup(id);
}

inline
size_t IdLinkRuleGroup::nLinkRules() const
{
  return linkRules_.length();
}

inline
const IdLinkRule &IdLinkRuleGroup::linkRule(size_t i) const
{
  return linkRules_[i];
}

#endif /* not Lpd_INCLUDED */
