// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifdef __GNUG__
#pragma implementation
#endif
#include "ParserState.H"
#include "InternalInputSource.H"
#include "MessageArg.H"
#include "macros.H"
#include "SgmlParser.H"
#include "IListIter.H"
#include "setAll.H"

#include "messageType.H"

const Location ParserState::nullLocation_;
const ShortReferenceMap ParserState::theEmptyMap = CString();

static const size_t eventSizes[] = {
  sizeof(MessageEvent),
  sizeof(DataEvent),
  sizeof(IgnoredCharsEvent),
  sizeof(StartElementEvent),
  sizeof(EndElementEvent),
  sizeof(PiEvent),
  sizeof(SdataEntityEvent),
  sizeof(ExternalDataEntityEvent),
  sizeof(SubdocEntityEvent),
  sizeof(ReOriginEvent),
  sizeof(AppinfoEvent),
  sizeof(SimpleLinkEvent),
  sizeof(ComplexLinkEvent),
  sizeof(UselinkEvent),
  sizeof(StartDtdEvent),
  sizeof(EndDtdEvent),
  sizeof(EndPrologEvent),
  sizeof(SgmlDeclEvent),
  sizeof(NotationDeclEvent),
  sizeof(ExternalEntityDeclEvent)
};

static
size_t maxSize(const size_t *v, size_t n)
{
  size_t max = 0;
  for (size_t i = 0; i < n; i++) {
    if (v[i] > max)
      max = v[i];
  }
  return max;
}

ParserState::ParserState(EntityManager *entityManager,
			 const ParserOptions &opt)
: entityManager_(entityManager),
  options_(opt),
  inInstance_(0),
  keepingMessages_(0),
  eventAllocator_(maxSize(eventSizes, SIZEOF(eventSizes)), 50),
  // FIXME compute maximum size of everything allocated
  // with internalAllocator
  internalAllocator_(sizeof(InternalInputSource), 50),
  handler_(&eventQueue_),
  subdocLevel_(0),
  documentElementContainer_(CString(), size_t(-1))
{
  init();
}

ParserState::ParserState(const SubdocFlag &, const ParserState &parent)
: entityManager_(parent.entityManager_),
  options_(parent.options_),
  inInstance_(0),
  keepingMessages_(0),
  eventAllocator_(maxSize(eventSizes, SIZEOF(eventSizes)), 50),
  // FIXME compute maximum size of everything allocated
  // with internalAllocator
  internalAllocator_(sizeof(InternalInputSource), 50),
  handler_(&eventQueue_),
  subdocLevel_(parent.subdocLevel_ + 1),
  documentElementContainer_(CString(), size_t(-1)),
  prologSyntax_(parent.prologSyntax_),
  instanceSyntax_(parent.instanceSyntax_),
  syntax_(parent.prologSyntax_),
  sd_(parent.sd_)
{
  // FIXME pass on includes_
  init();
  for (int i = 0; i < nModes; i++)
    recognizers_[i] = parent.recognizers_[i];
  if (subdocLevel() - 1 == sd().subdoc())
    message(Messages::subdocLevel, NumberMessageArg(sd().subdoc()));
}

void ParserState::init()
{
  inputLevel_ = 0;
  specialParseInputLevel_ = 0;
  markedSectionLevel_ = 0;
  markedSectionSpecialLevel_ = 0;
  currentMode_ = proMode;
  hadLpd_ = 0;
  resultAttributeSpecMode_ = 0;
  pass2_ = 0;
  activeLinkTypesSubsted_ = 0;
  allowPass2_ = 0;
  hadPass2Start_ = 0;
  pcdataRecovering_ = 0;
}

void ParserState::allDone()
{
  doFunction_ = 0;
}

void ParserState::setPass2Start()
{
  ASSERT(inputLevel_ == 1);
  if (hadPass2Start_)
    return;
  hadPass2Start_ = 1;
  if (!pass2() && sd().link() && activeLinkTypes_.length() > 0) {
    allowPass2_ = 1;
    pass1Handler_.init(handler_);
    handler_ = &pass1Handler_;
    Location location(currentLocation());
    const EntityOrigin *p = location.origin()->asEntityOrigin();
    NamedCharRef ref;
    Index index = location.index();
    while (p->lookupIndex(index, pass2StartOffset_, ref))
      index = ref.refStartIndex();
  }
  else {
    allowPass2_ = 0;
    currentInput()->willNotRewind();
  }
}

void ParserState::allLinkTypesActivated()
{
  if (activeLinkTypes_.length() == 0 && inputLevel_ == 1)
    currentInput()->willNotRewind();
}

Boolean ParserState::maybeStartPass2()
{
  if (pass2_ || !allowPass2_)
    return 0;
  handler_ = pass1Handler_.origHandler();
  if (!hadLpd() || pass1Handler_.hadError()) {
    while (!pass1Handler_.empty())
      pass1Handler_.get()->handle(*handler_);
    InputSource *top = 0;
    for (IListIter<InputSource> iter(inputStack_);
	 !iter.done();
	 iter.next())
      top = iter.cur();
    if (top)
      top->willNotRewind();
    return 0;
  }
  pass1Handler_.clear();
  while (inputLevel_ > 1) {
    InputSource *p = inputStack_.get();
    inputLevel_--;
    delete p;
  }
  // Caller will call allDone() if inputLevel_ is 0.
  if (inputLevel_ == 0)
    return 0;
  if (!inputStack_.head()->rewind(*this)) {
    inputLevel_ = 0;
    delete inputStack_.get();
    return 0;
  }
  inputStack_.head()->willNotRewind();
  for (; pass2StartOffset_ > 0; pass2StartOffset_--)
    if (inputStack_.head()->get(inputContext()) == InputSource::eE) {
      message(Messages::pass2Ee);
      inputLevel_ = 0;
      delete inputStack_.get();
      return 0;
    }
  init();
  inputLevel_ = 1;
  inInstance_ = 0;
  defDtd_.clear();
  defLpd_.clear();
  dtd_.clear();
  applicableEntitySets_.clear();
  dsEntity_.clear();
  currentDtd_.clear();
  doFunction_ = 0;
  pass2_ = 1;
  lpd_.moveTo(pass1Lpd_);
  allLpd_.clear();
  return 1;
}

ConstResourcePointer<AttributeValue> ParserState::makeImpliedAttributeValue()
{
  if (impliedAttributeValue_.isNull())
    impliedAttributeValue_ = new ImpliedAttributeValue;
  return impliedAttributeValue_;
}

Boolean ParserState::referenceDsEntity(const Location &loc)
{
  if (dsEntity_.isNull())
    return 0;
  ResourcePointer<EntityOrigin> origin
    = new (internalAllocator()) EntityOrigin(dsEntity_, loc, 0);
  dsEntity_->dsReference(*this, origin);
  dsEntity_.clear();
  return inputLevel() > 1;
}

void ParserState::startDtd(const CString &name)
{
  defDtd_ = new Dtd(name, instanceSyntax().nDelimShortref());
  // Assume all active LPDs have base DTD as source DTD.
  if (pass2() && dtd_.length() == 0) {
    firstApplicablePass1Lpd_ = 0;
    limApplicablePass1Lpd_ = pass1Lpd_.length();
    applicableEntitySets_.init(pass1Lpd_.length() + 1);
    for (size_t i = 0; i < pass1Lpd_.length(); i++)
      applicableEntitySets_[i] = pass1Lpd_[i].pointer();
    applicableEntitySets_[pass1Lpd_.length()] = defDtd_.pointer();
  }
  else {
    firstApplicablePass1Lpd_ = 0;
    limApplicablePass1Lpd_ = 0;
    applicableEntitySets_.init(1);
    applicableEntitySets_[0] = defDtd_.pointer();
  }
  for (size_t i = 0; i < includes_.length(); i++) {
    CString name = includes_[i];
    const SubstTable<Char> *subst = syntax().entitySubstTable();
    for (size_t j = 0; j < name.length(); j++)
      subst->subst(name[j]);
    Text text;
    text.addChars(syntax().reservedName(Syntax::rINCLUDE), Location());
    defDtd_->insertEntity(new InternalTextEntity(name,
						 Entity::parameterEntity,
						 Location(),
						 text,
						 InternalTextEntity::none));
  }
  currentDtd_ = defDtd_;
  currentMode_ = dsMode;
}

void ParserState::addInclude(const CString &str)
{
  includes_.grow() = str;
}

void ParserState::endDtd()
{
  dtd_.grow() = defDtd_;
  defDtd_.clear();
  currentDtd_.clear();
  applicableEntitySets_.clear();
  currentMode_ = proMode;
}

void ParserState::startLpd(ResourcePointer<Lpd> &lpd)
{
  defLpd_ = lpd;
  // What should we on pass 2 if the lpd is not active?
  if (pass2()
      && pass1Lpd_.length() > lpd_.length()
      && pass1Lpd_[lpd_.length()]->name() == lpd->name()) {
    // This LPD is active.
    applicableEntitySets_.init(pass1Lpd_.length() + 1);
    size_t i;
    for (i = 0; i < lpd_.length(); i++)
      applicableEntitySets_[i] = lpd_[i].pointer();
    applicableEntitySets_[i++] = defLpd_.pointer();
    firstApplicablePass1Lpd_ = i;
    for (; i < pass1Lpd_.length(); i++)
      applicableEntitySets_[i] = pass1Lpd_[i].pointer();
    limApplicablePass1Lpd_ = i;
  }
  else {
    firstApplicablePass1Lpd_ = 0;
    limApplicablePass1Lpd_ = 0;
    // if it's not active perhaps the DTD should come before the inactive LPD
    applicableEntitySets_.init(lpd_.length() + 2);
    for (size_t i = 0; i < lpd_.length(); i++)
      applicableEntitySets_[i] = lpd_[i].pointer();
    applicableEntitySets_[lpd_.length()] = defLpd_.pointer();
  }
  const EntitySet *&last
    = applicableEntitySets_[applicableEntitySets_.length() - 1];
  if (lpd->type() == Lpd::simple)
    last = dtd_.length() > 0 ? dtd_[0].pointer() : 0;
  else
    last = ((ComplexLpd &)*lpd).sourceDtd().pointer();
  currentMode_ = dsMode;
}

void ParserState::endLpd()
{
  applicableEntitySets_.clear();
  hadLpd_ = 1;
  if (defLpd_->active())
    lpd_.grow() = defLpd_;
  allLpd_.grow() = defLpd_;
  defLpd_.clear();
  currentMode_ = proMode;
}

void ParserState::popInputStack()
{
  ASSERT(inputLevel_ > 0);
  InputSource *p = inputStack_.get();
  inputLevel_--;
  delete p;
  if (specialParseInputLevel_ > 0 && inputLevel_ == specialParseInputLevel_)
    currentMode_ = specialParseMode_;
  if (currentMode_ == dsiMode
      && inputLevel_ == 1
      && markedSectionLevel_ == 0)
    currentMode_ = dsMode;
}

void ParserState::setSyntax(Syntax *syntax)
{
  syntax_ = syntax;
  prologSyntax_ = syntax;
  instanceSyntax_ = syntax;
}

void ParserState::setSyntaxes(Syntax *prologSyntax, Syntax *instanceSyntax)
{
  syntax_ = prologSyntax;
  prologSyntax_ = prologSyntax;
  instanceSyntax_ = instanceSyntax;
}

void ParserState::pushInput(InputSource *in)
{
  if (!in)
    return;
  if (!syntax_.isNull() && syntax_->multicode())
    in->setMarkupScanTable(syntax_->markupScanTable());
  inputStack_.insert(in);
  inputLevel_++;
  if (specialParseInputLevel_ > 0 && inputLevel_ > specialParseInputLevel_)
    currentMode_ = aliteMode;	// mode for rcdata in an entity
  else if (currentMode_ == dsMode)
    currentMode_ = dsiMode;
}

void ParserState::startMarkedSection()
{
  markedSectionLevel_++;
  if (currentMode_ == dsMode)
    currentMode_ = dsiMode;
  if (markedSectionSpecialLevel_)
    markedSectionSpecialLevel_++;
}

void ParserState::startSpecialMarkedSection(Mode mode)
{
  markedSectionLevel_++;
  specialParseInputLevel_ = inputLevel_;
  markedSectionSpecialLevel_ = 1;
  specialParseMode_ = currentMode_ = mode;
}

void ParserState::endMarkedSection()
{
  ASSERT(markedSectionLevel_ > 0);
  markedSectionLevel_--;
  if (markedSectionSpecialLevel_ > 0) {
    markedSectionSpecialLevel_--;
    if (markedSectionSpecialLevel_ > 0)
      return;			// remain in imsMode
    specialParseInputLevel_ = 0;
    if (inInstance_)
      currentMode_ = openElements_.head()->mode(netEnablingCount_ > 0);
    else
      currentMode_ = dsiMode;
  }
  if (currentMode_ == dsiMode
      && inputLevel_ == 1
      && markedSectionLevel_ == 0)
    currentMode_ = dsMode;
}

void ParserState::pushElement(OpenElement *e)
{
  tagLevel_++;
  openElementCount_[e->type()->index()]++;
  const ElementDefinition *def = e->type()->definition();
  if (def) {
    size_t i;
    for (i = 0; i < def->nInclusions(); i++)
      includeCount_[def->inclusion(i)->index()]++;
    for (i = 0; i < def->nExclusions(); i++) {
      excludeCount_[def->exclusion(i)->index()]++;
      totalExcludeCount_++;
    }
  }
  if (e->netEnabling())
    netEnablingCount_++;
  // the start tag of this element may have been implied by data
  // inside a cdata or rcdata marked section
  if (markedSectionSpecialLevel_ == 0) {
    currentMode_ = e->mode(netEnablingCount_ > 0);
    if (e->requiresSpecialParse()) {
      specialParseMode_ = currentMode_;
      specialParseInputLevel_ = inputLevel_;
    }
  }
  openElements_.insert(e);
  pcdataRecovering_ = 0;
}

// PCDATA was encountered somewhere where it was not allowed.
// Change the current mode to improve recovery.

void ParserState::pcdataRecover()
{
  switch (currentMode_) {
  case econMode:
    currentMode_ = mconMode;
    break;
  case econnetMode:
    currentMode_ = mconnetMode;
    break;
  default:
    break;
  }
  pcdataRecovering_ = 1;
}

OpenElement *ParserState::popSaveElement()
{
  ASSERT(tagLevel_ > 0);
  OpenElement *e = openElements_.get();
  tagLevel_--;
  openElementCount_[e->type()->index()]--;
  const ElementDefinition *def = e->type()->definition();
  if (def) {
    size_t i;
    for (i = 0; i < def->nInclusions(); i++)
      includeCount_[def->inclusion(i)->index()]--;
    for (i = 0; i < def->nExclusions(); i++) {
      excludeCount_[def->exclusion(i)->index()]--;
      totalExcludeCount_--;
    }
  }
  if (e->netEnabling())
    netEnablingCount_--;
  // the end tag of this element may have been implied by data
  // inside a cdata or rcdata marked section
  if (markedSectionSpecialLevel_ == 0) {
    currentMode_ = currentElement().mode(netEnablingCount_ > 0);
    specialParseInputLevel_ = 0;
  }
  lastEndedElementType_ = e->type();
  pcdataRecovering_ = 0;
  return e;
}

void ParserState::popElement()
{
  delete popSaveElement();
}
			      
Boolean ParserState::entityIsOpen(const Entity *entity) const
{
  for (IListIter<InputSource> iter(inputStack_);
       !iter.done();
       iter.next())
    if (iter.cur()->entity() == entity)
      return 1;
  return 0;
}

void ParserState::startInstance()
{
  if (!instanceSyntax_.isNull())
    syntax_ = instanceSyntax_;
  currentMode_ = econMode;
  currentDtd_ = dtd_[0];
  applicableEntitySets_.init(lpd_.length() + 1);
  for (size_t i = 0; i < lpd_.length(); i++)
    applicableEntitySets_[i] = lpd_[i].pointer();
  applicableEntitySets_[lpd_.length()] = dtd_[0].pointer();
  Vector<Owner<ContentToken> > tokens(1);
  tokens[0] = new ElementToken(currentDtd_->documentElementType(),
			       ContentToken::none);
  Owner<ModelGroup> model = new SeqModelGroup(tokens, ContentToken::none);
  Owner<CompiledModelGroup> compiledModel = new CompiledModelGroup(model);
  GrowableVector<ContentModelAmbiguity> ambiguities;
  compiledModel->compile(currentDtd().nElementTypeIndex(), ambiguities);
  ASSERT(ambiguities.length() == 0);
  ConstResourcePointer<ElementDefinition> def
    = new ElementDefinition(Location(),
			    0,
			    0,
			    ElementDefinition::modelGroup,
			    compiledModel);
  documentElementContainer_.setElementDefinition(def, 0);
  tagLevel_ = 0;
  while (!openElements_.empty())
    delete openElements_.get();
  openElements_.insert(new (internalAllocator_)
		         OpenElement(&documentElementContainer_,
				     0,
				     0,
				     &theEmptyMap,
				     Location()));
  inInstance_ = 1;
  if (sd().rank()) {
    CString empty;
    currentRank_.init(currentDtd().nRankStem());
    setAll(currentRank_.pointer(), currentRank_.length(), empty);
  }
  includeCount_.setLength(currentDtd().nElementTypeIndex());
  setAll(includeCount_.pointer(), includeCount_.length(), unsigned(0));
  excludeCount_.setLength(currentDtd().nElementTypeIndex());
  setAll(excludeCount_.pointer(), excludeCount_.length(), unsigned(0));
  openElementCount_.setLength(currentDtd().nElementTypeIndex());
  setAll(openElementCount_.pointer(), openElementCount_.length(),
	 unsigned(0));
  netEnablingCount_ = 0;
  totalExcludeCount_ = 0;
  lastEndedElementType_ = 0;
  undefinedElementTypeTable_.clear();
  currentAttributes_.init(currentDtd().nCurrentAttribute());
  idTable_.clear();
}

const ElementType *
ParserState::lookupCreateUndefinedElement(const CString &name)
{
  const ElementType *e = undefinedElementTypeTable_.lookup(name);
  if (e)
    return e;
  ElementType *p = new ElementType(name,
				   undefinedElementTypeTable_.count());
  p->setElementDefinition(new ElementDefinition(currentLocation(),
						ElementDefinition::undefinedIndex,
						0,
						ElementDefinition::any),
			  0);
  undefinedElementTypeTable_.insert(p);
  includeCount_.grow() = 0;
  excludeCount_.grow() = 0;
  openElementCount_.grow() = 0;
  return p;
}

#ifdef __GNUG__
typedef IListIter<OpenElement> Dummy_IListIter_OpenElement;
#endif

void ParserState::getOpenElementInfo(Vector<OpenElementInfo> &v) const
{
  if (!inInstance())
    return;
  v.init(tagLevel_);
  unsigned i = tagLevel_;
  for (IListIter<OpenElement> iter(openElements_);
       !iter.done() && i > 0;
       iter.next()) {
    OpenElementInfo &e = v[--i];
    e.gi = iter.cur()->type()->name();
    const LeafContentToken *token = iter.cur()->currentPosition();
    if (token && !token->isInitial()) {
      e.matchIndex = token->typeIndex() + 1;
      const ElementType *type = token->elementType();
      if (type)
	e.matchType = type->name();
      else {
	e.matchType = syntax().delimGeneral(Syntax::dRNI);
	e.matchType += syntax().reservedName(Syntax::rPCDATA);
      }
    }
    e.included = iter.cur()->included();
  }
}

Id *ParserState::lookupCreateId(const CString &name)
{
  Id *id = idTable_.lookup(name);
  if (!id) {
    id = new Id(name);
    idTable_.insert(id);
  }
  return id;
}

ConstResourcePointer<Entity>
ParserState::lookupEntity(Boolean isParameter,
			  const CString &name,
			  Boolean referenced,
			  Boolean &defaulted)
{
  ConstResourcePointer<Entity> entity;
  defaulted = 0;
  const EntitySet *resultEntitySet;
  const EntitySet **entitySets;
  unsigned short n;
  if (resultAttributeSpecMode_) {
    resultEntitySet = defComplexLpd().resultDtd().pointer();
    entitySets = &resultEntitySet;
    n = 1;
  }
  else {
    entitySets = applicableEntitySets_.pointer();
    n = applicableEntitySets_.length();
  }
  for (unsigned short i = 0; i < n; i++) {
    if (entitySets[i]) {
      entity = entitySets[i]->lookupEntity(isParameter, name);
      if (!entity.isNull()) {
	if (!inInstance_ && n > 1 && pass2() && referenced)
	  noteReferencedEntity(i, entity, 0);
	return entity;
      }
    }
  }
  if (!isParameter) {
    for (unsigned short i = 0; i < n; i++)
      if (entitySets[i]) {
	entity = entitySets[i]->defaultEntity();
	if (!entity.isNull()) {
	  Entity *p = entity->copy();
	  p->setName(name);
	  entity = p;
	  defaulted = 1;
	  if (!inInstance_ && n > 1 && pass2() && referenced)
	    noteReferencedEntity(i, entity, 1);
	  break;
	}
      }
  }
  return entity;
}

void ParserState::noteReferencedEntity(unsigned short entitySet,
				       const ConstResourcePointer<Entity> &entity,
				       Boolean defaulted)
{
  if (limApplicablePass1Lpd_ > firstApplicablePass1Lpd_) {
    LpdEntityRef ref;
    ref.entity = entity;
    ref.firstLpd = firstApplicablePass1Lpd_;
    ref.limLpd = limApplicablePass1Lpd_;
    ref.defaulted = defaulted;
    ref.found = (entitySet >= firstApplicablePass1Lpd_
		 && entitySet < limApplicablePass1Lpd_);
    LpdEntityRef *old = lpdEntityRefs_.lookup(ref);
    if (!old)
      lpdEntityRefs_.insert(new LpdEntityRef(ref));
  }
}

ConstResourcePointer<Entity>
ParserState::findApplicableLpdEntityRef(const LpdEntityRef &ref)
{
  ConstResourcePointer<Entity> entity;
  unsigned short lim = ref.limLpd;
  if (lim > lpd_.length())
    lim = lpd_.length();
  for (unsigned short j = ref.firstLpd; j < lim; j++)
    if (!lpd_[j].isNull()) {
      entity = lpd_[j]->lookupEntity(ref.entity->declType()
				     == Entity::parameterEntity,
				     ref.entity->name());
      if (!entity.isNull())
	return entity;
    }
  if (ref.defaulted) {
    for (unsigned short j = ref.firstLpd; j < lim; j++)
      if (!lpd_[j].isNull()) {
	entity = lpd_[j]->defaultEntity();
	if (!entity.isNull())
	  return entity;
      }
  }
  return entity;
}

// Compare entity definitions.
// e1 is the original (will not be an external non-text entity).
static
Boolean sameEntityDef(const Entity *e1, const Entity *e2)
{
  if (e1->dataType() != e2->dataType())
    return 0;
  const InternalEntity *i1 = e1->asInternalEntity();
  const InternalEntity *i2 = e2->asInternalEntity();
  if (i1) {
    if (!i2)
      return 0;
    if (i1->string() != i2->string())
      return 0;
    return 1;
  }
  else if (i2)
    return 0;
  const ExternalEntity *x1 = e1->asExternalEntity();
  const ExternalEntity *x2 = e2->asExternalEntity();
  const CString *s1 = x1->externalId().systemIdString();
  const CString *s2 = x2->externalId().systemIdString();
  if (s1) {
    if (!s2)
      return 0;
    if (*s1 != *s2)
      return 0;
  }
  else if (s2)
    return 0;
  s1 = x1->externalId().publicIdString();
  s2 = x2->externalId().publicIdString();
  if (s1) {
    if (!s2)
      return 0;
    if (*s1 != *s2)
      return 0;
  }
  else if (s2)
    return 0;
  return 1;
}

void ParserState::checkEntityStability()
{
  LpdEntityRefSetIter iter(lpdEntityRefs_);
  LpdEntityRef *ref;
  while ((ref = iter.next()) != 0) {
    ConstResourcePointer<Entity> newEntity = findApplicableLpdEntityRef(*ref);
    if (newEntity.isNull()
	? ref->found
	: !sameEntityDef(ref->entity.pointer(), newEntity.pointer()))
      message(((ref->entity->declType()
		== Entity::parameterEntity)
	       ? Messages::unstableLpdParameterEntity
	       : Messages::unstableLpdGeneralEntity),
	      StringMessageArg(ref->entity->name()));
  }
  lpdEntityRefs_.clear();
}    

Boolean ParserState::appendCurrentRank(CString &str, const RankStem *stem)
     const
{
  const CString &suffix(currentRank_[stem->index()]);
  if (suffix.length() > 0) {
    str += suffix;
    return 1;
  }
  return 0;
}

void ParserState::setCurrentRank(const RankStem *stem, const CString &suffix)
{
  currentRank_[stem->index()] = suffix;
}

void ParserState::getCurrentToken(const SubstTable<Char> *subst,
				  CString &str) const
{
  InputSource *in = currentInput();
  const Char *p = in->currentTokenStart();
  size_t count = in->currentTokenLength();
  str.init(count);
  Char *s = str.pointer();
  for (; count > 0; --count)
    *s++ = (*subst)[*p++];
}

void ParserState::queueMessage(MessageEvent *event)
{
  if (inInstance()) {
    Vector<OpenElementInfo> v;
    getOpenElementInfo(v);
    event->setOpenElementInfo(v);
  }
  if (keepingMessages_)
    keptMessages_.append(event);
  else
    handler_->message(event);
}

void ParserState::releaseKeptMessages()
{
  keepingMessages_ = 0;
  while (!keptMessages_.empty())
    handler_->message(keptMessages_.get());
}

void ParserState::discardKeptMessages()
{
  keepingMessages_ = 0;
  keptMessages_.clear();
}

void ParserState::inputContextMessage(const char *source,
				      const Location &loc,
				      unsigned flags,
				      int number,
				      const MessageArg **argv, int argc)
{
  MessageEvent::Type type
    = MessageEvent::Type(flags & InputContext::severityMask);
  Vector<Owner<MessageArg> > args(argc);
  for (int i = 0; i < argc; i++)
    args[i] = argv[i]->copy();
  Location l;
  if (loc.origin().isNull()) {
    l = currentLocation();
    if (flags & InputContext::parentLocation) {
      if (!l.origin().isNull())
	l = l.origin()->parent();
    }
  }
  else if (!(flags & InputContext::nullLocation))
    l = loc;
  queueMessage(new MessageEvent(source, type, l, number, args, Location()));
}

void ParserState::message(Messages::Type0 number)
{
  Vector<Owner<MessageArg> > args;
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				currentLocation(), number, args, Location()));
}

void ParserState::message(Messages::Type1 number, const MessageArg &arg1)
{
  Vector<Owner<MessageArg> > args(1);
  args[0] = arg1.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				currentLocation(), number, args, Location()));
}

void ParserState::message(Messages::Type2 number, const MessageArg &arg1,
			  const MessageArg &arg2)
{
  Vector<Owner<MessageArg> > args(2);
  args[0] = arg1.copy();
  args[1] = arg2.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				currentLocation(), number, args, Location()));
}

void ParserState::message(Messages::Type3 number, const MessageArg &arg1,
			  const MessageArg &arg2, const MessageArg &arg3)
{
  Vector<Owner<MessageArg> > args(3);
  args[0] = arg1.copy();
  args[1] = arg2.copy();
  args[2] = arg3.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				currentLocation(), number, args, Location()));
}

void ParserState::message(Messages::Type4 number, const MessageArg &arg1,
			  const MessageArg &arg2, const MessageArg &arg3,
			  const MessageArg &arg4)
{
  Vector<Owner<MessageArg> > args(4);
  args[0] = arg1.copy();
  args[1] = arg2.copy();
  args[2] = arg3.copy();
  args[3] = arg4.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				currentLocation(), number, args, Location()));
}

void ParserState::message(Messages::Type5 number, const MessageArg &arg1,
			  const MessageArg &arg2, const MessageArg &arg3,
			  const MessageArg &arg4, const MessageArg &arg5)
{
  Vector<Owner<MessageArg> > args(5);
  args[0] = arg1.copy();
  args[1] = arg2.copy();
  args[2] = arg3.copy();
  args[3] = arg4.copy();
  args[4] = arg5.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				currentLocation(), number, args, Location()));
}

void ParserState::message(Messages::Type6 number, const MessageArg &arg1,
			  const MessageArg &arg2, const MessageArg &arg3,
			  const MessageArg &arg4, const MessageArg &arg5,
			  const MessageArg &arg6)
{
  Vector<Owner<MessageArg> > args(6);
  args[0] = arg1.copy();
  args[1] = arg2.copy();
  args[2] = arg3.copy();
  args[3] = arg4.copy();
  args[4] = arg5.copy();
  args[5] = arg6.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				currentLocation(), number, args, Location()));
}

void ParserState::message(Messages::Type0L number, const Location &auxLoc)
{
  Vector<Owner<MessageArg> > args;
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				currentLocation(), number, args, auxLoc));
}

void ParserState::message(Messages::Type1L number, const MessageArg &arg1,
			  const Location &auxLoc)
{
  Vector<Owner<MessageArg> > args(1);
  args[0] = arg1.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				currentLocation(), number, args, auxLoc));
}

void ParserState::message(Messages::Type2L number, const MessageArg &arg1,
			  const MessageArg &arg2, const Location &auxLoc)
{
  Vector<Owner<MessageArg> > args(2);
  args[0] = arg1.copy();
  args[1] = arg2.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				currentLocation(), number, args, auxLoc));
}

void ParserState::message(Messages::Type3L number, const MessageArg &arg1,
			  const MessageArg &arg2, const MessageArg &arg3,
			  const Location &auxLoc)
{
  Vector<Owner<MessageArg> > args(3);
  args[0] = arg1.copy();
  args[1] = arg2.copy();
  args[2] = arg3.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				currentLocation(), number, args, auxLoc));
}

void ParserState::message(const Location &loc, Messages::Type0 number)
{
  Vector<Owner<MessageArg> > args;
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				loc, number, args, Location()));
}

void ParserState::message(const Location &loc, Messages::Type1 number,
			  const MessageArg &arg1)
{
  Vector<Owner<MessageArg> > args(1);
  args[0] = arg1.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				loc, number, args, Location()));
}

void ParserState::message(const Location &loc, Messages::Type2 number,
			  const MessageArg &arg1, const MessageArg &arg2)
{
  Vector<Owner<MessageArg> > args(2);
  args[0] = arg1.copy();
  args[1] = arg2.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				loc, number, args, Location()));
}

void ParserState::message(const Location &loc, Messages::Type3 number,
			  const MessageArg &arg1, const MessageArg &arg2,
			  const MessageArg &arg3)
{
  Vector<Owner<MessageArg> > args(3);
  args[0] = arg1.copy();
  args[1] = arg2.copy();
  args[2] = arg3.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				loc, number, args, Location()));
}

void ParserState::message(const Location &loc, Messages::Type0L number,
			  const Location &auxLoc)
{
  Vector<Owner<MessageArg> > args;
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				loc, number, args, auxLoc));
}

void ParserState::message(const Location &loc, Messages::Type1L number,
			  const MessageArg &arg1, const Location &auxLoc)
{
  Vector<Owner<MessageArg> > args(1);
  args[0] = arg1.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				loc, number, args, auxLoc));
}

void ParserState::message(const Location &loc, Messages::Type2L number,
			  const MessageArg &arg1, const MessageArg &arg2,
			  const Location &auxLoc)
{
  Vector<Owner<MessageArg> > args(2);
  args[0] = arg1.copy();
  args[1] = arg2.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				loc, number, args, auxLoc));
}

void ParserState::message(const Location &loc, Messages::Type3L number,
			  const MessageArg &arg1, const MessageArg &arg2,
			  const MessageArg &arg3, const Location &auxLoc)
{
  Vector<Owner<MessageArg> > args(3);
  args[0] = arg1.copy();
  args[1] = arg2.copy();
  args[2] = arg3.copy();
  queueMessage(new MessageEvent(SgmlParser::messageSource, messageType[number],
				loc, number, args, auxLoc));
}

void ParserState::message(InputContext &ic, const Location &loc,
			  Messages::Type0 number)
{
  ic.inputContextMessage(SgmlParser::messageSource, loc,
			 messageType[number], number, 0, 0);
}

void ParserState::message(InputContext &ic, const Location &loc,
			  Messages::Type1 number, const MessageArg &arg1)
{
  const MessageArg *p = &arg1;
  ic.inputContextMessage(SgmlParser::messageSource, loc,
			 messageType[number], number, &p, 1);
}

void ParserState::message(InputContext &ic, const Location &loc,
			  Messages::Type2 number,
			  const MessageArg &arg1, const MessageArg &arg2)
{
  const MessageArg *argv[2];
  argv[0] = &arg1;
  argv[1] = &arg2;
  ic.inputContextMessage(SgmlParser::messageSource, loc,
			 messageType[number], number, argv, 2);
}

Messages::Type1
ParserState::publicIdFormalErrorMessage(PublicId::FormalError formalError)
{
  Messages::Type1 message;
  switch (formalError) {
  case PublicId::missingField:
    message = Messages::fpiMissingField;
    break;
  case PublicId::missingTextClassSpace:
    message = Messages::fpiMissingTextClassSpace;
    break;
  case PublicId::invalidTextClass:
    message = Messages::fpiInvalidTextClass;
    break;
  case PublicId::invalidLanguage:
    message = Messages::fpiInvalidLanguage;
    break;
  case PublicId::illegalDisplayVersion:
    message = Messages::fpiIllegalDisplayVersion;
    break;
  case PublicId::extraField:
    message = Messages::fpiExtraField;
    break;
  default:
    CANNOT_HAPPEN();
  }
  return message;
}

AttributeList *
ParserState::allocAttributeList(const ConstResourcePointer<AttributeDefinitionList> &def,
				unsigned i)
{
  if (i < attributeLists_.length())
    attributeLists_[i]->init(def);
  else {
    attributeLists_.setLength(i + 1);
    attributeLists_[i] = new AttributeList(def);
  }
  return attributeLists_[i].pointer();
}

void ParserState::activateLinkType(const CString &name)
{
  if (!hadPass2Start_ && !pass2_)
    activeLinkTypes_.grow() = name;
  else
    message(Messages::linkActivateTooLate);
}

Boolean ParserState::shouldActivateLink(const CString &name) const
{
  if (!activeLinkTypesSubsted_) {
    // FIXME use mutable
    ParserState *state = (ParserState *)this;
    for (size_t i = 0; i < activeLinkTypes_.length(); i++)
      for (size_t j = 0; j < activeLinkTypes_[i].length(); j++)
	syntax().generalSubstTable()->subst(state->activeLinkTypes_[i][j]);
    state->activeLinkTypesSubsted_ = 1;
  }
  for (size_t i = 0; i < activeLinkTypes_.length(); i++)
    if (name == activeLinkTypes_[i])
      return 1;
  return 0;
}

ConstResourcePointer<Dtd> ParserState::lookupDtd(const CString &name)
{
  for (size_t i = 0; i < dtd_.length(); i++)
    if (dtd_[i]->name() == name)
      return dtd_[i];
  return ConstResourcePointer<Dtd>();
}

ConstResourcePointer<Lpd> ParserState::lookupLpd(const CString &name) const
{
  for (size_t i = 0; i < allLpd_.length(); i++)
    if (allLpd_[i]->name() == name)
      return allLpd_[i];
  return ConstResourcePointer<Lpd>();
}
