// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifndef ParserState_INCLUDED
#define ParserState_INCLUDED 1
#ifdef __GNUG__
#pragma interface
#endif

#include <stddef.h>
#include "Allocator.H"
#include "Attribute.H"
#include "Boolean.H"
#include "CString.H"
#include "Dtd.H"
#include "Entity.H"
#include "Event.H"
#include "EventQueue.H"
#include "GrowableVector.H"
#include "GrowableVectorD.H"
#include "Id.H"
#include "InputContext.H"
#include "InputSource.H"
#include "IList.H"
#include "IQueue.H"
#include "Location.H"
#include "Messages.H"
#include "Mode.H"
#include "OpenElement.H"
#include "OutputState.H"
#include "ParserOptions.H"
#include "ResourcePointer.H"
#include "Recognizer.H"
#include "Sd.H"
#include "Syntax.H"
#include "Vector.H"
#include "Lpd.H"
#include "LpdEntityRef.H"

class EntityManager;
class Parser;

class ParserState : private InputContext { 
public:
  struct SubdocFlag {
    SubdocFlag() { }
  };
  // entityManager will not.be deleted
  ParserState(EntityManager *entityManager, const ParserOptions &);
  ParserState(const SubdocFlag &, const ParserState &);
  void setHandler(EventHandler *);
  void unsetHandler();
  void addInclude(const CString &);
  Boolean inInstance() const;
  Boolean afterDocumentElement() const;
  Boolean hadDtd() const;
  void allDone();
  void startDtd(const CString &);
  void endDtd();
  void startInstance();
  unsigned subdocLevel() const;
  Boolean haveDefDtd() const;
  Dtd &defDtd();
  const ConstResourcePointer<Dtd> &defDtdPointer() const;
  Boolean haveCurrentDtd() const;
  const Dtd &currentDtd() const;
  const ConstResourcePointer<Dtd> &currentDtdPointer() const;
  void startLpd(ResourcePointer<Lpd> &lpd);
  void endLpd();
  Lpd &defLpd();
  size_t nActiveLink() const;
  const Lpd &activeLpd(size_t i) const;
  ComplexLpd &defComplexLpd();
  ConstResourcePointer<Dtd> lookupDtd(const CString &name);
  const Dtd &baseDtd() const;
  Boolean haveBaseDtd() const;
  void activateLinkType(const CString &);
  void allLinkTypesActivated();
  void setResultAttributeSpecMode();
  Boolean resultAttributeSpecMode() const;
  void clearResultAttributeSpecMode();
  Boolean haveApplicableDtd() const;
  Boolean hadLpd() const;
  Boolean pass2() const;
  void setPass2Start();
  Boolean maybeStartPass2();
  void checkEntityStability();
  ConstResourcePointer<Entity>
    findApplicableLpdEntityRef(const LpdEntityRef &ref);
  void noteReferencedEntity(unsigned short entitySet,
			    const ConstResourcePointer<Entity> &entity,
			    Boolean defaulted);
  const VectorBase<ConstResourcePointer<Lpd> > &activeLpds() const;
  ConstResourcePointer<Lpd> lookupLpd(const CString &name) const;
  Boolean shouldActivateLink(const CString &) const;
  Char currentChar() const;
  const Location &currentLocation() const;
  InputSource *currentInput() const;
  EntityManager *entityManager() const;
  void setSyntax(Syntax *);
  void setSyntaxes(Syntax *, Syntax *);
  void setSd(Sd *);
  const Syntax &syntax() const;
  const Syntax &instanceSyntax() const;
  const ConstResourcePointer<Syntax> &syntaxPointer() const;
  const ConstResourcePointer<Syntax> &instanceSyntaxPointer() const;
  const Sd &sd() const;
  const ConstResourcePointer<Sd> &sdPointer() const;
  typedef void (Parser::*DoFunction)();
  void setDoFunction(DoFunction func);
  DoFunction doFunction() const;
  Mode currentMode() const;
  void pcdataRecover();
  Boolean pcdataRecovering() const;
  void setRecognizer(Mode, ConstResourcePointer<Recognizer>);
  void setNormalMap(const XcharMap<PackedBoolean> &);
  const XcharMap<PackedBoolean> &normalMap() const;
  Xchar getChar();
  void skipChar();
  Token getToken(Mode mode);
  CString currentToken() const;
  void getCurrentToken(CString &) const;
  void getCurrentToken(const SubstTable<Char> *, CString &) const;
  unsigned inputLevel() const;
  unsigned specialParseInputLevel() const;
  unsigned markedSectionLevel() const;
  unsigned markedSectionSpecialLevel() const;
  Boolean entityIsOpen(const Entity *) const;
  void popInputStack();
  void pushInput(InputSource *);
  Boolean referenceDsEntity(const Location &);
  void setDsEntity(const ConstResourcePointer<Entity> &);
  Boolean eventQueueEmpty() const;
  Event *eventQueueGet();
  EventHandler &eventHandler();
  Boolean haveDoFunction() const;
  unsigned tagLevel() const;
  Boolean elementIsIncluded(const ElementType *) const;
  Boolean elementIsExcluded(const ElementType *) const;
  Boolean elementIsOpen(const ElementType *) const;
  const ElementType *lastEndedElementType() const;
  const unsigned *excludeCountVector() const;
  void pushElement(OpenElement *);
  OpenElement *popSaveElement();
  void popElement();
  OpenElement &currentElement();
  const OpenElement &currentElement() const;
  void getOpenElementInfo(Vector<OpenElementInfo> &) const;
  const ElementType *lookupCreateUndefinedElement(const CString &);
  Id *lookupCreateId(const CString &);
  ConstResourcePointer<Entity> lookupEntity(Boolean isParameter,
					    const CString &name,
					    Boolean referenced,
					    Boolean &defaulted);
  Boolean appendCurrentRank(CString &, const RankStem *) const;
  void setCurrentRank(const RankStem *, const CString &);
  ConstResourcePointer<AttributeValue> makeImpliedAttributeValue();
  ConstResourcePointer<AttributeValue> &currentAttribute(size_t);
  void startMarkedSection();
  void startSpecialMarkedSection(Mode);
  void endMarkedSection();
  void queueRe(const Location &);
  void noteMarkup();
  void noteData();
  void noteRs();
  void noteStartElement(Boolean included);
  void noteEndElement(Boolean included);
  // size of objects allocated with this must not exceed
  // sizeof(StartElementEvent)
  Allocator &eventAllocator();
  // size of objects allocated with this must not exceed
  // sizeof(OpenElement)
  Allocator &internalAllocator();
  AttributeList *allocAttributeList(const ConstResourcePointer<AttributeDefinitionList> &,
				    unsigned i);

  static void freeEvent(void *);
  Boolean esisPlus() const;
  CString &nameBuffer();
  typedef NamedTableIter<Id> IdTableIter;
  IdTableIter idTableIter();
  const ParserOptions &options() const;
  void keepMessages();
  void releaseKeptMessages();
  void discardKeptMessages();
  void message(Messages::Type0);
  void message(Messages::Type1, const MessageArg &);
  void message(Messages::Type2, const MessageArg &, const MessageArg &);
  void message(Messages::Type3, const MessageArg &, const MessageArg &,
	       const MessageArg &);
  void message(Messages::Type4, const MessageArg &, const MessageArg &,
	       const MessageArg &, const MessageArg &);
  void message(Messages::Type5, const MessageArg &, const MessageArg &,
	       const MessageArg &, const MessageArg &, const MessageArg &);
  void message(Messages::Type6, const MessageArg &, const MessageArg &,
	       const MessageArg &, const MessageArg &, const MessageArg &,
	       const MessageArg &);
  void message(Messages::Type0L, const Location &);
  void message(Messages::Type1L, const MessageArg &, const Location &);
  void message(Messages::Type2L, const MessageArg &, const MessageArg &,
	       const Location &);
  void message(Messages::Type3L, const MessageArg &, const MessageArg &,
	       const MessageArg &, const Location &);
  void message(const Location &, Messages::Type0);
  void message(const Location &, Messages::Type1, const MessageArg &);
  void message(const Location &, Messages::Type2, const MessageArg &,
	       const MessageArg &);
  void message(const Location &, Messages::Type3, const MessageArg &,
	       const MessageArg &, const MessageArg &);
  void message(const Location &, Messages::Type0L, const Location &);
  void message(const Location &, Messages::Type1L, const MessageArg &,
	       const Location &);
  void message(const Location &, Messages::Type2L, const MessageArg &,
	       const MessageArg &, const Location &);
  void message(const Location &, Messages::Type3L, const MessageArg &,
	       const MessageArg &, const MessageArg &, const Location &);
  static void message(InputContext &, const Location &, Messages::Type0);
  static void message(InputContext &, const Location &, Messages::Type1,
		      const MessageArg &);
  static void message(InputContext &, const Location &, Messages::Type2,
		      const MessageArg &, const MessageArg &);
  InputContext &inputContext();

  static Messages::Type1 publicIdFormalErrorMessage(PublicId::FormalError);
  static const ShortReferenceMap theEmptyMap;
private:
  ParserState(const ParserState &); // undefined
  void operator=(const ParserState &); // undefined
  void inputContextMessage(const char *source, const Location &,
			   unsigned flags,
			   int number, const MessageArg **args, int nArgs);
  void queueMessage(MessageEvent *);
  void init();

  ParserOptions options_;
  GrowableVectorD<CString> includes_;
  EventHandler *handler_;
  Pass1EventHandler pass1Handler_;
  Boolean allowPass2_;
  Offset pass2StartOffset_;
  Boolean hadPass2Start_;
  EventQueue eventQueue_;
  OutputState outputState_;
  ConstResourcePointer<Syntax> prologSyntax_;
  ConstResourcePointer<Syntax> instanceSyntax_;
  ConstResourcePointer<Sd> sd_;
  unsigned subdocLevel_;
  EntityManager *entityManager_;
  DoFunction doFunction_;
  Boolean inInstance_;
  ResourcePointer<Dtd> defDtd_;
  ResourcePointer<Lpd> defLpd_;
  GrowableVectorD<ConstResourcePointer<Lpd> > allLpd_;
  GrowableVectorD<ConstResourcePointer<Lpd> > lpd_; // active LPDs
  GrowableVectorD<ConstResourcePointer<Lpd> > pass1Lpd_;
  GrowableVectorD<CString> activeLinkTypes_;
  Boolean activeLinkTypesSubsted_;
  Boolean hadLpd_;
  Vector<const EntitySet *> applicableEntitySets_;
  Boolean resultAttributeSpecMode_;
  Boolean pass2_;
  unsigned short firstApplicablePass1Lpd_;
  unsigned short limApplicablePass1Lpd_;
  typedef OwnerTable<LpdEntityRef, LpdEntityRef, LpdEntityRef, LpdEntityRef>
    LpdEntityRefSet;
  typedef OwnerTableIter<LpdEntityRef, LpdEntityRef, LpdEntityRef, LpdEntityRef>
    LpdEntityRefSetIter;
  LpdEntityRefSet lpdEntityRefs_;
  // external entity to be referenced at the end of the declaration subset
  ConstResourcePointer<Entity> dsEntity_;
  ConstResourcePointer<AttributeValue> impliedAttributeValue_;
  Allocator eventAllocator_;
  Allocator internalAllocator_;
  GrowableVectorD<Owner<AttributeList> > attributeLists_;
  CString nameBuffer_;
  Boolean keepingMessages_;
  IQueue<MessageEvent> keptMessages_;
  Mode currentMode_;
  const ElementType *lastEndedElementType_;
  // if in a special parse (cdata, rcdata, ignore), the input level
  // at which the special parse started.
  unsigned specialParseInputLevel_;
  Mode specialParseMode_;
  unsigned markedSectionLevel_;
  unsigned markedSectionSpecialLevel_;
  ConstResourcePointer<Recognizer> recognizers_[nModes];
  XcharMap<PackedBoolean> normalMap_;
  unsigned inputLevel_;
  IList<InputSource> inputStack_;
  ConstResourcePointer<Dtd> currentDtd_;
  GrowableVectorD<ConstResourcePointer<Dtd> > dtd_;
  ConstResourcePointer<Syntax> syntax_;
  // These need to be growable in case undefined elements are found.
  GrowableVector<unsigned> openElementCount_;
  GrowableVector<unsigned> includeCount_;
  GrowableVector<unsigned> excludeCount_;
  unsigned totalExcludeCount_;
  unsigned tagLevel_;
  IList<OpenElement> openElements_;
  unsigned netEnablingCount_;
  Vector<CString> currentRank_;
  NamedTable<ElementType> undefinedElementTypeTable_;
  NamedTable<Id> idTable_;
  Vector<ConstResourcePointer<AttributeValue> > currentAttributes_;
  ElementType documentElementContainer_;
  Boolean pcdataRecovering_;
  static const Location nullLocation_;
  static MessageEvent::Type messageType[];
};

inline
InputContext &ParserState::inputContext()
{
  return *this;
}

inline
Boolean ParserState::esisPlus() const
{
  return options_.eventsWanted.wantEsisPlus();
}

inline
InputSource *ParserState::currentInput() const
{
  return inputStack_.head();
}

inline
const Location &ParserState::currentLocation() const
{
  InputSource *in = currentInput();
  return in ? in->currentLocation() : nullLocation_;
}

inline
unsigned ParserState::inputLevel() const
{
  return inputLevel_;
}

inline
unsigned ParserState::specialParseInputLevel() const
{
  return specialParseInputLevel_;
}

inline
unsigned ParserState::markedSectionLevel() const
{
  return markedSectionLevel_;
}

inline
unsigned ParserState::markedSectionSpecialLevel() const
{
  return markedSectionSpecialLevel_;
}

inline
Char ParserState::currentChar() const
{
  return currentInput()->currentTokenStart()[0];
}

inline
CString ParserState::currentToken() const
{
  return CString(currentInput()->currentTokenStart(),
		 currentInput()->currentTokenLength());
}

inline
void ParserState::getCurrentToken(CString &str) const
{
  InputSource *in = currentInput();
  str.set(in->currentTokenStart(), in->currentTokenLength());
}

inline
void ParserState::setSd(Sd *sd)
{
  sd_ = sd;
}

inline
void ParserState::setRecognizer(Mode mode, ConstResourcePointer<Recognizer> p)
{
  recognizers_[mode] = p;
}

inline
void ParserState::setNormalMap(const XcharMap<PackedBoolean> &map)
{
  normalMap_ = map;
}

inline
const XcharMap<PackedBoolean> &ParserState::normalMap() const
{
  return normalMap_;
}

inline
Boolean ParserState::haveDefDtd() const
{
  return !defDtd_.isNull();
}

inline
Boolean ParserState::haveCurrentDtd() const
{
  return !currentDtd_.isNull();
}

inline
Dtd &ParserState::defDtd()
{
  return *defDtd_;
}

inline
const Dtd &ParserState::currentDtd() const
{
  return *currentDtd_;
}

inline
const ConstResourcePointer<Dtd> &ParserState::defDtdPointer() const
{
  return defDtd_;
}

inline
const ConstResourcePointer<Dtd> &ParserState::currentDtdPointer() const
{
  return currentDtd_;
}

inline
Boolean ParserState::inInstance() const
{
  return inInstance_;
}

inline
const Syntax &ParserState::syntax() const
{
  return *syntax_;
}

inline
const Syntax &ParserState::instanceSyntax() const
{
  return *instanceSyntax_;
}

inline
const ConstResourcePointer<Syntax> &ParserState::syntaxPointer() const
{
  return syntax_;
}

inline
const ConstResourcePointer<Syntax> &ParserState::instanceSyntaxPointer() const
{
  return instanceSyntax_;
}

inline
const Sd &ParserState::sd() const
{
  return *sd_;
}

inline
const ConstResourcePointer<Sd> &ParserState::sdPointer() const
{
  return sd_;
}

inline
void ParserState::setDoFunction(DoFunction func)
{
  doFunction_ = func;
}

inline
Mode ParserState::currentMode() const
{
  return currentMode_;
}

inline
Xchar ParserState::getChar()
{
  return inputStack_.head()->get(inputContext());
}

inline
void ParserState::skipChar()
{
  (void)getChar();
}

inline
Token ParserState::getToken(Mode mode)
{
  return recognizers_[mode]->recognize(inputStack_.head(), inputContext());
}

inline
Boolean ParserState::hadDtd() const
{
  return dtd_.length() > 0;
}

inline
Boolean ParserState::eventQueueEmpty() const
{
  return eventQueue_.empty();
}

inline
Event *ParserState::eventQueueGet()
{
  return eventQueue_.get();
}

inline
ParserState::DoFunction ParserState::doFunction() const
{
  return doFunction_;
}

inline
Boolean ParserState::haveDoFunction() const
{
  return doFunction_ != 0;
}

inline
OpenElement &ParserState::currentElement()
{
  return *openElements_.head();
}

inline
const OpenElement &ParserState::currentElement() const
{
  return *openElements_.head();
}

inline
Boolean ParserState::elementIsOpen(const ElementType *e) const
{
  return openElementCount_[e->index()] != 0;
}

inline
Boolean ParserState::elementIsIncluded(const ElementType *e) const
{
  return includeCount_[e->index()] != 0 && excludeCount_[e->index()] == 0;
}

inline
Boolean ParserState::elementIsExcluded(const ElementType *e) const
{
  return excludeCount_[e->index()] != 0;
}

inline
const ElementType *ParserState::lastEndedElementType() const
{
  return lastEndedElementType_;
}

inline
unsigned ParserState::tagLevel() const
{
  return tagLevel_;
}

inline
Boolean ParserState::afterDocumentElement() const
{
  return tagLevel() == 0 && currentElement().isFinished();
}

inline
const unsigned *ParserState::excludeCountVector() const
{
  return excludeCount_.pointer();
}

inline
EntityManager *ParserState::entityManager() const
{
  return entityManager_;
}

inline
void ParserState::setDsEntity(const ConstResourcePointer<Entity> &entity)
{
  dsEntity_ = entity;
}

inline
ConstResourcePointer<AttributeValue> &ParserState::currentAttribute(size_t i)
{
  return currentAttributes_[i];
}

inline
Allocator &ParserState::eventAllocator()
{
  return eventAllocator_;
}

inline
Allocator &ParserState::internalAllocator()
{
  return internalAllocator_;
}

inline
CString &ParserState::nameBuffer()
{
  return nameBuffer_;
}

inline
void ParserState::setHandler(EventHandler *handler)
{
  handler_ = handler;
}

inline
void ParserState::unsetHandler()
{
  handler_ = &eventQueue_;
}

inline
void ParserState::queueRe(const Location &location)
{
  outputState_.handleRe(*handler_, eventAllocator_, options_.eventsWanted,
			syntax().standardFunction(Syntax::fRE),
			location);
}

inline
void ParserState::noteMarkup()
{
  if (inInstance_)
    outputState_.noteMarkup(*handler_, eventAllocator_, options_.eventsWanted);
}

inline
void ParserState::noteRs()
{
  outputState_.noteRs(*handler_, eventAllocator_, options_.eventsWanted);
}

inline
void ParserState::noteStartElement(Boolean included)
{
  outputState_.noteStartElement(included, *handler_, eventAllocator_,
				options_.eventsWanted);
}

inline
void ParserState::noteEndElement(Boolean included)
{
  outputState_.noteEndElement(included, *handler_, eventAllocator_,
			      options_.eventsWanted);
}

inline
void ParserState::noteData()
{
  outputState_.noteData(*handler_, eventAllocator_, options_.eventsWanted);
}

inline
unsigned ParserState::subdocLevel() const
{
  return subdocLevel_;
}

inline
EventHandler &ParserState::eventHandler()
{
  return *handler_;
}

inline
ParserState::IdTableIter ParserState::idTableIter()
{
  return IdTableIter(idTable_);
}

inline
const ParserOptions &ParserState::options() const
{
  return options_;
}

inline
void ParserState::keepMessages()
{
  keepingMessages_ = 1;
}

inline
Boolean ParserState::haveApplicableDtd() const
{
  return applicableEntitySets_.length() > 0;
}

inline
Boolean ParserState::hadLpd() const
{
  return hadLpd_;
}

inline
Boolean ParserState::pass2() const
{
  return pass2_;
}

inline
size_t ParserState::nActiveLink() const
{
  return lpd_.length();
}

inline
const Lpd &ParserState::activeLpd(size_t i) const
{
  return *lpd_[i];
}

inline
const VectorBase<ConstResourcePointer<Lpd> > &ParserState::activeLpds() const
{
  return lpd_;
}

inline
Lpd &ParserState::defLpd()
{
  return *defLpd_;
}

inline
ComplexLpd &ParserState::defComplexLpd()
{
  return (ComplexLpd &)defLpd();
}

inline
const Dtd &ParserState::baseDtd() const
{
  return *dtd_[0];
}

inline
Boolean ParserState::haveBaseDtd() const
{
  return dtd_.length() > 0 && !dtd_[0].isNull();
}

inline
void ParserState::setResultAttributeSpecMode()
{
  resultAttributeSpecMode_ = 1;
}

inline
void ParserState::clearResultAttributeSpecMode()
{
  resultAttributeSpecMode_ = 0;
}

inline
Boolean ParserState::resultAttributeSpecMode() const
{
  return resultAttributeSpecMode_;
}

inline
Boolean ParserState::pcdataRecovering() const
{
  return pcdataRecovering_;
}

#endif /* not ParserState_INCLUDED */
