// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifdef __GNUG__
#pragma implementation
#endif
#include "Syntax.H"
#include "Sd.H"
#include "UnivCharsetDesc.H"
#include "ISetIter.H"
#include "macros.H"
#include "MarkupScan.H"

const int Syntax::referenceQuantity_[] = {
  40,
  960,
  960,
  16,
  16,
  16,
  32,
  96,
  16,
  240,
  8,
  2,
  240,
  960,
  24
};

Syntax::Syntax(const Sd &sd)
: generalSubst_(0),
  entitySubst_(0),
  categoryTable_(otherCategory),
  shuncharControls_(0),
  multicode_(0),
  markupScanTable_(MarkupScan::normal)
{
  static const char lcletter[] = "abcdefghijklmnopqrstuvwxyz";
  static const char ucletter[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
  int i;
  for (i = 0; i < 26; i++) {
    Char lc = sd.execToDoc(lcletter[i]);
    Char uc = sd.execToDoc(ucletter[i]); 
    set_[nameStart] += lc;
    set_[nameStart] += uc;
    set_[minimumData] += lc;
    set_[minimumData] += uc;
    set_[significant] += lc;
    set_[significant] += uc;
    categoryTable_.setChar(lc, nameStartCategory);
    categoryTable_.setChar(uc, nameStartCategory);
    subst(lc, uc);
  }
  static const char digits[] = "0123456789";
  for (i = 0; i < 10; i++) {
    Char c = sd.execToDoc(digits[i]);
    set_[digit] += c;
    set_[minimumData] += c;
    set_[significant] += c;
    categoryTable_.setChar(c, digitCategory);
  }
  static const char special[] = "'()+,-./:=?";
  for (i = 0; i < special[i] != '\0'; i++) {
    Char c = special[i];
    set_[minimumData] += c;
    set_[significant] += c;
  }
  for (i = 0; i < nQuantity; i++)
    quantity_[i] = referenceQuantity_[i];
  for (i = 0; i < 3; i++)
    standardFunctionValid_[i] = 0;
}

void Syntax::addNmstrtPair(Char lc, Char uc)
{
  set_[nameStart] += lc;
  set_[nameStart] += uc;
  set_[significant] += lc;
  set_[significant] += uc;
  categoryTable_.setChar(lc, nameStartCategory);
  categoryTable_.setChar(uc, nameStartCategory);
  subst(lc, uc);
}

void Syntax::addNmcharPair(Char lc, Char uc)
{
  set_[nmchar] += lc;
  set_[nmchar] += uc;
  set_[significant] += lc;
  set_[significant] += uc;
  categoryTable_.setChar(lc, otherNameCategory);
  categoryTable_.setChar(uc, otherNameCategory);
  subst(lc, uc);
}

void Syntax::setStandardFunction(StandardFunction f, Char c)
{
  standardFunction_[f] = c;
  standardFunctionValid_[f] = 1;
  set_[minimumData] += c;
  set_[s] += c;
  categoryTable_.setChar(c, sCategory);
  set_[functionChar] += c;
  set_[significant] += c;
  switch (f) {
  case fSPACE:
    set_[blank] += c;
    break;
  case fRE:
  case fRS:
    break;
  }
}

void Syntax::enterStandardFunctionNames()
{
  static Syntax::ReservedName name[3] = {
    rRE, rRS, rSPACE
  };
  for (int i = 0; i < 3; i++)
    if (standardFunctionValid_[i])
      functionTable_.insert(reservedName(name[i]), standardFunction_[i]);
}

void Syntax::setDelimGeneral(int i, const CString &str)
{
  delimGeneral_[i] = str;
  for (int j = 0; j < str.length(); j++)
    set_[significant] += str[j];
}

void Syntax::addDelimShortref(const CString &str)
{
  delimShortref_.grow() = str;
  for (int i = 0; i < str.length(); i++)
    set_[significant] += str[i];
}

void Syntax::addFunctionChar(const CString &str, FunctionClass fun, Char c)
{
  switch (fun) {
  case cFUNCHAR:
    break;
  case cSEPCHAR:
    set_[s] += c;
    categoryTable_.setChar(c, sCategory);
    set_[blank] += c;
    set_[sepchar] += c;
    break;
  case cMSOCHAR:
    multicode_ = 1;
    markupScanTable_.setChar(c, MarkupScan::out);
    break;
  case cMSICHAR:
    // don't need to do anything special if we just have MSICHARs
    markupScanTable_.setChar(c, MarkupScan::in);
    break;
  case cMSSCHAR:
    multicode_ = 1;
    markupScanTable_.setChar(c, MarkupScan::suppress);
    break;
  }
  set_[functionChar] += c;
  set_[significant] += c;
  functionTable_.insert(str, c);
}

void Syntax::setName(int i, const CString &str)
{
  names_[i] = str;
  nameTable_.insert(str, i);
}

void Syntax::setNamecaseGeneral(Boolean b)
{
  namecaseGeneral_ = b;
  generalSubst_ = b ? &upperSubst_ : &identitySubst_;
}

void Syntax::setNamecaseEntity(Boolean b)
{
  namecaseEntity_ = b;
  entitySubst_ = b ? &upperSubst_ : &identitySubst_;
}

void Syntax::subst(Char from, Char to)
{
  upperSubst_.addSubst(from, to);
}

void Syntax::addShunchar(Char c)
{
  shunchar_.add(c);
}

Boolean Syntax::lookupReservedName(const CString &str,
				   ReservedName *result) const
{
  const int *tem = nameTable_.lookup(str);
  if (tem) {
    *result = ReservedName(*tem);
    return 1;
  }
  else
    return 0;
}

Boolean Syntax::lookupFunctionChar(const CString &name, Char *result) const
{
  const Char *p = functionTable_.lookup(name);
  if (p) {
    *result = *p;
    return 1;
  }
  else
    return 0;
}

Boolean Syntax::lookupShortref(const CString &str, int *index) const
{
  for (int i = 0; i < delimShortref_.length(); i++)
    if (str == delimShortref_[i]) {
      *index = i;
      return 1;
    }
  return 0;
}


void Syntax::implySgmlChar(const CharsetInfo &docCharset)
{
  docCharset.getDescSet(set_[sgmlChar]);
  ISet<WideChar> invalid;
  checkSgmlChar(docCharset, 0, invalid);
  ISetIter<WideChar> iter(invalid);
  WideChar min, max;
  while (iter.next(min, max)) {
    do {
      if (min <= Char(-1))
	set_[sgmlChar].remove(Char(min));
    } while (min++ != max);
  }
}

void Syntax::checkSgmlChar(const CharsetInfo &docCharset,
			   const Syntax *otherSyntax,
			   ISet<WideChar> &invalid) const
{
  ISetIter<Char> iter(shunchar_);
  Char min, max;
  while (iter.next(min, max)) {
    if (min <= max) {
      do {
	if (!set_[significant].contains(min)
	    && (!otherSyntax || !otherSyntax->set_[significant].contains(min))
	    && set_[sgmlChar].contains(min))
	  invalid += min;
      } while (min++ != max);
    }
  }
  if (shuncharControls_) {
    UnivChar i;
    for (i = 0; i < 32; i++)
      checkUnivControlChar(i, docCharset, otherSyntax, invalid);
    for (i = 127; i < 160; i++)
      checkUnivControlChar(i, docCharset, otherSyntax, invalid);
  }
}

void Syntax::checkUnivControlChar(UnivChar univChar,
				  const CharsetInfo &docCharset,
				  const Syntax *otherSyntax,
				  ISet<WideChar> &invalid) const
{
  WideChar c;
  ISet<WideChar> set;
  switch (docCharset.univToDesc(univChar, c, set)) {
  case 0:
    break;
  case 1:
    set += c;
    // fall through
  default:
    {
      ISetIter<WideChar> iter(set);
      WideChar min, max;
      while (iter.next(min, max)) {
	do {
	  if (min > Char(-1))
	    break;
	  Char ch = Char(min);
	  if (!set_[significant].contains(ch)
	      && (!otherSyntax
		  || !otherSyntax->set_[significant].contains(ch))
	      && set_[sgmlChar].contains(ch))
	    invalid += ch;
	} while (min++ != max);
      }
    }
  }
}

CString Syntax::rniReservedName(ReservedName i) const
{
  CString result = delimGeneral(dRNI);
  result += reservedName(i);
  return result;
}
