// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#ifndef Text_INCLUDED
#define Text_INCLUDED 1
#ifdef __GNUG__
#pragma interface
#endif

#include "types.H"
#include "CString.H"
#include "CopyGrowableVectorD.H"
#include "Location.H"
#include <stddef.h>

struct TextItem {
  TextItem();
  void destruct();
  enum Type {
    data,
    entityStart,
    entityEnd,
    sdataEntityStart,
    sdataEntityEnd,
    cdataEntityStart,
    cdataEntityEnd,
    ignore
    };
  Type type;
  // char that was ignored
  Char c;
  // location of this item
  // data - location of first char
  // (c/sdata)entityStart - location of first char of entity
  // (c/sdata)entityEnd - location of entity end in entity
  // ignore - location of ignored character
  Location loc;
  // index of character in chars_ to which this applies
  size_t index;
};

// This is used to represent literals and attribute values.

class Text {
public:
  Text();
  void clear();
  void moveTo(Text &to);
  void addChar(Char c, const Location &);
  void addChars(const CString &, const Location &);
  void addChars(const Char *, size_t, const Location &);
  void insertChars(const CString &, const Location &);
  void ignoreChar(Char, const Location &);
  void ignoreLastChar();
  void addEntityStart(const Location &);
  void addEntityEnd(const Location &);
  void addCdataEntityStart(const Location &);
  void addCdataEntityEnd(const Location &);
  void addSdataEntityStart(const Location &);
  void addSdataEntityEnd(const Location &);
  Location charLocation(size_t i) const;
  size_t length() const;
  Char lastChar() const;
  void destruct();		// for GrowableVectorD
  const CString &string() const;
  size_t nDataEntities() const;
  Boolean fixedEqual(const Text &) const;
private:
  void addSimple(TextItem::Type, const Location &);
  CString chars_;
  CopyGrowableVectorD<TextItem> items_;
  friend class EsisTextIter;
};

class EsisTextIter {
public:
  EsisTextIter(const Text &);
  enum Type {
    data,
    sdata
    };
  Boolean next(Type &, const Char *&, size_t &);
private:
  const Text *ptr_;
  size_t charIndex_;
  size_t itemIndex_;
};

inline
size_t Text::length() const
{
  return chars_.length();
}

inline
Char Text::lastChar() const
{
  return chars_[chars_.length() - 1];
}

inline
const CString &Text::string() const
{
  return chars_;
}

inline
void Text::addEntityStart(const Location &loc)
{
  addSimple(TextItem::entityStart, loc);
}

inline
void Text::addEntityEnd(const Location &loc)
{
  addSimple(TextItem::entityEnd, loc);
}

inline
void Text::addCdataEntityStart(const Location &loc)
{
  addSimple(TextItem::cdataEntityStart, loc);
}

inline
void Text::addCdataEntityEnd(const Location &loc)
{
  addSimple(TextItem::cdataEntityEnd, loc);
}

inline
void Text::addSdataEntityStart(const Location &loc)
{
  addSimple(TextItem::sdataEntityStart, loc);
}

inline
void Text::addSdataEntityEnd(const Location &loc)
{
  addSimple(TextItem::sdataEntityEnd, loc);
}

inline
void Text::addChars(const CString &s, const Location &loc)
{
  addChars(s.pointer(), s.length(), loc);
}

#endif /* not Text_INCLUDED */
