// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#include "Parser.H"
#include "MessageArg.H"
#include "token.H"
#include "macros.H"

Boolean Parser::parseAttributeSpec(Boolean inDecl,
				   AttributeList &atts,
				   Token &closeToken)

{
  unsigned specLength = 0;
  AttributeParameter parm1, parm2;
  AttributeParameter *curParm = &parm1, *nextParm = &parm2;

  if (!parseAttributeParameter(inDecl, 0, *curParm))
    return 0;
  while (curParm->type != AttributeParameter::end) {
    switch (curParm->type) {
    case AttributeParameter::name:
      {
	Text text;
	text.addChars(currentInput()->currentTokenStart(),
		      currentInput()->currentTokenLength(),
		      currentLocation());
	if (!parseAttributeParameter(inDecl, 1, *nextParm))
	  return 0;
	if (nextParm->type == AttributeParameter::vi) {
	  specLength += curParm->token.length() + syntax().normsep();
	  if (!parseAttributeValueSpec(inDecl, curParm->token, text.string(),
				       atts, specLength))
	    return 0;
	  // setup for next attribute
	  if (!parseAttributeParameter(inDecl, 0, *curParm))
	    return 0;
	}
	else {
	  handleAttributeNameToken(curParm->token, text, atts, specLength);
	  AttributeParameter *tem = curParm;
	  curParm = nextParm;
	  nextParm = tem;
	}
      }
      break;
    case AttributeParameter::nameToken:
      {
	Text text;
	text.addChars(currentInput()->currentTokenStart(),
		      currentInput()->currentTokenLength(),
		      currentLocation());

	handleAttributeNameToken(curParm->token, text, atts, specLength);
	if (!parseAttributeParameter(inDecl, 0, *curParm))
	  return 0;
      }
      break;
    default:
      CANNOT_HAPPEN();
    }
  }
  closeToken = curParm->closeToken;
  atts.finish(*this);
  if (specLength > syntax().attsplen())
    message(Messages::attsplen,
	    NumberMessageArg(syntax().attsplen()),
	    NumberMessageArg(specLength));
  return 1;
}

void Parser::handleAttributeNameToken(CString &token,
				      Text &text,
				      AttributeList &atts,
				      unsigned &specLength)
{
  unsigned index;
  if (!atts.tokenIndex(token, index))
    message(Messages::noSuchAttributeToken, StringMessageArg(token));
  else {
    if (!sd().shorttag())
      message(Messages::attributeNameShorttag);
    atts.setSpec(index, *this);
    atts.setValue(index, token, text, *this, specLength);
  }
}

Boolean Parser::parseAttributeValueSpec(Boolean inDecl,
					const CString &name,
					const CString &origName,
					AttributeList &atts,
					unsigned &specLength)
{
  unsigned index;
  Boolean valid = atts.attributeIndex(name, index);
  if (!valid)
    message(Messages::noSuchAttribute, StringMessageArg(name));
  else
    atts.setSpec(index, origName, *this);
  Mode mode = inDecl ? asMode : tagMode;
  Token token = getToken(mode);
  while (token == tokenS)
    token = getToken(mode);
  Text text;
  switch (token) {
  case tokenUnrecognized:
    if (!reportNonSgmlCharacter())
      message(Messages::attributeSpecCharacter,
	      StringMessageArg(currentToken()));
    return 0;
  case tokenEe:
    message(Messages::attributeSpecEntityEnd);
    return 0;
  case tokenEtago:
  case tokenStago:
  case tokenNet:
  case tokenTagc:
  case tokenDsc:
  case tokenVi:
    message(Messages::attributeValueExpected);
    return 0;
  case tokenNameStart:
  case tokenDigit:
  case tokenLcUcNmchar:
    if (!sd().shorttag())
      message(Messages::attributeValueShorttag);
    extendNameToken(syntax().litlen() >= syntax().normsep()
		    ? syntax().litlen() - syntax().normsep()
		    : 0,
		    Messages::attributeValueLength);
    text.addChars(currentInput()->currentTokenStart(),
		  currentInput()->currentTokenLength(),
		  currentLocation());
    break;
  case tokenLit:
  case tokenLita:
    Boolean lita;
    lita = (token == tokenLita);
    if (!((valid ? atts.tokenized(index) : 1)
	  ? parseTokenizedAttributeValueLiteral(lita, text)
	  : parseAttributeValueLiteral(lita, text)))
      return 0;
    break;
  default:
      CANNOT_HAPPEN();
  }
  if (valid)
    atts.setValue(index, text, *this, specLength);
  return 1;
}


Boolean Parser::parseAttributeParameter(Boolean inDecl,
					Boolean allowVi,
					AttributeParameter &result)
{
  Mode mode = inDecl ? asMode : tagMode;
  Token token = getToken(mode);
  while (token == tokenS)
    token = getToken(mode);
  switch (token) {
  case tokenUnrecognized:
    if (!reportNonSgmlCharacter())
      message(Messages::attributeSpecCharacter,
	      StringMessageArg(currentToken()));
    return 0;
  case tokenEe:
    message(Messages::attributeSpecEntityEnd);
    return 0;
  case tokenEtago:
  case tokenStago:
  case tokenNet:
  case tokenTagc:
  case tokenDsc:
    result.closeToken = token;
    result.type = AttributeParameter::end;
    break;
  case tokenNameStart:
    extendNameToken(syntax().namelen(), Messages::nameTokenLength);
    getCurrentToken(syntax().generalSubstTable(), result.token);
    result.type = AttributeParameter::name;
    break;
  case tokenDigit:
  case tokenLcUcNmchar:
    extendNameToken(syntax().namelen(), Messages::nameTokenLength);
    getCurrentToken(syntax().generalSubstTable(), result.token);
    result.type = AttributeParameter::nameToken;
    break;
  case tokenLit:
  case tokenLita:
    message(allowVi
	    ? Messages::attributeSpecLiteral
	    : Messages::attributeSpecNameTokenExpected);
    return 0;
  case tokenVi:
    if (!allowVi) {
      message(Messages::attributeSpecNameTokenExpected);
      return 0;
    }
    result.type = AttributeParameter::vi;
    break;
  default:
    CANNOT_HAPPEN();
  }
  return 1;
}

Boolean Parser::parseAttributeValueLiteral(Boolean lita, Text &text)
{
  size_t maxLength = (syntax().litlen() > syntax().normsep()
		      ? syntax().litlen() - syntax().normsep()
		      : 0);
  if (parseLiteral(lita ? alitaMode : alitMode, aliteMode,
		   maxLength,
		   Messages::attributeValueLength, 0, text)) {
    if (text.length() == 0
	&& syntax().normsep() > syntax().litlen())
      message(Messages::attributeValueLengthNeg,
	      NumberMessageArg(syntax().normsep() - syntax().litlen()));
    return 1;
  }
  else
    return 0;
}

Boolean Parser::parseTokenizedAttributeValueLiteral(Boolean lita, Text &text)
{
  size_t maxLength = (syntax().litlen() > syntax().normsep()
		      ? syntax().litlen() - syntax().normsep()
		      : 0);
  if (parseLiteral(lita ? talitaMode : talitMode, taliteMode,
		   maxLength,
		   Messages::tokenizedAttributeValueLength,
		   literalSingleSpace, text)) {
    if (text.length() == 0
	&& syntax().normsep() > syntax().litlen())
      message(Messages::tokenizedAttributeValueLengthNeg,
	      NumberMessageArg(syntax().normsep() - syntax().litlen()));
    return 1;
  }
  else
    return 0;
}


Boolean Parser::skipAttributeSpec(Token &closeToken)
{
  AttributeParameter parm;
  if (!parseAttributeParameter(0, 0, parm))
    return 0;
  while (parm.type != AttributeParameter::end) {
    if (parm.type == AttributeParameter::name) {
      if (!parseAttributeParameter(0, 1, parm))
	return 0;
      if (parm.type == AttributeParameter::vi) {
	Token token = getToken(tagMode);
	while (token == tokenS)
	  token = getToken(tagMode);
	switch (token) {
	case tokenUnrecognized:
	  if (!reportNonSgmlCharacter())
	    message(Messages::attributeSpecCharacter,
		    StringMessageArg(currentToken()));
	  return 0;
	case tokenEe:
	  message(Messages::attributeSpecEntityEnd);
	  return 0;
	case tokenEtago:
	case tokenStago:
	case tokenNet:
	case tokenTagc:
	case tokenDsc:
	case tokenVi:
	  message(Messages::attributeValueExpected);
	  return 0;
	case tokenNameStart:
	case tokenDigit:
	case tokenLcUcNmchar:
	  if (!sd().shorttag())
	    message(Messages::attributeValueShorttag);
	  extendNameToken(syntax().litlen() >= syntax().normsep()
			  ? syntax().litlen() - syntax().normsep()
			  : 0,
			  Messages::attributeValueLength);
	  break;
	case tokenLit:
	case tokenLita:
	  {
	    Mode mode = token == tokenLita ? talitaMode : talitMode;
	    for (;;) {
	      Token token = getToken(mode);
	      if (token == tokenEe) {
		message(Messages::literalLevel);
		return 0;
	      }
	      if (token == tokenLit || token == tokenLita)
		break;
	      // Possible to enter ending delimiter with named char ref.
	      if (token == tokenCroNameStart) {
		if (!parseNamedCharRef())
		  return 0;
	      }
	    }
	  }
	  break;
	default:
	  CANNOT_HAPPEN();
	}
	if (!parseAttributeParameter(0, 0, parm))
	  return 0;
      }
      else if (!sd().shorttag())
	message(Messages::attributeNameShorttag);
    }
    else {
      // It's a name token.
      if (!parseAttributeParameter(0, 0, parm))
	return 0;
      if (!sd().shorttag())
	message(Messages::attributeNameShorttag);
    }
  }
  closeToken = parm.closeToken;
  return 1;
}
