// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

// Prolog, dtd and declaration parsing.

#include "Parser.H"
#include "Param.H"
#include "MessageArg.H"
#include "TokenMessageArg.H"
#include "token.H"
#include "macros.H"

static const AllowedParams allowMdc(Param::mdc);
static const AllowedParams allowName(Param::name);
static const AllowedParams allowParamLiteral(Param::paramLiteral);
static const AllowedParams allowNameNameGroup(Param::name, Param::nameGroup);
static const AllowedParams allowDsoMdc(Param::dso, Param::mdc);
static AllowedParams allowNameMdc(Param::name, Param::mdc);
static AllowedParams
  allowExplicitLinkRuleMdc(Param::mdc,
			   Param::name,
			   Param::nameGroup,
			   Param::indicatedReservedName + Syntax::rIMPLIED);
static AllowedParams
  allowNameNameGroupMdc(Param::name, Param::nameGroup, Param::mdc);

static const AllowedParams
  allowLinkSetSpec(Param::name,
		   Param::indicatedReservedName + Syntax::rINITIAL,
		   Param::indicatedReservedName + Syntax::rEMPTY,
		   Param::indicatedReservedName + Syntax::rRESTORE);

void Parser::doProlog()
{
  const unsigned maxTries = 10;
  unsigned tries = 0;
  for (;;) {
    Token token = getToken(proMode);
    switch (token) {
    case tokenUnrecognized:
      if (reportNonSgmlCharacter())
	break;
      if (hadDtd()) {
	currentInput()->ungetToken();
	endProlog();
	return;
      }
      if (++tries >= maxTries) {
	message(Messages::noDtd);
	giveUp();
	return;
      }
      message(Messages::prologCharacter, StringMessageArg(currentToken()));
      prologRecover();
      break;
    case tokenEe:
      if (hadDtd()) {
	endProlog();
	return;
      }
      message(Messages::documentEndProlog);
      allDone();
      return;
    case tokenMdoMdc:
      break;
    case tokenMdoCom:
      if (!parseCommentDecl())
	prologRecover();
      break;
    case tokenMdoNameStart:
      setPass2Start();
      Syntax::ReservedName name;
      if (parseDeclarationName(&name)) {
	switch (name) {
	case Syntax::rDOCTYPE:
	  if (!parseDoctypeDeclStart())
	    giveUp();
	  return;
	case Syntax::rLINKTYPE:
	  if (!parseLinktypeDeclStart())
	    giveUp();
	  return;
	case Syntax::rELEMENT:
	case Syntax::rATTLIST:
	case Syntax::rENTITY:
	case Syntax::rNOTATION:
	case Syntax::rSHORTREF:
	case Syntax::rUSEMAP:
	case Syntax::rUSELINK:
	case Syntax::rLINK:
	case Syntax::rIDLINK:
	  message(Messages::prologDeclaration,
		  StringMessageArg(syntax().reservedName(name)));
	  if (!hadDtd())
	    tries++;
	  prologRecover();
	  break;
	default:
	  message(Messages::noSuchDeclarationType,
		  StringMessageArg(syntax().reservedName(name)));
	  prologRecover();
	  break;
	}
      }
      else
	prologRecover();
      break;
    case tokenPio:
      if (!parseProcessingInstruction())
	prologRecover();
      break;
    case tokenS:
      break;
    default:
      CANNOT_HAPPEN();
    }
  }
}

void Parser::endProlog()
{
  eventHandler().endProlog(new (eventAllocator())
			   EndPrologEvent(currentLocation()));
  if (maybeStartPass2())
    setDoFunction(&Parser::doProlog);
  else {
    if (inputLevel() == 0) {
      allDone();
      return;
    }
    if (pass2())
      checkEntityStability();
    setDoFunction(&Parser::doInstanceStart);
  }
}

void Parser::prologRecover()
{
  unsigned skipCount = 0;
  const unsigned skipMax = 250;
  for (;;) {
    Token token = getToken(proMode);
    skipCount++;
    if (token == tokenUnrecognized) {
      token = getToken(mdMode);
      if (token == tokenMdc) {
	token = getToken(proMode);
	if (token == tokenS)
	  return;
      }
    }
    switch (token) {
    case tokenUnrecognized:
      (void)getChar();
      break;
    case tokenEe:
      return;
    case tokenMdoMdc:
    case tokenMdoCom:
    case tokenMdoNameStart:
    case tokenPio:
      currentInput()->ungetToken();
      return;
    case tokenS:
      if (currentChar() == syntax().standardFunction(Syntax::fRE)
	  && skipCount >= skipMax)
	return;
    default:
      break;
    }
  }
}

void Parser::doDeclSubset()
{
  for (;;) {
    Token token = getToken(currentMode());
    unsigned startLevel = inputLevel();
    Boolean inDtd = haveDefDtd();
    switch (token) {
    case tokenUnrecognized:
      if (reportNonSgmlCharacter())
	break;
      message(Messages::declSubsetCharacter, StringMessageArg(currentToken()));
      declSubsetRecover(startLevel);
      break;
    case tokenEe:
      if (inputLevel() == specialParseInputLevel()) {
	// FIXME have separate messages for each type of special parse
	message(Messages::specialParseEntityEnd);
      }
      if (inputLevel() == 2) {
	const Entity *e = currentInput()->entity();
	if (e
	    && (e->declType() == Entity::doctype
		|| e->declType() == Entity::linktype)) {
	  popInputStack();
	  if (!(inDtd ? parseDoctypeDeclEnd() : parseLinktypeDeclEnd()))
	    ;			// FIXME recover
	  setDoFunction(&Parser::doProlog);
	  return;
	}
      }
      if (inputLevel() == 1) {
	// Give message before popping stack.
	message(inDtd
		? Messages::documentEndDtdSubset
		: Messages::documentEndLpdSubset);
	popInputStack();
	allDone();
      }
      else
	popInputStack();
      return;
    case tokenDsc:		// end of declaration subset
      // FIXME what's the right location?
      if (!referenceDsEntity(currentLocation())) {
	if (!(inDtd ? parseDoctypeDeclEnd() : parseLinktypeDeclEnd()))
	  ;			// FIXME recover
	setDoFunction(&Parser::doProlog);
      }
      return;
    case tokenMdoNameStart:	// named markup declaration
      Syntax::ReservedName name;
      Boolean result;
      if (parseDeclarationName(&name)) {
	switch (name) {
	case Syntax::rELEMENT:
	  if (inDtd)
	    result = parseElementDecl();
	  else {
	    message(Messages::lpdSubsetDeclaration,
		    StringMessageArg(syntax().reservedName(name)));
	    result = 0;
	  }
	  break;
	case Syntax::rATTLIST:
	  result = inDtd ? parseAttlistDecl() : parseLinkAttlistDecl();
	  break;
	case Syntax::rENTITY:
	  result = parseEntityDecl();
	  break;
	case Syntax::rNOTATION:
	  if (inDtd)
	    result = parseNotationDecl();
	  else {
	    message(Messages::lpdSubsetDeclaration,
		    StringMessageArg(syntax().reservedName(name)));
	    result = 0;
	  }
	  break;
	case Syntax::rSHORTREF:
	  if (inDtd)
	    result = parseShortrefDecl();
	  else {
	    message(Messages::lpdSubsetDeclaration,
		    StringMessageArg(syntax().reservedName(name)));
	    result = 0;
	  }
	  break;
	case Syntax::rUSEMAP:
	  if (inDtd)
	    result = parseUsemapDecl();
	  else {
	    message(Messages::lpdSubsetDeclaration,
		    StringMessageArg(syntax().reservedName(name)));
	    result = 0;
	  }
	  break;
	case Syntax::rLINK:
	  if (inDtd) {
	    message(Messages::dtdSubsetDeclaration,
		    StringMessageArg(syntax().reservedName(name)));
	    result = 0;
	  }
	  else
	    result = parseLinkDecl();
	  break;
	case Syntax::rIDLINK:
	  if (inDtd) {
	    message(Messages::dtdSubsetDeclaration,
		    StringMessageArg(syntax().reservedName(name)));
	    result = 0;
	  }
	  else
	    result = parseIdlinkDecl();
	  break;
	case Syntax::rDOCTYPE:
	case Syntax::rLINKTYPE:
	case Syntax::rUSELINK:
	  result = 0;
	  message(inDtd
		  ? Messages::dtdSubsetDeclaration
		  : Messages::lpdSubsetDeclaration,
		  StringMessageArg(syntax().reservedName(name)));
	  break;
	default:
	  result = 0;
	  message(Messages::noSuchDeclarationType,
		  StringMessageArg(syntax().reservedName(name)));
	  break;
	}
      }
      else
	result = 0;
      if (!result)
	declSubsetRecover(startLevel);
      break;
    case tokenMdoMdc:		// empty comment declaration
      break;
    case tokenMdoCom:		// comment declaration
      if (!parseCommentDecl())
	declSubsetRecover(startLevel);
      break;
    case tokenMdoDso:		// marked section declaration
      if (!parseMarkedSectionDeclStart())
	declSubsetRecover(startLevel);
      break;
    case tokenMscMdc:
      handleMarkedSectionEnd();
      break;
    case tokenPeroGrpo:		// parameter entity reference with name group
      message(Messages::peroGrpoProlog);
      // fall through
    case tokenPeroNameStart:	// parameter entity reference
      {
	ConstResourcePointer<Entity> entity;
	ResourcePointer<EntityOrigin> origin;
	if (parseEntityReference(1, token == tokenPeroGrpo, entity, origin)) {
	  if (!entity.isNull())
	    entity->dsReference(*this, origin);
	}
	else
	  declSubsetRecover(startLevel);
      }
      break;
    case tokenPio:		// processing instruction
      if (!parseProcessingInstruction())
	declSubsetRecover(startLevel);
      break;
    case tokenS:		// white space in element content
      extendS();
      break;
    case tokenIgnoredChar:
      // from an ignored marked section
      break;
    case tokenRe:
    case tokenRs:
    case tokenCroDigit:
    case tokenCroNameStart:
    case tokenEroNameStart:
    case tokenEroGrpo:
    case tokenChar:
      // these can occur in a cdata or rcdata marked section
      message(Messages::dataMarkedSectionDeclSubset);
      declSubsetRecover(startLevel);
      break;
    default:
      CANNOT_HAPPEN();
    }
  }
}

void Parser::declSubsetRecover(unsigned startLevel)
{
  for (;;) {
    Token token = getToken(currentMode());
    switch (token) {
    case tokenUnrecognized:
      (void)getChar();
      break;
    case tokenEe:
      if (inputLevel() <= startLevel)
	return;
      popInputStack();
      break;
    case tokenMdoCom:
    case tokenDsc:
    case tokenMdoNameStart:
    case tokenMdoMdc:
    case tokenMdoDso:
    case tokenMscMdc:
    case tokenPio:
      if (inputLevel() == startLevel) {
	currentInput()->ungetToken();
	return;
      }
      break;
    default:
      break;
    }
  }
}

Boolean Parser::parseDeclarationName(Syntax::ReservedName *result)
{
  currentInput()->discardInitial();
  extendNameToken(syntax().namelen(), Messages::nameLength);
  CString &name(nameBuffer());
  getCurrentToken(syntax().generalSubstTable(), name);
  if (!syntax().lookupReservedName(name, result)) {
    message(Messages::noSuchDeclarationType, StringMessageArg(name));
    return 0;
  }
  return 1;
}

Boolean Parser::parseElementDecl()
{
  unsigned declInputLevel = inputLevel();
  Location location(currentLocation());
  Param parm;
  if (!parseParam(allowNameNameGroup, declInputLevel, parm))
    return 0;
  GrowableVectorD<NameToken> nameVector;
  if (parm.type == Param::nameGroup)
    parm.nameTokenVector.moveTo(nameVector);
  else
    parm.token.moveTo(nameVector.grow().name);
  static AllowedParams
    allowRankOmissionContent(Param::number,
			     Param::reservedName + Syntax::rO,
			     Param::minus,
			     Param::reservedName + Syntax::rCDATA,
			     Param::reservedName + Syntax::rRCDATA,
			     Param::reservedName + Syntax::rEMPTY,
			     Param::reservedName + Syntax::rANY,
			     Param::modelGroup);
  if (!parseParam(allowRankOmissionContent, declInputLevel, parm))
    return 0;
  CString rankSuffix;
  Vector<ElementType *> elements(nameVector.length());
  Vector<RankStem *> rankStems;
  Vector<const RankStem *> constRankStems;
  int i;
  if (parm.type == Param::number) {
    parm.token.moveTo(rankSuffix);
    rankStems.init(nameVector.length());
    constRankStems.init(nameVector.length());
    for (i = 0; i < elements.length(); i++) {
      CString name(nameVector[i].name);
      name += rankSuffix;
      if (name.length() > syntax().namelen()
	  && nameVector[i].name.length() <= syntax().namelen())
	message(Messages::genericIdentifierLength,
		NumberMessageArg(syntax().namelen()));
      elements[i] = lookupCreateElement(name);
      rankStems[i] = lookupCreateRankStem(nameVector[i].name);
      constRankStems[i] = rankStems[i];
    }
    static AllowedParams
      allowOmissionContent(Param::reservedName + Syntax::rO,
			   Param::minus,
			   Param::reservedName + Syntax::rCDATA,
			   Param::reservedName + Syntax::rRCDATA,
			   Param::reservedName + Syntax::rEMPTY,
			   Param::reservedName + Syntax::rANY,
			   Param::modelGroup);
    Token token = getToken(mdMinusMode);
    if (token == tokenNameStart)
      message(Messages::psRequired);
    currentInput()->ungetToken();
    if (!parseParam(allowOmissionContent, declInputLevel, parm))
      return 0;
  }
  else {
    for (i = 0; i < elements.length(); i++)
      elements[i] = lookupCreateElement(nameVector[i].name);
  }
  for (i = 0; i < elements.length(); i++)
    if (defDtd().lookupRankStem(elements[i]->name()))
      message(Messages::rankStemGenericIdentifier,
	      StringMessageArg(elements[i]->name()));
  unsigned char omitFlags = 0;
  if (parm.type == Param::minus
      || parm.type == Param::reservedName + Syntax::rO) {
    omitFlags |= ElementDefinition::omitSpec;
    if (parm.type != Param::minus)
      omitFlags |= ElementDefinition::omitStart;
    static AllowedParams allowOmission(Param::reservedName + Syntax::rO,
				       Param::minus);
    if (!parseParam(allowOmission, declInputLevel, parm))
      return 0;
    if (parm.type != Param::minus)
      omitFlags |= ElementDefinition::omitEnd;
    static AllowedParams allowContent(Param::reservedName + Syntax::rCDATA,
				      Param::reservedName + Syntax::rRCDATA,
				      Param::reservedName + Syntax::rEMPTY,
				      Param::reservedName + Syntax::rANY,
				      Param::modelGroup);
    if (!parseParam(allowContent, declInputLevel, parm))
      return 0;
  }
  else {
    if (sd().omittag())
      message(Messages::missingTagMinimization);
  }
  ResourcePointer<ElementDefinition> def;
  switch (parm.type) {
  case Param::reservedName + Syntax::rCDATA:
    def = new ElementDefinition(location,
				defDtd().allocElementDefinitionIndex(),
				omitFlags,
				ElementDefinition::cdata);
    if (!parseParam(allowMdc, declInputLevel, parm))
      return 0;
    break;
  case Param::reservedName + Syntax::rRCDATA:
    def = new ElementDefinition(location,
				defDtd().allocElementDefinitionIndex(),
				omitFlags,
				ElementDefinition::rcdata);
    if (!parseParam(allowMdc, declInputLevel, parm))
      return 0;
    break;
  case Param::reservedName + Syntax::rEMPTY:
    def = new ElementDefinition(location,
				defDtd().allocElementDefinitionIndex(),
				omitFlags,
				ElementDefinition::empty);
    if ((omitFlags & ElementDefinition::omitSpec)
	&& !(omitFlags & ElementDefinition::omitEnd)
	&& options().warnShould)
      message(Messages::emptyOmitEndTag);
    if (!parseParam(allowMdc, declInputLevel, parm))
      return 0;
    break;
  case Param::reservedName + Syntax::rANY:
    def = new ElementDefinition(location,
				defDtd().allocElementDefinitionIndex(),
				omitFlags,
				ElementDefinition::any);
    if (!parseExceptions(declInputLevel, def))
      return 0;
    break;
  case Param::modelGroup:
    {
      unsigned long cnt = parm.modelGroupPtr->grpgtcnt();
      // The outermost model group isn't formally a content token.
      if (cnt - 1 > syntax().grpgtcnt())
	message(Messages::grpgtcnt, NumberMessageArg(syntax().grpgtcnt()));
      Owner<CompiledModelGroup> modelGroup 
	= new CompiledModelGroup(parm.modelGroupPtr);
      GrowableVector<ContentModelAmbiguity> ambiguities;
      modelGroup->compile(currentDtd().nElementTypeIndex(), ambiguities);
      for (size_t i = 0; i < ambiguities.length(); i++) {
	const ContentModelAmbiguity &a(ambiguities[i]);
	reportAmbiguity(a.from, a.to1, a.to2, a.andDepth);
      }
      def = new ElementDefinition(location,
				  defDtd().allocElementDefinitionIndex(),
				  omitFlags,
				  ElementDefinition::modelGroup,
				  modelGroup);
      if (!parseExceptions(declInputLevel, def))
	return 0;
    }
    break;
  }
  if (rankSuffix.length() > 0)
    def->setRank(rankSuffix, constRankStems);
  ConstResourcePointer<ElementDefinition> constDef(def);
  for (i = 0; i < elements.length(); i++) {
    if (elements[i]->definition() != 0)
      message(Messages::duplicateElementDefinition,
	      StringMessageArg(elements[i]->name()));
    else {
      elements[i]->setElementDefinition(constDef, i);
      if (!elements[i]->attributeDef().isNull())
	checkElementAttribute(elements[i]);
    }
    if (rankStems.length() > 0)
      rankStems[i]->addDefinition(constDef);
  }
  return 1;
}

void Parser::reportAmbiguity(const LeafContentToken *from,
			     const LeafContentToken *to1,
			     const LeafContentToken *to2,
			     unsigned ambigAndDepth)
{
  CString toName;
  const ElementType *toType = to1->elementType();
  if (toType)
    toName = toType->name();
  else {
    toName = syntax().delimGeneral(Syntax::dRNI);
    toName += syntax().reservedName(Syntax::rPCDATA);
  }
  unsigned to1Index = to1->typeIndex() + 1;
  unsigned to2Index = to2->typeIndex() + 1;
  if (from->isInitial())
    message(Messages::ambiguousModelInitial,
	    StringMessageArg(toName),
	    OrdinalMessageArg(to1Index),
	    OrdinalMessageArg(to2Index));
  else {
    CString fromName;
    const ElementType *fromType = from->elementType();
    if (fromType)
      fromName = fromType->name();
    else {
      fromName = syntax().delimGeneral(Syntax::dRNI);
      fromName += syntax().reservedName(Syntax::rPCDATA);
    }
    unsigned fromIndex = from->typeIndex() + 1;
    unsigned andMatches = from->andDepth() - ambigAndDepth;
    if (andMatches == 0)
      message(Messages::ambiguousModel,
	      StringMessageArg(fromName),
	      OrdinalMessageArg(fromIndex),
	      StringMessageArg(toName),
	      OrdinalMessageArg(to1Index),
	      OrdinalMessageArg(to2Index));
    else if (andMatches == 1)
      message(Messages::ambiguousModelSingleAnd,
	      StringMessageArg(fromName),
	      OrdinalMessageArg(fromIndex),
	      StringMessageArg(toName),
	      OrdinalMessageArg(to1Index),
	      OrdinalMessageArg(to2Index));
    else
      message(Messages::ambiguousModelMultipleAnd,
	      StringMessageArg(fromName),
	      OrdinalMessageArg(fromIndex),
	      NumberMessageArg(andMatches),
	      StringMessageArg(toName),
	      OrdinalMessageArg(to1Index),
	      OrdinalMessageArg(to2Index));
  }
}


// Check the compatibility of the attribute definition with
// the element definition.

void Parser::checkElementAttribute(const ElementType *e)
{
  const AttributeDefinitionList *attDef = e->attributeDef().pointer();
  Boolean conref = 0;
  ASSERT(e != 0);
  const ElementDefinition *edef = e->definition();
  ASSERT(edef != 0);
  ASSERT(attDef != 0);
  size_t attDefLength = attDef->length();
  for (size_t i = 0; i < attDefLength; i++) {
    const AttributeDefinition *p = attDef->def(i);
    if (p->isConref())
      conref = 1;
    if (p->isNotation()
	&& edef->declaredContent() == ElementDefinition::empty)
      message(Messages::notationEmpty, StringMessageArg(e->name()));
  }
  if (conref) {
    if (edef->omittedTagSpec() && !edef->canOmitEndTag()
	&& options().warnShould)
      message(Messages::conrefOmitEndTag, StringMessageArg(e->name()));
    if (edef->declaredContent() == ElementDefinition::empty)
      message(Messages::conrefEmpty, StringMessageArg(e->name()));
  }
}

ElementType *Parser::lookupCreateElement(const CString &name)
{
  ElementType *e = defDtd().lookupElementType(name);
  if (!e) {
    e = new ElementType(name, defDtd().nElementTypeIndex());
    defDtd().insertElementType(e);
  }
  return e;
}

RankStem *Parser::lookupCreateRankStem(const CString &name)
{
  RankStem *r = defDtd().lookupRankStem(name);
  if (!r) {
    r = new RankStem(name, defDtd().nRankStem());
    defDtd().insertRankStem(r);
    const ElementType *e = defDtd().lookupElementType(name);
    if (e && e->definition() != 0)
      message(Messages::rankStemGenericIdentifier, StringMessageArg(name));
  }
  return r;
}

Boolean Parser::parseExceptions(unsigned declInputLevel,
				ResourcePointer<ElementDefinition> &def)
{
  Param parm;
  static AllowedParams
    allowExceptionsMdc(Param::mdc, Param::exclusions, Param::inclusions);
  if (!parseParam(allowExceptionsMdc, declInputLevel, parm))
    return 0;
  if (parm.type == Param::exclusions) {
    def->setExclusions(parm.elementVector);
    static AllowedParams allowInclusionsMdc(Param::mdc, Param::inclusions);
    if (!parseParam(allowInclusionsMdc, declInputLevel, parm))
      return 0;
  }
  if (parm.type == Param::inclusions) {
    def->setInclusions(parm.elementVector);
    if (!parseParam(allowMdc, declInputLevel, parm))
      return 0;
  }
  return 1;
}

Boolean Parser::parseAttlistDecl()
{
  unsigned declInputLevel = inputLevel();
  Param parm;
  size_t attcnt = 0;
  int idCount = 0;
  int notationCount = 0;
  
  Boolean isNotation;
  Vector<Attributed *> attributed;
  if (!parseAttributed(declInputLevel, parm, attributed, isNotation))
    return 0;
  GrowableVectorD<Owner<AttributeDefinition> > defs;
  if (!parseParam(allowName, declInputLevel, parm))
    return 0;
  do {
    CString attributeName;
    parm.token.moveTo(attributeName);
    attcnt++;
    Boolean duplicate = 0;
    for (size_t i = 0; i < defs.length(); i++)
      if (defs[i]->name() == attributeName) {
	message(Messages::duplicateAttributeDef,
		StringMessageArg(attributeName));
	duplicate = 1;
	break;
      }
    Owner<DeclaredValue> declaredValue;
    if (!parseDeclaredValue(declInputLevel, isNotation, parm, declaredValue))
      return 0;
    if (declaredValue->isId()) {
      if (++idCount == 2)
	message(Messages::multipleIdAttributes);
    }
    else if (declaredValue->isNotation()) {
      if (++notationCount == 2)
	message(Messages::multipleNotationAttributes);
    }
    const CString *tokens;
    size_t nTokens = declaredValue->getTokens(tokens);
    for (i = 0; i < nTokens; i++) {
      for (size_t j = 0; j < defs.length(); j++)
	if (defs[j]->containsToken(tokens[i])) {
	  message(Messages::duplicateAttributeToken,
		  StringMessageArg(tokens[i]));
	  break;
	}
    }
    attcnt += nTokens;
    Owner<AttributeDefinition> def;
    if (!parseDefaultValue(declInputLevel, isNotation, parm, attributeName,
			   declaredValue, def))
      return 0;
    if (!duplicate)
      defs.grow() = def.extract();
    static AllowedParams allowNameMdc(Param::name, Param::mdc);
    if (!parseParam(allowNameMdc, declInputLevel, parm))
      return 0;
  } while (parm.type != Param::mdc);
  ConstResourcePointer<AttributeDefinitionList> adl
    = new AttributeDefinitionList(defs,
				  defDtd().allocAttributeDefinitionListIndex());
  if (attcnt > syntax().attcnt())
    message(Messages::attcnt,
	    NumberMessageArg(attcnt),
	    NumberMessageArg(syntax().attcnt()));
    
  for (size_t i = 0; i < attributed.length(); i++) {
    if (attributed[i]->attributeDef().isNull()) {
      attributed[i]->setAttributeDef(adl);
      if (!isNotation) {
	ElementType *e = (ElementType *)attributed[i];
	if (e->definition() != 0)
	  checkElementAttribute(e);
      }
    }
    else {
      if (isNotation)
	message(Messages::duplicateAttlistNotation,
		StringMessageArg(((Notation *)attributed[i])->name()));
      else
	message(Messages::duplicateAttlistElement,
		StringMessageArg(((ElementType *)attributed[i])->name()));
    }
  }
  if (isNotation) {
    EntitySet::EntityIter entityIter(defDtd().generalEntityIter());
    for (;;) {
      ResourcePointer<Entity> entity(entityIter.next());
      if (entity.isNull())
	break;
      const ExternalDataEntity *external = entity->asExternalDataEntity();
      if (external) {
	const Notation *entityNotation = external->notation();
	for (size_t  i = 0; i < attributed.length(); i++)
	  if ((Notation *)attributed[i] == entityNotation) {
	    AttributeList attributes(entityNotation->attributeDef());
	    attributes.finish(*this);
	    ((ExternalDataEntity *)entity.pointer())->setAttributes(attributes);
	  }
      }
    }
  }
  return 1;
}


Boolean Parser::parseAttributed(unsigned declInputLevel,
				Param &parm,
				Vector<Attributed *> &attributed,
				Boolean &isNotation)
{
  static AllowedParams
    allowNameGroupNotation(Param::name,
			   Param::nameGroup,
			   Param::indicatedReservedName + Syntax::rNOTATION);
  if (!parseParam(allowNameGroupNotation, declInputLevel, parm))
    return 0;
  if (parm.type == Param::indicatedReservedName + Syntax::rNOTATION) {
    isNotation = 1;
    if (!parseParam(allowNameNameGroup, declInputLevel, parm))
      return 0;
    if (parm.type == Param::name) {
      attributed.init(1);
      attributed[0] = lookupCreateNotation(parm.token);
    }
    else {
      attributed.init(parm.nameTokenVector.length());
      for (size_t i = 0; i < attributed.length(); i++)
	attributed[i] = lookupCreateNotation(parm.nameTokenVector[i].name);
    }
  }
  else {
    isNotation = 0;
    if (parm.type == Param::name) {
      attributed.init(1);
      attributed[0] = lookupCreateElement(parm.token);
    }
    else {
      attributed.init(parm.nameTokenVector.length());
      for (size_t i = 0; i < attributed.length(); i++)
	attributed[i] = lookupCreateElement(parm.nameTokenVector[i].name);
    }
  }
  return 1;
}

Boolean Parser::parseLinkAttlistDecl()
{
  unsigned declInputLevel = inputLevel();
  Param parm;
  size_t attcnt = 0;

  Vector<const ElementType *> elements;
  if (!parseParam(allowNameNameGroup, declInputLevel, parm))
    return 0;
  if (parm.type == Param::name) {
    elements.init(1);
    elements[0] = lookupSourceElementType(parm.token);
  }
  else {
    elements.init(parm.nameTokenVector.length());
    for (size_t i = 0; i < elements.length(); i++)
      elements[i] = lookupSourceElementType(parm.nameTokenVector[i].name);
  }

  GrowableVectorD<Owner<AttributeDefinition> > defs;
  if (!parseParam(allowName, declInputLevel, parm))
    return 0;
  do {
    CString attributeName;
    parm.token.moveTo(attributeName);
    attcnt++;
    Boolean duplicate = 0;
    for (size_t i = 0; i < defs.length(); i++)
      if (defs[i]->name() == attributeName) {
	message(Messages::duplicateAttributeDef,
		StringMessageArg(attributeName));
	duplicate = 1;
	break;
      }
    Owner<DeclaredValue> declaredValue;
    if (!parseDeclaredValue(declInputLevel, 0, parm, declaredValue))
      return 0;
    const CString *tokens;
    size_t nTokens = declaredValue->getTokens(tokens);
    for (i = 0; i < nTokens; i++) {
      for (size_t j = 0; j < defs.length(); j++)
	if (defs[j]->containsToken(tokens[i])) {
	  message(Messages::duplicateAttributeToken,
		  StringMessageArg(tokens[i]));
	  break;
	}
    }
    attcnt += nTokens;
    Owner<AttributeDefinition> def;
    if (!parseDefaultValue(declInputLevel, 0, parm, attributeName,
			   declaredValue, def))
      return 0;
    if (defLpd().type() == Lpd::simple && !def->isFixed())
      message(Messages::simpleLinkFixedAttribute);
    if (!duplicate)
      defs.grow() = def.extract();
    if (!parseParam(allowNameMdc, declInputLevel, parm))
      return 0;
  } while (parm.type != Param::mdc);
  if (attcnt > syntax().attcnt())
    message(Messages::attcnt,
	    NumberMessageArg(attcnt),
	    NumberMessageArg(syntax().attcnt()));
  
  if (defLpd().type() == Lpd::simple) {
    if (haveBaseDtd()) {
      for (size_t i = 0; i < elements.length(); i++)
	if (elements[i]) {
	  if (elements[i]->name() == baseDtd().name()) {
	    SimpleLpd &lpd = (SimpleLpd &)defLpd();
	    if (lpd.attributeDef().isNull())
	      lpd.setAttributeDef(new AttributeDefinitionList(defs, 0));
	    else
	      message(Messages::duplicateAttlistElement,
		      StringMessageArg(elements[i]->name()));
	  }
	  else
	    message(Messages::simpleLinkAttlistElement,
		    StringMessageArg(elements[i]->name()));
	}
    }
  }
  else {
    ConstResourcePointer<AttributeDefinitionList> adl
      = new AttributeDefinitionList(defs,
				    defComplexLpd().allocAttributeDefinitionListIndex());
    for (size_t i = 0; i < elements.length(); i++)
      if (elements[i]) {
	if (defComplexLpd().attributeDef(elements[i]).isNull())
	  defComplexLpd().setAttributeDef(elements[i], adl);
	else
	  message(Messages::duplicateAttlistElement,
		  StringMessageArg(elements[i]->name()));
      }
  }
  return 1;
}

Boolean Parser::parseDeclaredValue(unsigned declInputLevel,
				   Boolean isNotation,
				   Param &parm,
				   Owner<DeclaredValue> &declaredValue)
{
  static Param::Type declaredValues[] = {
    Param::reservedName + Syntax::rCDATA,
    Param::reservedName + Syntax::rENTITY,
    Param::reservedName + Syntax::rENTITIES,
    Param::reservedName + Syntax::rID,
    Param::reservedName + Syntax::rIDREF,
    Param::reservedName + Syntax::rIDREFS,
    Param::reservedName + Syntax::rNAME,
    Param::reservedName + Syntax::rNAMES,
    Param::reservedName + Syntax::rNMTOKEN,
    Param::reservedName + Syntax::rNMTOKENS,
    Param::reservedName + Syntax::rNUMBER,
    Param::reservedName + Syntax::rNUMBERS,
    Param::reservedName + Syntax::rNUTOKEN,
    Param::reservedName + Syntax::rNUTOKENS,
    Param::reservedName + Syntax::rNOTATION,
    Param::nameTokenGroup
    };
  static AllowedParams allowDeclaredValue(declaredValues,
					  SIZEOF(declaredValues));
  if (!parseParam(allowDeclaredValue, declInputLevel, parm))
    return 0;
  enum { asDataAttribute = 01, asLinkAttribute = 02 };
  unsigned allowedFlags = asDataAttribute|asLinkAttribute;
  switch (parm.type) {
  case Param::reservedName + Syntax::rCDATA:
    declaredValue = new CdataDeclaredValue;
    break;
  case Param::reservedName + Syntax::rENTITY:
    declaredValue = new EntityDeclaredValue(0);
    allowedFlags = asLinkAttribute;
    break;
  case Param::reservedName + Syntax::rENTITIES:
    declaredValue = new EntityDeclaredValue(1);
    allowedFlags = asLinkAttribute;
    break;
  case Param::reservedName + Syntax::rID:
    declaredValue = new IdDeclaredValue;
    allowedFlags = 0;
    break;
  case Param::reservedName + Syntax::rIDREF:
    declaredValue = new IdrefDeclaredValue(0);
    allowedFlags = 0;
    break;
  case Param::reservedName + Syntax::rIDREFS:
    declaredValue = new IdrefDeclaredValue(1);
    allowedFlags = 0;
    break;
  case Param::reservedName + Syntax::rNAME:
    declaredValue
      = new TokenizedDeclaredValue(TokenizedDeclaredValue::name, 0);
    break;
  case Param::reservedName + Syntax::rNAMES:
    declaredValue
      = new TokenizedDeclaredValue(TokenizedDeclaredValue::name, 1);
    break;
  case Param::reservedName + Syntax::rNMTOKEN:
    declaredValue
      = new TokenizedDeclaredValue(TokenizedDeclaredValue::nameToken, 0);
    break;
  case Param::reservedName + Syntax::rNMTOKENS:
    declaredValue
      = new TokenizedDeclaredValue(TokenizedDeclaredValue::nameToken, 1);
    break;
  case Param::reservedName + Syntax::rNUMBER:
    declaredValue
      = new TokenizedDeclaredValue(TokenizedDeclaredValue::number, 0);
    break;
  case Param::reservedName + Syntax::rNUMBERS:
    declaredValue
      = new TokenizedDeclaredValue(TokenizedDeclaredValue::number, 1);
    break;
  case Param::reservedName + Syntax::rNUTOKEN:
    declaredValue
      = new TokenizedDeclaredValue(TokenizedDeclaredValue::numberToken, 0);
    break;
  case Param::reservedName + Syntax::rNUTOKENS:
    declaredValue
      = new TokenizedDeclaredValue(TokenizedDeclaredValue::numberToken, 1);
    break;
  case Param::reservedName + Syntax::rNOTATION:
    {
      static AllowedParams allowNameGroup(Param::nameGroup);
      if (!parseParam(allowNameGroup, declInputLevel, parm))
	return 0;
      Vector<CString> group(parm.nameTokenVector.length());
      for (size_t i = 0; i < group.length(); i++)
	parm.nameTokenVector[i].name.moveTo(group[i]);
      declaredValue = new NotationDeclaredValue(group);
      allowedFlags = 0;
    }
    break;
  case Param::nameTokenGroup:
    {
      Vector<CString> group(parm.nameTokenVector.length());
      for (size_t i = 0; i < group.length(); i++)
	parm.nameTokenVector[i].name.moveTo(group[i]);
      declaredValue = new NameTokenGroupDeclaredValue(group);
    }
    break;
  default:
    CANNOT_HAPPEN();
  }
  if (isNotation) {
    if (!(allowedFlags & asDataAttribute))
      message(Messages::dataAttributeDeclaredValue);
  }
  else if (!haveDefDtd() && !(allowedFlags & asLinkAttribute))
    message(Messages::linkAttributeDeclaredValue);
  return 1;
}

Boolean Parser::parseDefaultValue(unsigned declInputLevel,
				  Boolean isNotation,
				  Param &parm,
				  const CString &attributeName,
				  Owner<DeclaredValue> &declaredValue,
				  Owner<AttributeDefinition> &def)
{
  // default value
  static AllowedParams
    allowDefaultValue(Param::indicatedReservedName + Syntax::rFIXED,
		      Param::indicatedReservedName + Syntax::rREQUIRED,
		      Param::indicatedReservedName + Syntax::rCURRENT,
		      Param::indicatedReservedName + Syntax::rCONREF,
		      Param::indicatedReservedName + Syntax::rIMPLIED,
		      Param::attributeValue,
		      Param::attributeValueLiteral);
  static AllowedParams
    allowTokenDefaultValue(Param::indicatedReservedName + Syntax::rFIXED,
		      Param::indicatedReservedName + Syntax::rREQUIRED,
		      Param::indicatedReservedName + Syntax::rCURRENT,
		      Param::indicatedReservedName + Syntax::rCONREF,
		      Param::indicatedReservedName + Syntax::rIMPLIED,
		      Param::attributeValue,
		      Param::tokenizedAttributeValueLiteral);
  if (!parseParam(declaredValue->tokenized()
		  ? allowTokenDefaultValue
		  : allowDefaultValue, declInputLevel, parm))
    return 0;
  switch (parm.type) {
  case Param::indicatedReservedName + Syntax::rFIXED:
    {
      static AllowedParams allowValue(Param::attributeValue,
				      Param::attributeValueLiteral);
      static AllowedParams
	allowTokenValue(Param::attributeValue,
			Param::tokenizedAttributeValueLiteral);
      if (!parseParam(declaredValue->tokenized()
		      ? allowTokenValue
		      : allowValue, declInputLevel, parm))
	return 0;
      unsigned specLength = 0;
      AttributeValue *value = declaredValue->makeValue(parm.literalText,
						       *this,
						       attributeName,
						       specLength);
      if (declaredValue->isId())
	message(Messages::idDeclaredValue);
      def = new FixedAttributeDefinition(attributeName,
					 declaredValue.extract(),
					 value);
    }
    break;
  case Param::attributeValue:
  case Param::attributeValueLiteral:
  case Param::tokenizedAttributeValueLiteral:
    {
      unsigned specLength = 0;
      AttributeValue *value = declaredValue->makeValue(parm.literalText,
						       *this,
						       attributeName,
						       specLength);
      if (declaredValue->isId())
	message(Messages::idDeclaredValue);
      def = new DefaultAttributeDefinition(attributeName,
					   declaredValue.extract(),
					   value);
    }
    break;
  case Param::indicatedReservedName + Syntax::rREQUIRED:
    def = new RequiredAttributeDefinition(attributeName,
					  declaredValue.extract());
    break;
  case Param::indicatedReservedName + Syntax::rCURRENT:
    if (declaredValue->isId())
      message(Messages::idDeclaredValue);
    def = new CurrentAttributeDefinition(attributeName,
					 declaredValue.extract(),
					 defDtd().allocCurrentAttributeIndex());
    if (isNotation)
      message(Messages::dataAttributeDefaultValue);
    else if (!haveDefDtd())
      message(Messages::linkAttributeDefaultValue);
    break;
  case Param::indicatedReservedName + Syntax::rCONREF:
    if (declaredValue->isId())
      message(Messages::idDeclaredValue);
    def = new ConrefAttributeDefinition(attributeName,
					declaredValue.extract());
    if (isNotation)
      message(Messages::dataAttributeDefaultValue);
    else if (!haveDefDtd())
      message(Messages::linkAttributeDefaultValue);
    break;
  case Param::indicatedReservedName + Syntax::rIMPLIED:
    def = new ImpliedAttributeDefinition(attributeName,
					 declaredValue.extract());
    break;
  default:
    CANNOT_HAPPEN();
  }
  return 1;
}

// parm contains either system or public

Boolean Parser::parseExternalId(const AllowedParams &sysidAllow,
				const AllowedParams &endAllow,
				unsigned declInputLevel,
				Param &parm,
				ExternalId &id)
{
  if (parm.type == Param::reservedName + Syntax::rPUBLIC) {
    static AllowedParams allowMinimumLiteral(Param::minimumLiteral);
    if (!parseParam(allowMinimumLiteral, declInputLevel, parm))
      return 0;
    PublicId::FormalError err;
    if (!id.setPublic(parm.literalText, sd().docCharset(), syntax().space(),
		      err)
	&& sd().formal())
      message(publicIdFormalErrorMessage(err),
	      StringMessageArg(*id.publicIdString()));
  }
  if (!parseParam(sysidAllow, declInputLevel, parm))
    return 0;
  if (parm.type == Param::systemIdentifier) {
    id.setSystem(parm.literalText);
    if (!parseParam(endAllow, declInputLevel, parm))
      return 0;
  }
  return 1;
}

Boolean Parser::parseNotationDecl()
{
  unsigned declInputLevel = inputLevel();
  Param parm;
  if (!parseParam(allowName, declInputLevel, parm))
    return 0;
  Notation *nt = lookupCreateNotation(parm.token);
  if (nt->defined())
    message(Messages::duplicateNotationDeclaration,
	    StringMessageArg(parm.token));
  static AllowedParams
    allowPublicSystem(Param::reservedName + Syntax::rPUBLIC,
		      Param::reservedName + Syntax::rSYSTEM);
  if (!parseParam(allowPublicSystem, declInputLevel, parm))
    return 0;


  static AllowedParams allowSystemIdentifierMdc(Param::systemIdentifier,
						Param::mdc);

  ExternalId id;
  if (!parseExternalId(allowSystemIdentifierMdc, allowMdc,
		       declInputLevel, parm, id))
    return 0;
  if (sd().formal()) {
    PublicId::TextClass textClass;
    const PublicId *publicId = id.publicId();
    if (publicId
	&& publicId->getTextClass(textClass)
	&& textClass != PublicId::NOTATION)
      message(Messages::notationIdentifierTextClass);
  }
  if (!nt->defined()) {
    nt->setExternalId(id);
    eventHandler().notationDecl(new (eventAllocator())
				NotationDeclEvent(nt));
  }
  return 1;
}

Boolean Parser::parseEntityDecl()
{
  unsigned declInputLevel = inputLevel();
  Param parm;

  Location location(currentLocation());
  static AllowedParams
    allowEntityNamePero(Param::entityName,
			Param::indicatedReservedName + Syntax::rDEFAULT,
			Param::pero);

  if (!parseParam(allowEntityNamePero, declInputLevel, parm))
    return 0;

  Entity::DeclType declType;
  CString name;			// empty for default entity
  if (parm.type == Param::pero) {
    declType = Entity::parameterEntity;
    static AllowedParams allowParamEntityName(Param::paramEntityName);
    if (!parseParam(allowParamEntityName, declInputLevel, parm))
      return 0;
    parm.token.moveTo(name);
  }
  else {
    declType = Entity::generalEntity;
    if (parm.type == Param::entityName)
      parm.token.moveTo(name);
  }
  static AllowedParams
    allowEntityTextType(Param::paramLiteral,
			Param::reservedName + Syntax::rCDATA,
			Param::reservedName + Syntax::rSDATA,
			Param::reservedName + Syntax::rPI,
			Param::reservedName + Syntax::rSTARTTAG,
			Param::reservedName + Syntax::rENDTAG,
			Param::reservedName + Syntax::rMS,
			Param::reservedName + Syntax::rMD,
			Param::reservedName + Syntax::rSYSTEM,
			Param::reservedName + Syntax::rPUBLIC);

  if (!parseParam(allowEntityTextType, declInputLevel, parm))
    return 0;
  Location typeLocation(currentLocation());
  Entity::DataType dataType = Entity::sgmlText;
  InternalTextEntity::Bracketed bracketed = InternalTextEntity::none;
  switch (parm.type) {
  case Param::reservedName + Syntax::rSYSTEM:
  case Param::reservedName + Syntax::rPUBLIC:
    return parseExternalEntity(location, name, declType,
			       declInputLevel, parm);
  case Param::reservedName + Syntax::rCDATA:
    dataType = Entity::cdata;
    break;
  case Param::reservedName + Syntax::rSDATA:
    dataType = Entity::sdata;
    break;
  case Param::reservedName + Syntax::rPI:
    dataType = Entity::pi;
    break;
  case Param::reservedName + Syntax::rSTARTTAG:
    bracketed = InternalTextEntity::starttag;
    break;
  case Param::reservedName + Syntax::rENDTAG:
    bracketed = InternalTextEntity::endtag;
    break;
  case Param::reservedName + Syntax::rMS:
    bracketed = InternalTextEntity::ms;
    break;
  case Param::reservedName + Syntax::rMD:
    bracketed = InternalTextEntity::md;
    break;
  }
  if (parm.type != Param::paramLiteral) {
    if (!parseParam(allowParamLiteral, declInputLevel, parm))
      return 0;
  }
  Text text;
  parm.literalText.moveTo(text);
  if (bracketed != InternalTextEntity::none) {
    CString open;
    CString close;
    switch (bracketed) {
    case InternalTextEntity::starttag:
      open = syntax().delimGeneral(Syntax::dSTAGO);
      close = syntax().delimGeneral(Syntax::dTAGC);
      break;
    case InternalTextEntity::endtag:
      open = syntax().delimGeneral(Syntax::dETAGO);
      close = syntax().delimGeneral(Syntax::dTAGC);
      break;
    case InternalTextEntity::ms:
      open = syntax().delimGeneral(Syntax::dMDO);
      open += syntax().delimGeneral(Syntax::dDSO);
      close = syntax().delimGeneral(Syntax::dMSC);
      close += syntax().delimGeneral(Syntax::dMDC);
      break;
    case InternalTextEntity::md:
      open = syntax().delimGeneral(Syntax::dMDO);
      close = syntax().delimGeneral(Syntax::dMDC);
      break;
    default:
      CANNOT_HAPPEN();
    }
    text.insertChars(open, Location(new BracketOrigin(typeLocation,
						      BracketOrigin::open),
				    0));
    text.addChars(close, Location(new BracketOrigin(typeLocation,
						    BracketOrigin::close),
				  0));
    if (text.length() > syntax().litlen()
	&& text.length() - open.length() - close.length() <= syntax().litlen())
      message(Messages::bracketedLitlen,
	      NumberMessageArg(syntax().litlen()));
  }
  if (!parseParam(allowMdc, declInputLevel, parm))
    return 0;
  if (declType == Entity::parameterEntity
      && (dataType == Entity::cdata || dataType == Entity::sdata)) {
    message(Messages::internalParameterDataEntity,
	    StringMessageArg(name));
    return 1;
  }
  ResourcePointer<Entity> entity;
  switch (dataType) {
  case Entity::cdata:
    entity = new InternalCdataEntity(name, location, text);
    break;
  case Entity::sdata:
    entity = new InternalSdataEntity(name, location, text);
    break;
  case Entity::pi:
    entity = new PiEntity(name, declType, location, text);
    break;
  case Entity::sgmlText:
    entity = new InternalTextEntity(name, declType, location, text, bracketed);
    break;
  default:
    CANNOT_HAPPEN();
    break;
  }
  maybeDefineEntity(entity);
  return 1;
}

Boolean Parser::parseExternalEntity(const Location &location,
				    CString &name,
				    Entity::DeclType declType,
				    unsigned declInputLevel,
				    Param &parm)
{
  static AllowedParams
    allowSystemIdentifierEntityTypeMdc(Param::systemIdentifier,
				       Param::reservedName + Syntax::rSUBDOC,
				       Param::reservedName + Syntax::rCDATA,
				       Param::reservedName + Syntax::rSDATA,
				       Param::reservedName + Syntax::rNDATA,
				       Param::mdc);
  static AllowedParams
    allowEntityTypeMdc(Param::reservedName + Syntax::rSUBDOC,
		       Param::reservedName + Syntax::rCDATA,
		       Param::reservedName + Syntax::rSDATA,
		       Param::reservedName + Syntax::rNDATA,
		       Param::mdc);
  
  ExternalId id;
  if (!parseExternalId(allowSystemIdentifierEntityTypeMdc, allowEntityTypeMdc,
		       declInputLevel, parm, id))
    return 0;
  if (parm.type == Param::mdc) {
    maybeDefineEntity(new ExternalTextEntity(name, declType, location, id));
    return 1;
  }
  ResourcePointer<Entity> entity;
  if (parm.type == Param::reservedName + Syntax::rSUBDOC) {
    if (sd().subdoc() == 0)
      message(Messages::subdocEntity, StringMessageArg(name));
    if (!parseParam(allowMdc, declInputLevel, parm))
      return 0;
    entity = new SubdocEntity(name, location, id);
  }
  else {
    Entity::DataType dataType;
    switch (parm.type) {
    case Param::reservedName + Syntax::rCDATA:
      dataType = Entity::cdata;
      break;
    case Param::reservedName + Syntax::rSDATA:
      dataType = Entity::sdata;
      break;
    case Param::reservedName + Syntax::rNDATA:
      dataType = Entity::ndata;
      break;
    default:
      CANNOT_HAPPEN();
    }
    if (!parseParam(allowName, declInputLevel, parm))
      return 0;
    ConstResourcePointer<Notation> notation;
    if (haveDefDtd())
      notation = lookupCreateNotation(parm.token);
    else {
      const Dtd *dtd = (defLpd().type() == Lpd::simple
			? (haveBaseDtd() ? &baseDtd() : 0)
			: defComplexLpd().sourceDtd().pointer());
      if (dtd) {
	notation = dtd->lookupNotation(parm.token);
	if (notation.isNull())
	  message(Messages::notationUndefinedSourceDtd,
		  StringMessageArg(parm.token));
      }
      if (notation.isNull())
	notation = new Notation(parm.token);
    }
    if (!parseParam(allowDsoMdc, declInputLevel, parm))
      return 0;
    AttributeList attributes(notation->attributeDef());
    if (parm.type == Param::dso) {
      if (attributes.length() == 0)
	message(Messages::notationNoAttributes,
		StringMessageArg(notation->name()));
      Token closeToken;
      if (!parseAttributeSpec(1, attributes, closeToken))
	return 0;
      if (attributes.nSpec() == 0)
	message(Messages::emptyDataAttributeSpec);
      if (!parseParam(allowMdc, declInputLevel, parm))
	return 0;
    }
    else
      attributes.finish(*this);
    entity = new ExternalDataEntity(name, dataType, location, id, notation,
				    attributes);
  }
  if (declType == Entity::parameterEntity) {
    message(Messages::externalParameterDataSubdocEntity,
	    StringMessageArg(name));
    return 1;
  }
  maybeDefineEntity(entity);
  return 1;
}

Notation *Parser::lookupCreateNotation(const CString &name)
{
  ResourcePointer<Notation> nt = defDtd().lookupNotation(name);
  if (nt.isNull()) {
    nt = new Notation(name);
    defDtd().insertNotation(nt);
  }
  return nt.pointer();
}

void Parser::maybeDefineEntity(const ResourcePointer<Entity> &entity)
{
  EntitySet &defEntitySet = (haveDefDtd()
			     ? (EntitySet &)defDtd() 
			     : (EntitySet &)defLpd());
  if (entity->name().length() == 0) {
    // the default entity
    if (!defEntitySet.defaultEntity().isNull()) {
      if (options().warnDuplicateEntity)
	message(Messages::duplicateEntityDeclaration,
		StringMessageArg(syntax().rniReservedName(Syntax::rDEFAULT)));
    }
    else
      defEntitySet.setDefaultEntity(entity);
  }
  ResourcePointer<Entity> prevEntity = defEntitySet.insertEntity(entity);
  if (!prevEntity.isNull() && options().warnDuplicateEntity)
    message(entity->declType() == Entity::parameterEntity
	    ? Messages::duplicateParameterEntityDeclaration
	    : Messages::duplicateEntityDeclaration,
	    StringMessageArg(entity->name()));
  if (entity->asExternalEntity())
    eventHandler()
      .externalEntityDecl(new (eventAllocator())
			  ExternalEntityDeclEvent(entity,
						  !prevEntity.isNull()));
}

Boolean Parser::parseShortrefDecl()
{
  if (haveBaseDtd())
    message(Messages::shortrefOnlyInBaseDtd);

  unsigned declInputLevel = inputLevel();
  Param parm;

  if (!parseParam(allowName, declInputLevel, parm))
    return 0;
  ShortReferenceMap *map = lookupCreateMap(parm.token);
  int valid = 1;
  if (map->defined()) {
    message(Messages::duplicateShortrefDeclaration,
	    StringMessageArg(parm.token));
    valid = 0;
  }
  if (!parseParam(allowParamLiteral, declInputLevel, parm))
    return 0;
  Vector<CString> vec(instanceSyntax().nDelimShortref());
  do {
    CString delim(parm.literalText.string());
    const SubstTable<Char> *table = instanceSyntax().generalSubstTable();
    for (size_t i = 0; i < delim.length(); i++)
      table->subst(delim[i]);
    int srIndex;
    if (!instanceSyntax().lookupShortref(delim, &srIndex)) {
      message(Messages::unknownShortrefDelim, StringMessageArg(delim));
      valid = 0;
    }
    else
      defDtd().noteShortrefUsed(srIndex);
    static const AllowedParams allowEntityName(Param::entityName);
    if (!parseParam(allowEntityName, declInputLevel, parm))
      return 0;
    if (valid) {
      if (vec[srIndex].length() > 0) {
	message(Messages::delimDuplicateMap, StringMessageArg(parm.token));
	valid = 0;
      }
      else
	parm.token.moveTo(vec[srIndex]);
    }
    static AllowedParams allowParamLiteralMdc(Param::paramLiteral, Param::mdc);
    if (!parseParam(allowParamLiteralMdc, declInputLevel, parm))
      return 0;
  } while (parm.type != Param::mdc);
  if (valid)
    map->setNameMap(vec);
  return 1;
}

ShortReferenceMap *Parser::lookupCreateMap(const CString &name)
{
  ShortReferenceMap *map = defDtd().lookupShortReferenceMap(name);
  if (!map) {
    map = new ShortReferenceMap(name);
    defDtd().insertShortReferenceMap(map);
  }
  return map;
}

Boolean Parser::parseUsemapDecl()
{
  if (haveBaseDtd() && haveDefDtd())
    message(Messages::usemapOnlyInBaseDtd);

  unsigned declInputLevel = inputLevel();
  Param parm;
  
  static AllowedParams
    allowNameEmpty(Param::name,
		   Param::indicatedReservedName + Syntax::rEMPTY);
  if (!parseParam(allowNameEmpty, declInputLevel, parm))
    return 0;
  const ShortReferenceMap *map;
  if (parm.type == Param::name) {
    if (inInstance()) {
      map = currentDtd().lookupShortReferenceMap(parm.token);
      if (!map)
	message(Messages::undefinedShortrefMapInstance,
		StringMessageArg(parm.token));
    }
    else
      map = lookupCreateMap(parm.token);
  }
  else
    map = &theEmptyMap;
  static AllowedParams
    allowNameNameGroupMdc(Param::name, Param::nameGroup, Param::mdc);
  if (!parseParam(allowNameNameGroupMdc, declInputLevel, parm))
    return 0;
  if (parm.type != Param::mdc) {
    if (inInstance())
      message(Messages::usemapAssociatedElementTypeInstance);
    else if (map) {
      if (parm.type == Param::name) {
	ElementType *e = lookupCreateElement(parm.token);
	if (!e->map())
	  e->setMap(map);
      }
      else {
	for (size_t i = 0; i < parm.nameTokenVector.length(); i++) {
	  ElementType *e
	    = lookupCreateElement(parm.nameTokenVector[i].name);
	  if (!e->map())
	    e->setMap(map);
	}
      }
    }
    if (!parseParam(allowMdc, declInputLevel, parm))
      return 0;
  }
  else {
    if (!inInstance())
      message(Messages::usemapAssociatedElementTypeDtd);
    else if (map) {
      if (map != &theEmptyMap && !map->defined())
	message(Messages::undefinedShortrefMapInstance,
		StringMessageArg(map->name()));
      else
	currentElement().setMap(map);
    }
  }
  return 1;
}

Boolean Parser::parseDoctypeDeclStart()
{
  if (hadDtd() && !sd().concur() && !sd().explicit())
    message(Messages::multipleDtds);
  if (hadLpd())
    message(Messages::dtdAfterLpd);
  unsigned declInputLevel = inputLevel();
  Param parm;
  Location location(currentLocation());
  
  if (!parseParam(allowName, declInputLevel, parm))
    return 0;
  CString name;
  parm.token.moveTo(name);
  if (!lookupDtd(name).isNull())
    message(Messages::duplicateDtd, StringMessageArg(name));
  static AllowedParams
    allowPublicSystemDsoMdc(Param::reservedName + Syntax::rPUBLIC,
			    Param::reservedName + Syntax::rSYSTEM,
			    Param::dso,
			    Param::mdc);
  if (!parseParam(allowPublicSystemDsoMdc, declInputLevel, parm))
    return 0;
  ConstResourcePointer<Entity> entity;
  if (parm.type == Param::reservedName + Syntax::rPUBLIC
      || parm.type == Param::reservedName + Syntax::rSYSTEM) {
    static AllowedParams allowSystemIdentifierDsoMdc(Param::systemIdentifier,
						     Param::dso, Param::mdc);
    ExternalId id;
    if (!parseExternalId(allowSystemIdentifierDsoMdc, allowDsoMdc,
			 declInputLevel, parm, id))
      return 0;
    entity = new ExternalTextEntity(name, Entity::doctype, location, id);
    eventHandler()
      .externalEntityDecl(new (eventAllocator())
			  ExternalEntityDeclEvent(entity, 0));
  }
  else if (parm.type == Param::mdc) {
    message(Messages::noDtdSubset, StringMessageArg(name));
    return 1;
  }
  eventHandler().startDtd(new (eventAllocator())
			  StartDtdEvent(name, location));
  startDtd(name);
  if (parm.type == Param::mdc) {
    size_t tokenLength = currentInput()->currentTokenLength();
    // unget the mdc
    currentInput()->ungetToken();
    // reference the entity
    ResourcePointer<EntityOrigin> origin
      = new (internalAllocator()) EntityOrigin(entity, currentLocation(), 0);
    entity->dsReference(*this, origin);
    if (inputLevel() == 1) {	// reference failed
      currentInput()->endToken(tokenLength);
      doEndDtd();
      return 1;
    }
  }
  else if (!entity.isNull())
    setDsEntity(entity);
  setDoFunction(&Parser::doDeclSubset);
  return 1;
}

Boolean Parser::parseDoctypeDeclEnd()
{
  doEndDtd();
  Param parm;
  // End DTD before parsing final param so parameter entity reference
  // not allowed between ] and >.
  return parseParam(allowMdc, inputLevel(), parm);
}

void Parser::doEndDtd()
{
  checkDtd(defDtd());
  eventHandler().endDtd(new (eventAllocator()) EndDtdEvent(defDtdPointer(),
							   currentLocation()));
  endDtd();
}

void Parser::checkDtd(Dtd &dtd)
{
  Dtd::ElementTypeIter elementIter(dtd.elementTypeIter());
  ElementType *p;
  ConstResourcePointer<ElementDefinition> def;
  int i = 0;
  while ((p = elementIter.next()) != 0) {
    if (p->definition() == 0) {
      if (options().warnUndefinedElement)
	message(Messages::dtdUndefinedElement, StringMessageArg(p->name()));
      if (def.isNull())
	def = new ElementDefinition(currentLocation(),
				    ElementDefinition::undefinedIndex,
				    0,
				    ElementDefinition::any);
      p->setElementDefinition(def, i++);
    }
    const ShortReferenceMap *map = p->map();
    if (map != 0 && map != &theEmptyMap && !map->defined()) {
      message(Messages::undefinedShortrefMapDtd,
	      StringMessageArg(map->name()),
	      StringMessageArg(p->name()));
      p->setMap(0);
    }
  }
  EntitySet::ConstEntityIter entityIter(((const Dtd &)dtd).generalEntityIter());
  for (;;) {
    ConstResourcePointer<Entity> entity(entityIter.next());
    if (entity.isNull())
      break;
    const ExternalDataEntity *external = entity->asExternalDataEntity();
    if (external) {
      const Notation *notation = external->notation();
      if (!notation->defined()) {
	message(Messages::entityNotationUndefined,
		StringMessageArg(notation->name()),
		StringMessageArg(external->name()));
      }
    }
  }
  Dtd::ConstNotationIter notationIter(dtd.notationIter());
  for (;;) {
    ConstResourcePointer<Notation> notation(notationIter.next());
    if (notation.isNull())
      break;
    if (!notation->defined() && !notation->attributeDef().isNull())
      message(Messages::attlistNotationUndefined,
	      StringMessageArg(notation->name()));
  }
  Dtd::ShortReferenceMapIter mapIter(dtd.shortReferenceMapIter());
  int nDelimShortref = instanceSyntax().nDelimShortref();
  for (;;) {
    ShortReferenceMap *map = mapIter.next();
    if (!map)
      break;
    Vector<ConstResourcePointer<Entity> > entityMap(nDelimShortref);
    for (int i = 0; i < nDelimShortref; i++) {
      const CString *entityName = map->entityName(i);
      if (entityName) {
	Boolean defaulted;
	ConstResourcePointer<Entity> entity
	  = lookupEntity(0, *entityName, 0, defaulted);
	if (entity.isNull())
	  message(Messages::mapEntityUndefined,
		  StringMessageArg(*entityName),
		  StringMessageArg(map->name()));
	else {
	  if (defaulted && options().warnDefaultEntityReference)
	    message(Messages::mapDefaultEntity,
		    StringMessageArg(*entityName),
		    StringMessageArg(map->name()));
	  entityMap[i] = entity;
	}
      }
    }
    map->setEntityMap(entityMap);
  }
}

Boolean Parser::parseLinktypeDeclStart()
{
  if (!haveBaseDtd())
    message(Messages::lpdBeforeBaseDtd);
  unsigned declInputLevel = inputLevel();
  Param parm;
  Location location(currentLocation());
  
  if (!parseParam(allowName, declInputLevel, parm))
    return 0;
  CString name;
  parm.token.moveTo(name);
  if (!lookupDtd(name).isNull())
    message(Messages::duplicateDtdLpd, StringMessageArg(name));
  else if (!lookupLpd(name).isNull())
    message(Messages::duplicateLpd, StringMessageArg(name));
  static AllowedParams
    allowSimpleName(Param::indicatedReservedName + Syntax::rSIMPLE,
		    Param::name);
  if (!parseParam(allowSimpleName, declInputLevel, parm))
    return 0;
  Boolean simple;
  ConstResourcePointer<Dtd> sourceDtd;
  if (parm.type == Param::indicatedReservedName + Syntax::rSIMPLE)
    simple = 1;
  else {
    simple = 0;
    sourceDtd = lookupDtd(parm.token);
    if (sourceDtd.isNull())
      message(Messages::noSuchDtd, StringMessageArg(parm.token));
  }
  static AllowedParams
    allowImpliedName(Param::indicatedReservedName + Syntax::rIMPLIED,
		     Param::name);
  if (!parseParam(allowImpliedName, declInputLevel, parm))
    return 0;
  ConstResourcePointer<Dtd> resultDtd;
  Boolean implied = 0;
  if (parm.type == Param::indicatedReservedName + Syntax::rIMPLIED) {
    if (simple) {
      if (!sd().simple())
	message(Messages::simpleLinkFeature);
    }
    else {
      implied = 1;
      if (!sd().implicit())
	message(Messages::implicitLinkFeature);
    }
  }
  else {
    if (simple)
      message(Messages::simpleLinkResultNotImplied);
    else {
      if (!sd().explicit())
	message(Messages::explicitLinkFeature);
      resultDtd = lookupDtd(parm.token);
      if (resultDtd.isNull())
	message(Messages::noSuchDtd, StringMessageArg(parm.token));
    }
  }
  static AllowedParams
    allowPublicSystemDsoMdc(Param::reservedName + Syntax::rPUBLIC,
			    Param::reservedName + Syntax::rSYSTEM,
			    Param::dso,
			    Param::mdc);
  if (!parseParam(allowPublicSystemDsoMdc, declInputLevel, parm))
    return 0;
  ConstResourcePointer<Entity> entity;
  if (parm.type == Param::reservedName + Syntax::rPUBLIC
      || parm.type == Param::reservedName + Syntax::rSYSTEM) {
    static AllowedParams allowSystemIdentifierDsoMdc(Param::systemIdentifier,
						     Param::dso, Param::mdc);
    ExternalId id;
    if (!parseExternalId(allowSystemIdentifierDsoMdc, allowDsoMdc,
			 declInputLevel, parm, id))
      return 0;
    entity = new ExternalTextEntity(name, Entity::linktype, location, id);
    eventHandler()
      .externalEntityDecl(new (eventAllocator())
			  ExternalEntityDeclEvent(entity, 0));
  }
  else if (parm.type == Param::mdc) {
    message(Messages::noLpdSubset, StringMessageArg(name));
    return 1;
  }
  ResourcePointer<Lpd> lpd;
  if (simple)
    lpd = new SimpleLpd(name, location);
  else
    lpd = new ComplexLpd(name,
			 implied ? Lpd::implicit : Lpd::explicit,
			 location,
			 syntax(),
			 sourceDtd,
			 resultDtd);
  if (haveBaseDtd() && shouldActivateLink(name)) {
    size_t nActive = nActiveLink();
    if (simple) {
      size_t nSimple = 0;
      for (size_t i = 0; i < nActive; i++)
	if (activeLpd(i).type() == Lpd::simple)
	  nSimple++;
      if (nSimple == sd().simple())
	message(Messages::simpleLinkCount, NumberMessageArg(sd().simple()));
      lpd->activate();
    }
    else {
      Boolean haveImplicit = 0;
      Boolean haveExplicit = 0;
      size_t i;
      for (i = 0; i < nActive; i++) {
	if (activeLpd(i).type() == Lpd::implicit)
	  haveImplicit = 1;
	else if (activeLpd(i).type() == Lpd::explicit)
	  haveExplicit = 1;
      }
      const Dtd *sourceDtd
	= ((ComplexLpd *)lpd.pointer())->sourceDtd().pointer();
      if (implied && haveImplicit)
	message(Messages::oneImplicitLink);
      else if (sd().explicit() <= 1
	       && (!sourceDtd || sourceDtd->name() != baseDtd().name()))
	message(sd().explicit() == 0
		? Messages::explicitNoRequiresSourceTypeBase
		: Messages::explicit1RequiresSourceTypeBase,
		StringMessageArg(lpd->name()));
      else if (sd().explicit() == 1 && haveExplicit && !implied)
	message(Messages::duplicateExplicitChain);
      else if (haveExplicit || haveImplicit
	       || !sourceDtd || sourceDtd->name() != baseDtd().name())
	message(Messages::sorryLink, StringMessageArg(lpd->name()));
      else
	lpd->activate();
    }
  }
  startLpd(lpd);
  if (parm.type == Param::mdc) {
    size_t tokenLength = currentInput()->currentTokenLength();
    // unget the mdc
    currentInput()->ungetToken();
    // reference the entity
    ResourcePointer<EntityOrigin> origin
      = new (internalAllocator()) EntityOrigin(entity, currentLocation(), 0);
    entity->dsReference(*this, origin);
    if (inputLevel() == 1) {	// reference failed
      currentInput()->endToken(tokenLength);
      doEndLpd();
      return 1;
    }
  }
  else if (!entity.isNull())
    setDsEntity(entity);
  setDoFunction(&Parser::doDeclSubset);
  return 1;
}

Boolean Parser::parseLinktypeDeclEnd()
{
  unsigned declInputLevel = inputLevel();
  Param parm;
  // Can have parameter entity reference between ] and >.
  Boolean result = parseParam(allowMdc, declInputLevel, parm);
  doEndLpd();
  return result;
}

void Parser::doEndLpd()
{
  if (defLpd().type() != Lpd::simple) {
    if (!defComplexLpd().initialLinkSet()->defined())
      message(Messages::noInitialLinkSet,
	      StringMessageArg(defLpd().name()));
    ComplexLpd::ConstLinkSetIter iter = defComplexLpd().linkSetIter();
    const LinkSet *linkSet;
    while ((linkSet = iter.next()) != 0)
      if (!linkSet->defined())
	message(Messages::undefinedLinkSet, StringMessageArg(linkSet->name()));
  }
  endLpd();
}



Boolean Parser::parseLinkDecl()
{
  return parseLinkSet(0);
}

Boolean Parser::parseIdlinkDecl()
{
  return parseLinkSet(1);
}

// This will only get called if we're defining a complex lpd.

Boolean Parser::parseLinkSet(Boolean idlink)
{
  if (defLpd().type() == Lpd::simple) {
    message(idlink ? Messages::idlinkDeclSimple : Messages::linkDeclSimple);
    return 0;
  }
  if (idlink) {
    if (defComplexLpd().hadIdLinkSet())
      message(Messages::duplicateIdLinkSet);
    else
      defComplexLpd().setHadIdLinkSet();
  }
  unsigned declInputLevel = inputLevel();
  Param parm;
  
  Boolean explicit = (defLpd().type() == Lpd::explicit);
  LinkSet *linkSet;
  if (idlink) {
    if (!parseParam(allowName, declInputLevel, parm))
      return 0;
    linkSet = 0;
  }
  else {
    static AllowedParams
      allowNameInitial(Param::name,
		       Param::indicatedReservedName + Syntax::rINITIAL);
    if (!parseParam(allowNameInitial, declInputLevel, parm))
      return 0;
    if (parm.type == Param::name)
      linkSet = lookupCreateLinkSet(parm.token);
    else
      linkSet = defComplexLpd().initialLinkSet();
    if (linkSet->defined())
      message(Messages::duplicateLinkSet, StringMessageArg(linkSet->name()));
    static AllowedParams
      allowExplicitLinkRule(Param::name,
			    Param::nameGroup,
			    Param::indicatedReservedName + Syntax::rIMPLIED);
    if (!parseParam(explicit ? allowExplicitLinkRule : allowNameNameGroup,
		    declInputLevel, parm))
      return 0;
  }

  do {
    CString id;
    if (idlink) {
      parm.token.moveTo(id);
      if (!parseParam(explicit ? allowExplicitLinkRuleMdc : allowNameNameGroupMdc,
		      declInputLevel, parm))
	return 0;
    }
    if (parm.type == Param::indicatedReservedName + Syntax::rIMPLIED) {
      if (!parseParam(allowName, declInputLevel, parm))
	return 0;
      Boolean resultImplied;
      const ElementType *resultType;
      AttributeList resultAttributes;
      if (!parseResultElementSpec(declInputLevel,
				  parm,
				  idlink,
				  resultImplied,
				  resultType,
				  resultAttributes))
	return 0;
      if (resultType) {
	const AttributeList *dummy;
	if (linkSet->impliedResultAttributes(resultType, dummy))
	  message(Messages::duplicateImpliedResult,
		  StringMessageArg(resultType->name()));
	else
	  linkSet->addImplied(resultType, resultAttributes);
      }
    }
    else {
      SourceLinkRule *linkRule = 0;
      IdLinkRule idLinkRule;
      ResourcePointer<SourceLinkRuleResource> linkRuleResource;
      if (idlink)
	linkRule = &idLinkRule;
      else {
	linkRuleResource = new SourceLinkRuleResource;
	linkRule = linkRuleResource.pointer();
      }
      Vector<const ElementType *> assocElementTypes;
      if (parm.type == Param::name) {
	assocElementTypes.init(1);
	assocElementTypes[0] = lookupSourceElementType(parm.token);
      }
      else {
	assocElementTypes.init(parm.nameTokenVector.length());
	for (size_t i = 0; i < assocElementTypes.length(); i++)
	  assocElementTypes[i]
	    = lookupSourceElementType(parm.nameTokenVector[i].name);
      }
      static AllowedParams
	allow2i(Param::indicatedReservedName + Syntax::rUSELINK,
		Param::indicatedReservedName + Syntax::rPOSTLINK,
		Param::dso,
		Param::mdc,
		Param::name,
		Param::nameGroup);
      static AllowedParams
	allow2id(Param::indicatedReservedName + Syntax::rUSELINK,
		 Param::indicatedReservedName + Syntax::rPOSTLINK,
		 Param::dso,
		 Param::mdc,
		 Param::name);
      static AllowedParams
	allow2e(Param::indicatedReservedName + Syntax::rUSELINK,
		Param::indicatedReservedName + Syntax::rPOSTLINK,
		Param::dso,
		Param::name,
		Param::indicatedReservedName + Syntax::rIMPLIED);

      if (!parseParam(explicit
		      ? allow2e
		      : (idlink ? allow2id : allow2i), declInputLevel, parm))
	return 0;
      if (parm.type == Param::indicatedReservedName + Syntax::rUSELINK) {
	static AllowedParams
	  allowLinkSetEmpty(Param::name,
			    Param::indicatedReservedName + Syntax::rINITIAL,
			    Param::indicatedReservedName + Syntax::rEMPTY);
	if (!parseParam(allowLinkSetEmpty, declInputLevel, parm))
	  return 0;
	const LinkSet *uselink;
	if (parm.type == Param::name)
	  uselink = lookupCreateLinkSet(parm.token);
	else if (parm.type == Param::indicatedReservedName + Syntax::rINITIAL)
	  uselink = defComplexLpd().initialLinkSet();
	else
	  uselink = defComplexLpd().emptyLinkSet();
	linkRule->setUselink(uselink);
	static AllowedParams
	  allow3i(Param::indicatedReservedName + Syntax::rPOSTLINK,
		  Param::dso,
		  Param::mdc,
		  Param::name,
		  Param::nameGroup);
	static AllowedParams
	  allow3id(Param::indicatedReservedName + Syntax::rPOSTLINK,
		   Param::dso,
		   Param::mdc,
		   Param::name);
	static AllowedParams
	  allow3e(Param::indicatedReservedName + Syntax::rPOSTLINK,
		  Param::dso,
		  Param::name,
		  Param::indicatedReservedName + Syntax::rIMPLIED);
	
	if (!parseParam(explicit
			? allow3e
			: (idlink ? allow3id : allow3i),
			declInputLevel, parm))
	  return 0;
      }
      if (parm.type == Param::indicatedReservedName + Syntax::rPOSTLINK) {
	if (!parseParam(allowLinkSetSpec, declInputLevel, parm))
	  return 0;
	const LinkSet *postlink;
	if (parm.type == Param::indicatedReservedName + Syntax::rRESTORE)
	  linkRule->setPostlinkRestore();
	else {
	  if (parm.type == Param::name)
	    postlink = lookupCreateLinkSet(parm.token);
	  else if (parm.type
		   == Param::indicatedReservedName + Syntax::rINITIAL)
	    postlink = defComplexLpd().initialLinkSet();
	  else
	    postlink = defComplexLpd().emptyLinkSet();
	  linkRule->setPostlink(postlink);
	}
	static AllowedParams
	  allow4i(Param::dso,
		  Param::mdc,
		  Param::name,
		  Param::nameGroup);
	static AllowedParams
	  allow4id(Param::dso,
		   Param::mdc,
		   Param::name);
	static AllowedParams
	  allow4e(Param::dso,
		  Param::name,
		  Param::indicatedReservedName + Syntax::rIMPLIED);
	if (!parseParam(explicit
			? allow4e
			: (idlink ? allow4id : allow4i),
			declInputLevel, parm))
	  return 0;
      }
      AttributeList attributes;
      ConstResourcePointer<AttributeDefinitionList> attDef;
      for (size_t i = 0; i < assocElementTypes.length(); i++) {
	const ElementType *e = assocElementTypes[i];
	if (e) {
	  if (i == 0)
	    attDef = defComplexLpd().attributeDef(e);
	  else if (attDef != defComplexLpd().attributeDef(e))
	    message(Messages::assocElementDifferentAtts);
	  // FIXME recover from this
	}
      }
      attributes.init(attDef);
      
      if (parm.type == Param::dso) {
	Token closeToken;
	if (!parseAttributeSpec(1, attributes, closeToken))
	  return 0;
	static AllowedParams
	  allow5e(Param::name,
		  Param::indicatedReservedName + Syntax::rIMPLIED);
	if (!parseParam(explicit
			? allow5e
			: (idlink ? allowNameMdc : allowNameNameGroupMdc),
			declInputLevel, parm))
	  return 0;
      }
      else
	attributes.finish(*this);
      linkRule->setLinkAttributes(attributes);
      if (explicit) {
	Boolean resultImplied;
	const ElementType *resultType;
	AttributeList resultAttributes;
	if (!parseResultElementSpec(declInputLevel,
				    parm,
				    idlink,
				    resultImplied,
				    resultType,
				    resultAttributes))
	  return 0;
	if (!resultImplied)
	  linkRule->setResult(resultType, resultAttributes);
      }
      // Install the link rule.
      if (idlink) {
	idLinkRule.setAssocElementTypes(assocElementTypes);
	addIdLinkRule(id, idLinkRule);
      }
      else {
	if (!linkSet->defined()) {
	  for (size_t i = 0; i < assocElementTypes.length(); i++)
	    if (assocElementTypes[i])
	      addLinkRule(linkSet, assocElementTypes[i], linkRuleResource);
	}
      }
    }
  } while (parm.type != Param::mdc);
  if (linkSet)
    linkSet->setDefined();
  return 1;
}

void Parser::addIdLinkRule(const CString &id,
			   IdLinkRule &rule)
{
  IdLinkRuleGroup *group = defComplexLpd().lookupCreateIdLink(id);
  size_t nRules = group->nLinkRules();
  if ((nRules == 1 && group->linkRule(0).attributes().nSpec() == 0)
      || nRules >= 1 && rule.attributes().nSpec() == 0)
    message(Messages::multipleIdLinkRuleAttribute,
	    StringMessageArg(id));
  group->addLinkRule(rule);
}

void Parser::addLinkRule(LinkSet *linkSet,
			 const ElementType *sourceElement,
			 const ConstResourcePointer<SourceLinkRuleResource> &linkRule)
{
  size_t nRules = linkSet->nLinkRules(sourceElement);
  if ((nRules == 1
       && linkSet->linkRule(sourceElement, 0).attributes().nSpec() == 0)
      || nRules >= 1 && linkRule->attributes().nSpec() == 0)
    message(Messages::multipleLinkRuleAttribute,
	    StringMessageArg(sourceElement->name()));
  linkSet->addLinkRule(sourceElement, linkRule);
}

class ResultAttributeSpecModeSetter {
public:
  ResultAttributeSpecModeSetter(ParserState *state) : state_(state) {
    state_->setResultAttributeSpecMode();
  }
  ~ResultAttributeSpecModeSetter() { clear(); }
  void clear() {
    if (state_) {
      state_->clearResultAttributeSpecMode();
      state_ = 0;
    }
  }
private:
  ParserState *state_;
};

Boolean Parser::parseResultElementSpec(unsigned declInputLevel,
				       Param &parm,
				       Boolean idlink,
				       Boolean &implied,
				       const ElementType *&resultType,
				       AttributeList &attributes)
{
  if (parm.type == Param::indicatedReservedName + Syntax::rIMPLIED) {
    if (!parseParam(idlink ? allowNameMdc : allowExplicitLinkRuleMdc,
		    declInputLevel, parm))
      return 0;
    implied = 1;
  }
  else {
    implied = 0;
    resultType = lookupResultElementType(parm.token);
    static AllowedParams
      allow(Param::dso,
	    Param::mdc,
	    Param::name,
	    Param::nameGroup,
	    Param::indicatedReservedName + Syntax::rIMPLIED);
    static AllowedParams
      allowNameDsoMdc(Param::dso,
		      Param::mdc,
		      Param::name);
    if (!parseParam(idlink ? allowNameDsoMdc : allow,
		    declInputLevel, parm))
      return 0;
    if (resultType)
      attributes.init(resultType->attributeDef());
    if (parm.type == Param::dso) {
      ResultAttributeSpecModeSetter modeSetter(this);
      Token closeToken;
      if (!parseAttributeSpec(1, attributes, closeToken))
	return 0;
      modeSetter.clear();
      if (attributes.nSpec() == 0)
	message(Messages::emptyResultAttributeSpec);
      if (!parseParam(idlink ? allowNameMdc : allowExplicitLinkRuleMdc,
		      declInputLevel, parm))
	return 0;
    }
    else {
      // For entity and notation attributes.
      ResultAttributeSpecModeSetter modeSetter(this);
      attributes.finish(*this);
      modeSetter.clear();
    }
  }
  return 1;
}

const ElementType *Parser::lookupSourceElementType(const CString &name)
{
  const Dtd *dtd = (defLpd().type() == Lpd::simple
		    ? (haveBaseDtd() ? &baseDtd() : 0)
		    : defComplexLpd().sourceDtd().pointer());
  if (!dtd)
    return 0;
  const ElementType *e = dtd->lookupElementType(name);
  if (!e)
    message(Messages::noSuchSourceElement, StringMessageArg(name));
  return e;
}

const ElementType *Parser::lookupResultElementType(const CString &name)
{
  const Dtd *dtd = defComplexLpd().resultDtd().pointer();
  if (!dtd)
    return 0;
  const ElementType *e = dtd->lookupElementType(name);
  if (!e)
    message(Messages::noSuchResultElement, StringMessageArg(name));
  return e;
}

Boolean Parser::parseUselinkDecl()
{
  unsigned declInputLevel = inputLevel();
  Param parm;
  Location location(currentLocation());
  if (!parseParam(allowLinkSetSpec, declInputLevel, parm))
    return 0;
  Param parm2;
  if (!parseParam(allowName, declInputLevel, parm2))
    return 0;
  CString linkType;
  parm2.token.moveTo(linkType);
  if (!parseParam(allowMdc, declInputLevel, parm2))
    return 0;
  ConstResourcePointer<Lpd> lpd = lookupLpd(linkType);
  if (lpd.isNull()) 
    message(Messages::uselinkBadLinkType, StringMessageArg(linkType));
  else if (lpd->type() == Lpd::simple)
    message(Messages::uselinkSimpleLpd, StringMessageArg(linkType));
  else {
    const ComplexLpd *complexLpd = (const ComplexLpd *)lpd.pointer();
    const LinkSet *linkSet;
    Boolean restore = 0;
    if (parm.type == Param::name) {
      linkSet = complexLpd->lookupLinkSet(parm.token);
      if (!linkSet) {
	message(Messages::uselinkBadLinkSet,
		StringMessageArg(complexLpd->name()),
		StringMessageArg(parm.token));
	return 1;
      }
    }
    else if (parm.type == Param::indicatedReservedName + Syntax::rINITIAL)
      linkSet = complexLpd->initialLinkSet();
    else if (parm.type == Param::indicatedReservedName + Syntax::rEMPTY)
      linkSet = complexLpd->emptyLinkSet();
    else {
      linkSet = 0;
      restore = 1;
    }
    if (lpd->active())
      eventHandler().uselink(new (eventAllocator())
			     UselinkEvent(lpd, linkSet, restore, location));
  }
  return 1;
}

LinkSet *Parser::lookupCreateLinkSet(const CString &name)
{
  LinkSet *linkSet = defComplexLpd().lookupLinkSet(name);
  if (!linkSet) {
    linkSet = new LinkSet(name, defComplexLpd().sourceDtd().pointer());
    defComplexLpd().insertLinkSet(linkSet);
  }
  return linkSet;
}

Boolean Parser::parseMarkedSectionDeclStart()
{
  unsigned declInputLevel = inputLevel();
  Param parm;
  enum Status {
    include,
    rcdata,
    cdata,
    ignore
    } status = include;

  if (markedSectionLevel() == syntax().taglvl())
    message(Messages::markedSectionLevel,
	    NumberMessageArg(syntax().taglvl()));
  if (markedSectionSpecialLevel() > 0) {
    startMarkedSection();
    return 1;
  }
  static AllowedParams allowStatusDso(Param::dso,
				      Param::reservedName + Syntax::rCDATA,
				      Param::reservedName + Syntax::rRCDATA,
				      Param::reservedName + Syntax::rIGNORE,
				      Param::reservedName + Syntax::rINCLUDE,
				      Param::reservedName + Syntax::rTEMP);
  for (;;) {
    if (!parseParam(allowStatusDso, declInputLevel, parm))
      return 0;
    if (parm.type == Param::dso)
      break;
    switch (parm.type) {
    case Param::reservedName + Syntax::rCDATA:
      if (status < cdata)
	status = cdata;
      break;
    case Param::reservedName + Syntax::rRCDATA:
      if (status < rcdata)
	status = rcdata;
      break;
    case Param::reservedName + Syntax::rIGNORE:
      if (status < ignore)
	status = ignore;
      break;
    }
  }
  // FIXME this disallows
  // <!entity % e "include [ stuff ">
  // ...
  // <![ %e; ]]>
  // which I think is legal.

  if (inputLevel() > declInputLevel)
    message(Messages::parameterEntityNotEnded);
  switch (status) {
  case include:
    startMarkedSection();
    break;
  case cdata:
    startSpecialMarkedSection(cmsMode);
    break;
  case rcdata:
    startSpecialMarkedSection(rcmsMode);
    break;
  case ignore:
    startSpecialMarkedSection(imsMode);
    break;
  }
  return 1;
}

void Parser::handleMarkedSectionEnd()
{
  if (markedSectionLevel() == 0)
    message(Messages::markedSectionEnd);
  else
    endMarkedSection();
}

Boolean Parser::parseCommentDecl()
{
  if (!parseComment(comMode))
    return 0;
  for (;;) {
    Token token = getToken(mdMode);
    switch (token) {
    case tokenS:
      break;
    case tokenCom:
      if (!parseComment(comMode))
	return 0;
      break;
    case tokenMdc:
      goto done;
    case tokenEe:
      message(Messages::declarationLevel);
      return 0;
    case tokenUnrecognized:
      if (reportNonSgmlCharacter())
	break;
      // braces to work round Sun C++ 4.0 bug
      {
	message(Messages::commentDeclarationCharacter,
		StringMessageArg(currentToken()));
      }
      return 0;
    default:
      // braces to work round Sun C++ 4.0 bug
      {
	message(Messages::commentDeclInvalidToken,
		TokenMessageArg(token, mdMode, syntaxPointer(), sdPointer()));
      }
      return 0;
    }
  }
 done:
  return 1;
}

