// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#include "Parser.H"
#include "MessageArg.H"
#include "TokenMessageArg.H"
#include "token.H"
#include "macros.H"

void Parser::doInstanceStart()
{
  // FIXME check here that we have a valid dtd
  startInstance();
  outputLinks();
  compileInstanceModes();
  setDoFunction(&Parser::doContent);
  Token token = getToken(currentMode());
  switch (token) {
  case tokenEe:
  case tokenStagoNameStart:
  case tokenStagoTagc:
  case tokenStagoGrpo:
  case tokenEtagoNameStart:
  case tokenEtagoTagc:
  case tokenEtagoGrpo:
    break;
  default:
    if (sd().omittag()) {
      unsigned startImpliedCount = 0;
      unsigned attributeListIndex = 0;
      IList<Undo> undoList;
      IList<ElementEvent> eventList;
      if (!tryImplyTag(currentLocation(),
		       startImpliedCount,
		       attributeListIndex,
		       undoList,
		       eventList))
	CANNOT_HAPPEN();
      queueElementEvents(eventList);
    }
    else
      message(Messages::instanceStartOmittag);
  }
  currentInput()->ungetToken();
}

void Parser::outputLinks()
{
  size_t nLink = nActiveLink();
  for (size_t i = 0; i < nLink; i++)
    if (activeLpd(i).type() == Lpd::simple) {
      const SimpleLpd &lpd = (SimpleLpd &)activeLpd(i);
      AttributeList attributes(lpd.attributeDef());
      attributes.finish(*this);
      eventHandler().simpleLink(new (eventAllocator())
				SimpleLinkEvent(lpd.name(),
						attributes,
						lpd.location()));
    }
    else
      eventHandler().complexLink(
	  new (eventAllocator())
	       ComplexLinkEvent((ComplexLpd *)&activeLpd(i), // ugly
				activeLpds()));
}

void Parser::endInstance()
{
  // Do checking before popping entity stack so that there's a
  // current location for error messages.
  endAllElements();
  while (markedSectionLevel() > 0) {
    message(Messages::unclosedMarkedSection);
    endMarkedSection();
  }
  checkIdrefs();
  popInputStack();
  allDone();
}

void Parser::checkIdrefs()
{
  IdTableIter iter(idTableIter());
  Id *id;
  while ((id = iter.next()) != 0) {
    if (id->referenced() && !id->defined())
      message(Messages::missingId, StringMessageArg(id->name()),
	      id->refLocation());
  }
}

void Parser::doContent()
{
  do {
    Token token = getToken(currentMode());
    switch (token) {
    case tokenEe:
      if (inputLevel() == specialParseInputLevel()) {
	// FIXME have separate messages for each type of special parse
	// perhaps force end of marked section or element
	message(Messages::specialParseEntityEnd);
      }
      if (inputLevel() == 1) {
	endInstance();
	return;
      }
      if (afterDocumentElement())
	message(Messages::afterDocumentElementEntityEnd);
      popInputStack();
      break;
    case tokenCroDigit:
      {
	if (afterDocumentElement())
	  message(Messages::characterReferenceAfterDocumentElement);
	Char ch;
	Location loc;
	if (parseNumericCharRef(ch, loc)) {
	  noteData();
	  eventHandler().data(new (eventAllocator())
			      ImmediateDataEvent(Event::characterData,
						 &ch, 1, loc, 1));
	  break;
	}
      }
      break;
    case tokenCroNameStart:
      if (afterDocumentElement())
	  message(Messages::characterReferenceAfterDocumentElement);
      parseNamedCharRef();
      break;
    case tokenEroGrpo:
    case tokenEroNameStart:
      {
	if (afterDocumentElement())
	  message(Messages::entityReferenceAfterDocumentElement);
	ConstResourcePointer<Entity> entity;
	ResourcePointer<EntityOrigin> origin;
	if (parseEntityReference(0, token == tokenEroGrpo, entity, origin)) {
	  if (!entity.isNull()) {
	    if (entity->isCharacterData())
	      acceptPcdata(Location(origin.pointer(), 0));
	    if (inputLevel() == specialParseInputLevel())
	      entity->rcdataReference(*this, origin);
	    else
	      entity->contentReference(*this, origin);
	  }
	}
	noteMarkup();
      }
      break;
    case tokenEtagoNameStart:
      parseEndTag();
      break;
    case tokenEtagoTagc:
      parseEmptyEndTag();
      break;
    case tokenEtagoGrpo:
      parseGroupEndTag();
      break;
    case tokenMdoNameStart:
      Syntax::ReservedName name;
      Boolean result;
      unsigned startLevel;
      startLevel = inputLevel();
      if (parseDeclarationName(&name)) {
	switch (name) {
	case Syntax::rUSEMAP:
	  if (afterDocumentElement())
	    message(Messages::declarationAfterDocumentElement,
		    StringMessageArg(syntax().reservedName(name)));
	  result = parseUsemapDecl();
	  break;
	case Syntax::rUSELINK:
	  if (afterDocumentElement())
	    message(Messages::declarationAfterDocumentElement,
		    StringMessageArg(syntax().reservedName(name)));
	  result = parseUselinkDecl();
	  break;
	case Syntax::rDOCTYPE:
	case Syntax::rLINKTYPE:
	case Syntax::rELEMENT:
	case Syntax::rATTLIST:
	case Syntax::rENTITY:
	case Syntax::rNOTATION:
	case Syntax::rSHORTREF:
	case Syntax::rLINK:
	case Syntax::rIDLINK:
	  message(Messages::instanceDeclaration,
		  StringMessageArg(syntax().reservedName(name)));
	  result = 0;
	  break;
	default:
	  message(Messages::noSuchDeclarationType,
		  StringMessageArg(syntax().reservedName(name)));
	  result = 0;
	  break;
	}
      }
      else
	result = 0;
      if (!result)
	skipDeclaration(startLevel);
      noteMarkup();
      break;
    case tokenMdoMdc:
      // empty comment
      noteMarkup();
      break;
    case tokenMdoCom:
      parseCommentDecl();
      noteMarkup();
      break;
    case tokenMdoDso:
      if (afterDocumentElement())
	message(Messages::markedSectionAfterDocumentElement);
      parseMarkedSectionDeclStart();
      noteMarkup();
      break;
    case tokenMscMdc:
      handleMarkedSectionEnd();
      noteMarkup();
      break;
    case tokenNet:
      parseNullEndTag();
      break;
    case tokenPio:
      parseProcessingInstruction();
      break;
    case tokenStagoNameStart:
      parseStartTag();
      break;
    case tokenStagoTagc:
      parseEmptyStartTag();
      break;
    case tokenStagoGrpo:
      parseGroupStartTag();
      break;
    case tokenRe:
      acceptPcdata(currentLocation());
      queueRe(currentLocation());
      break;
    case tokenRs:
      acceptPcdata(currentLocation());
      noteRs();
      if (esisPlus())
	eventHandler().ignoredChars(new (eventAllocator())
				        RsEvent(currentInput()->currentTokenStart(),
						currentLocation(), 0));
      break;
    case tokenS:
      extendContentS();
      if (esisPlus())
	eventHandler().ignoredChars(new (eventAllocator())
				    IgnoredCharsEvent(currentInput()->currentTokenStart(),
						      currentInput()->currentTokenLength(),
						      currentLocation(),
						      0));
      break;
    case tokenIgnoredChar:
      extendData();
      if (esisPlus())
	eventHandler().ignoredChars(new (eventAllocator())
				    IgnoredCharsEvent(currentInput()->currentTokenStart(),
						      currentInput()->currentTokenLength(),
						      currentLocation(),
						      0));
      break;
    case tokenUnrecognized:
      reportNonSgmlCharacter();
      // fall through
    case tokenChar:
      parsePcdata();
      break;
    default:
      ASSERT(token >= tokenFirstShortref);
      handleShortref(token - tokenFirstShortref);
      break;
    }
  } while (eventQueueEmpty());
}

void Parser::skipDeclaration(unsigned startLevel)
{
  const unsigned skipMax = 250;
  unsigned skipCount = 0;
  for (;;) {
    Token token = getToken(mdMode);
    if (inputLevel() == startLevel)
      skipCount++;
    switch (token) {
    case tokenUnrecognized:
      (void)getChar();
      break;
    case tokenEe:
      if (inputLevel() <= startLevel)
	return;
      popInputStack();
      return;
    case tokenMdc:
      if (inputLevel() == startLevel)
	return;
      break;
    case tokenS:
      if (inputLevel() == startLevel && skipCount >= skipMax
	  && currentChar() == syntax().standardFunction(Syntax::fRE))
	return;
      break;
    default:
      break;
    }
  }
}

void Parser::handleShortref(int index)
{
  const ConstResourcePointer<Entity> &entity(currentElement().map()->entity(index));
  if (!entity.isNull()) {
    ResourcePointer<EntityOrigin> origin
      = new (internalAllocator())
          EntityOrigin(entity,
		       currentLocation(),
		       currentInput()->currentTokenLength());
    entity->contentReference(*this, origin);
    return;
  }
  InputSource *in = currentInput();
  size_t length = in->currentTokenLength();
  const Char *s = in->currentTokenStart();
  size_t i = 0;
  if (currentMode() == econMode || currentMode() == econnetMode) {
    // FIXME do this in advance (what about B sequence?)
    for (i = 0; i < length && syntax().isS(s[i]); i++)
      ;
    if (i > 0 && esisPlus())
      eventHandler().ignoredChars(new (eventAllocator())
				  IgnoredCharsEvent(s, i, currentLocation(),
						    // copy if more chars
						    i < length));
  }
  if (i < length) {
    Location location(currentLocation());
    location += i;
    s += i;
    length -= i;
    acceptPcdata(location);
    // FIXME speed this up
    for (; length > 0; location += 1, length--, s++) {
      if (*s == syntax().standardFunction(Syntax::fRS)) {
	noteRs();
	if (esisPlus())
	  eventHandler().ignoredChars(new (eventAllocator())
				      RsEvent(s, location, 0));
      }
      else if (*s == syntax().standardFunction(Syntax::fRE))
	queueRe(location);
      else {
	noteData();
	eventHandler().data(new (eventAllocator())
			    ImmediateDataEvent(Event::characterData, s, 1,
					       location, 0));
      }
    }
  }
}

void Parser::parsePcdata()
{
  extendData();
  acceptPcdata(currentLocation());
  noteData();
  eventHandler().data(new (eventAllocator())
		      ImmediateDataEvent(Event::characterData,
					 currentInput()->currentTokenStart(),
					 currentInput()->currentTokenLength(),
					 currentLocation(),
					 0));
}

void Parser::parseStartTag()
{
  InputSource *in = currentInput();
  Location location(in->currentLocation());
  in->discardInitial();
  extendNameToken(syntax().namelen(), Messages::nameLength);
  CString origName;
  if (esisPlus())
    getCurrentToken(origName);
  CString &name(nameBuffer());
  getCurrentToken(syntax().generalSubstTable(), name);
  const ElementType *e = currentDtd().lookupElementType(name);
  if (sd().rank()) {
    if (!e)
      e = completeRankStem(name);
    else if (e->isRankedElement())
      handleRankedElement(e);
  }
  if (!e) 
    e = lookupCreateUndefinedElement(name);
  ElementEvent::Minimization min;
  AttributeList *attributes = allocAttributeList(e->attributeDef(), 0);
  Token closeToken;
  if (parseStartTagClose(*attributes, closeToken)) {
    // tokenTagc will be much the most common case
    if (closeToken == tokenTagc)
      min = ElementEvent::complete;
    else {
      if (!sd().shorttag())
	message(Messages::minimizedStartTag);
      if (closeToken == tokenNet)
	min = ElementEvent::net;
      else {
	min = ElementEvent::unclosed;
	in->ungetToken();
      }
    }
    // The difference between the indices will be the difference
    // in offsets plus 1 for each named character reference.
    if (in->currentLocation().index() - location.index()
	> syntax().taglen())
      checkTaglen(location.index());
  }
  else
    min = ElementEvent::unclosed;	  
  acceptStartTag(e, 
		 new (eventAllocator())
		   StartElementEvent(e,
				     currentDtdPointer(),
				     location,
				     in->nextIndex(),
				     min,
				     attributes,
				     origName));
}

const ElementType *Parser::completeRankStem(const CString &name)
{
  const RankStem *rankStem = currentDtd().lookupRankStem(name);
  if (rankStem) {
    CString name(rankStem->name());
    if (!appendCurrentRank(name, rankStem))
      message(Messages::noCurrentRank, StringMessageArg(name));
    else
      return currentDtd().lookupElementType(name);
  }
  return 0;
}

void Parser::handleRankedElement(const ElementType *e)
{
  CString rankSuffix(e->definition()->rankSuffix());
  const RankStem *rankStem = e->rankedElementRankStem();
  for (size_t i = 0; i < rankStem->nDefinitions(); i++) {
    const ElementDefinition *def = rankStem->definition(i);
    for (size_t j = 0; j < def->nRankStems(); j++)
      setCurrentRank(def->rankStem(j), rankSuffix);
  }
}

Boolean Parser::parseStartTagClose(AttributeList &attributes,
                                   Token &closeToken)
{
  // optimize the common cases
  closeToken = getToken(tagMode);
  if (closeToken == tokenTagc || closeToken == tokenNet) {
    attributes.finish(*this);
    return 1;
  }
  currentInput()->ungetToken();
  return parseAttributeSpec(0, attributes, closeToken);
}

void Parser::checkTaglen(Index tagStartIndex)
{
  const EntityOrigin *origin = currentLocation().origin()->asEntityOrigin();
  ASSERT(origin != 0);
  NamedCharRef ref;
  Offset startOffset, endOffset;
  EntityOrigin::IndexType type
    = origin->lookupIndex(tagStartIndex
			  + syntax().delimGeneral(Syntax::dSTAGO).length(),
			  startOffset, ref);
  ASSERT(type == EntityOrigin::normalCharIndex);
  type = origin->lookupIndex(currentLocation().index(), endOffset, ref);
  ASSERT(type == EntityOrigin::normalCharIndex);
  if (endOffset - startOffset > syntax().taglen())
    message(Messages::taglen, NumberMessageArg(syntax().taglen()));
}

void Parser::parseEmptyStartTag()
{
  // FIXME error if not in base.
  const ElementType *e = 0;
  if (!sd().omittag()) 
    e = lastEndedElementType();
  else if (tagLevel() > 0)
    e = currentElement().type();
  if (!e)
    e = currentDtd().documentElementType();
  AttributeList *attributes = allocAttributeList(e->attributeDef(), 0);
  attributes->finish(*this);
  acceptStartTag(e,
		 new (eventAllocator())
		   StartElementEvent(e, currentDtdPointer(), currentLocation(),
				     currentInput()->nextIndex(),
				     ElementEvent::empty,
				     attributes));
}

void Parser::parseGroupStartTag()
{
  Boolean active;
  if (!parseTagNameGroup(active))
    return;
  InputSource *in = currentInput();
  // Location startLocation = in->currentLocation();
  in->startToken();
  Xchar c = in->tokenChar(inputContext());
  if (!syntax().isNameStartCharacter(c)) {
    message(Messages::startTagMissingName);
    return;
  }
  in->discardInitial();
  extendNameToken(syntax().namelen(), Messages::nameLength);
  Token closeToken;
  if (skipAttributeSpec(closeToken)) {
    if (closeToken == tokenNet)
      message(Messages::startTagGroupNet);
    else if (closeToken != tokenTagc) {
      if (!sd().shorttag())
	message(Messages::minimizedStartTag);
      in->ungetToken();
    }
  }
  noteMarkup();
}

void Parser::parseGroupEndTag()
{
  Boolean active;
  if (!parseTagNameGroup(active))
    return;
  InputSource *in = currentInput();
  // Location startLocation = in->currentLocation();
  in->startToken();
  Xchar c = in->tokenChar(inputContext());
  if (!syntax().isNameStartCharacter(c)) {
    message(Messages::endTagMissingName);
    return;
  }
  in->discardInitial();
  extendNameToken(syntax().namelen(), Messages::nameLength);
  ElementEvent::Minimization min;
  parseEndTagClose(min);
  noteMarkup();
}

void Parser::acceptPcdata(const Location &startLocation)
{
  if (currentElement().tryTransitionPcdata())
    return;
  // Need to test here since implying tags may turn off pcdataRecovering.
  if (pcdataRecovering())
    return;
  IList<Undo> undoList;
  IList<ElementEvent> eventList;
  unsigned startImpliedCount = 0;
  unsigned attributeListIndex = 0;
  keepMessages();
  while (tryImplyTag(startLocation, startImpliedCount, attributeListIndex,
		     undoList, eventList))
    if (currentElement().tryTransitionPcdata()) {
      queueElementEvents(eventList);
      return;
    }
  discardKeptMessages();
  undo(undoList);
  message(Messages::pcdataNotAllowed);
  pcdataRecover();
}

void Parser::acceptStartTag(const ElementType *e,
			    StartElementEvent *event)
{
  if (elementIsExcluded(e)) {
    keepMessages();
    checkExclusion(e);
  }
  else {
    if (currentElement().tryTransition(e)) {
      pushElementCheck(e, event);
      return;
    }
    if (elementIsIncluded(e)) {
      event->setIncluded();
      pushElementCheck(e, event);
      return;
    }
    keepMessages();
  }
  IList<Undo> undoList;
  IList<ElementEvent> eventList;
  unsigned startImpliedCount = 0;
  unsigned attributeListIndex = 1;
  while (tryImplyTag(event->location(), startImpliedCount,
		     attributeListIndex, undoList, eventList))
    if (tryStartTag(e, event, eventList))
      return;
  discardKeptMessages();
  undo(undoList);
  message(Messages::elementNotAllowed, StringMessageArg(e->name()));
  // If element couldn't occur because it was excluded, then
  // do the transition here.
  (void)currentElement().tryTransition(e);
  pushElementCheck(e, event);
}

void Parser::undo(IList<Undo> &undoList)
{
  while (!undoList.empty()) {
    Undo *p = undoList.get();
    p->undo(this);
    delete p;
  }
}

void Parser::queueElementEvents(IList<ElementEvent> &events)
{
  releaseKeptMessages();
  // FIXME provide IList<T>::reverse function
  // reverse it
  IList<ElementEvent> tem;
  while (!events.empty())
    tem.insert(events.get());
  while (!tem.empty()) {
    ElementEvent *e = tem.get();
    if (e->type() == Event::startElement) {
      noteStartElement(e->included());
      eventHandler().startElement((StartElementEvent *)e);
    }
    else {
      noteEndElement(e->included());
      eventHandler().endElement((EndElementEvent *)e);
    }
  }

}

void Parser::checkExclusion(const ElementType *e)
{
  const LeafContentToken *token = currentElement().invalidExclusion(e);
  if (token)
    message(Messages::invalidExclusion,
	    OrdinalMessageArg(token->typeIndex() + 1),
	    StringMessageArg(token->elementType()->name()),
	    StringMessageArg(currentElement().type()->name()));
}

Boolean Parser::tryStartTag(const ElementType *e,
			    StartElementEvent *event,
			    IList<ElementEvent> &impliedEvents)
{
  if (elementIsExcluded(e)) {
    checkExclusion(e);
    return 0;
  }
  if (currentElement().tryTransition(e)) {
    queueElementEvents(impliedEvents);
    pushElementCheck(e, event);
    return 1;
  }
  if (elementIsIncluded(e)) {
    queueElementEvents(impliedEvents);
    event->setIncluded();
    pushElementCheck(e, event);
    return 1;
  }
  return 0;
}

Boolean Parser::tryImplyTag(const Location &loc,
			    unsigned &startImpliedCount,
			    unsigned &attributeListIndex,
			    IList<Undo> &undo,
			    IList<ElementEvent> &eventList)
{
  if (!sd().omittag())
    return 0;
  if (currentElement().isFinished()) {
    if (tagLevel() == 0)
      return 0;
    // imply an end tag
    if (startImpliedCount > 0) {
      message(Messages::startTagEmptyElement,
	      StringMessageArg(currentElement().type()->name()));
      startImpliedCount--;
    }
    const ElementDefinition *def = currentElement().type()->definition();
    if (def && !def->canOmitEndTag())
      message(Messages::omitEndTagDeclare,
	      StringMessageArg(currentElement().type()->name()),
	      currentElement().startLocation());
    EndElementEvent *event
      = new (eventAllocator()) EndElementEvent(currentElement().type(),
					       currentDtdPointer(),
					       loc,
					       loc.index(),
					       ElementEvent::omittedOmittag);
    eventList.insert(event);
    undo.insert(new (internalAllocator()) UndoEndTag(popSaveElement()));
    return 1;
  }
  const LeafContentToken *token = currentElement().impliedStartTag();
  if (!token)
    return 0;
  const ElementType *e = token->elementType();
  if (elementIsExcluded(e))
    message(Messages::requiredElementExcluded,
	    OrdinalMessageArg(token->typeIndex() + 1),
	    StringMessageArg(e->name()),
	    StringMessageArg(currentElement().type()->name()));
  if (tagLevel() != 0)
    undo.insert(new (internalAllocator())
		     UndoTransition(currentElement().matchState()));
  currentElement().doRequiredTransition();
  const ElementDefinition *def = e->definition();
  if (def && def->declaredContent() != ElementDefinition::modelGroup
      && def->declaredContent() != ElementDefinition::any)
    message(Messages::omitStartTagDeclaredContent,
	    StringMessageArg(e->name()));
  if (def && !def->canOmitStartTag())
    message(Messages::omitStartTagDeclare, StringMessageArg(e->name()));
  AttributeList *attributes
    = allocAttributeList(e->attributeDef(),
			 attributeListIndex++);
  // this will give an error if the element has a required attribute
  attributes->finish(*this);
  startImpliedCount++;
  StartElementEvent *event
    = new (eventAllocator()) StartElementEvent(e,
					       currentDtdPointer(),
					       loc,
					       loc.index(),
					       ElementEvent::omittedOmittag,
					       attributes);
  pushElementCheck(e, event, undo, eventList);
  return 1;
}

void Parser::pushElementCheck(const ElementType *e, StartElementEvent *event)
{
  if (tagLevel() == syntax().taglvl())
    message(Messages::taglvlOpenElements, NumberMessageArg(syntax().taglvl()));
  const ElementDefinition *def = e->definition();
  if (def->undefined())
    message(Messages::undefinedElement, StringMessageArg(e->name()));
  noteStartElement(event->included());
  if (def->declaredContent() == ElementDefinition::empty
      || event->attributes().conref()) {
    ElementEvent::Minimization min 
      = (def->declaredContent() == ElementDefinition::empty
	 ? ElementEvent::omittedEmpty
	 : ElementEvent::omittedConref);
    EndElementEvent *end
      = new (eventAllocator()) EndElementEvent(e,
					       currentDtdPointer(),
					       Location(event->location().origin(),
							event->nextIndex()),
					       event->nextIndex(),
					       min);
    if (event->included()) {
      end->setIncluded();
      noteEndElement(1);
    }
    else
      noteEndElement(0);
    eventHandler().startElement(event);
    eventHandler().endElement(end);
  }
  else {
    const ShortReferenceMap *map = e->map();
    if (!map)
      map = currentElement().map();
    pushElement(new (internalAllocator()) OpenElement(e,
						      event->netEnabling(),
						      event->included(),
						      map,
						      event->location()));
    // Can't access event after it's passed to the event handler.
    eventHandler().startElement(event);
  }
}

void Parser::pushElementCheck(const ElementType *e, StartElementEvent *event,
			      IList<Undo> &undoList,
			      IList<ElementEvent> &eventList)
{
  if (tagLevel() == syntax().taglvl())
    message(Messages::taglvlOpenElements, NumberMessageArg(syntax().taglvl()));
  const ElementDefinition *def = e->definition();
  if (def->undefined())
    message(Messages::undefinedElement, StringMessageArg(e->name()));
  eventList.insert(event);
  if (def->declaredContent() == ElementDefinition::empty
      || event->attributes().conref()) {
    ElementEvent::Minimization min 
      = (def->declaredContent() == ElementDefinition::empty
	 ? ElementEvent::omittedEmpty
	 : ElementEvent::omittedConref);
    EndElementEvent *end
      = new (eventAllocator()) EndElementEvent(e,
					       currentDtdPointer(),
					       Location(event->location().origin(),
							event->nextIndex()),
					       event->nextIndex(),
					       min);
    if (event->included())
      end->setIncluded();
    eventList.insert(end);
  }
  else {
    undoList.insert(new (internalAllocator()) UndoStartTag);
    const ShortReferenceMap *map = e->map();
    if (!map)
      map = currentElement().map();
    pushElement(new (internalAllocator()) OpenElement(e,
						      event->netEnabling(),
						      event->included(),
						      map,
						      event->location()));
  }
}

void Parser::parseEndTag()
{
  // Location location(currentLocation());
  currentInput()->discardInitial();
  extendNameToken(syntax().namelen(), Messages::nameLength);
  CString origName;
  if (esisPlus())
    getCurrentToken(origName);
  CString &name(nameBuffer());
  getCurrentToken(syntax().generalSubstTable(), name);
  const ElementType *e = currentDtd().lookupElementType(name);
  if (sd().rank()) {
    if (!e)
      e = completeRankStem(name);
  }
  if (!e) 
    e = lookupCreateUndefinedElement(name);
  ElementEvent::Minimization min;
  parseEndTagClose(min);
  acceptEndTag(e,
	       new (eventAllocator()) EndElementEvent(e,
						      currentDtdPointer(),
						      currentLocation(),
						      currentInput()->nextIndex(),
						      min,
						      origName));
}

void Parser::parseEndTagClose(ElementEvent::Minimization &min)
{
  min = ElementEvent::complete;
  for (;;) {
    Token token = getToken(tagMode);
    switch (token) {
    case tokenUnrecognized:
      if (!reportNonSgmlCharacter())
	message(Messages::endTagCharacter, StringMessageArg(currentToken()));
      return;
    case tokenEe:
      message(Messages::endTagEntityEnd);
      return;
    case tokenEtago:
    case tokenStago:
      if (!sd().shorttag())
	message(Messages::minimizedEndTag);
      min = ElementEvent::unclosed;
      currentInput()->ungetToken();
      return;
    case tokenTagc:
      return;
    case tokenS:
      break;
    default:
      message(Messages::endTagInvalidToken,
	      TokenMessageArg(token, tagMode, syntaxPointer(), sdPointer()));
      return;
    }
  }
}

void Parser::parseEmptyEndTag()
{
  // FIXME what to do if not in base
  if (tagLevel() == 0)
    message(Messages::emptyEndTagNoOpenElements);
  else
    acceptEndTag(currentElement().type(),
		 new (eventAllocator()) EndElementEvent(currentElement().type(),
							currentDtdPointer(),
							currentLocation(),
							currentInput()->nextIndex(),
							ElementEvent::empty));
}

void Parser::parseNullEndTag()
{
  // If a null end tag was recognized, then there must be a net enabling
  // element on the stack.
  for (;;) {
    ASSERT(tagLevel() > 0);
    if (currentElement().netEnabling())
      break;
    if (!currentElement().isFinished())
      message(Messages::elementNotFinished,
	      StringMessageArg(currentElement().type()->name()));
    implyCurrentElementEnd(currentLocation());
  }
  if (!currentElement().isFinished())
    message(Messages::elementEndTagNotFinished,
	    StringMessageArg(currentElement().type()->name()));
  acceptEndTag(currentElement().type(),
	       new (eventAllocator()) EndElementEvent(currentElement().type(),
						      currentDtdPointer(),
						      currentLocation(),
						      currentInput()->nextIndex(),
						      ElementEvent::net));
}

void Parser::endAllElements()
{
  while (tagLevel() > 0) {
    if (!sd().omittag() || !currentElement().isFinished())
      message(Messages::elementNotFinishedDocumentEnd,
	      StringMessageArg(currentElement().type()->name()));
    implyCurrentElementEnd(currentLocation());
  }
  if (!currentElement().isFinished())
    message(Messages::noDocumentElement);
}

void Parser::acceptEndTag(const ElementType *e,
			  EndElementEvent *event)
{
  if (!elementIsOpen(e)) {
    message(Messages::elementNotOpen, StringMessageArg(e->name()));
    delete event;
    return;
  }
  for (;;){
    if (currentElement().type() == e)
      break;
    if (!currentElement().isFinished())
      message(Messages::elementNotFinished,
	      StringMessageArg(currentElement().type()->name()));
    implyCurrentElementEnd(event->location());
  }
  if (!currentElement().isFinished())
    message(Messages::elementEndTagNotFinished,
	    StringMessageArg(currentElement().type()->name()));
  if (currentElement().included())
    event->setIncluded();
  noteEndElement(event->included());
  eventHandler().endElement(event);
  popElement();
}

void Parser::implyCurrentElementEnd(const Location &loc)
{
  const ElementDefinition *def = currentElement().type()->definition();
  if (def && !def->canOmitEndTag())
    message(Messages::omitEndTagDeclare,
	    StringMessageArg(currentElement().type()->name()),
	    currentElement().startLocation());
  EndElementEvent *event
    = new (eventAllocator()) EndElementEvent(currentElement().type(),
					     currentDtdPointer(),
					     loc,
					     loc.index(),
					     ElementEvent::omittedOmittag);
  if (currentElement().included())
    event->setIncluded();
  noteEndElement(event->included());
  eventHandler().endElement(event);
  popElement();
}

void Parser::extendData()
{
  XcharMap<PackedBoolean> isNormal(normalMap());
  InputSource *in = currentInput();
  size_t length = in->currentTokenLength();
  // This is one of the parser's inner loops, so it needs to be fast.
  while (isNormal[in->tokenChar(inputContext())])
    length++;
  in->endToken(length);
}

void Parser::extendContentS()
{
  InputSource *in = currentInput();
  size_t length = in->currentTokenLength();
  XcharMap<PackedBoolean> isNormal(normalMap());
  for (;;) {
    Xchar ch = in->tokenChar(inputContext());
    if (!syntax().isS(ch) || !isNormal[ch])
      break;
    length++;
  }
  in->endToken(length);
}

