// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#include "Parser.H"
#include "MessageArg.H"
#include "TokenMessageArg.H"
#include "ModeInfo.H"
#include "Partition.H"
#include "SrInfo.H"
#include "Vector.H"
#include "ISetIter.H"
#include "token.H"
#include "TrieBuilder.H"
#include "setAll.H"

enum {
  modeUsedInSd = 01,
  modeUsedInProlog = 02,
  modeUsedInInstance = 04,
  modeUsesSr = 010
};

static struct {
  Mode mode;
  unsigned flags;
} modeTable[] = {
  { grpMode, modeUsedInProlog|modeUsedInInstance },
  { alitMode, modeUsedInProlog|modeUsedInInstance },
  { alitaMode, modeUsedInProlog|modeUsedInInstance },
  { aliteMode, modeUsedInProlog|modeUsedInInstance },
  { talitMode, modeUsedInProlog|modeUsedInInstance },
  { talitaMode, modeUsedInProlog|modeUsedInInstance },
  { taliteMode, modeUsedInProlog|modeUsedInInstance },
  { mdMode, modeUsedInProlog|modeUsedInInstance|modeUsedInSd },
  { mdMinusMode, modeUsedInProlog },
  { mdPeroMode, modeUsedInProlog },
  { comMode, modeUsedInProlog|modeUsedInInstance },
  { sdcomMode, modeUsedInSd },
  { piMode, modeUsedInProlog|modeUsedInInstance },
  { refMode, modeUsedInProlog|modeUsedInInstance|modeUsedInSd },
  { imsMode, modeUsedInProlog|modeUsedInInstance },
  { cmsMode, modeUsedInProlog|modeUsedInInstance },
  { rcmsMode, modeUsedInProlog|modeUsedInInstance },
  { proMode, modeUsedInProlog },
  { dsMode, modeUsedInProlog },
  { dsiMode, modeUsedInProlog },
  { plitMode, modeUsedInProlog },
  { plitaMode, modeUsedInProlog },
  { pliteMode, modeUsedInProlog },
  { sdplitMode, modeUsedInSd },
  { sdplitaMode, modeUsedInSd },
  { grpsufMode, modeUsedInProlog },
  { mlitMode, modeUsedInProlog|modeUsedInSd },
  { mlitaMode, modeUsedInProlog|modeUsedInSd },
  { asMode, modeUsedInProlog },
  { slitMode, modeUsedInProlog },
  { slitaMode, modeUsedInProlog },
  { cconMode, modeUsedInInstance },
  { rcconMode, modeUsedInInstance },
  { cconnetMode, modeUsedInInstance },
  { rcconnetMode, modeUsedInInstance },
  { tagMode, modeUsedInInstance },
  { econMode, modeUsedInInstance|modeUsesSr },
  { mconMode, modeUsedInInstance|modeUsesSr },
  { econnetMode, modeUsedInInstance|modeUsesSr },
  { mconnetMode, modeUsedInInstance|modeUsesSr },
};

void Parser::compileSdModes()
{
  Mode modes[nModes];
  int n = 0;
  for (int i = 0; i < nModes; i++)
    if (modeTable[i].flags & modeUsedInSd)
      modes[n++] = Mode(i);
  compileModes(modes, n, 0);
}

void Parser::compilePrologModes()
{
  Boolean scopeInstance = sd().scopeInstance();
  if (subdocLevel() > 0 && !scopeInstance)
    return;
  Boolean haveSr = syntax().nDelimShortref() > 0;
  Mode modes[nModes];
  int n = 0;
  for (int i = 0; i < nModes; i++) {
    if (scopeInstance) {
      if (modeTable[i].flags & modeUsedInProlog)
	modes[n++] = Mode(i);
    }
    else if (haveSr) {
      if ((modeTable[i].flags & (modeUsedInInstance|modeUsedInProlog))
	  && !(modeTable[i].flags & modeUsesSr))
	modes[n++] = Mode(i);
    }
    else {
      if (modeTable[i].flags & (modeUsedInInstance|modeUsedInProlog))
	modes[n++] = Mode(i);
    }
  }
  compileModes(modes, n, 0);
}

void Parser::compileInstanceModes()
{
  Vector<PackedBoolean> needShortref(syntax().nDelimShortref());
  Boolean scopeInstance = sd().scopeInstance();
  Boolean haveSr = syntax().nDelimShortref() > 0;

  if (haveSr)
    determineNeededShortrefs(currentDtd().usedShortrefs(),
			     needShortref.pointer());
  compileNormalMap(needShortref.pointer());
  if (!scopeInstance && !haveSr)
    return;
  Mode modes[nModes];
  int n = 0;
  for (int i = 0; i < nModes; i++) {
    if (scopeInstance) {
      if (modeTable[i].flags & modeUsedInInstance)
	modes[n++] = Mode(i);
    }
    else {
      if (modeTable[i].flags & modeUsesSr)
	modes[n++] = Mode(i);
    }
  }
  compileModes(modes, n, needShortref.pointer());
}

void Parser::compileModes(const Mode *modes,
			  int n,
			  const PackedBoolean *needShortref)
{
  PackedBoolean sets[Syntax::nSet];
  PackedBoolean delims[Syntax::nDelimGeneral];
  PackedBoolean functions[3];
  int i;
  Boolean includesShortref = 0;
  for (i = 0; i < Syntax::nSet; i++)
    sets[i] = 0;
  for (i = 0; i < Syntax::nDelimGeneral; i++)
    delims[i] = 0;
  for (i = 0; i < 3; i++)
    functions[i] = 0;
  
  for (i = 0; i < n; i++) {
    ModeInfo iter(modes[i], sd());
    TokenInfo ti;
    while (iter.nextToken(&ti)) {
      switch (ti.type) {
      case TokenInfo::delimType:
	delims[ti.delim1] = 1;
	break;
      case TokenInfo::delimDelimType:
	delims[ti.delim1] = 1;
	delims[ti.delim2] = 1;
	break;
      case TokenInfo::delimSetType:
	delims[ti.delim1] = 1;
	// fall through
      case TokenInfo::setType:
	sets[ti.set] = 1;
	break;
      case TokenInfo::functionType:
	functions[ti.function] = 1;
	break;
      }
    }
    if (!includesShortref && iter.includesShortref())
      includesShortref = 1;
  }

  ISet<Char> chars;

  for (i = 0; i < 3; i++)
    if (functions[i])
      chars.add(syntax().standardFunction(i));
  for (i = 0; i < Syntax::nDelimGeneral; i++)
    if (delims[i]) {
      const CString &str = syntax().delimGeneral(i);
      for (int j = 0; j < str.length(); j++)
	chars.add(str[j]);
    }
  if (includesShortref) {
    size_t n = syntax().nDelimShortref();
    for (size_t i = 0; i < n; i++) 
      if (needShortref[i]) {
	const CString &delim = syntax().delimShortref(i);
	size_t len = delim.length();
	for (size_t j = 0; j < len; j++)
	  if (delim[j] == sd().execToDoc('B'))
	    sets[Syntax::blank] = 1;
	  else
	    chars.add(delim[j]);
      }
  }

  const ISet<Char> *csets[Syntax::nSet];
  int usedSets = 0;
  for (i = 0; i < Syntax::nSet; i++)
    if (sets[i])
      csets[usedSets++] = syntax().charSet(i);

  Partition partition(chars, csets, usedSets, *syntax().generalSubstTable());

  String<EquivCode> setCodes[Syntax::nSet];
  
  int j = 0;
  for (i = 0; i < Syntax::nSet; i++)
    if (sets[i])
      setCodes[i] = partition.setCodes(j++);

  String<EquivCode> delimCodes[Syntax::nDelimGeneral];
  for (i = 0; i < Syntax::nDelimGeneral; i++)
    if (delims[i]) {
      CString str = syntax().delimGeneral(i);
      for (int j = 0; j < str.length(); j++)
	delimCodes[i] += partition.charCode(str[j]);
    }

  String<EquivCode> functionCode[3];
  for (i = 0; i < 3; i++)
    if (functions[i])
      functionCode[i] += partition.charCode(syntax().standardFunction(i));

  Vector<SrInfo> srInfo;
      
  int nShortref = syntax().nDelimShortref();

  if (includesShortref) {
    srInfo.init(nShortref);

    for (int i = 0; i < nShortref; i++)
      if (needShortref[i]) {
	const CString delim = syntax().delimShortref(i);
	SrInfo *p = &srInfo[i];
	for (j = 0; j < delim.length(); j++) {
	  if (delim[j] == sd().execToDoc('B'))
	    break;
	  p->chars += partition.charCode(delim[j]);
	}
	if (j < delim.length()) {
	  p->bSequenceLength = 1;
	  for (++j; j < delim.length(); j++) {
	    if (delim[j] != sd().execToDoc('B'))
	      break;
	    p->bSequenceLength += 1;
	  }
	  for (; j < delim.length(); j++)
	    p->chars2 += partition.charCode(delim[j]);
	}
	else
	  p->bSequenceLength = 0;
      }
  }

  const String<EquivCode> emptyString;
  Boolean multicode = syntax().multicode();
  for (i = 0; i < n; i++) {
    TrieBuilder tb(partition.maxCode() + 1);
    TrieBuilder::TokenVector ambiguities;
    Vector<Token> suppressTokens;
    if (multicode) {
      suppressTokens.init(partition.maxCode() + 1);
      setAll(suppressTokens.pointer(), suppressTokens.length(), Token(0));
      suppressTokens[partition.eECode()] = tokenEe;
    }
    tb.recognizeEE(partition.eECode(), tokenEe);
    ModeInfo iter(modes[i], sd());
    TokenInfo ti;
    // We try to handle the possibility that some delimiters may be empty;
    // this might happen when compiling recognizers for the SGML declaration.
    while (iter.nextToken(&ti)) {
      switch (ti.type) {
      case TokenInfo::delimType:
	if (delimCodes[ti.delim1].length() > 0)
	  tb.recognize(delimCodes[ti.delim1], ti.token,
		       ti.priority, ambiguities);
	break;
      case TokenInfo::delimDelimType:
	{
	  String<EquivCode> str(delimCodes[ti.delim1]);
	  if (str.length() > 0 && delimCodes[ti.delim2].length() > 0) {
	    str += delimCodes[ti.delim2];
	    tb.recognize(str, ti.token, ti.priority, ambiguities);
	  }
	}
	break;
      case TokenInfo::delimSetType:
	if (delimCodes[ti.delim1].length() > 0)
	  tb.recognize(delimCodes[ti.delim1], setCodes[ti.set],
		       ti.token, ti.priority, ambiguities);
	break;
      case TokenInfo::setType:
	tb.recognize(emptyString, setCodes[ti.set], ti.token,
		     ti.priority, ambiguities);
	if (multicode) {
	  const String<EquivCode> &equivCodes(setCodes[ti.set]);
	  for (size_t j = 0; j < equivCodes.length(); j++)
	    suppressTokens[equivCodes[j]] = ti.token;
	}
	break;
      case TokenInfo::functionType:
	tb.recognize(functionCode[ti.function], ti.token,
		     ti.priority, ambiguities);
	if (multicode)
	  suppressTokens[functionCode[ti.function][0]] = ti.token;
	break;
      }
    }
    if (iter.includesShortref()) {
      for (int j = 0; j < nShortref; j++)
	if (needShortref[j]) {
	  const SrInfo *p = &srInfo[j];
	  if (p->bSequenceLength > 0)
	    tb.recognizeB(p->chars, p->bSequenceLength,
			  syntax().quantity(Syntax::qBSEQLEN),
			  setCodes[Syntax::blank],
			  p->chars2, tokenFirstShortref + j,
			  ambiguities);
	  else
	    tb.recognize(p->chars, tokenFirstShortref + j,
			 Priority::delim, ambiguities);
	}
    }
    setRecognizer(modes[i],
		  (multicode
		   ? new Recognizer(tb.extractTrie(), partition.map(),
				    suppressTokens)
		   : new Recognizer(tb.extractTrie(), partition.map())));
    // FIXME give more information
    for (int j = 0; j < ambiguities.length(); j += 2)
      message(Messages::lexicalAmbiguity,
	      TokenMessageArg(ambiguities[j], modes[i], syntaxPointer(),
			      sdPointer()),
	      TokenMessageArg(ambiguities[j + 1], modes[i], syntaxPointer(),
			      sdPointer()));
  }
}

void Parser::compileNormalMap(const PackedBoolean *needShortref)
{
  XcharMap<PackedBoolean> map(0);
  ISetIter<Char> sgmlCharIter(*syntax().charSet(Syntax::sgmlChar));
  Char min, max;
  while (sgmlCharIter.next(min, max))
    map.setRange(min, max, 1);
  ModeInfo iter(mconnetMode, sd());
  TokenInfo ti;
  while (iter.nextToken(&ti)) {
    switch (ti.type) {
    case TokenInfo::delimType:
    case TokenInfo::delimDelimType:
    case TokenInfo::delimSetType:
      {
	Char c = syntax().delimGeneral(ti.delim1)[0];
	map.setChar(c, 0);
	String<Char> str(syntax().generalSubstTable()->inverse(c));
	for (int i = 0; i < str.length(); i++)
	  map.setChar(str[i], 0);
      }
      break;
    case TokenInfo::setType:
      if (ti.token != tokenChar) {
	ISetIter<Char> setIter(*syntax().charSet(ti.set));
	Char min, max;
	while (setIter.next(min, max))
	  map.setRange(min, max, 0);
      }
      break;
    case TokenInfo::functionType:
      if (ti.token != tokenChar)
	map.setChar(syntax().standardFunction(ti.function), 0);
      break;
    }
  }
  int nShortref = syntax().nDelimShortref();
  for (int i = 0; i < nShortref; i++) {
    if (needShortref[i]) {
      Char c = syntax().delimShortref(i)[0];
      if (c == sd().execToDoc('B')) {
	ISetIter<Char> setIter(*syntax().charSet(Syntax::blank));
	Char min, max;
	while (setIter.next(min, max))
	  map.setRange(min, max, 0);
      }
      else {
	map.setChar(c, 0);
	String<Char> str(syntax().generalSubstTable()->inverse(c));
	for (int i = 0; i < str.length(); i++)
	  map.setChar(str[i], 0);
      }
    }
  }
  setNormalMap(map);
}

void Parser::determineNeededShortrefs(const PackedBoolean *used,
				      PackedBoolean *needed)
{
  PackedBoolean delimRelevant[Syntax::nDelimGeneral];
  for (size_t i = 0; i < Syntax::nDelimGeneral; i++)
    delimRelevant[i] = 0;
  ModeInfo iter(mconnetMode, sd());
  TokenInfo ti;
  while (iter.nextToken(&ti)) {
    switch (ti.type) {
    case TokenInfo::delimType:
    case TokenInfo::delimDelimType:
    case TokenInfo::delimSetType:
      delimRelevant[ti.delim1] = 1;
      break;
    default:
      break;
    }
  }

  size_t nShortref = syntax().nDelimShortref();
  // A short reference delimiter is needed if it is used or if it can
  // contains some other shorter delimiter that is either a relevant general
  // delimiter or a shortref delimiter that is used.

  for (i = 0; i < nShortref; i++) {
    needed[i] = 0;
    if (used[i]
	// PIO and NET are the only delimiters that are recognized in con
	// mode without context.  If a short reference delimiter is
	// identical to one of these delimiters, then we'll have an
	// ambiguity.   We make such a short reference delimiter needed
	// to ensure that this ambiguity is reported.
	|| syntax().delimShortref(i) == syntax().delimGeneral(Syntax::dPIO)
	|| syntax().delimShortref(i) == syntax().delimGeneral(Syntax::dNET))
      needed[i] = 1;
    else {
      size_t j;
      for (j = 0; j < Syntax::nDelimGeneral; j++)
	if (delimRelevant[j]
	    && shortrefCanContainDelim(syntax().delimShortref(i),
				       syntax().delimGeneral(j),
				       0)) {
	  needed[i] = 1;
	  break;
	}
      if (!needed[i]) {
	for (j = 0; j < nShortref; j++)
	  if (used[j]
	      && shortrefCanContainDelim(syntax().delimShortref(i),
					 syntax().delimShortref(j),
					 1)) {
	    needed[i] = 1;
	    break;
	  }
      } 
    }
  }
}

Boolean Parser::shortrefCanContainDelim(const CString &sr,
					const CString &d,
					Boolean dIsSr)
{
  Char letterB = sd().execToDoc('B');
  if (sr.length() == 1 && sr[0] != letterB)
    return 0;
  for (size_t i = 0; i < sr.length(); i++) {
    size_t j = 0;
    size_t k = i;
    for (;;) {
      if (j == d.length())
	return 1;
      if (k >= sr.length())
	break;
      if (sr[k] == letterB) {
	if (dIsSr && d[j] == letterB) {
	  j++;
	  k++;
	}
	else if (syntax().isB(d[j])) {
	  j++;
	  k++;
	  if (k == sr.length() || sr[k] != letterB) {
	    // it was the last B in the sequence
	    while (j < d.length() && syntax().isB(d[j]))
	      j++;
	  }
	}
	else
	  break;
      }
      else if (dIsSr && d[j] == letterB) {
	if (syntax().isB(sr[k])) {
	  ++j;
	  ++k;
	  if (j < d.length() && d[j] != letterB) {
	    while (k < sr.length() && syntax().isB(sr[k]))
	      k++;
	  }
	}
	else
	  break;
      }
      else if (d[j] == sr[k]) {
	j++;
	k++;
      }
      else
	break;
    }
  }
  return 0;
}
