// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#include "Parser.H"
#include "Param.H"
#include "Group.H"
#include "MessageArg.H"
#include "TokenMessageArg.H"
#include "token.H"
#include "macros.H"

Boolean Parser::parseParam(const AllowedParams &allow,
			   unsigned declInputLevel,
			   Param &parm)
{
  for (;;) {
    Token token = getToken(allow.mainMode());
    switch (token) {
    case tokenUnrecognized:
      if (reportNonSgmlCharacter())
	break;
      {
	message(Messages::markupDeclarationCharacter,
		StringMessageArg(currentToken()),
		AllowedParamsMessageArg(allow, syntaxPointer()));
      }
      return 0;
    case tokenEe:
      if (inputLevel() <= declInputLevel) {
	message(Messages::declarationLevel);
	return 0;
      }
      popInputStack();
      break;
    case tokenCom:
      if (!parseComment(comMode))
	return 0;
      break;
    case tokenDso:
      if (!allow.dso()) {
	paramInvalidToken(tokenDso, allow);
	return 0;
      }
      parm.type = Param::dso;
      return 1;
    case tokenGrpo:
      switch (allow.group()) {
      case Param::invalid:
	paramInvalidToken(tokenGrpo, allow);
	return 0;
      case Param::modelGroup:
	{
	  ModelGroup *group;
	  if (!parseModelGroup(1, declInputLevel, group, grpsufMode))
	    return 0;
	  parm.type = Param::modelGroup;
	  parm.modelGroupPtr = group;
	}
	break;
      case Param::nameGroup:
	if (!parseNameGroup(declInputLevel, parm))
	  return 0;
	break;
      case Param::nameTokenGroup:
	if (!parseNameTokenGroup(declInputLevel, parm))
	  return 0;
	break;
      default:
	CANNOT_HAPPEN();
      }
      parm.type = allow.group();
      return 1;
    case tokenLita:
    case tokenLit:
      parm.type = allow.literal();
      parm.lita = token == tokenLita;
      switch (allow.literal()) {
      case Param::invalid:
	paramInvalidToken(tokenMinusGrpo, allow);
	return 0;
      case Param::minimumLiteral:
	if (!parseMinimumLiteral(parm.lita, parm.literalText))
	  return 0;
	break;
      case Param::attributeValueLiteral:
	if (!parseAttributeValueLiteral(parm.lita, parm.literalText))
	  return 0;
	break;
      case Param::tokenizedAttributeValueLiteral:
	if (!parseTokenizedAttributeValueLiteral(parm.lita, parm.literalText))
	  return 0;
	break;
      case Param::systemIdentifier:
	if (!parseSystemIdentifier(parm.lita, parm.literalText))
	  return 0;
	break;
      case Param::paramLiteral:
	if (!parseParameterLiteral(parm.lita, parm.literalText))
	  return 0;
	break;
      }
      return 1;
    case tokenMdc:
      if (!allow.mdc()) {
	paramInvalidToken(tokenMdc, allow);
	return 0;
      }
      if (inputLevel() > declInputLevel)
	message(Messages::parameterEntityNotEnded);
      parm.type = Param::mdc;
      return 1;
    case tokenMinus:
      parm.type = Param::minus;
      return 1;
    case tokenMinusGrpo:
      if (!allow.exclusions()) {
	paramInvalidToken(tokenMinusGrpo, allow);
	return 0;
      }
      parm.type = Param::exclusions;
      return parseElementNameGroup(declInputLevel, parm);
    case tokenPero:
      parm.type = Param::pero;
      return 1;
    case tokenPeroGrpo:
      if (!inInstance())
	message(Messages::peroGrpoProlog);
      // fall through
    case tokenPeroNameStart:
      {
	ConstResourcePointer<Entity> entity;
	ResourcePointer<EntityOrigin> origin;
	if (!parseEntityReference(1, token == tokenPeroGrpo, entity, origin))
	  return 0;
	if (!entity.isNull())
	  entity->declReference(*this, origin);
      }
      break;
    case tokenPlusGrpo:
      if (!allow.inclusions()) {
	paramInvalidToken(tokenPlusGrpo, allow);
	return 0;
      }
      parm.type = Param::inclusions;
      return parseElementNameGroup(declInputLevel, parm);
    case tokenRni:
      if (!allow.rni()) {
	paramInvalidToken(tokenRni, allow);
	return 0;
      }
      return parseIndicatedReservedName(allow, parm);
    case tokenS:
      extendS();
      break;
    case tokenNameStart:
      switch (allow.nameStart()) {
      case Param::invalid:
	paramInvalidToken(tokenNameStart, allow);
	return 0;
      case Param::reservedName:
	return parseReservedName(allow, parm);
      case Param::name:
	extendNameToken(syntax().namelen(), Messages::nameLength);
	parm.type = Param::name;
	getCurrentToken(syntax().generalSubstTable(), parm.token);
	return 1;
      case Param::entityName:
	extendNameToken(syntax().namelen(), Messages::nameLength);
	parm.type = Param::entityName;
	getCurrentToken(syntax().entitySubstTable(), parm.token);
	return 1;
      case Param::paramEntityName:
	extendNameToken(syntax().penamelen(), Messages::parameterEntityNameLength);
	parm.type = Param::paramEntityName;
	getCurrentToken(syntax().entitySubstTable(), parm.token);
	return 1;
      case Param::attributeValue:
	return parseAttributeValueParam(parm);
      }
      break;
    case tokenDigit:
      switch (allow.digit()) {
      case Param::invalid:
	paramInvalidToken(tokenDigit, allow);
	return 0;
      case Param::number:
	extendNumber(syntax().namelen(), Messages::numberLength);
	parm.type = Param::number;
	getCurrentToken(parm.token);
	return 1;
      case Param::attributeValue:
	return parseAttributeValueParam(parm);
      }
      break;
    case tokenLcUcNmchar:
      switch (allow.nmchar()) {
      case Param::invalid:
	paramInvalidToken(tokenLcUcNmchar, allow);
	return 0;
      case Param::attributeValue:
	return parseAttributeValueParam(parm);
      }
      break;
    default:
      CANNOT_HAPPEN();
    }
  }
}

void Parser::paramInvalidToken(Token token, const AllowedParams &allow)
{
  message(Messages::paramInvalidToken,
	  TokenMessageArg(token, allow.mainMode(),
			  syntaxPointer(), sdPointer()),
	  AllowedParamsMessageArg(allow, syntaxPointer()));
}

Boolean Parser::parseGroupToken(const AllowedGroupTokens &allow,
				unsigned nestingLevel,
				unsigned declInputLevel,
				unsigned groupInputLevel,
				GroupToken &gt)
{
  for (;;) {
    Token token = getToken(grpMode);
    switch (token) {
    case tokenEe:
      if (inputLevel() <= groupInputLevel) {
	message(Messages::groupLevel);
	if (inputLevel() <= declInputLevel)
	  return 0;
      }
      else
	message(Messages::groupEntityEnd);
      popInputStack();
      break;
    case tokenPeroGrpo:
      if (!inInstance())
	message(Messages::peroGrpoProlog);
      // fall through
    case tokenPeroNameStart:
      {
	ConstResourcePointer<Entity> entity;
	ResourcePointer<EntityOrigin> origin;
	if (!parseEntityReference(1, token == tokenPeroGrpo, entity, origin))
	  return 0;
	if (!entity.isNull())
	  entity->declReference(*this, origin);
      }
      break;
    case tokenUnrecognized:
      if (reportNonSgmlCharacter())
	break;
      {
	message(Messages::groupCharacter,
		StringMessageArg(currentToken()),
		AllowedGroupTokensMessageArg(allow, syntaxPointer()));
      }
      return 0;
    case tokenDtgo:
      if (!allow.groupToken(GroupToken::dataTagGroup)) {
	groupTokenInvalidToken(tokenDtgo, allow);
	return 0;
      }
      return parseDataTagGroup(nestingLevel + 1, declInputLevel, gt);
    case tokenGrpo:
      switch (allow.group()) {
      case GroupToken::modelGroup:
	{
	  ModelGroup *modelGroup;
	  if (!parseModelGroup(nestingLevel + 1, declInputLevel, modelGroup,
			       grpMode))
	    return 0;
	  gt.model = modelGroup;
	  gt.type = GroupToken::modelGroup;
	  return 1;
	}
      case GroupToken::dataTagTemplateGroup:
	return parseDataTagTemplateGroup(nestingLevel + 1, declInputLevel, gt);
      default:
	groupTokenInvalidToken(tokenGrpo, allow);
	return 0;
      }
      break;
    case tokenRni:
      if (!allow.groupToken(GroupToken::pcdata)) {
	groupTokenInvalidToken(tokenRni, allow);
	return 0;
      }
      Syntax::ReservedName rn;
      if (!getIndicatedReservedName(&rn))
	return 0;
      if (rn != Syntax::rPCDATA) {
	CString token(syntax().delimGeneral(Syntax::dRNI));
	token += syntax().reservedName(Syntax::rPCDATA);
	message(Messages::invalidToken, StringMessageArg(token));
	return 0;
      }
      gt.type = GroupToken::pcdata;
      gt.contentToken = new PcdataToken;
      return 1;
    case tokenS:
      break;
    case tokenNameStart:
      switch (allow.nameStart()) {
      case GroupToken::elementToken:
	{
	  extendNameToken(syntax().namelen(), Messages::nameLength);
	  gt.type = GroupToken::elementToken;
	  CString &buffer(nameBuffer());
	  getCurrentToken(syntax().generalSubstTable(), buffer);
	  const ElementType *e = lookupCreateElement(buffer);
	  ContentToken::OccurrenceIndicator oi = getOccurrenceIndicator(grpMode);
	  gt.contentToken = new ElementToken(e, oi);
	  return 1;
	}
      case GroupToken::name:
      case GroupToken::nameToken:
	extendNameToken(syntax().namelen(),
			token == GroupToken::name
			? Messages::nameLength
			: Messages::nameTokenLength);
	getCurrentToken(syntax().generalSubstTable(), gt.token);
	gt.type = allow.nameStart();
	return 1;
      default:
	groupTokenInvalidToken(tokenNameStart, allow);
	return 0;
      }
    case tokenDigit:
    case tokenLcUcNmchar:
      if (!allow.groupToken(GroupToken::nameToken)) {
	groupTokenInvalidToken(token, allow);
	return 0;
      }
      extendNameToken(syntax().namelen(), Messages::nameTokenLength);
      getCurrentToken(syntax().generalSubstTable(), gt.token);
      gt.type = GroupToken::nameToken;
      return 1;
    case tokenLit:
    case tokenLita:
      // parameter literal in data tag pattern
      if (!allow.groupToken(GroupToken::dataTagLiteral)) {
	groupTokenInvalidToken(token, allow);
	return 0;
      }
      if (!parseDataTagParameterLiteral(token == tokenLita, gt.text))
	return 0;
      gt.type = GroupToken::dataTagLiteral;
      return 1;
    case tokenAnd:
    case tokenSeq:
    case tokenOr:
    case tokenDtgc:
    case tokenGrpc:
    case tokenOpt:
    case tokenPlus:
    case tokenRep:
      groupTokenInvalidToken(token, allow);
      return 0;
    }
  }
}


void Parser::groupTokenInvalidToken(Token token, const AllowedGroupTokens &allow)
{
  message(Messages::groupTokenInvalidToken,
	  TokenMessageArg(token, grpMode, syntaxPointer(), sdPointer()),
	  AllowedGroupTokensMessageArg(allow, syntaxPointer()));
}


Boolean Parser::parseGroupConnector(const AllowedGroupConnectors &allow,
				    unsigned declInputLevel,
				    unsigned groupInputLevel,
				    GroupConnector &gc)
{
  for (;;) {
    Token token = getToken(grpMode);
    switch (token) {
    case tokenEe:
      if (inputLevel() <= groupInputLevel) {
	message(Messages::groupLevel);
	if (inputLevel() <= declInputLevel)
	  return 0;
      }
      popInputStack();
      break;
    case tokenS:
      break;
    case tokenPeroGrpo:
      if (inInstance()) {
	message(Messages::peroGrpoProlog);
	break;
      }
      // fall through
    case tokenPeroNameStart:
      message(Messages::groupEntityReference);
      break;
    case tokenUnrecognized:
      if (reportNonSgmlCharacter())
	break;
      {
	message(Messages::groupCharacter,
		StringMessageArg(currentToken()),
		AllowedGroupConnectorsMessageArg(allow, syntaxPointer()));
      }
      return 0;
    case tokenAnd:
      if (!allow.groupConnector(GroupConnector::and)) {
	groupConnectorInvalidToken(tokenAnd, allow);
	return 0;
      }
      gc.type = GroupConnector::and;
      return 1;
    case tokenSeq:
      if (!allow.groupConnector(GroupConnector::seq)) {
	groupConnectorInvalidToken(tokenSeq, allow);
	return 0;
      }
      gc.type = GroupConnector::seq;
      return 1;
    case tokenOr:
      if (!allow.groupConnector(GroupConnector::or)) {
	groupConnectorInvalidToken(tokenOr, allow);
	return 0;
      }
      gc.type = GroupConnector::or;
      return 1;
    case tokenDtgc:
      if (!allow.groupConnector(GroupConnector::dtgc)) {
	groupConnectorInvalidToken(tokenDtgc, allow);
	return 0;
      }
      gc.type = GroupConnector::dtgc;
      if (inputLevel() > groupInputLevel)
	message(Messages::groupParameterEntityNotEnded);
      return 1;
    case tokenGrpc:
      if (!allow.groupConnector(GroupConnector::grpc)) {
	groupConnectorInvalidToken(tokenGrpc, allow);
	return 0;
      }
      gc.type = GroupConnector::grpc;
      if (inputLevel() > groupInputLevel)
	message(Messages::groupParameterEntityNotEnded);
      return 1;
    default:
      groupConnectorInvalidToken(token, allow);
      return 0;
    }
  }
}

void Parser::groupConnectorInvalidToken(Token token,
					const AllowedGroupConnectors &allow)
{
  message(Messages::connectorInvalidToken,
	  TokenMessageArg(token, grpMode, syntaxPointer(), sdPointer()),
	  AllowedGroupConnectorsMessageArg(allow, syntaxPointer()));
}

Boolean Parser::parseElementNameGroup(unsigned declInputLevel, Param &parm)
{
  if (!parseNameGroup(declInputLevel, parm))
    return 0;
  parm.elementVector.init(parm.nameTokenVector.length());
  for (size_t i = 0; i < parm.nameTokenVector.length(); i++)
    parm.elementVector[i] = lookupCreateElement(parm.nameTokenVector[i].name);
  return 1;
}

Boolean Parser::parseEntityReferenceNameGroup(Boolean &active)
{
  Param parm;
  if (!parseNameGroup(inputLevel(), parm))
    return 0;
  if (inInstance()) {
    for (size_t i = 0; i < parm.nameTokenVector.length(); i++) {
      const Lpd *lpd = lookupLpd(parm.nameTokenVector[i].name).pointer();
      if (lpd && lpd->active()) {
	active = 1;
	return 1;
      }
    }
  }
  active = 0;
  return 1;
}

Boolean Parser::parseTagNameGroup(Boolean &active)
{
  Param parm;
  if (!parseNameGroup(inputLevel(), parm))
    return 0;
  active = 0;
  return 1;
}

Boolean Parser::parseNameGroup(unsigned declInputLevel, Param &parm)
{
  static AllowedGroupTokens allowName(GroupToken::name);
  return parseGroup(allowName, declInputLevel, parm);
}

Boolean Parser::parseNameTokenGroup(unsigned declInputLevel, Param &parm)
{
  static AllowedGroupTokens allowNameToken(GroupToken::nameToken);
  return parseGroup(allowNameToken, declInputLevel, parm);
}

static
Boolean groupContains(const VectorBase<NameToken> &vec, const CString &str)
{
  for (size_t i = 0; i < vec.length(); i++)
    if (vec[i].name == str)
      return 1;
  return 0;
}

Boolean Parser::parseGroup(const AllowedGroupTokens &allowToken,
			   unsigned declInputLevel,
			   Param &parm)
{
  unsigned groupInputLevel = inputLevel();
  int nDuplicates = 0;
  GrowableVectorD<NameToken> &vec = parm.nameTokenVector;
  vec.clear();
  GroupConnector::Type connector = GroupConnector::grpc;
  GroupToken gt;
  for (;;) {
    if (!parseGroupToken(allowToken, 0, declInputLevel, groupInputLevel, gt))
      return 0;
    if (groupContains(vec, gt.token)) {
      nDuplicates++;
      message(Messages::duplicateGroupToken);
    }
    else {
      vec.grow();
      gt.token.moveTo(vec.last().name);
      getCurrentToken(vec.last().origName);
      vec.last().loc = currentLocation();
    }
    GroupConnector gc;
    static AllowedGroupConnectors allowAnyConnectorGrpc(GroupConnector::or,
							GroupConnector::and,
							GroupConnector::seq,
							GroupConnector::grpc);
			    
    if (!parseGroupConnector(allowAnyConnectorGrpc, declInputLevel,
			     groupInputLevel, gc))
      return 0;
    if (gc.type == GroupConnector::grpc)
      break;
    if (connector == GroupConnector::grpc)
      connector = gc.type;
    else if (gc.type != connector) {
      if (options().warnShould)
	message(Messages::mixedConnectors);
      connector = gc.type;
    }
  }
  if (nDuplicates + vec.length() > syntax().grpcnt())
    message(Messages::groupCount, NumberMessageArg(syntax().grpcnt()));
  return 1;
}

Boolean Parser::parseDataTagGroup(unsigned nestingLevel,
				  unsigned declInputLevel, GroupToken &result)
{
  if (nestingLevel - 1 == syntax().grplvl())
    message(Messages::grplvl, NumberMessageArg(syntax().grplvl()));
  unsigned groupInputLevel = inputLevel();
  GroupToken gt;
  static AllowedGroupTokens allowName(GroupToken::name);
  if (!parseGroupToken(allowName, nestingLevel, declInputLevel,
		       groupInputLevel, gt))
    return 0;
  const ElementType *element = lookupCreateElement(gt.token);
  GroupConnector gc;
  static AllowedGroupConnectors allowSeq(GroupConnector::seq);
  if (!parseGroupConnector(allowSeq, declInputLevel, groupInputLevel, gc))
    return 0;
  static AllowedGroupTokens
    allowDataTagLiteralDataTagTemplateGroup(GroupToken::dataTagLiteral,
					    GroupToken::dataTagTemplateGroup);
  GrowableVectorD<Text> templates;
  if (!parseGroupToken(allowDataTagLiteralDataTagTemplateGroup,
		       nestingLevel,
		       declInputLevel,
		       groupInputLevel,
		       gt))
    return 0;
  if (gt.type == GroupToken::dataTagTemplateGroup)
    gt.textVector.moveTo(templates);
  else
    gt.text.moveTo(templates.grow());
  static AllowedGroupConnectors allowSeqDtgc(GroupConnector::seq,
					     GroupConnector::dtgc);
  if (!parseGroupConnector(allowSeqDtgc, declInputLevel, groupInputLevel, gc))
    return 0;
  Vector<Owner<ContentToken> > vec(2);
  vec[1] = new PcdataToken;
  if (gc.type != GroupConnector::dtgc) {
    static AllowedGroupTokens allowDataTagLiteral(GroupToken::dataTagLiteral);
    if (!parseGroupToken(allowDataTagLiteral,
			 nestingLevel,
			 declInputLevel,
			 groupInputLevel,
			 gt))
      return 0;
    vec[0] = new DataTagElementToken(element, templates, gt.text);
    static AllowedGroupConnectors allowDtgc(GroupConnector::dtgc);
    if (!parseGroupConnector(allowDtgc, declInputLevel, groupInputLevel, gc))
      return 0;
  }
  else
    vec[0] = new DataTagElementToken(element, templates);
  ContentToken::OccurrenceIndicator oi = getOccurrenceIndicator(grpMode);
  result.contentToken = new DataTagGroup(vec, oi);
  result.type = GroupToken::dataTagGroup;
  return 1;
}

Boolean Parser::parseDataTagTemplateGroup(unsigned nestingLevel,
					  unsigned declInputLevel,
					  GroupToken &result)
{
  if (nestingLevel - 1 == syntax().grplvl())
    message(Messages::grplvl, NumberMessageArg(syntax().grplvl()));
  unsigned groupInputLevel = inputLevel();
  GrowableVectorD<Text> &vec(result.textVector);
  for (;;) {
    GroupToken gt;
    static AllowedGroupTokens allowDataTagLiteral(GroupToken::dataTagLiteral);
    if (!parseGroupToken(allowDataTagLiteral,
			 nestingLevel,
			 declInputLevel,
			 groupInputLevel,
			 gt))
      return 0;
    if (vec.length() == syntax().grpcnt())
      message(Messages::groupCount, NumberMessageArg(syntax().grpcnt()));
    gt.text.moveTo(vec.grow());
    static AllowedGroupConnectors allowOrGrpc(GroupConnector::or,
					      GroupConnector::grpc);
    GroupConnector gc;
    if (!parseGroupConnector(allowOrGrpc, declInputLevel, groupInputLevel, gc))
      return 0;
    if (gc.type == GroupConnector::grpc)
      break;
  }
  return 1;
}

Boolean Parser::parseModelGroup(unsigned nestingLevel, unsigned declInputLevel,
				ModelGroup *&group, Mode oiMode)
{
  if (nestingLevel - 1 == syntax().grplvl())
    message(Messages::grplvl, NumberMessageArg(syntax().grplvl()));
  unsigned groupInputLevel = inputLevel();
  GroupToken gt;
  GrowableVectorD<Owner<ContentToken> > tokenVector;
  GroupConnector::Type connector = GroupConnector::grpc;
  
  static AllowedGroupTokens allowContentToken(GroupToken::pcdata,
					      GroupToken::dataTagGroup,
					      GroupToken::elementToken,
					      GroupToken::modelGroup);
  static AllowedGroupConnectors allowAnyConnectorGrpc(GroupConnector::or,
						      GroupConnector::and,
						      GroupConnector::seq,
						      GroupConnector::grpc);
			    
  static AllowedGroupConnectors allowOrGrpc(GroupConnector::or,
					    GroupConnector::grpc);
  static AllowedGroupConnectors allowAndGrpc(GroupConnector::and,
					     GroupConnector::grpc);
  static AllowedGroupConnectors allowSeqGrpc(GroupConnector::seq,
					     GroupConnector::grpc);
  const AllowedGroupConnectors *connectorp = &allowAnyConnectorGrpc;

  GroupConnector gc;
  do {
    if (!parseGroupToken(allowContentToken, nestingLevel, declInputLevel,
			 groupInputLevel, gt))
      return 0;
    ContentToken *contentToken;
    if (gt.type == GroupToken::modelGroup)
      contentToken = gt.model.extract();
    else
      contentToken = gt.contentToken.extract();
    if (tokenVector.length() == syntax().grpcnt())
      message(Messages::groupCount, NumberMessageArg(syntax().grpcnt()));
    tokenVector.grow() = contentToken;
    if (!parseGroupConnector(*connectorp, declInputLevel, groupInputLevel, gc))
      return 0;
    if (tokenVector.length() == 1) {
      connector = gc.type;
      switch (gc.type) {
      case GroupConnector::or:
	connectorp = &allowOrGrpc;
	break;
      case GroupConnector::seq:
	connectorp = &allowSeqGrpc;
	break;
      case GroupConnector::and:
	connectorp = &allowAndGrpc;
	break;
      default:
	break;
      }
    }
  } while (gc.type != GroupConnector::grpc);
  ContentToken::OccurrenceIndicator oi
    = getOccurrenceIndicator(oiMode);
  switch (connector) {
  case GroupConnector::or:
    group = new OrModelGroup(tokenVector, oi);
    break;
  case GroupConnector::grpc:
  case GroupConnector::seq:
    group = new SeqModelGroup(tokenVector, oi);
    break;
  case GroupConnector::and:
    group = new AndModelGroup(tokenVector, oi);
    break;
  default:
    break;
  }
  return 1;
}

ContentToken::OccurrenceIndicator
Parser::getOccurrenceIndicator(Mode oiMode)
{
  Token token = getToken(oiMode);
  switch (token) {
  case tokenPlus:
    return ContentToken::plus;
  case tokenOpt:
    return ContentToken::opt;
  case tokenRep:
    return ContentToken::rep;
  default:
    currentInput()->ungetToken();
    return ContentToken::none;
  }
}

Boolean Parser::parseMinimumLiteral(Boolean lita, Text &text)
{
  return parseLiteral(lita ? mlitaMode : mlitMode, mlitMode,
		      Syntax::referenceQuantity(Syntax::qLITLEN),
		      Messages::minimumLiteralLength,
		      literalSingleSpace|literalMinimumData, text);
}

Boolean Parser::parseSystemIdentifier(Boolean lita, Text &text)
{
  return parseLiteral(lita ? slitaMode : slitMode, slitMode, syntax().litlen(),
		      Messages::systemIdentifierLength, 0, text);
}

Boolean Parser::parseParameterLiteral(Boolean lita, Text &text)
{
  return parseLiteral(lita ? plitaMode : plitMode, pliteMode, syntax().litlen(), 
		      Messages::parameterLiteralLength, 0, text);
}

Boolean Parser::parseDataTagParameterLiteral(Boolean lita, Text &text)
{
  return parseLiteral(lita ? plitaMode : plitMode, pliteMode, syntax().dtemplen(), 
		      Messages::dataTagPatternLiteralLength, literalDataTag, text);
}

Boolean Parser::parseIndicatedReservedName(const AllowedParams &allow,
					   Param &parm)
{
  Syntax::ReservedName rn;
  if (!getIndicatedReservedName(&rn))
    return 0;
  if (!allow.reservedName(rn)) {
    message(Messages::invalidReservedName,
	    StringMessageArg(syntax().reservedName(rn)));
    return 0;
  }
  parm.type = Param::indicatedReservedName + rn;
  return 1;
}

Boolean Parser::parseReservedName(const AllowedParams &allow,
				  Param &parm)
{
  Syntax::ReservedName rn;
  if (!getReservedName(&rn))
    return 0;
  if (!allow.reservedName(rn)) {
    message(Messages::invalidReservedName,
	    StringMessageArg(syntax().reservedName(rn)));
    return 0;
  }
  parm.type = Param::reservedName + rn;
  return 1;
}


Boolean Parser::parseAttributeValueParam(Param &parm)
{
  extendNameToken(syntax().litlen() > syntax().normsep()
		  ? syntax().litlen() - syntax().normsep()
		  : 0,
		  Messages::attributeValueLength);
  parm.type = Param::attributeValue;
  Text text;
  text.addChars(currentInput()->currentTokenStart(),
		currentInput()->currentTokenLength(),
		currentLocation());
  text.moveTo(parm.literalText);
  return 1;
}

Boolean Parser::getIndicatedReservedName(Syntax::ReservedName *result)
{
  InputSource *in = currentInput();
  in->startToken();
  if (!syntax().isNameStartCharacter(in->tokenChar(inputContext()))) {
    message(Messages::rniNameStart);
    return 0;
  }
  extendNameToken(syntax().namelen(), Messages::nameLength);
  CString &buffer(nameBuffer());
  getCurrentToken(syntax().generalSubstTable(), buffer);
  if (!syntax().lookupReservedName(buffer, result)) {
    message(Messages::noSuchReservedName, StringMessageArg(buffer));
    return 0;
  }
  return 1;
}

Boolean Parser::getReservedName(Syntax::ReservedName *result)
{
  extendNameToken(syntax().namelen(), Messages::nameLength);
  CString &buffer(nameBuffer());
  getCurrentToken(syntax().generalSubstTable(), buffer);
  if (!syntax().lookupReservedName(buffer, result)) {
    message(Messages::noSuchReservedName, StringMessageArg(buffer));
    return 0;
  }
  return 1;
}

