// Copyright (c) 1994 James Clark
// See the file COPYING for copying permission.

#include "SgmlParser.H"
#include "Event.H"
#include "MessageReporter.H"
#include "RastEventHandler.H"
#include "OutputCharStream.H"
#include "EasyParser.H"
#include "Boolean.H"
#include "programName.H"
#include "macros.H"
#include "getopt.h"

#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>

static Boolean exotericaMode = 0;

static void usage();

class XRastEventHandler : public RastEventHandler {
public:
  XRastEventHandler(SgmlParser *,
		    const char *filename,
		    const OutputCodingSystem *,
		    MessageReporter *reporter);
  void reportMessage(const MessageEvent &);
  void message(MessageEvent *);
  void truncateOutput();
private:
  MessageReporter *reporter_;
  // file_ must come before os_ so it gets inited first
  filebuf file_;
  IosOutputCharStream os_;
  const char *filename_;
};

int main(int argc, char **argv)
{
  programName = argv[0];
  int opt;
  EasyParser easy;

  while ((opt = getopt(argc, argv, "ewxc:")) >= 0) {
    switch (opt) {
    case 'c':
      easy.addCatalogSysid(optarg);
      break;
    case 'e':
      // describe open entities in error messages
      easy.addOption(MessageReporter::openEntities);
      break;
    case 'x':
      exotericaMode = 1;
      break;
    case 'w':
      easy.options.warnShould = 1;
      easy.options.warnUndefinedElement = 1;
      easy.options.warnSgmlDecl = 1;
      easy.options.warnDuplicateEntity = 1;
      easy.options.warnDefaultEntityReference = 1;
      break;
    case '?':
      usage();
    default:
      CANNOT_HAPPEN();
    }
  }
  if (argc - optind < 2)
    usage();
  easy.init(argc - optind - 1, argv + optind);
  SgmlParser &parser = easy.parser();
  XRastEventHandler handler(&parser, argv[argc - 1], easy.codingSystem(),
			    &easy);
  parser.parseAll(handler);
  handler.end();
  return handler.hadError();
}

static
void usage()
{
  cerr << "usage: "
       << programName
       << " [-ewx] [-c catalog] filename ... rastfile\n";
  exit(1);
}

XRastEventHandler::XRastEventHandler(SgmlParser *parser,
				     const char *filename,
				     const OutputCodingSystem *codingSystem,
				     MessageReporter *reporter)
: RastEventHandler(parser, reporter),
  reporter_(reporter),
  filename_(filename)
{
  if (!file_.open(filename, ios::out|ios::trunc)) {
    // FIXME
    cerr << programName << ": couldn't open " << filename << '\n';
    exit(1);
  }
  os_.open(&file_, codingSystem);
  setOutputStream(&os_);
}

void XRastEventHandler::truncateOutput()
{
  os_.flush();
  file_.close();		// FIXME handle error
  if (!file_.open(filename_, ios::out|ios::trunc)) {
    // FIXME
    cerr << programName << ": couldn't open " << filename_ << '\n';
    exit(1);
  }
}

void XRastEventHandler::message(MessageEvent *event)
{
  // FIXME
  if (exotericaMode
      && event->source() == SgmlParser::messageSource
      && event->number() == 139) {
    delete event;
    return;
  }
  reporter_->reportMessage(*event);
  RastEventHandler::message(event);
}
