/*
 * XMLStreamWriterImpl.java
 *
 * Copyright 2004 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.xml.stream.writers;

import com.sun.xml.stream.Constants;
import java.io.*;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.util.Random;
import java.util.ArrayList;
import javax.xml.namespace.NamespaceContext;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.transform.stream.StreamResult;
import com.sun.xml.stream.xerces.util.NamespaceSupport;
import com.sun.xml.stream.xerces.xni.QName;
import com.sun.xml.stream.PropertyManager;
import com.sun.xml.stream.util.ReadOnlyIterator;
import java.util.Iterator;
import java.util.Vector;
import com.sun.xml.stream.xerces.util.SymbolTable;
import javax.xml.XMLConstants;
import javax.xml.stream.XMLOutputFactory;


/**
 *
 * @author Neeraj Bajaj,K.Venugopal Sun Microsystems.
 */

/*
    TODO::-Venu
1.restructuring of stream writer classes.
2.Performance improvement.(use of appropriate datastructures).
3.encoding and valid xml char check based on implementation specific properties
4.identify empty elements.
 
 
 */

public class XMLStreamWriterImpl extends WriterUtility implements XMLStreamWriter {
            
    private ElementState fCurrentElementState                 = null;
    private ElementState fillElementState                     = null;
    //attribute related variable
    private ArrayList fAttributeCache                         = null;
    
    //contains prefix of namespace decl to be written.    
    private ArrayList fNamespaceDecls                         = null;
    /* Namespace context encapsulating user specified context
     * and context built by the writer  */
    private NamespaceContextImpl fNamespaceContext            = null;    
    private NamespaceSupport fInternalNamespaceContext        = null ;
    private Random fPrefixGen                                 = null;
    
    /** Reference to PropertyManager */
    private PropertyManager fPropertyManager                  = null;
    
    /** Flag to track if start tag is opened */
    private boolean fStartTagOpened                           = false;
    
    /** Flag for the value of repairNamespace property */
    private boolean fIsRepairingNamespace                     = false;
    
    /* boolean flag  to indicate, if instance can be reused */
    private boolean fReuse ;
    
    private SymbolTable fSymbolTable = new SymbolTable();        
    private ElementStack fElementStack = new ElementStack();    //Change this .-Venu    
    final private String DEFAULT_PREFIX = fSymbolTable.addSymbol("");    
    private final ReadOnlyIterator fReadOnlyIterator = new ReadOnlyIterator();
    private QName fTempQname = new QName();
    //debug flags
    private static final boolean DEBUG = false;
    
    /**
     * Creates a new instance of XMLStreamWriterImpl
     * @param outputStream Stream to write to.
     * @param props properties to be used by this writer.
     */
    public XMLStreamWriterImpl(OutputStream outputStream , PropertyManager props) throws java.io.IOException{
        this(outputStream, null, props);
    }
        
    /**
     *
     * @param props properties to be used by this writer.
     * @param outputStream outputStream Stream to write to.
     * @param encoding character encoding to be used.
     * @throws IOException
     */
    public XMLStreamWriterImpl(OutputStream outputStream, String encoding,PropertyManager props) throws java.io.IOException{        
        this(new OutputStreamWriter(outputStream,encoding), props);
    }
    
    /**
     * @param props props properties to be used by this writer.
     * @param writer data will be written to this writer.
     */
    public XMLStreamWriterImpl(Writer writer, PropertyManager props) throws java.io.IOException {
        this(new StreamResult(writer), null, props);
    }
    
    public XMLStreamWriterImpl(StreamResult sr, String encoding, PropertyManager props)throws java.io.IOException{
        setOutput(sr, encoding);
        fPropertyManager = props;
        init();
    }
    
    private void init(){
        fReuse = false;        
        if(fNamespaceDecls == null) fNamespaceDecls = new ArrayList();
        if(fPrefixGen == null) fPrefixGen = new Random();        
        if(fAttributeCache == null) fAttributeCache = new ArrayList();
        if(fInternalNamespaceContext == null) fInternalNamespaceContext = new NamespaceSupport();        
        if(fNamespaceContext == null) fNamespaceContext = new NamespaceContextImpl();
        fNamespaceContext.internalContext = fInternalNamespaceContext;        
        if(fillElementState == null) fillElementState = new ElementState();        
        if(fPropertyManager != null ){
            Boolean ob =(Boolean) fPropertyManager.getProperty(XMLOutputFactory.IS_REPAIRING_NAMESPACES);
            fIsRepairingNamespace = ob.booleanValue();
            ob = (Boolean) fPropertyManager.getProperty(Constants.ESCAPE_CHARACTERS);
            setEscapeCharacters(ob.booleanValue());
        }
    }
    
    /** If the properties have changed, serializer checks again from 
     *  the given PropertyManager to find out the property values.
     *  
     *  //xxx: use this function to call reset() from Factory when
     *  property has changed. 
     */
    void reset(boolean resetProperties){
        if(!fReuse){
            throw new java.lang.IllegalStateException("close() Must be called before calling reset()");
        }
        //set to fReuse to false;
        fReuse = false;
               
        fNamespaceDecls.clear();
        fAttributeCache.clear();
        fInternalNamespaceContext.reset();
        fillElementState.clear();
        fStartTagOpened = false;
        fNamespaceContext.userContext = null ;        
        //No need to reset fIsRepairingNamespace , fEscapeCharacters
        //unless called by application
        if(resetProperties){            
            Boolean ob = (Boolean) fPropertyManager.getProperty(XMLOutputFactory.IS_REPAIRING_NAMESPACES);
            fIsRepairingNamespace = ob.booleanValue();
            ob = (Boolean) fPropertyManager.getProperty(Constants.ESCAPE_CHARACTERS);
            setEscapeCharacters(ob.booleanValue());            
        }
    }
    
    /** Set the output (<code>StreamResult</code> object) where this writer would 
     *  write. If both the byte stream and character stream is set in StreamResult,
     *  byte stream is given preference over character stream. 
     *
     *  Encoding information is used to encode the characters to bytes. 
     *  If encoding information is not available, platform default encoding is used.
     *
     * @param StreamResult StreamResult object 
     * @param String       encoding to use.
     */
    public void setOutput(StreamResult sr, String encoding) throws IOException{
        //if stream is provided, give preference to stream
        if(sr.getOutputStream() != null){    
            if(encoding != null){
                setWriter(new OutputStreamWriter(sr.getOutputStream(), encoding));
            }else{
                setWriter(new OutputStreamWriter(sr.getOutputStream()));
            }
            
        }//else use writer
        else if(sr.getWriter() != null){
            setWriter(sr.getWriter());
        }
        else if(sr.getSystemId() != null){
            //Use FileOutputStream than FileWriter because it uses platform default encoding
            if(encoding != null){
                setWriter(new OutputStreamWriter(
                            new FileOutputStream(sr.getSystemId()) ,encoding));
            }else{
                setWriter(new OutputStreamWriter(
                            new FileOutputStream(sr.getSystemId())));
            }            
        }        
    }//setOutput
        
    
    /** Resets this instance so that it can be reused again.
     *  One must call setOutput() before invoking any other
     *  function of <code>XMLStreamWriter</code>.
     *
     *  This instance must have been closed by calling <strong>close()</strong> 
     *  before calling reset(). If reset() is called without closing this instance, 
     *  an <code>IllegalStateException</code> would be thrown.
     *
     *  @see #setOutput()
     *  @throws java.lang.IllegalStateException
     */
    public void reset(){
        reset(false);
    }
    
    
    /** Can this instance be reused
     *
     * @return boolean boolean value to indicate if this instance can be reused or not
     */
    public boolean canReuse(){
        return fReuse;
    }
    
    /**
     * Close the underlying stream to which data is being written.
     * @throws XMLStreamException if an error occurs while closing the stream.
     */
    public void close() throws XMLStreamException {
        fReuse = true;
        try{
            fWriter.close();
        }catch(IOException ioexception){
            throw new XMLStreamException(ioexception);
        }
    }
    
    /**
     * Flush underlying stream.
     *
     * @throws XMLStreamException if an error occurs while flushing the stream.
     */
    public void flush() throws XMLStreamException {
        try{
            fWriter.flush();
        }catch(IOException ioexception){
            throw new XMLStreamException(ioexception);
        }
    }
    
    /**
     * Return <code>NamespaceContext</code> being used by the writer.
     *
     * @return NamespaceContext
     */
    public NamespaceContext getNamespaceContext() {
        return fNamespaceContext;
    }
    
    
    /**
     * Return the prefix associated with specified uri.
     *
     * @param uri NamespaceURI
     * @throws XMLStreamException if uri specified is "" or null.
     * @return Return current prefix associated with the specified NamespaceURI.
     * @see javax.xml.namespace#getPrefix
     */
    public String getPrefix(String uri) throws XMLStreamException {
        return fNamespaceContext.getPrefix(uri);
    }
    
    /**
     * Return value associated with the specified property name.
     *
     * @param str Property name
     * @throws IllegalArgumentException if the specified property is not supported.
     * @return value associated with the specified property.
     */
    public Object getProperty(String str) throws java.lang.IllegalArgumentException {
        if(str == null) throw new NullPointerException();
        if(!fPropertyManager.containsProperty(str))
            throw new java.lang.IllegalArgumentException(str+" is not supported");
        
        return fPropertyManager.getProperty(str);
    }
    
    /**
     * Set the specified URI as default namespace in the current namespace context.
     *
     * @param uri Namespace URI
     * @throws XMLStreamException
     */
    public void setDefaultNamespace(String uri) throws XMLStreamException {
        if(uri!= null )uri = fSymbolTable.addSymbol(uri);
        if(fIsRepairingNamespace){
            if(isDefaultNamespace(uri)){
                return;
            }
            QName qname = new QName();
            qname.setValues(DEFAULT_PREFIX,"xmlns", null,uri);
            fNamespaceDecls.add(qname);
            return;
        }else
            fInternalNamespaceContext.declarePrefix(DEFAULT_PREFIX, uri);
    }
    
    /**
     * Sets the current <code>NamespaceContext</code> for prefix and uri bindings. 
     * This context becomes the root namespace context for writing and 
     * will replace the current root namespace context. Subsequent calls 
     * to setPrefix and setDefaultNamespace will bind namespaces using 
     * the context passed to the method as the root context for resolving 
     * namespaces. This method may only be called once at the start of the 
     * document. It does not cause the namespaces to be declared. If a 
     * namespace URI to prefix mapping is found in the namespace context 
     * it is treated as declared and the prefix may be used by the <code>XMLStreamWriter</code>.
     *
     * @param namespaceContext the namespace context to use for this writer, may not be null
     * @throws XMLStreamException
     */
    public void setNamespaceContext(NamespaceContext namespaceContext) throws XMLStreamException {
        fNamespaceContext.userContext = namespaceContext;
    }
    
    /**
     * Sets the prefix the uri is bound to. This prefix is bound in the scope of 
     * the current START_ELEMENT / END_ELEMENT pair. If this method is called before 
     * a START_ELEMENT has been written the prefix is bound in the root scope.
     *
     * @param prefix
     * @param uri
     * @throws XMLStreamException
     */
    public void setPrefix(String prefix, String uri) throws XMLStreamException {
        
        if(prefix == null ) throw new XMLStreamException("Prefix cannot be null");
        if(uri == null ) throw new XMLStreamException("URI cannot be null");
        
        prefix = fSymbolTable.addSymbol(prefix);
        uri = fSymbolTable.addSymbol(uri);
        
        if(fIsRepairingNamespace){
            String tmpURI = fInternalNamespaceContext.getURI(prefix);
            if(tmpURI != null && tmpURI == uri){
                return;
            }
            
            if(checkUserNamespaceContext(prefix,uri))
                return;
            QName qname = new QName();
            qname.setValues(prefix,XMLConstants.XMLNS_ATTRIBUTE, null,uri);
            fNamespaceDecls.add(qname);
            
            return;
        }
        
        fInternalNamespaceContext.declarePrefix(prefix, uri);
        
    }
    
    /**
     * @param localName
     * @param value
     * @throws XMLStreamException
     */
    public void writeAttribute(String localName, String value) throws XMLStreamException {
        try{
            if(!fStartTagOpened)
                throw new XMLStreamException("Attribute not associated with any element");
            if(fIsRepairingNamespace){
                Attribute attr = new Attribute(value); // Revisit:Dont create new one's. Reuse.-Venu
                attr.setValues(null,localName, null,null);
                fAttributeCache.add(attr);
                return;
            }
            fWriter.write(" ");
            fWriter.write(localName);
            fWriter.write("=");
            fWriter.write("\"");
            writeXMLAttributeValue(value);
            fWriter.write("\"");
        }catch(IOException ie){
            if(DEBUG)ie.printStackTrace();
            throw new XMLStreamException(ie);
        }
    }
    
    /**
     * @param namespaceURI
     * @param localName
     * @param value
     * @throws XMLStreamException
     */
    public void writeAttribute(String namespaceURI, String localName, String value) throws XMLStreamException {
        try{
            if(!fStartTagOpened)
                throw new XMLStreamException("Attribute not associated with any element");
            if(namespaceURI == null )
                throw new XMLStreamException("NamespaceURI cannot be null");
            namespaceURI = fSymbolTable.addSymbol(namespaceURI);
            String prefix = fInternalNamespaceContext.getPrefix(namespaceURI);
            if(!fIsRepairingNamespace){
                if(prefix == null)
                    throw new XMLStreamException("Prefix cannot be null");
            }else{
                Attribute attr = new Attribute(value);
                attr.setValues(null,localName, null, namespaceURI);
                fAttributeCache.add(attr);
                return;
            }
            writeAttributeWithPrefix(prefix,localName,value);
        }catch(IOException ie){
            if(DEBUG)ie.printStackTrace();
            throw new XMLStreamException(ie);
        }
    }
    
    private void writeAttributeWithPrefix(String prefix,String localName,String value)throws IOException{
        fWriter.write(SPACE);
        if(prefix != null && prefix != XMLConstants.DEFAULT_NS_PREFIX){
            fWriter.write(prefix);
            fWriter.write(":");
        }
        fWriter.write(localName);
        fWriter.write("=");
        fWriter.write("\"");
        writeXMLAttributeValue(value);
        fWriter.write("\"");
    }
    
    /**
     * @param prefix
     * @param namespaceURI
     * @param localName
     * @param value
     * @throws XMLStreamException
     */
    public void writeAttribute(String prefix, String namespaceURI, String localName, String value) throws XMLStreamException {
        try{
            if(!fStartTagOpened)
                throw new XMLStreamException("Attribute not associated with any element");
            
            if(namespaceURI == null )
                throw new XMLStreamException("NamespaceURI cannot be null");
            if(localName == null)
                throw new XMLStreamException("Local name cannot be null");
            
            
            if(!fIsRepairingNamespace){
                if(prefix == null)
                    throw new XMLStreamException("prefix cannot be null");
            } else {
                if(prefix != null) prefix = fSymbolTable.addSymbol(prefix );
                namespaceURI = fSymbolTable.addSymbol(namespaceURI );
                Attribute attr = new Attribute(value);
                attr.setValues(prefix,localName, null, namespaceURI);
                fAttributeCache.add(attr);
                return;
            }
            writeAttributeWithPrefix(prefix,localName,value);
        }catch(IOException ie){
            if(DEBUG)ie.printStackTrace();
            throw new XMLStreamException(ie);
        }
    }
    
    /**
     * @param cdata
     * @throws XMLStreamException
     */
    public void writeCData(String cdata) throws XMLStreamException {
        try{
            
            if(cdata == null)
                throw new XMLStreamException("cdata cannot be null");
            if(fStartTagOpened){
                closeStartTag();
            }
            fWriter.write(START_CDATA);
            fWriter.write(cdata);
            fWriter.write(END_CDATA);
        }catch(IOException ie){
            if(DEBUG) ie.printStackTrace();
            throw new XMLStreamException(ie);
        }
    }
    
    /**
     * @param data
     * @throws XMLStreamException
     */
    public void writeCharacters(String data) throws XMLStreamException {
        try{
            if(fStartTagOpened){
                closeStartTag();
            }
            writeXMLContent(data);
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
    }
    
    /**
     * @param data
     * @param start
     * @param len
     * @throws XMLStreamException
     */
    public void writeCharacters(char[] data, int start, int len) throws XMLStreamException {
        try{
            if(fStartTagOpened){
                closeStartTag();
            }
            writeXMLContent(data,start,len);
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
        
    }
    
    /**
     * @param comment
     * @throws XMLStreamException
     */
    public void writeComment(String comment) throws XMLStreamException {
        try{
            fWriter.write(START_COMMENT);
            if(comment!=null)
                fWriter.write(comment);
            fWriter.write(END_COMMENT);
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
    }
    
    /**
     * @param dtd
     * @throws XMLStreamException
     */
    public void writeDTD(String dtd) throws XMLStreamException {
        try{
            if(fStartTagOpened){
                closeStartTag();
            }
            fWriter.write(dtd);
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
    }
    
    /**
     * @param uri
     * @throws XMLStreamException
     */
    public void writeDefaultNamespace(String uri) throws XMLStreamException {
        try{
            if(!fStartTagOpened)
                throw new XMLStreamException("Namespace Attribute not associated with any element");
            if(fIsRepairingNamespace){
                QName  qname = new QName();
                qname.setValues(XMLConstants.DEFAULT_NS_PREFIX,XMLConstants.XMLNS_ATTRIBUTE,null, uri);
                fNamespaceDecls.add(qname);
                return;
            }
            fWriter.write(SPACE);
            fWriter.write("xmlns");
            fWriter.write("=");
            fWriter.write("\"");
            fWriter.write(uri);
            fWriter.write("\"");
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
    }
    
    /**
     * @param localName
     * @throws XMLStreamException
     */
    public void writeEmptyElement(String localName) throws XMLStreamException {
        try{
            if(fStartTagOpened){
                closeStartTag();
            }
            openStartTag();
            fElementStack.push(null, localName, null, null,true);
            fInternalNamespaceContext.pushContext();
            if(fIsRepairingNamespace){
                return;
            }
            fWriter.write(localName);
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
    }
    
    /**
     * @param namespaceURI
     * @param localName
     * @throws XMLStreamException
     */
    public void writeEmptyElement(String namespaceURI, String localName) throws XMLStreamException {
        if( namespaceURI == null)
            throw new XMLStreamException("NamespaceURI cannot be null");
        namespaceURI = fSymbolTable.addSymbol(namespaceURI);
        
        String prefix = fNamespaceContext.getPrefix(namespaceURI);
        writeEmptyElement(prefix,localName,namespaceURI);
    }
    
    /**
     * @param prefix
     * @param localName
     * @param namespaceURI
     * @throws XMLStreamException
     */
    public void writeEmptyElement(String prefix, String localName, String namespaceURI) throws XMLStreamException {
        try{
            if( localName == null )
                throw new XMLStreamException("Local Name cannot be null");
            
            if( namespaceURI == null)
                throw new XMLStreamException("NamespaceURI cannot be null");
            if(prefix != null) prefix = fSymbolTable.addSymbol(prefix);
            namespaceURI = fSymbolTable.addSymbol(namespaceURI);
            
            if(fStartTagOpened){
                closeStartTag();
            }
            
            openStartTag();
            
            fElementStack.push(prefix, localName, null, namespaceURI,true);
            fInternalNamespaceContext.pushContext();
            if(!fIsRepairingNamespace){
                if(prefix == null )
                    throw new XMLStreamException("NamespaceURI "+namespaceURI+" has not been bound to any prefix");
            }else{
                return;
            }
            
            
            if(prefix != null && prefix != XMLConstants.DEFAULT_NS_PREFIX ){
                fWriter.write(prefix);
                fWriter.write(":");
            }
            
            fWriter.write(localName);
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
    }
    
    /**
     * @throws XMLStreamException
     */
    public void writeEndDocument() throws XMLStreamException {
        try{
            if(fStartTagOpened){
                closeStartTag();
            }
            ElementState elem = null;
            while(!fElementStack.empty()){
                elem =(ElementState) fElementStack.pop();
                fInternalNamespaceContext.popContext();
                if(elem.isEmpty){
                    //fWriter.write(CLOSE_EMPTY_ELEMENT);
                }else{
                    fWriter.write(OPEN_END_TAG);
                    if(elem.prefix != null && ! (elem.prefix).equals("")){
                        fWriter.write(elem.prefix);
                        fWriter.write(":");
                    }
                    fWriter.write(elem.localpart);
                    fWriter.write(CLOSE_END_TAG);
                }
            }
        }catch(IOException ie){
            if(DEBUG) ie.printStackTrace();
            throw new XMLStreamException(ie);
        }catch(java.lang.ArrayIndexOutOfBoundsException ae){
            throw new XMLStreamException("No more elements to write");
        }
    }
    
    /**
     * @throws XMLStreamException
     */
    public void writeEndElement() throws XMLStreamException {
        try{
            if(fStartTagOpened){
                closeStartTag();
            }
            fCurrentElementState = (ElementState)fElementStack.pop();
            if(fCurrentElementState == null )
                throw new XMLStreamException("No element was found to write");
            if(fCurrentElementState.isEmpty){
                //fWriter.write(CLOSE_EMPTY_ELEMENT);
                return;
            }
            fWriter.write(OPEN_END_TAG);
            if(fCurrentElementState.prefix != null && ! (fCurrentElementState.prefix).equals("")){
                fWriter.write(fCurrentElementState.prefix);
                fWriter.write(":");
            }
            fWriter.write(fCurrentElementState.localpart);
            fWriter.write(CLOSE_END_TAG);
            fInternalNamespaceContext.popContext();
        }catch(IOException ie) {
            if(DEBUG) ie.printStackTrace();
            throw new XMLStreamException(ie);
        }catch(java.lang.ArrayIndexOutOfBoundsException ae){
            throw new XMLStreamException("No element was found to write");
        }
    }
    
    /**
     * @param refName
     * @throws XMLStreamException
     */
    public void writeEntityRef(String refName) throws XMLStreamException {
        try {
            if(fStartTagOpened){
                closeStartTag();
            }
            fWriter.write('&');
            fWriter.write(refName);
            fWriter.write(';');
        }
        catch(IOException ex) {
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
    }
    
    /**
     * @param prefix
     * @param namespaceURI
     * @throws XMLStreamException
     */
    public void writeNamespace(String prefix, String namespaceURI) throws XMLStreamException {
        try{
            QName qname = null;
            
            if(!fStartTagOpened)
                throw new XMLStreamException("Namespace Attribute not associated with any element");
            
            if(prefix == null || prefix.equals("") || prefix.equals("xmlns")){
                writeDefaultNamespace(namespaceURI);
                return;
            }
            
            if(namespaceURI == null)
                throw new XMLStreamException("Namespace`uri cannot be null");
            
            prefix = fSymbolTable.addSymbol(prefix);
            namespaceURI = fSymbolTable.addSymbol(namespaceURI);
            
            
            if(fIsRepairingNamespace){
                String tmpURI = fInternalNamespaceContext.getURI(prefix);
                if(tmpURI != null && tmpURI == namespaceURI)
                    return;
                qname = new QName();
                qname.setValues(prefix,XMLConstants.XMLNS_ATTRIBUTE,null, namespaceURI);
                fNamespaceDecls.add(qname);
                return;
            }
            
            String tmp = fInternalNamespaceContext.getURI(prefix);
            if(tmp == null || !(tmp != namespaceURI)){
                fInternalNamespaceContext.declarePrefix(prefix,namespaceURI);
            }
            writenamespace(prefix,namespaceURI);
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
    }
    
    
    private void writenamespace(String prefix , String namespaceURI) throws IOException{
        fWriter.write(SPACE);
        fWriter.write("xmlns");
        if(prefix != null && prefix != XMLConstants.DEFAULT_NS_PREFIX){
            fWriter.write(":");
            fWriter.write(prefix);
        }
        fWriter.write("=");
        fWriter.write("\"");
        fWriter.write(namespaceURI);
        fWriter.write("\"");
    }
    
    /**
     * @param target
     * @throws XMLStreamException
     */
    public void writeProcessingInstruction(String target) throws XMLStreamException {
        try{
            if(fStartTagOpened){
                closeStartTag();
            }
            if(target != null ){
                fWriter.write("<?");
                fWriter.write(target);
                fWriter.write("?>");
                return;
            }
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
        throw new  XMLStreamException("PI target cannot be null");
    }
    
    /**
     * @param target
     * @param data
     * @throws XMLStreamException
     */
    public void writeProcessingInstruction(String target, String data) throws XMLStreamException {
        try{
            if(fStartTagOpened){
                closeStartTag();
            }
            if(target == null || data == null ){
                throw new  XMLStreamException("PI target cannot be null");
            }
            fWriter.write("<?");
            fWriter.write(target);
            fWriter.write(SPACE);
            fWriter.write(data);
            fWriter.write("?>");
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
    }
    
    /**
     * @throws XMLStreamException
     */
    public void writeStartDocument() throws XMLStreamException {
        try{
            fWriter.write(DEFAULT_XMLDECL);
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
    }
    
    
    /**
     * @param version
     * @throws XMLStreamException
     */
    public void writeStartDocument(String version) throws XMLStreamException {
        try{
            if(version == null || version.equals("")){
                writeStartDocument();
                return;
            }
            fWriter.write("<?xml version=\"");
            fWriter.write(version);
            fWriter.write("\"");
            //fWriter.write(DEFAULT_ENCODING);
            fWriter.write("?>");
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
    }
    
    /**
     * @param encoding
     * @param version
     * @throws XMLStreamException
     */
    public void writeStartDocument(String encoding, String version) throws XMLStreamException {
        //Revisit : What about standalone ?
        try{
            if( encoding == null  &&  version == null ){
                writeStartDocument();
                return;
            }
            
            if(encoding == null){
                writeStartDocument(version);
                return;
            }
            fWriter.write("<?xml version=\"");
            if(version == null || version.equals(""))
                fWriter.write(DEFAULT_XML_VERSION);
            else
                fWriter.write(version);
            
            if(!encoding.equals("")){
                fWriter.write("\" encoding=\"");
                fWriter.write(encoding);
            }
            
            fWriter.write('\"');
            fWriter.write("?>");
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
    }
    
    /**
     * @param localName
     * @throws XMLStreamException
     */
    public void writeStartElement(String localName) throws XMLStreamException {
        try{
            if( localName == null )
                throw new XMLStreamException("Local Name cannot be null");
            
            if(fStartTagOpened){
                closeStartTag();
            }
            openStartTag();
            fElementStack.push(null, localName, null, null,false);
            fInternalNamespaceContext.pushContext();
            if(fIsRepairingNamespace)
                return;
            fWriter.write(localName);
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
        
    }
    
    /**
     * @param namespaceURI
     * @param localName
     * @throws XMLStreamException
     */
    public void writeStartElement(String namespaceURI, String localName) throws XMLStreamException {
        if( localName == null )
            throw new XMLStreamException("Local Name cannot be null");
        
        if( namespaceURI == null)
            throw new XMLStreamException("NamespaceURI cannot be null");
        
        namespaceURI = fSymbolTable.addSymbol(namespaceURI);
        String prefix = null;
        if(!fIsRepairingNamespace){
            prefix = fNamespaceContext.getPrefix(namespaceURI);
            if(prefix != null) prefix = fSymbolTable.addSymbol(prefix);
        }
        writeStartElement(prefix,localName,namespaceURI);
    }
    
    /**
     * @param prefix
     * @param localName
     * @param namespaceURI
     * @throws XMLStreamException
     */
    public void writeStartElement(String prefix, String localName, String namespaceURI) throws XMLStreamException {
        try{
            if( localName == null )
                throw new XMLStreamException("Local Name cannot be null");
            
            if( namespaceURI == null)
                throw new XMLStreamException("NamespaceURI cannot be null");
            
            if(!fIsRepairingNamespace){
                if(prefix == null)
                    throw new XMLStreamException("Prefix cannot be null");
            }
            
            if(fStartTagOpened){
                closeStartTag();
            }
            
            openStartTag();
            namespaceURI = fSymbolTable.addSymbol(namespaceURI);
            if(prefix != null) prefix = fSymbolTable.addSymbol(prefix);
            fElementStack.push(prefix, localName, null, namespaceURI,false);
            fInternalNamespaceContext.pushContext();
            String tmpPrefix = fNamespaceContext.getPrefix(namespaceURI);
            if(prefix != null && (tmpPrefix == null  || !prefix.equals(tmpPrefix)))
                fInternalNamespaceContext.declarePrefix(prefix,namespaceURI);
            
            if(fIsRepairingNamespace){
                if(prefix == null || (tmpPrefix != null  && prefix.equals(tmpPrefix)))
                    return;
                
                QName qname = new QName();
                qname.setValues(prefix,XMLConstants.XMLNS_ATTRIBUTE, null,namespaceURI);
                fNamespaceDecls.add(qname);
                return;
            }
            
            if(prefix != null && prefix != XMLConstants.DEFAULT_NS_PREFIX){
                fWriter.write(prefix);
                fWriter.write(":");
            }
            
            fWriter.write(localName);
            
        }catch(IOException ex){
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
        
    }
    
    /**
     * marks close of start tag and writes the same into the writer.
     */
    private void closeStartTag() throws XMLStreamException{
        try{
            fCurrentElementState = fElementStack.peek();
            if(fIsRepairingNamespace){
                repair();
                correctPrefix(fCurrentElementState);
                if(fCurrentElementState.prefix != null &&  fCurrentElementState.prefix != XMLConstants.DEFAULT_NS_PREFIX){
                    fWriter.write(fCurrentElementState.prefix);
                    fWriter.write(":");
                }
                fWriter.write(fCurrentElementState.localpart);
                int len = fNamespaceDecls.size();
                QName qname = null;
                
                for(int i=0; i<len ; i++){
                    qname =(QName) fNamespaceDecls.get(i);
                    if(qname != null){
                        fInternalNamespaceContext.declarePrefix(qname.prefix,qname.uri);
                        writenamespace(qname.prefix,qname.uri);
                    }
                }
                fNamespaceDecls.clear();
                Attribute attr = null;
                for(int j=0; j< fAttributeCache.size();j++){
                    attr = (Attribute)fAttributeCache.get(j);
                    if(attr.prefix != null && attr.uri != null) {
                        String tmp = fInternalNamespaceContext.getPrefix(attr.uri);
                        if(tmp == null || tmp != attr.prefix){
                            fInternalNamespaceContext.declarePrefix(attr.prefix,attr.uri);
                            writenamespace(attr.prefix,attr.uri);
                        }
                    }
                    writeAttributeWithPrefix(attr.prefix,attr.localpart, attr.value);
                }
                fAttributeCache.clear();
            }
            if(fCurrentElementState.isEmpty){
                fElementStack.pop();
                fWriter.write(CLOSE_EMPTY_ELEMENT);
            }else
                fWriter.write(CLOSE_START_TAG);
            fStartTagOpened = false;
            
        }catch(IOException ex){
            fStartTagOpened = false;
            if(DEBUG) ex.printStackTrace();
            throw new XMLStreamException(ex);
        }
    }
    
    
    /**
     * marks open of start tag and writes the same into the writer.
     */
    private void openStartTag() throws IOException{
        fStartTagOpened = true;
        fWriter.write(OPEN_START_TAG);
    }
    
    
    /**
     *
     * @param uri
     * @return
     */
    private void correctPrefix(QName attr) {
        String tmpPrefix = null;
        String prefix;
        String uri;
        prefix = attr.prefix;
        uri = attr.uri;
        
        if( prefix == null){
            if(uri== null)
                return;
            
            uri = fSymbolTable.addSymbol(uri);
            QName decl =null;
            
            for(int i =0; i<fNamespaceDecls.size();i++){
                decl = (QName)fNamespaceDecls.get(i);
                if(decl != null && decl.uri == attr.uri){
                    attr.prefix = decl.prefix;
                    return;
                }
            }
            
            tmpPrefix = fNamespaceContext.getPrefix(uri);
            
            if(tmpPrefix == XMLConstants.DEFAULT_NS_PREFIX)
                return;
            if(tmpPrefix == null ){
                StringBuffer genPrefix = new StringBuffer("zdef");
                for(int i=0; i<1 ; i++)
                    genPrefix.append(fPrefixGen.nextInt());
                prefix = genPrefix.toString();
                prefix = fSymbolTable.addSymbol(prefix);
            }else{
                prefix = fSymbolTable.addSymbol(tmpPrefix);
            }
            
            if( tmpPrefix == null ){
                QName qname = new QName();
                qname.setValues(prefix, XMLConstants.XMLNS_ATTRIBUTE, null,uri);
                fNamespaceDecls.add(qname);
                fInternalNamespaceContext.declarePrefix(fSymbolTable.addSymbol(prefix),uri);
            }
        }
        attr.prefix = prefix;
    }
    
    
    /**
     * @param uri
     * @return
     */
    private boolean isDefaultNamespace(String uri){
        String defaultNamespace = fInternalNamespaceContext.getURI(DEFAULT_PREFIX);
        if(uri == defaultNamespace){
            return  true;
        }
        return false;
    }
    
    /**
     * @param prefix
     * @param uri
     * @return
     */
    private boolean checkUserNamespaceContext(String prefix,String uri){
        if(fNamespaceContext.userContext != null){
            String tmpURI =  fNamespaceContext.userContext.getNamespaceURI(prefix);
            if(tmpURI != null && tmpURI.equals(uri))
                return true;
        }
        return false;
    }
    
    /**
     * Correct's namespaces  as per requirements of isReparisingNamespace property.
     * @param uri
     * @param prefix
     * @param localname
     * @param value
     */
    protected void repair() {
        Attribute attr = null;
        Attribute attr2 = null;
        fCurrentElementState = fElementStack.peek();
        removeDuplicateDecls();
        for(int i=0 ; i< fAttributeCache.size();i++){
            attr = (Attribute)fAttributeCache.get(i);
            if(attr.prefix != null && fCurrentElementState.prefix != null){
                correctPrefix(fCurrentElementState,attr);
            }
        }
        if(!isDeclared(fCurrentElementState)){
            if(fCurrentElementState.prefix != null && fCurrentElementState.uri != null)
                fNamespaceDecls.add(fCurrentElementState);
        }
        
        for(int i=0 ; i< fAttributeCache.size();i++){
            attr = (Attribute)fAttributeCache.get(i);
            for(int j=i+1;j<fAttributeCache.size();j++){
                attr2 = (Attribute)fAttributeCache.get(j);
                if(attr.prefix != null && attr2.prefix != null)
                    correctPrefix(attr,attr2);
            }
        }
        
        repairNamespaceDecl(fCurrentElementState);
        int i=0;
        for ( i =0 ; i< fAttributeCache.size() ; i++){
            attr = (Attribute)fAttributeCache.get(i);
            repairNamespaceDecl(attr);
        }
        
        QName qname = null;
        for(i=0; i<fNamespaceDecls.size(); i++){
            qname =(QName) fNamespaceDecls.get(i);
            if(qname != null){
                fInternalNamespaceContext.declarePrefix(qname.prefix,qname.uri);
            }
        }
        
        for ( i =0 ; i< fAttributeCache.size() ; i++){
            attr = (Attribute)fAttributeCache.get(i);
            correctPrefix(attr);
        }
    }
    
    
   /*
    *If element and/or attribute names in the same start or empty-element tag
    *are bound to different namespace URIs and are using the same prefix then
    *the element or the first occurring attribute retains the original prefix
    *and the following attributes have their prefixes replaced with a new prefix
    *that is bound to the namespace URIs of those attributes.
    */
    void correctPrefix(QName attr1 , QName attr2){
        String tmpPrefix = null;
        QName decl = null;
        boolean done = false;
        
        if(attr1.prefix.equals(attr2.prefix) && !(attr1.uri.equals(attr2.uri))){
            
            tmpPrefix = fNamespaceContext.getPrefix(attr2.uri);
            
            if(tmpPrefix != null ){
                attr2.prefix = fSymbolTable.addSymbol(tmpPrefix);
            }else{
                decl = null;
                for(int n=0;n<fNamespaceDecls.size();n++){
                    decl = (QName)fNamespaceDecls.get(n);
                    if(decl.uri == attr2.uri){
                        attr2.prefix = decl.prefix;
                        return;
                    }
                }
                //No namespace mapping found , so declare prefix.
                StringBuffer genPrefix = new StringBuffer("zdef");
                for(int k=0; k<1 ; k++)
                    genPrefix.append(fPrefixGen.nextInt());
                tmpPrefix = genPrefix.toString();
                tmpPrefix = fSymbolTable.addSymbol(tmpPrefix);
                attr2.prefix = tmpPrefix;
                QName qname = new QName();
                qname.setValues(tmpPrefix,XMLConstants.XMLNS_ATTRIBUTE, null,attr2.uri);
                fNamespaceDecls.add(qname);
            }
        }
    }
    
    
    void removeDuplicateDecls(){
        QName decl1,decl2;
        for(int i =0;i<fNamespaceDecls.size();i++){
            decl1 = (QName)fNamespaceDecls.get(i);
            for(int j=i+1;j<fNamespaceDecls.size();j++){
                decl2 = (QName)fNamespaceDecls.get(j);
                if(decl1 != null && decl1.equals(decl2))
                    fNamespaceDecls.set(j, null);
            }
        }
    }
   /*
    *If an element or attribute name is bound to a prefix and there is a namespace
    *declaration that binds that prefix to a different URI then that namespace declaration
    *is either removed if the correct mapping is inherited from the parent context of that element,
    *or changed to the namespace URI of the element or attribute using that prefix.
    *
    */
    void repairNamespaceDecl(QName attr){
        QName decl = null;
        String tmpURI;
        //check for null prefix.
        for(int j=0;j<fNamespaceDecls.size();j++){
            decl = (QName)fNamespaceDecls.get(j);
            if(decl != null){
                if(attr.prefix != null && (attr.prefix.equals(decl.prefix) && !(attr.uri.equals(decl.uri)))){
                    tmpURI = fNamespaceContext.getNamespaceURI(attr.prefix);
                    //see if you need to add to symbole table.
                    if(tmpURI != null){
                        if(tmpURI.equals(attr.uri)){
                            fNamespaceDecls.set(j,null);
                        }else{
                            decl.uri = attr.uri;
                        }
                    }
                }
            }
        }
    }
    
    boolean isDeclared(QName attr){
        QName decl =null;
        for(int n=0;n<fNamespaceDecls.size();n++){
            decl = (QName)fNamespaceDecls.get(n);
            if(attr.prefix != null && (attr.prefix == decl.prefix && decl.uri == attr.uri)){
                return true;
            }
        }
        if(attr.uri != null ){
            if(fNamespaceContext.getPrefix(attr.uri) != null)
                return true;
        }
        return false;
    }
    
    
    /*
     * Start of Internal classes.
     *
     */
    protected  class ElementStack {
        
        /** The stack data. */
        protected ElementState[] fElements;
        
        /** The size of the stack. */
        protected short fDepth;
        
        /** Default constructor. */
        public ElementStack() {
            fElements = new ElementState[10];
            for (int i = 0; i < fElements.length; i++) {
                fElements[i] = new ElementState();
            }
        }
        
        /**
         * Pushes an element on the stack.
         * <p>
         * <strong>Note:</strong> The QName values are copied into the
         * stack. In other words, the caller does <em>not</em> orphan
         * the element to the stack. Also, the QName object returned
         * is <em>not</em> orphaned to the caller. It should be
         * considered read-only.
         *
         * @param element The element to push onto the stack.
         *
         * @return Returns the actual QName object that stores the
         */
        public ElementState push(ElementState element) {
            if (fDepth == fElements.length) {
                ElementState[] array = new ElementState[fElements.length * 2];
                System.arraycopy(fElements, 0, array, 0, fDepth);
                fElements = array;
                for (int i = fDepth; i < fElements.length; i++) {
                    fElements[i] = new ElementState();
                }
            }
            fElements[fDepth].setValues(element);
            return fElements[fDepth++];
        }
        
        /**
         *
         * @param prefix
         * @param localpart
         * @param rawname
         * @param uri
         * @param isEmpty
         * @return
         */
        public ElementState push(String prefix, String localpart, String rawname,
        String uri,boolean isEmpty){
            if (fDepth == fElements.length) {
                ElementState[] array = new ElementState[fElements.length * 2];
                System.arraycopy(fElements, 0, array, 0, fDepth);
                fElements = array;
                for (int i = fDepth; i < fElements.length; i++) {
                    fElements[i] = new ElementState();
                }
            }
            fElements[fDepth].setValues(prefix,localpart,rawname,uri,isEmpty);
            return fElements[fDepth++];
        }
        
        /**
         * Pops an element off of the stack by setting the values of
         * the specified QName.
         * <p>
         * <strong>Note:</strong> The object returned is <em>not</em>
         * orphaned to the caller. Therefore, the caller should consider
         * the object to be read-only.
         */
        public ElementState pop() {
            return fElements[--fDepth] ;
            
        }
        
        /** Clears the stack without throwing away existing QName objects. */
        public void clear() {
            fDepth = 0;
        }
        
        /**
         * This function is as a result of optimization done for endElement --
         * we dont need to set the value for every end element we encouter.
         * For Well formedness checks we can have the same QName object that was pushed.
         * the values will be set only if application need to know about the endElement
         * -- neeraj.bajaj@sun.com
         */
        
        public ElementState peek(){
            return fElements[fDepth-1];
        }
        
        /**
         *
         * @return
         */
        public boolean empty(){
            return fDepth > 0 ? false : true;
        }
    }
    
    /**
     * Maintains element state . localName for now.
     */
    
    class ElementState extends QName {
        public boolean isEmpty = false;
        public ElementState(){
        }
        
        public ElementState(String prefix, String localpart, String rawname, String uri) {
            super(prefix,localpart,rawname,uri);
        }
        
        public void setValues(String prefix, String localpart, String rawname,String uri,boolean isEmpty){
            super.setValues(prefix,localpart, rawname, uri);
            this.isEmpty = isEmpty;
            
        }
    }
    
    /**
     * Attributes
     */
    class Attribute extends QName{
        String value;
        Attribute(String value){
            super();
            this.value = value;
        }
    }
    
    /**
     * Implementation of NamespaceContext .
     *
     */
    class NamespaceContextImpl implements NamespaceContext{
        //root namespace context set by user.
        NamespaceContext userContext = null;
        //context built by the writer.
        NamespaceSupport internalContext = null;
        
        public String getNamespaceURI(String prefix) {
            String uri = null;
            if(prefix != null) prefix = fSymbolTable.addSymbol(prefix);
            
            if(internalContext != null){
                uri =  internalContext.getURI(prefix);
                if(uri != null)return uri;
            }
            
            if(userContext != null){
                uri = userContext.getNamespaceURI(prefix);
                return uri;
            }
            
            return null;
        }
        
        public String getPrefix(String uri) {
            String prefix = null;
            if(uri != null) uri = fSymbolTable.addSymbol(uri);
            if(internalContext != null){
                prefix = internalContext.getPrefix(uri);
                if(prefix != null) return prefix;
            }
            if(userContext != null){
                return userContext.getPrefix(uri);
            }
            
            return null;
        }
        
        public java.util.Iterator getPrefixes(String uri) {
            Vector prefixes = null;
            Iterator itr = null;
            if(uri != null) uri = fSymbolTable.addSymbol(uri);
            if(userContext != null){
                itr = userContext.getPrefixes(uri);
            }
            
            if(internalContext != null){
                prefixes = internalContext.getPrefixes(uri);
            }
            if(prefixes == null && itr != null)
                return itr;
            else if(prefixes != null && itr == null){
                return new ReadOnlyIterator(prefixes.iterator());
            }else if(prefixes != null && itr != null){
                String ob = null;
                while(itr.hasNext()){
                    ob =(String) itr.next();
                    if(ob != null) ob = fSymbolTable.addSymbol(ob);
                    if(!prefixes.contains(ob))
                        prefixes.add(ob);
                    
                }
                return new ReadOnlyIterator(prefixes.iterator());
            }
            return fReadOnlyIterator;
        }
    }
}
