#include "savage_driver.h"
#include "savage_vbe.h"

Bool vbeModeInit( vbeInfoPtr, int );
static int SavageGetDevice( SavagePtr psav );
static int SavageGetTVType( SavagePtr psav );

static void
SavageClearVM86Regs( xf86Int10InfoPtr pInt )
{
    pInt->ax = 0;
    pInt->bx = 0;
    pInt->cx = 0;
    pInt->dx = 0;
    pInt->si = 0;
    pInt->di = 0;
    pInt->es = 0xc000;
    pInt->num = 0x10;
}

void
SavageSetTextMode( SavagePtr psav )
{
    /* Restore display device if changed */
    if (psav->iDevInfo != psav->iDevInfoPrim) {
      SavageClearVM86Regs( psav->pInt10 );
      psav->pInt10->ax = 0x4f14;
      psav->pInt10->bx = 0x0003;
      psav->pInt10->cx = psav->iDevInfoPrim;
      xf86ExecX86int10( psav->pInt10 );
    }

    SavageClearVM86Regs( psav->pInt10 );

    psav->pInt10->ax = 0x83;

    xf86ExecX86int10( psav->pInt10 );
}


void
SavageSetVESAMode( SavagePtr psav, int n, int Refresh )
{
    int iDevInfo;
    static int iCount;		/* for saving Devinfo before first set mode */

    /* Get current device status, set new wanted device status */
    iDevInfo = SavageGetDevice(psav);
    psav->iDevInfo = iDevInfo;
    if (iCount == 0) psav->iDevInfoPrim = iDevInfo;
    iCount++;
    if (psav->CrtOnly == TRUE)	psav->iDevInfo = SAVAGE_CRT_ON;
    if (psav->TvOn == TRUE) 	psav->iDevInfo = SAVAGE_TV_ON;

    /* Establish the refresh rate for this mode. */
    SavageClearVM86Regs( psav->pInt10 );
    psav->pInt10->ax = 0x4f14;	/* S3 extensions */
    psav->pInt10->bx = 0x0001;	/* Set default refresh rate */
    psav->pInt10->cx = n ;
    psav->pInt10->di = Refresh & 0xFFFF;
    xf86ExecX86int10( psav->pInt10 );

    /* set TV type if TV on */
    if (psav->TvOn == TRUE) {
      SavageClearVM86Regs( psav->pInt10 );
      psav->pInt10->ax = 0x4f14;	/* S3 extensions */
      psav->pInt10->bx = 0x0007;	/* Set default refresh rate */
      if (psav->PAL == FALSE)
	psav->pInt10->cx = 0x04;
      else
	psav->pInt10->cx = 0x08;
      psav->pInt10->dx = 0x0c;
      xf86ExecX86int10( psav->pInt10 );
    }

    /* Change device if needed */
    if (psav->iDevInfo != iDevInfo) {
      SavageClearVM86Regs( psav->pInt10 );
      psav->pInt10->ax = 0x4f14;
      psav->pInt10->bx = 0x0003;
      psav->pInt10->cx = psav->iDevInfo;
      xf86ExecX86int10( psav->pInt10 );

      /* set actual device status */
      psav->iDevInfo = SavageGetDevice(psav);
      iDevInfo = psav->iDevInfo;
      psav->CrtOnly = (iDevInfo == 1 ? TRUE : FALSE);
      iDevInfo &= 0x04;		/* simplify for complier */
      psav->TvOn = (iDevInfo != 0 ? TRUE : FALSE);
    }

    if( !vbeModeInit( psav->pVbe, n ) )
    {
	ErrorF("Set video mode failed\n");
    }
}

/* Function to get device which supported
 */
static int SavageGetDevice( SavagePtr psav )
{
    SavageClearVM86Regs( psav->pInt10 );
    psav->pInt10->ax = 0x4f14;	/* S3 extensions */
    psav->pInt10->bx = 0x0103;	/* get display device at present */

    xf86ExecX86int10( psav->pInt10 );

    return ((psav->pInt10->cx) & 0xF);
}

void
SavageFreeBIOSModeTable( SavagePtr psav, SavageModeTablePtr* ppTable )
{
    int i;
    SavageModeEntryPtr pMode = (*ppTable)->Modes;

    for( i = (*ppTable)->NumModes; i--; )
    {
	if( pMode->RefreshRate )
	{
	    xfree( pMode->RefreshRate );
	    pMode->RefreshRate = NULL;
	}
    }

    xfree( *ppTable );
}


SavageModeTablePtr
SavageGetBIOSModeTable( SavagePtr psav, int iDepth )
{
    SavageModeTablePtr pTable;
    
    pTable = (SavageModeTablePtr) xcalloc( 1, sizeof(SavageModeTableRec) + 
		    (psav->VesaModeNum) * sizeof(SavageModeEntry) );
    if( pTable ) {
	pTable->NumModes = SavageGetBIOSModes( psav, iDepth, pTable->Modes );
    }
    return pTable;
}


unsigned short
SavageGetBIOSModes( 
    SavagePtr psav,
    int iDepth,
    SavageModeEntryPtr s3vModeTable )
{
    int 		vbeReal;
    pointer 		vbeLinear = NULL;
    struct 		vbe_mode_info_block * vmib;
    unsigned short	VesaMode;
    unsigned short	iModeCount;
    int 		i;

    if( !psav->pVbe )
	return 0;

    if ((vbeLinear = xf86Int10AllocPages( psav->pInt10, 1, &vbeReal ))==NULL) {
	ErrorF("Cannot allocate one scratch page in real mode memory.\n");
	return (FALSE);
    }
    vmib = (struct vbe_mode_info_block *) vbeLinear;
    
    iModeCount = 0;
    for (i = 0; i < psav->VesaModeNum; i++) {
        VesaMode = psav->VesaModeList[i];
	/*
	 * This is a HACK to work around what I believe is a BUG in the
	 * Toshiba Satellite BIOSes in 08/2000 and 09/2000.  The BIOS
	 * table for 1024x600 says it has six refresh rates, when in fact
	 * it only has 3.  When I ask for rate #4, the BIOS goes into an 
	 * infinite loop until the user interrupts it, usually by pressing
	 * Ctrl-Alt-F1.  For now, we'll just punt everything with a VESA
	 * number greater than or equal to 0200.
	 *
	 * This also prevents some strange and unusual results seen with
	 * the later ProSavage/PM133 BIOSes directly from S3/VIA.
	 */
	if( VesaMode >= 0x0200 )
	    continue;

	SavageClearVM86Regs( psav->pInt10 );

	psav->pInt10->ax = 0x4f01;
	psav->pInt10->cx = VesaMode;
	psav->pInt10->es = SEG_ADDR(vbeReal);
	psav->pInt10->di = SEG_OFF(vbeReal);
	psav->pInt10->num = 0x10;

	xf86ExecX86int10( psav->pInt10 );

	if( 
	   (vmib->bits_per_pixel == iDepth) &&
	   (
	      (vmib->memory_model == VBE_MODEL_256) ||
	      (vmib->memory_model == VBE_MODEL_PACKED) ||
	      (vmib->memory_model == VBE_MODEL_RGB)
	   )
	)
	{
	    iModeCount++;
	    /* If we're supposed to fetch information, do it now. */

	    if( s3vModeTable )
	    {
	        int iRefresh = 0;

		s3vModeTable->Width = vmib->x_resolution;
		s3vModeTable->Height = vmib->y_resolution;
		s3vModeTable->VesaMode = VesaMode;
		
		/* Query the refresh rates at this mode. */

		psav->pInt10->cx = VesaMode;
		psav->pInt10->dx = 0;

		do
		{
		    if( (iRefresh % 8) == 0 )
		    {
			if( s3vModeTable->RefreshRate )
			{
			    s3vModeTable->RefreshRate = (unsigned char *)
				xrealloc( 
				    s3vModeTable->RefreshRate,
				    (iRefresh+8) * sizeof(unsigned char)
				);
			}
			else
			{
			    s3vModeTable->RefreshRate = (unsigned char *)
				xcalloc( 
				    sizeof(unsigned char),
				    (iRefresh+8)
				);
			}
		    }

		    psav->pInt10->ax = 0x4f14;	/* S3 extended functions */
		    psav->pInt10->bx = 0x0201;	/* query refresh rates */
		    psav->pInt10->num = 0x10;
		    xf86ExecX86int10( psav->pInt10 );

		    s3vModeTable->RefreshRate[iRefresh++] = psav->pInt10->di;
		}
		while( psav->pInt10->dx );

		s3vModeTable->RefreshCount = iRefresh;

	    	s3vModeTable++;
	    }
	}
    }

    xf86Int10FreePages( psav->pInt10, vbeLinear, 1 );

    return iModeCount;
}
