#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# VMServerd.pm
# 
# A basic class that contains utility routines needed by vmserverd.
#

package VMware::VMServerd;
use strict;
use VMware::DOMAccess;
use VMware::Control::Profiler;
use URI::Escape;
use VMware::Config;
use VMware::ExtHelpers qw(&internal_dirname &internal_basename &getFileSeparator);
use VMware::HConfig::CommonConstant qw(@VMX_CFG_EXTENSIONS @VMX_STD_EXTENSIONS);

use VMware::VMServerd::AuthPolicy qw(doPolicy undoPolicy);

#Do a backtrace on errors
use Carp qw( croak confess );
$SIG{ __DIE__ } = \&die_handler;

use Exporter   ();
use vars       qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

BEGIN {
    # set the version for version checking
    $VERSION     = 1.00;
    @ISA         = qw(Exporter);
    @EXPORT_OK   = qw(&Log
		      &Warning
		      &LogEvent
		      &Panic
                      &errorReset
                      &haveError
                      &errorAppend
                      &errorPost
		      &GetDb
		      %gConfig
		      $VMSTATE_ERROR 
		      $VMSTATE_OFF 
		      $VMSTATE_ON 
		      $VMSTATE_SUSPEND 
		      $VMSTATE_STUCK 
		      %VMSTATE_NAMES
		      );
}

use vars qw(%gConfig
	    $VMSTATE_ERROR 
	    $VMSTATE_OFF 
	    $VMSTATE_ON 
	    $VMSTATE_SUSPEND 
	    $VMSTATE_STUCK 
	    %VMSTATE_NAMES
	    );

bootstrap VMware::VMServerd '0.01';

#############################################################
# Global variables and state
#############################################################
$gConfig{OPERATION_TABLE} = {};
$gConfig{VERBOSE} = 0;                 # verbose output on server side
$gConfig{REQUEST_COUNT} = 0;           # Number of exec requests handled by this serverd
$gConfig{POLLCALLBACK_TABLE} = {};
$gConfig{POLLCALLBACK_ID} = 0;
my $gVmdbDb = undef;

# Per request data that is stored as a global.
my %gRequestData;
$gRequestData{MSG} = undef;
$gRequestData{USERNAME} = undef;       # The username of the authenticated user.

# List of VMs -- config name --> VM object
my %vm_list;
my $p = VMware::Control::Profiler::getInstance("serverd");

# State constants for each of the various VM states.  Don't modify
# these as they are used in the MUI in the XSL templates.
$VMSTATE_ERROR = -1;
$VMSTATE_OFF = 0;
$VMSTATE_ON = 1;
$VMSTATE_SUSPEND = 2;
$VMSTATE_STUCK = 3;

%VMSTATE_NAMES = ( $VMSTATE_ERROR => 'error',
                   $VMSTATE_OFF => 'powered off',
		   $VMSTATE_ON => 'powered on',
		   $VMSTATE_SUSPEND => 'suspended',
		   $VMSTATE_STUCK => 'stuck'
		   );

sub die_handler($) {
    my ($msg) = @_;

    if (&VMware::VMServerd::AuthPolicy::IN_PRIVILEGED_SECTION()) {
	&VMware::VMServerd::AuthPolicy::END_PRIVILEGED();
    }
    &confess($msg);
}

#############################################################
# Functions
#############################################################
# Gets the product title
sub productTitle() {
  return($VMware::VMServerd::PRODUCT);
}

# Gets the product name
sub product() {
  # The order of the check is important as 
  # ESX also contains the Server string
  if ($VMware::VMServerd::PRODUCT =~ /ESX/) {
    return "ESX";
  } elsif (($VMware::VMServerd::PRODUCT =~ /GSX/) || 
           ($VMware::VMServerd::PRODUCT =~ /Server/i)) {
    return "GSX";
  } else {
    die "Can't figure out product - $VMware::VMServerd::PRODUCT";
  }
}

# Get the build number.
sub build() {
  my( $doc ) = @_;
  my $version = $VMware::VMServerd::VERSION;
  my $build;

  if( $version =~ /^([0-9A-Za-z]+\.[0-9A-Za-z]+\.[0-9A-Za-z]+)$/ ) {
      $build = '000';
  } elsif( $version =~ /^([0-9A-Za-z]+\.[0-9A-Za-z]+\.[0-9A-Za-z]+) build-([0-9]+)$/ ) {
      $build = $2;
  } else {
      &errorPost("Could not get build number.", "warning");
      return;
  }

  return($build);
}

# Set the production information for each request.
sub version() {
  my( $doc ) = @_;
  my $version = $VMware::VMServerd::VERSION;

  if( $version =~ /^[0-9A-Za-z]+\.[0-9A-Za-z]+\.[0-9A-Za-z]+/ ) {
      return($version);
  } else {
      &errorPost("Could not get product version information.", "warning");
      return;
  }

  return($version);
}


sub isESX() {
    return( product() eq "ESX" );
}

sub isGSX() {
    return( product() eq "GSX" );
}

sub setVerbose($) {
    my ($verbose) = @_;
    $gConfig{VERBOSE} = $verbose;
    &VMware::DOMAccess::setVerbose($verbose);
}

sub isVerbose() {
    return($gConfig{VERBOSE});
}

sub getRequestMsgData() {
    return($gRequestData{MSG});
}

sub Log($) {
    my ($msg) = @_;
    if (isVerbose()) {
	if ($^O ne "MSWin32") {
	    print STDERR $msg;
	}
    }
    LogToFile($msg);
}

sub perlLogEvent($) {
    my ($msg) = @_;
    LogEvent($msg);
    Log($msg);
}

sub setRequestMsgData($) {
    my ($domdoc) = @_;
    $gRequestData{MSG} = $domdoc;
}

sub clearRequestData() {
    my $msg = getRequestMsgData();

    if( defined($msg) ) {
	$msg->dispose();
	setRequestMsgData(undef);
    }
}

# Set request specific data
sub setRequestData($$) {
    my($key, $value) = @_;
    $gRequestData{$key} = $value;
}

# Get request specific data
sub getRequestData($) {
    my($key) = @_;
    return($gRequestData{$key});
}

my $errStr = "";

sub errorAppend($) {
    my ($error) = @_;

    if( isVerbose() ) {
	if ($^O ne "MSWin32") {
	    print STDERR "errorAppend: $error";
	}
    }

    $errStr = $error . "\n" . $errStr;
}

sub errorPost($$) {
    my ($error, $severity) = @_;
    my $msg = getRequestMsgData();

    if( !defined($msg) ) {
	if ($^O ne "MSWin32") {
	    print STDERR "errorPost: msg is undefined.\n";
	}
        $errStr = "";
	return;
    }

    $errStr = $error . "\n" . $errStr;

    if (!defined($severity)) {
      $severity = "error";
    }

    Log("errorPost: [severity=$severity] $errStr");
    
    my @errors = sort($msg->listElementNames("messages"));
    my $next_errno = $#errors + 1;

    $msg->setValue(".msg[$next_errno]", $errStr);
    $msg->setAttribute(".msg[$next_errno]", "severity", $severity);

    $errStr = "";
}

sub errorReset() {
    if( isVerbose() ) {
	if ($^O ne "MSWin32") {
	    print STDERR "errorReset: $errStr";
	}
    }
    $errStr = "";
}

sub haveError() {
    return($errStr ne "");
}

sub incRequestCount() {
    $gConfig{REQUEST_COUNT}++;
}

sub getRequestCount() {
    return($gConfig{REQUEST_COUNT});
}

sub addOperation(%) {
    my(%map) = @_;

    my $opName = $map{OPNAME};
    my $perlFunc = $map{PERLFUNC};
    my $policy = $map{POLICY};

    $gConfig{OPERATION_TABLE}->{$opName}->{FUNCTION} = $perlFunc;
    $gConfig{OPERATION_TABLE}->{$opName}->{POLICY} = $policy;
    $gConfig{OPERATION_TABLE}->{$opName}->{NumRequests} = 0;
    $gConfig{OPERATION_TABLE}->{$opName}->{NumErrors} = 0;
    $gConfig{OPERATION_TABLE}->{$opName}->{HandlerTime} = 0;
    $gConfig{OPERATION_TABLE}->{$opName}->{TotalTime} = 0;
    return(1);
} 

sub incOperationStat($$$) {
    my($opName, $statName, $increment) = @_;
    if (!defined($gConfig{OPERATION_TABLE}->{$opName})) {
	Log("incOperationStat: Invalid operation $opName specified.\n");
    }
    if (!defined($gConfig{OPERATION_TABLE}->{$opName}->{$statName})) {
	Log("incOperationStat: Invalid statistic '$statName' for operation $opName specified.\n");
    }
    $gConfig{OPERATION_TABLE}->{$opName}->{$statName} += $increment;
}

sub removeOperation($) {
    my($opName) = @_;

    delete( $gConfig{OPERATION_TABLE}->{$opName} );
} 

sub lookupOperation($) {
    my($opName) = @_;

    return( $gConfig{OPERATION_TABLE}->{$opName} );
}   

sub dumpOperation() {
    my $s = "";

    my %OperationTable = %{$gConfig{OPERATION_TABLE}};

    my $key;
    foreach $key (sort(keys(%OperationTable))) {
	my $op = lookupOperation($key);
	my $perlFunc = $op->{FUNCTION};
	my $policy = (defined($op->{POLICY}) ? " (" . $op->{POLICY} . ")" : "");
	$s .= "Op $key -> $perlFunc" . $policy . "\n";
    }

    return($s);
}

# Does the user have access to this VM?
#
# Takes in a VM object and a bitmask for the necessary permissions.
#
# Args:
#   vmcfg := the name of the config file of the VM
#   request := the permissions requested for this VM
sub accessVM($$) {
    my($vmcfg, $request) = @_;
    my $username = getRequestData('USERNAME');

    if( !defined($vmcfg) ) {
	Warning("accessVM: null config file was specified.\n");
	return(0);
    }

    if( !defined($username) ) {
	# ASSERT fail
	Warning("accessVM: null username was specified.\n");
	return(0);
    }

    my $access = 0;
    $access = &VMware::VMServerd::GetAccessBits($vmcfg);

    Log("accessVM: check cfg=$vmcfg request=$request username=$username access=$access\n");

    if( ($request & $access) == $request ) {
	return(1);
    }

    return(0);
}

sub getOp($) {
    my $doc = VMware::DOMAccess->newXML(shift);
    my $op = $doc->getValue("exec.op");
    $doc->dispose();
    return($op);
}

# Set the product information for each request.
sub setProductInfo {
  my( $doc ) = @_;
  my $product = $VMware::VMServerd::PRODUCT;
  my $version = $VMware::VMServerd::VERSION;
  my $build;

  $doc->setValue(".product.title", $product);

  my $p = VMware::VMServerd::product();
  
  $doc->setValue(".product.version", version());
  $doc->setValue(".product.build", build());
  $doc->setAttribute(".product", "id", $p);
}

# Called by vmserverd to handle an exec request.
sub handleExecRequest($$) {
    my $stub = $p->profile();
    my $username = shift;
    my $XML_in = shift;
    my $doc_in = VMware::DOMAccess->newXML($XML_in);
    my $doc_out = VMware::DOMAccess->new("exec");
    my $ret = 0;            # Return value from the exec call
    my $op;                 # The name of the operation
    my $handlerTime = 0;    # The amount of time spent in the handler
    my $totalTime = 0;      # The total amount of time spent handling this request
    my $success = 0;        # Flag indicating success of request

    my $totalStartTime = GetRealTime();

    # Get the subtrees for the input and output
    my $in = $doc_in->getSubTree("exec.in");
    if( !defined($in) ) {
	# Create it and try getting it again
	$doc_in->setValue("exec.in", "");
	$in = $doc_in->getSubTree("exec.in");
	if( !defined($in) ) {
	    die "Internal serverd error.  Unable to get the arguments for the request.";
	}
    }

    my $out = VMware::DOMAccess->new("out");
    my $msg = VMware::DOMAccess->new("messages");

    setRequestMsgData($msg);
    setRequestData('USERNAME', $username);
    setProductInfo($doc_out);
    incRequestCount();

    if (defined($in)) {
	$op = getOp($XML_in);
	my $opMap = lookupOperation($op);
	if (defined($opMap)) {
	    my $perlFunc = $opMap->{FUNCTION};
	    
	    # Report success by default
	    $doc_out->setValue("exec.returncode", 1);
	    
	    if (defined($perlFunc) && $perlFunc ne "") {
		Log("handleExecRequest: Executing operation $op ($perlFunc) for user $username with args '" . 
		    $in->getValue("") . "'\n");

		my $policy = $opMap->{POLICY};

		if (!doPolicy($policy, $username)) {
		    errorPost("Permission denied.\n", "error");
		    Log("handleExecRequest: Permission policy $policy FAILED. username=$username.\n");
		} else {
		    my $handlerStartTime = GetRealTime();
		    # Execute the function with the proper arguments 
		    $p->start($perlFunc);
		    eval "\$ret = $perlFunc(\$in, \$out);";
		    $p->end();
		    my $handlerEndTime = GetRealTime();
		    if ($handlerEndTime < $handlerStartTime) {
			Log("handleExecRequest: Request '$op' handler end time $handlerEndTime < start time " .
			    "$handlerStartTime\n");
		    } else {
			$handlerTime = $handlerEndTime - $handlerStartTime;
		    }
		}

		if ($@) {
		    # Report failure
		    $doc_out->setValue("exec.returncode", 0);
		    if ("$@" ne "") {
			chomp($@);
			errorPost("$@", "error");
		    } else {
			errorPost("The operation failed to complete for some reason.", "error");
		    }
		} else {
		    if (!defined($ret)) {
			$doc_out->setValue("exec.returncode", 0);
		    } else {
			$doc_out->setValue("exec.returncode", $ret);
			$success = 1;
		    }
		    Log("handleExecRequest: Operation completed in $handlerTime ms with return value '" . 
			$out->prettyPrint() . "'\n");
		}

		if (!undoPolicy($policy, $username)) {
		    Log("handleExecRequest: Could not undo permission policy $policy\n");
		}
	    } else { # if( defined($perlFunc) && $perlFunc ne "" )
		$doc_out->setValue("exec.returncode", 0);
		errorPost("Invalid operation '$op' requested.  Perl function not found.", "error");
	    }
	    # Check to make sure that the error buffer is clean.  We don't want error messages from 
	    # this request polluting future requests.
	    if (haveError()) {
		Log("handleExecRequest: Error buffer not clean after $op request.  Buffer contains: $errStr\n");
		errorReset();
	    }
	} else { # if( defined($opMap) ) 
	    $doc_out->setValue("exec.returncode", 0);
	    errorPost("Invalid operation '$op' request.  Operation not defined.\n", "error");
	}
    } else { # if( defined($in) )
	$doc_out->setValue("exec.returncode", 0);
	errorPost("Input arguments not defined or missing.", "error");
    }
    
    # Add the outputs if any exist.
    if ($out->hasChildElement("out")) {
	$doc_out->addSubTree("", $out);
    }

    # Add the messages if any exist.
    if ($msg->hasChildElement("messages")) {
	$doc_out->addSubTree("", $msg);
    }

    my $XML_out = $doc_out->toEncodedString("");

    # Dispose of any XML structures to clean up the memory.
    clearRequestData();
    $doc_in->dispose();
    $doc_out->dispose();
    $in->dispose();
    $out->dispose();

    # Clear the per request data
    setRequestData('USERNAME', undef);

    my $totalEndTime = GetRealTime();
    if ($totalEndTime < $totalStartTime) {
	Log("handleExecRequest: Request '$op' total end time $totalEndTime < start time " .
	    "$totalStartTime\n");
    } else {
	$totalTime = $totalEndTime - $totalStartTime;
    }

    # Performing accounting work
    incOperationStat($op, "NumRequests", 1);
    incOperationStat($op, "NumErrors", $success == 1 ? 0 : 1);
    incOperationStat($op, "HandlerTime", $handlerTime);
    incOperationStat($op, "TotalTime", $totalTime);

    return($XML_out);
}

sub canonicalize($) {
    my ($cfg) = @_;

    my $canon_cfg = &VMware::VMServerd::GetCanonicalPath($cfg);
    if (defined($canon_cfg)) {
        return $canon_cfg;
    }

    return $cfg;
}

# Puts a VM object into the list of connected VMs
sub setVM($@) {
    my $vm = shift;
    my @cfgs = @_;

    if (! UNIVERSAL::isa($vm,"VMware::Control::VM")) {
        croak("Invalid VM object");
    }

    #print STDERR "setVM: $cfg $vm\n"; #DEBUG
    
    my $cfg;
    foreach $cfg (@cfgs) {
	$vm_list{$cfg} = $vm;
    }
}

# Gets the VM object given its name
sub findVM($) {
    my $cfg = shift;

    $cfg = canonicalize($cfg);

    return $vm_list{$cfg};
}

sub listConnectedVMs() {
    return(keys %vm_list);
}

# Takes a list of config files and returns only the ones for which the user
# has access
sub filter_vmlist(@) {
    my @vmlist = @_;
    # XXX Use a constant define here
    my @newlist = grep( accessVM($_, 4), @vmlist );
    return(@newlist);
}

# Resolve Symbolic links to absolute path
sub resolveSymLinks($) {
    my ($path) = @_;

    # XXX TODO to be implemented
    return $path;
}

#
# Gets the state of the VM.  Input arg is a config file.
#
# Returns
#   ($state, $error) where $state is one of $VMSTATE_XXXX and $error is a string
#   detailing the error if $state is $VMSTATE_ERROR
#
sub getVMState($) {
    my($config) = @_;
    my $error;
    my $state = $VMSTATE_ERROR;;
    my $s;

    my($vm);
  
    $vm = findVM($config);

    if (defined($vm) && $vm->is_connected()) {
	# Assumption: Running VMs must be connected

	$s = $vm->get("Status.power");
	if (!defined($s)) {
	    my ($err, $errstr) = $vm->get_last_error();
	    $error = "Could not get power state for VM $config.  Error $err: $errstr.\n";
	} elsif ($s eq "off") {
	    $state = $VMSTATE_OFF;
	} elsif ($s eq "on") {
	    $state = $VMSTATE_ON;
	} elsif ($s eq "suspended") {
	    $state = $VMSTATE_SUSPEND;
	} elsif ($s eq "stuck") {
	    $state = $VMSTATE_STUCK;
	} else {
	    my ($err, $errstr) = $vm->get_last_error();
	    $error = "Invalid power state '$s' for VM $config.  Error $err: $errstr.\n";
	}
    } else {
	($state, $error) = getVMStateNoVMX($config);
    }

    return($state, $error);
}


#
# Gets the state of the VM.  Input arg is a config file.
#
# Returns
#   ($state, $error) where $state is one of $VMSTATE_XXXX and $error is a string
#   detailing the error if $state is $VMSTATE_ERROR
#
# XXX This attempts to approximate what a VMX would do to detect its current
#     state, but does so only when the VMX is not running.  Do we really want
#     to keep duplicate logic like this around? -jhu
#
sub getVMStateNoVMX($) {
    my($config) = @_;
    my $error;
    my $state = -1;

    # If there is no VMX, then the VM must be either off or suspended.  The VM is
    # suspended if we find a suspend to disk file for it.  Otherwise, it is off.

    if (! -f $config) {
	$error = "Config file for VM $config does not exist.";
	$state = $VMSTATE_ERROR;
	goto end;
    }

    my $std = getVMSuspendFile($config);
    if (defined($std)) {
	$state = $VMSTATE_SUSPEND;
    } else {
	$state = $VMSTATE_OFF;
    }

  end:
    return ($state, $error);
}


#
# Looks for the suspend to disk file for the VM.  Gets the file by looking at the
# file system directly.
#
# Returns
#   the file name of the suspend file or undef if none found.
#
# XXX This attempts to approximate what a VMX would do to find its suspend to
#     disk file.  Do we really want to keep duplicate logic like this around? -jhu
#
sub getVMSuspendFile($) {
    my($config) = @_;

    my $suspend_dir = internal_dirname($config);
    my $suspend_base = internal_basename($config);
    foreach my $ext (@VMX_CFG_EXTENSIONS) {
	if ($suspend_base =~ /\.$ext$/) {
	    $suspend_base =~ s/\.$ext$//;
	    last;
	}
    }
    if (-e $config && -f $config) {
	my $c = new VMware::Config;
	if ($c->readin($config)) {
	    $suspend_dir = $c->get("suspend.directory", $suspend_dir);
            $suspend_base = $c->get("checkpoint.cptConfigName", $suspend_base);
	}
    } else {
	return undef;
    }

    foreach my $ext (@VMX_STD_EXTENSIONS) {
	my $std = $suspend_dir . getFileSeparator() . $suspend_base . '.' . $ext;
	
	if (-e $std && -f $std) {
	    return $std;
	}
    }

    return undef;
}

sub getVMWorldID($) {
    my($config) = @_;
    my $worldID = -1;

    # We have a config file.  Make sure it's valid.
    my ($state, $error) = getVMState($config);

    if( $state == $VMSTATE_ERROR ) {
	return(-1);
    } elsif( $state == $VMSTATE_SUSPEND ) {
	return(-1);
    }

    if( $state != $VMSTATE_OFF ) {
	# If the VM is running, get its world ID.  The world ID must exist if the world is running.
	my $vm = findVM($config);
	
	if( !defined($vm) ) {
	    # XXX This could happen if the VM is shutdown between the calls to getVMState from above -jhu

	    # Error: VM state is on or stuck, but the VM control object for this VM '$configFile' could not be found.");
	    return(-1);
	}	    

	if( !$vm->is_connected() ) {
	    #my ( $err, $errstr ) = $vm->get_last_error( );
	    #&VMware::VMServerd::errorAppend("GetVMResources_Handler: VMControl error $err for $configFile " .
	    #				    "while getting world ID for VM: $errstr.");
	    return(-1);
	}

	$worldID = $vm->get('Status.id');
	if( !defined($worldID) || $worldID == -1 ) {
	    #&VMware::VMServerd::errorAppend("GetVMResources_Handler: Could not get the worldID for the VM '$configFile'");
	    return(-1);
	}
    }

    return($worldID);
}

# Escapes a config file string
sub toSafeCfgString($) {
    my($cfg) = @_;
    return(URI::Escape::uri_escape($cfg, "^a-zA-Z0-9\\*\\@\\-_\\.\\/"));
}

# Cleans up memory
sub cleanup {
    my $id = shift;
    foreach my $func (@_) {
        if( ref($func) eq 'CODE' ) {
            my $count = &$func;
            Warning("cleanup: cleaned up $count objects\n") if $count > 0;
        } else {
            Warning("cleanup: handler [$func] is not a CODE reference.");
        }
    }
}

#######################################################
# Poll Callback module
#######################################################

# $id = VMware::VMServerd::RegisterPollCallback("POLL_FD", $fd, \&handler, $arg1, $arg2);
# $id = VMware::VMServerd::RegisterPollCallback("POLL_PERIODIC", $millisecs, \&handler);
# VMware::VMServerd::UnregisterPollCallback($id);

sub POLL_FD { return 0; }
sub POLL_PERIODIC { return 1; }

sub RegisterPollCallback {
  my $type = shift;
  my $value = shift;
  my $handler = shift;
  my @args = @_;

  $gConfig{POLLCALLBACK_ID}++;
  my $id = $gConfig{POLLCALLBACK_ID};

  $gConfig{POLLCALLBACK_TABLE}->{$id} = {TYPE => $type, 
                                         VALUE => $value, 
                                         HANDLER => $handler,
                                         ARGS => \@args};
  
  VMware::VMServerd::_RegisterPollCallback($type, $value, $id);
  
  return $id;
}

sub UnregisterPollCallback {
  my($id) = @_;
  my $type = $gConfig{POLLCALLBACK_TABLE}->{$id}->{TYPE};
  VMware::VMServerd::_RegisterPollCallback($type, -1, $id);
}

sub _PollCallbackHandler {
  my ($id) = @_;
  if (defined($gConfig{POLLCALLBACK_TABLE}->{$id})) {
    my $func = $gConfig{POLLCALLBACK_TABLE}->{$id}->{HANDLER};
    my $args = $gConfig{POLLCALLBACK_TABLE}->{$id}->{ARGS};
    &{$func}($id, @{$args});
  }
}


#############################################################
# Define a debug operation handler.
#############################################################
#
# Gets the number of requests that have been made on the 
# serverd.
#
# Input format:
#   No input args.
# 
# Output format:
#
#   .count := number of request that have been made on serverd 
#             the one being made to get the count.
#
sub GetRequestCount_Handler($$) {
    my $in = shift;
    my $out = shift; 

    my $count = getRequestCount();
    $out->setValue(".count", $count);

    return(1);
}

#
# Gets the operations exported by this instance of serverd.
#
# Input format:
#   No input args.
# 
# Output format:
#
#   .operation[].name := the name of the operation
#   .operation[].function := the name of the perl function
#   .operation[].policy := the authentication policy
#
sub GetOperationTable_Handler($$) {
    my $in = shift;
    my $out = shift; 

    my %OperationTable = %{$gConfig{OPERATION_TABLE}};

    my $name;
    my $count = 0;
    foreach $name (sort(keys(%OperationTable))) {
	my $op = lookupOperation($name);
	my $perlFunc = $op->{FUNCTION};
	my $policy = (defined($op->{POLICY}) ? $op->{POLICY} : "");

	my $key = '.operation[' . $count . ']';
	$out->setValue($key . '.name', $name);
	$out->setValue($key . '.function', $perlFunc);
	$out->setValue($key . '.policy', $policy);
	$out->setValue($key . '.NumRequests', $op->{NumRequests});
	$out->setValue($key . '.NumErrors', $op->{NumErrors});
	$out->setValue($key . '.HandlerTime', $op->{HandlerTime});
	$out->setValue($key . '.TotalTime', $op->{TotalTime});
	$count++;
    }

    return(1);
}

#
# Gets the product that serverd is managing.
#
# XXX It would be nice to have the version and build number too.
#
# Input format:
#   No input args.
# 
# Output format:
#
#   .product := ESX | GSX
#
sub GetProductInfo_Handler($$) {
    my $in = shift;
    my $out = shift; 

    $out->setValue('.product', productTitle() . ' Version ' . version());

    return(1);
}


#
# Gets the list of connected VMs
#
# Input format:
#   No input args.
# 
# Output format:
#
#   .vm[] := config file for VM
#
sub GetConnectedVMList_Handler($$) {
    my $in = shift;
    my $out = shift; 
    
    my $vm;
    foreach $vm (keys(%vm_list)) {
        if( accessVM($vm, 4) ) {
            my $dom = VMware::DOMAccess->new("vm");
            $dom->setValue(".cfg", $vm);
            $dom->setValue(".safe_cfg", toSafeCfgString($vm));
            $out->addSubTree(".", $dom);
        }
    }

    return(1);
}


#####################################################################
#
# InitVmdb --
#
#      Initializes VMDB in Perl. Initializes the $gVmdbDb DB reference.
# 
# Results
#      Returns 1 if successful, 0 for failure.
#
#####################################################################

sub InitVmdb($) {
    my ($rawDbPtrStr) = @_;
    
    my $rawDbPtr = int($rawDbPtrStr);
    $gVmdbDb = \$rawDbPtr;

    if (!defined(bless($gVmdbDb, "VMware::VmdbPerl::VmdbDb"))) {
	return 0;
    }

    return 1;
}

#####################################################################
#
# GetDb --
#
#      Returns the Vmdb Db ref.
#
#####################################################################

sub GetDb() {
    return $gVmdbDb;
}

#
# Output format: 
#   .adminuser := bool.  true if user has admin privileges, false if user doesn't.
# 
sub IsAdminUser_Handler($$) {
   my ($in, $out) = @_;
   $out->setValue(".adminuser", &VMware::VMServerd::IsAdminUser());
   return 1;
}

VMware::VMServerd::addOperation( OPNAME	=> 'VMServerd_GetRequestCount',
				 PERLFUNC  => 'VMware::VMServerd::GetRequestCount_Handler',
				 POLICY => 'authuser' );

VMware::VMServerd::addOperation( OPNAME	=> 'VMServerd_GetOperationTable',
				 PERLFUNC  => 'VMware::VMServerd::GetOperationTable_Handler',
				 POLICY => 'authuser' );

VMware::VMServerd::addOperation( OPNAME	=> 'VMServerd_GetProductInfo',
				 PERLFUNC  => 'VMware::VMServerd::GetProductInfo_Handler',
				 POLICY => 'authuser' );

VMware::VMServerd::addOperation( OPNAME => 'VMServerd_GetConnectedVMList',
				 PERLFUNC => 'VMware::VMServerd::GetConnectedVMList_Handler',
				 POLICY => 'authuser' );

VMware::VMServerd::addOperation( OPNAME => 'VMServerd_IsAdminUser',
				  PERLFUNC => 'VMware::VMServerd::IsAdminUser_Handler',
				  POLICY => 'authuser');

&VMware::VMServerd::RegisterPollCallback(VMware::VMServerd::POLL_PERIODIC,
                                         1000, \&cleanup,
                                         \&VMware::DOMAccess::execDispose,
                                         );

1;

