#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################
#
# GSXHostInfo.pm
# 
# VMServerd wrappers to get information for GSX host machine
#
package VMware::VMServerd::GSXHostInfo;

use strict;
use VMware::DOMAccess;
use VMware::VMServerd; 
use VMware::ExtHelpers qw(&internal_dirname);

use Exporter   ();
use vars       qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

BEGIN {
    # set the version for version checking
    $VERSION     = 1.00;
    @ISA         = qw(Exporter);
}

# Gets information about the host
#
# Input format:
#   No input args.
# 
# Output format:
#   .hostOS := Type of hostOS
#   .hostOSVersion := Host OS version (major number)
#   .vmDefaultDir := Default directory where VM will be created
#   .cpu := Number of CPUs on the host
#   .memory.total := Total physical memory in the host (in MB)
#   .memory.avail := Total physical memory available (in MB)
#   .devices.device[].type  := floppy | cdrom | hard-disk
#   .devices.device[].path  := path to the device (A:\ or /dev/fda)
#   .drives.drive[].path  := path to the file system drive (A:\ or /dev/hda)
#   .drives.drive[].space := Free space on the drive (in MB) 
#

sub Query_Handler($$) {
    my ($in, $out) = @_;

    my $username = $in->getValue('.username');

    my $hostOS = $^O;
    my $hostOSVersion = &VMware::VMServerd::GetHostOSVersion(0);
    my $vmDefaultDir = getVMDefaultDir();
    $out->setValue('.hostOS', $hostOS);
    $out->setValue('.hostOSVersion', $hostOSVersion);
    $out->setValue('.vmDefaultDir', $vmDefaultDir);

    my $cpu = getCPUOnHost();
    $out->setValue('.cpu', $cpu);

    my $memory = getMemoryOnHost();
    my $memNode = VMware::DOMAccess->new('memory');
    $memNode->setValue('.total', $memory->{TOTAL});
    $memNode->setAttribute('.total', 'units', 'MB');

    $out->addSubTree('.', $memNode);

    $memNode->dispose();

    my @deviceList = getDevicesOnHost();    
    foreach my $device (@deviceList) {
        my $deviceNode = VMware::DOMAccess->new('device');
        $deviceNode->setValue('.type', $device->{DEVICE_TYPE});
        $deviceNode->setValue('.path', $device->{DEVICE_PATH});
        $out->addSubTree('.devices', $deviceNode);
	$deviceNode->dispose();
    }

    my @driveList = getDiskDrivesOnHost();
    foreach my $drive (@driveList) {
        my $driveNode = VMware::DOMAccess->new('drive');
        $driveNode->setValue('.path', $drive->{DRIVE_PATH});
        $driveNode->setValue('.space', $drive->{DRIVE_SPACE});
        $driveNode->setAttribute('.space', 'units', 'MB');
	$out->addSubTree('.drives', $driveNode);
	$driveNode->dispose();
    }

    return 1;
}

sub getVMDefaultDir() {
    my $path = &VMware::VMServerd::GetDefaultVMPath();
    if (!defined($path) || ($path eq " ")) {
	&VMware::VMServerd::Warning("Could not get default VM path for the user");
	return ("");
    }

    return $path;
}

sub getCPUOnHost() {
    my $cpu = &VMware::VMServerd::GetCPUOnHost();
    if ($cpu == -1) {
	# Some error occurred.  Just return 0.
	$cpu = 0;
    }
    return $cpu;
}

sub getMemoryOnHost() {
    my %hash;

    my ($total, $avail) = &VMware::VMServerd::GetMemoryOnHost();
    $hash{TOTAL} = $total;
    $hash{AVAILABLE} = $avail;
 
    return \%hash;
}

sub getDevicesOnHost() {

    my @deviceArray = &VMware::VMServerd::GetDevicesOnHost();
    my $path;
    my $type;
    my @retArray;

    while (defined($path = shift(@deviceArray))) {
	$type = shift(@deviceArray);
	
        my %hash;
 
	$hash{DEVICE_PATH} = $path;
    	$hash{DEVICE_TYPE} = $type;
	push (@retArray, \%hash);
    }

    return @retArray;
}

sub getDiskDrivesOnHost() {

    my @drivesArray = &VMware::VMServerd::GetDiskDrivesOnHost();
    my $path;
    my $space;
    my @retArray;

    while (defined($path = shift(@drivesArray))) {
	$space = shift(@drivesArray);
        my %hash;
 
	$hash{DRIVE_PATH} = $path;
        $hash{DRIVE_SPACE} = $space;
	push (@retArray, \%hash);
    }

    return @retArray;
}

# Gets available disk space for a given directory
#
# Input format:
#   .directory := Directory for which disk space is being requested.
# 
# Output format:
#   .diskSpace := Disk space in MB 
sub GetDiskSpace_Handler($$) {
   my ($in, $out) = @_;

   my $directory = $in->getValue(".directory");

   if (!defined($directory)) {
      &VMware::VMServerd::errorPost("Invalid directory $directory");
      return 0;
   }

   my $diskSpace = &VMware::VMServerd::GetDiskSpace($directory);

   $out->setValue(".diskSpace", $diskSpace);

   return 1;
}

&VMware::VMServerd::addOperation( OPNAME => 'GSXHostInfo_Query',
				  PERLFUNC => 'VMware::VMServerd::GSXHostInfo::Query_Handler',
				  POLICY => 'authuser');

&VMware::VMServerd::addOperation( OPNAME => 'GSXHostInfo_GetDiskSpace',
				  PERLFUNC => 'VMware::VMServerd::GSXHostInfo::GetDiskSpace_Handler',
				  POLICY => 'authuser');

1;
