#
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#

package VMware::Control;

#require 5.005_62;
use strict;
#use warnings; #Doesn't work in 5.005_03
use Carp;

require Exporter;
use AutoLoader;
use VMware::VmPerl;

#Note: "our" is not supported in 5.005_03
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK $AUTOLOAD);

$VERSION = '1.01';

@ISA = qw(Exporter);

@EXPORT = qw(
             VM_EXECUTION_STATE_OFF
             VM_EXECUTION_STATE_ON
             VM_EXECUTION_STATE_STUCK
             VM_EXECUTION_STATE_SUSPENDED
             VM_EXECUTION_STATE_UNKNOWN
             VM_E_ALREADYCONNECTED
             VM_E_BADRESPONSE
             VM_E_BADSTATE
             VM_E_BADVERSION
             VM_E_DISCONNECT
             VM_E_GARBAGE
             VM_E_INSUFFICIENT_RESOURCES
             VM_E_INVALIDARGS
             VM_E_INVALIDVM
             VM_E_NEEDINPUT
             VM_E_NETFAIL
             VM_E_NOACCESS
             VM_E_NOEXECVMAUTHD
             VM_E_NOMEM
             VM_E_NOPROPERTY
             VM_E_NOSUCHVM
             VM_E_NOTCONNECTED
             VM_E_NOTSUPPORTED
             VM_E_PROXYFAIL
             VM_E_SUCCESS
             VM_E_TIMEOUT
             VM_E_UNSPECIFIED
             VM_E_VMBUSY
             VM_E_VMEXISTS
             VM_E_VMINITFAILED
             VM_PLATFORM_LINUX
             VM_PLATFORM_UNKNOWN
             VM_PLATFORM_VMNIX
             VM_PLATFORM_WINDOWS
             VM_POWEROP_MODE_HARD
             VM_POWEROP_MODE_SOFT
             VM_POWEROP_MODE_TRYSOFT
             VM_PRODINFO_BUILD
             VM_PRODINFO_PLATFORM
             VM_PRODINFO_PRODUCT
             VM_PRODINFO_VERSION_MAJOR
             VM_PRODINFO_VERSION_MINOR
             VM_PRODINFO_VERSION_REVISION
             VM_PRODUCT_ESX
             VM_PRODUCT_GSX
             VM_PRODUCT_UNKNOWN
             VM_PRODUCT_WS
             VM_TIMEOUT_ID_DEFAULT
             );

@EXPORT_OK = qw(
                get_error_message
                get_default_tcp_port
                ); 

sub AUTOLOAD {
    # This AUTOLOAD is used to 'autoload' constants from the constant()
    # XS function.  If a constant is not found then control is passed
    # to the AUTOLOAD in AutoLoader.

    my $constname;
    ($constname = $AUTOLOAD) =~ s/.*:://;
    croak "& not defined" if $constname eq 'constant';
    my $val = VMware::VmPerl::constant($constname, @_ ? $_[0] : 0);
    if ($! != 0) {
        if ($! =~ /Invalid/ || $!{EINVAL}) {
            $AutoLoader::AUTOLOAD = $AUTOLOAD;
            goto &AutoLoader::AUTOLOAD;
        }
        else {
            croak "Your vendor has not defined VMware::Control macro $constname";
        }
    }
    {
        no strict 'refs';
        *$AUTOLOAD = sub { $val };
    }
    goto &$AUTOLOAD;
}

# bootstrap VMware::Control $VERSION;

# Preloaded methods go here.

sub version {
    return VMware::VmPerl::Version();
}

sub get_error_message {
    return GetErrorMessage(shift);
}

sub get_default_tcp_port {
  return GetDefaultTCPPort();
}

#Force version compatability
die "Perl API Version does not match dynamic library version."
    unless (version() eq $VERSION);

VMware::Control::InitMemoryRoutines();

1;
   
__END__

=head1 NAME

VMware::Control - Interface for controlling VMware server products,
both locally and remotely.

=head1 SYNOPSIS

  use VMware::Control;

  $ver = VMware::Control::version();
   

=head1 DESCRIPTION

The VMware::Control package and its sub-packages provide the ability
to enumerate, query, and control the virtual machines running on a
given host.

CAUTION: This API is preliminary and will change significantly between
this Beta and the final release version.

=head1 Error Codes

(This section is not yet complete.)

=over 8

=item VMware::Control::VM_E_SUCCESS

The operation succeeded.

=item VMware::Control::VM_E_NETFAIL

A network failure or misconfiguration prevented the operation from
completing.

=item VMware::Control::VM_E_INVALIDARGS

The arguments specified are not valid for this operation.

=item VMware::Control::VM_E_NOACCESS

The operation could not be completed because of an access violation (a
permissions problem).

=item VMware::Control::VM_E_NOTCONNECTED

An operation was attempted on a disconnected $vm.

=item VMware::Control::VM_E_DISCONNECT

The network connection to the VM was lost.

=item VMware::Control::VM_E_BADVERSION

The version of the VMware::Control module and the VMware server
product are not compatable.

=item VMware::Control::VM_E_TIMEOUT

The operation timed out.

=item VMware::Control::VM_E_NOCREATE

Failure to create a VM.

=item VMware::Control::VM_E_ALREADYCREATED

VMs are instantiated with configuration files.  This error is returned
if a VM instance for this configuration file aleady exists.

=item VMware::Control::VM_E_NOFILE

The indicated file did not exist.

=item VMware::Control::VM_E_NOSUCHVM

The indicated VM does not exist.

=item VMware::Control::VM_E_NOPROPERTY

The desired property does not exist.

=item VMware::Control::VM_E_BADSTATE

An attempt was made to restore a non-suspended VM, power on an already
powered on VM, etc.

=item VMware::Control::VM_E_NEEDINPUT

The operation did not complete because the VM is stuck waiting for
user input (e.g., there is a question that needs to be answered).

=item VMware::Control::VM_E_NOTSUPPORTED

The operation is not supported for this server or virtual machine.

=head1 AUTHORS

John Haggerty, Daniel Kerns, Brett Vasconcellos.

=head1 COPYRIGHT

    (c) 2000 VMware Incorporated.  All rights reserved.

=head1 VERSION

Version 0.2	   21 Sep  2000

C<% pod2html --header --title="VMware::Control -- the VMware Perl API" E<lt> Control.pm E<gt> perlAPI.html>

=head1 SEE ALSO

perl(1).

=cut
