#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# Disk.pm : Disks related operations 
#
# XXX TODO This module should be re-written to use the disk library.
#

package VMware::VMServerd::Disk;
use strict;

use VMware::DOMAccess;
use VMware::VMServerd qw(Warning
                         errorPost);
use VMware::VMServerd::FileInfo qw(&fileExists
				   &isVmfsFile
				   );

use VMware::ExtHelpers qw(&internal_dirname 
                          &internal_basename
                          &absolute_path
                          &remove_dots_from_path
			  &getSizeMB
			  &shell_string
			  &getFileSeparator
			  &getFileSepRegEx
			  );

use Exporter   ();
use vars       qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

BEGIN {
    if ($VMware::VMServerd::PRODUCT =~ /ESX/i) {
	require VMware::VMServerd::ESXVMConfigEdit;
    }
    
    # set the version for version checking
    $VERSION     = 1.00;
    @ISA         = qw(Exporter);

}

my @validExtensions = qw(dsk vmdk pln dat);

# XXX This function exists to provide the hack for VMFS type disks.
# Once diskLib supports VMFS disks as well, we can get rid of this
# function and directly call GetDiskType().
sub getDiskType($) {
    my ($name) = @_;

    if (!hasValidDiskExtension($name)) {
	return undef;
    }
 
    my $diskType = &VMware::VMServerd::GetDiskType($name);
    if (defined($diskType)) {
	return $diskType;
    }

    if (isVmfsFile($name)) {
	return "VMFS";
    }

    return undef;
}


# Converts a disk name to the file name that represents it.
# Converts to absolute path, resolves symlinks and goes to REDO
# parent root.       
sub getCanonicalName($$) {
    my ($diskname, $config) = @_;

    my $name = absolute_path($diskname, internal_dirname($config));
    my $cname;

    my $diskType = getDiskType($name);
    if (!defined ($diskType)) {
      return undef;
    }

    $name = remove_dots_from_path($name);

    if ( ($diskType eq "SPARSE")   ||
         ($diskType eq "FLAT") ||
         ($diskType eq "DEVICE") ) {
        my $absname = VMware::VMServerd::resolveSymLinks($name);
	$cname = getREDOParentRoot($absname);
	return $cname;
      }

    if ($diskType eq "VMFS") {
        $cname = getCanonicalVMFSDiskName($name);
	return $cname;
      }

    return undef;
}

sub getUniqCanonicalDiskList($$) {
    my ($diskListRef, $config) = @_;

    my @canonicalDiskNames;
    foreach my $d (@$diskListRef) {
        push (@canonicalDiskNames, getCanonicalName($d, $config));
    }

    return @canonicalDiskNames;
}

sub diskIsDeletableByUser($) {
  my ($cname) = @_;

  # $cname is Canonical Name for the disk

  my $dir = internal_dirname($cname);
  
  if (-e $cname && -f $cname &&
      -e $dir && -d $dir && -w $dir) {
    return 1;
  }

  return 0;
}

sub RemoveCOWDisk($) {
    my ($disk) = @_;

    return &VMware::VMServerd::RemoveDisk($disk);
}

sub RemoveDisk($) {
  my($disk) = @_;

  &VMware::VMServerd::errorReset();

  my $diskType = getDiskType($disk);

  # XXX Disk code should not be in ESXVMConfigEdit.pm. It should be
  # moved to this module.
  if ($diskType eq "VMFS") {
     return VMware::VMServerd::ESXVMConfigEdit::RemoveDisk($disk);
  }

  if ( ($diskType eq "SPARSE")   ||
       ($diskType eq "FLAT") ||
       ($diskType eq "DEVICE") ) {
      return RemoveCOWDisk($disk);
  }

  return 0;
}

#######################################################################
#
# Description:
#    Gets lock files on the cow disk members and REDO logs
#
#######################################################################
sub getLockFiles($) {
    my ($ref) = @_;
    my @files = @$ref;
    my @newlist;
    my @extens = (".lck", ".WRITELOCK", ".READLOCK");

    foreach my $ext (@extens) {
        foreach my $f (@files) {
            if (-e $f.$ext && -f $f.$ext) {
	        push (@newlist, $f.$ext);
	    }
	}
    }
    return @newlist;
}
    
#######################################################################
#
# Description:
#    Returns TRUE if there exist READ or WRITE locks on any of the COW disk members
#
#######################################################################
sub isDiskLocked($) {
    my ($diskname) = @_;

    my @files = getCOWDiskFiles($diskname);

    my @redoLogs = getREDOLogs($diskname);
    push(@files, @redoLogs);

    my @lockFiles = getLockFiles(\@files);

    return (scalar(@lockFiles) > 0);
}

sub getREDOParentRoot($) {
    my ($name) = @_;
    my $prev = $name;

    while (defined($name)) {
	$prev = $name;
	$name = getREDOParent($name);
    }
    return $prev;
}

sub getREDOParent($) {
    my ($name) = @_;
    # XXX TODO: Handle redoLogDir config directive.

    if ($name =~ /\.REDO/) {
        $name =~ s/\.REDO//;
        return $name;
    }
    return undef;
}

#######################################################################
#
# Description:
#    Returns the REDO file name
#
# Note: 
#    XXX Does not handle redoLogDir config directive.
#
#######################################################################
sub getREDOFileName($$) {
    my ($name, $level) = @_;

    my $rlogName = $name;
    if ($level == 0) {
        &VMware::VMServerd::errorAppend("Internal error: level=0 in getREDOFileName");
	return undef;
    }

    for (my $i=0; $i<$level; $i++) {
	$name .= ".REDO";
    }
    return $name;
}

#######################################################################
#
# Description:
#    Gets the REDO logs of a given COW disk
#
# Note: 
#    XXX Does not handle redoLogDir config directive.
#
# Returns:
#    An array of REDO files
#
#######################################################################
sub getREDOLogs($) {
    my ($name) = @_;
    my @REDOLogs;
  
    my $level = 1;
    while ($level != 0) {
        my $rlogName = getREDOFileName($name, $level);
        if (-e $rlogName && -f $rlogName) {
            push (@REDOLogs, $rlogName);
	    # Yup! This is how it is in big_cowdisk.c
	    my @redoMemberFiles = getCOWDiskMemberFiles($rlogName);
	    push (@REDOLogs, @redoMemberFiles);
	    $level++;
  	}
	else {
	    $level = 0;
	    last;
	}
    }

    return @REDOLogs;
}


#######################################################################
#
# Description:
#    Gets all the big COW disk member files
#
# Returns:
#    An array of all the member files. Does not include the head file itself.
#
#######################################################################
sub getCOWDiskMemberFiles($) {
    my ($name) = @_;
    my @files;

    my $disknum = 2;
    my $memberFile;
    while (1) {
  	$memberFile = getCOWDiskMemberFileName($name, $disknum);
        if (! defined($memberFile)) {
	    last;
        }
	push (@files, $memberFile);
        $disknum++;
    } 

    return @files;
}

#######################################################################
#
# Description:
#    Converts disk name and number to the big COW disk name, it it exists
#
# Returns:
#    The big COW disk member file name
#
#######################################################################
sub getCOWDiskMemberFileName($$) {
    my ($fileName, $diskNum) = @_;

    # The naming convention for big COW disks is in lib/disk/big_cowdisk.c
    # If that ever changes, make the changes here as well.
    # If the file name is <fname>.<ext>, then the big cowdisk names
    # are <fname>-0x.<ext>, where x goes from 2 through 9, and
    # <fname>-N.<ext>, where N starts and 10 and goes up. If ".<ext>"
    # does not exist in file name, then it doesn't exist in derivatives.

    # XXX TODO: Invoke the C function in libdisk to get the member 
    # file name instead of doing it here.

    # The "#" makes sure we convert <disk.> to <disk-02.> and not <disk.-02>
    my @segments = split(/\./, $fileName."#");
    my $numsegs = @segments;
    my $baseFileName;
    my $ext;
    if ($numsegs == 1) {
	$baseFileName = $fileName;
    } else {
	$ext = $segments[$numsegs - 1];
	$#segments--;
	$baseFileName = join('.', @segments);
    }
    my $memberName = $baseFileName . '-';
    if ($diskNum <= 9) {
	$memberName .= '0'.$diskNum;
    }
    else {
	$memberName .= $diskNum;
    }
    if (defined($ext)) {
	$memberName .= ".";
        if ($ext ne "#") {
	    chop($ext);
	    $memberName .= $ext;
        }
    }

    if (-e $memberName && -f $memberName) {
  	return $memberName;
    }
    return undef;
}

#####################################################
###
### The following are only needed for ESX
###
#####################################################

sub getCanonicalVMFSDiskName {
    my($diskname) = @_;

    # Break the string into two parts separated by a colon.  The second
    # part should contain no colons.
    if( $diskname !~ /^(\S+):([^:]+)$/ ) {
#	&VMware::VMServerd::errorAppend("Invalid disk name '$diskname' received.");
	return(undef);
    }
    my($vmfsname, $filename) = ($1, $2);

    # Check for the existence of the VMFS
    my $vmfs = &VMware::VMServerd::ESXVMHostInfo::getESXHost()->getMFSLookup($vmfsname);
    if( !defined($vmfs) ) {
#	&VMware::VMServerd::errorAppend("Invalid disk name $diskname received.  Could not find a " .
#					"VMFS for name $vmfsname.");
	return(undef);
    }

    return(VMware::HConfig::RawDisk::makeVmkVmfsFileName($vmfs->getVmkName(), $filename));
}

#####################################################
###
### Given a list of disk names, builds a list of unique, canonical disk names.
###
#####################################################

sub getUniqCanonicalVMFSDiskList {
    my(@diskNameList) = @_;
    my $diskName;
    my @newDiskNameList;

    foreach $diskName (@diskNameList) {
	my $canonicalName = getCanonicalVMFSDiskName($diskName);
	if( !defined($canonicalName) ) {
#	    &VMware::VMServerd::errorAppend("Invalid VMFS disk $diskName specified.");
	    next;
	}
	if( !grep($_ eq $canonicalName, @newDiskNameList) ) {
	    push(@newDiskNameList, $canonicalName);
	}
    }
    return(@newDiskNameList);
}

#######################################################################
#
# Description:
#    Checks to see if the disk has a valid disk extension 
#
# Input:
#
# Output:
#    Returns 1 or 0 depending on if the extension is va
#
#######################################################################
sub hasValidDiskExtension($) {
    my ($filename) = @_;

    # If the file has the right extension, it could be a disk file
    my $basename = internal_basename($filename);
    my @parts = split(/\./, $basename);

    if (scalar(@parts) < 2) {
	return 0;
    }
    my $fileExtension = $parts[$#parts];

    if (!grep($_ eq $fileExtension, @validExtensions)) {
	return 0;
    }

    return 1;
}

#######################################################################
#
# Description:
#    Checks if the given file is a COW disk
#
#######################################################################
sub isCOWDisk($) {
    my ($filename) = @_;

    my $diskType = getDiskType($filename);
    if (defined($diskType)) {
	return (($diskType eq "SPARSE") || ($diskType eq "FLAT"));
    }

    return 0;
}

#######################################################################
#
# Description:
#    Checks if the given file is a VMFS disk
#
#######################################################################
sub isVmfsDisk($) {
    my ($filename) = @_;

    my $diskType = getDiskType($filename);
    if (defined($diskType)) {
	return ($diskType eq "VMFS");
    }

    return 0;
}


#######################################################################
#
# Description:
#    Checks if the given file is a COW disk member file. 
#
# Returns:
#    1 if it is a member (REDO log, big_cowdisk member (-02, ...), 
#    or read/write lock
#
#######################################################################
sub isCOWDiskMember($) {
    my ($filename) = @_;

    my $pos = rindex($filename, ".");
    if ($pos == -1) {
	return 0;
    }

    my $diskExtension = substr($filename, $pos+1);
    my $diskName = substr($filename, 0, $pos);

    if (!defined($diskExtension)) {
	return 0;
    }

    for my $ext (@validExtensions) {
	if ($ext eq $diskExtension) {
	    my $parentDisk = undef;
	    # "SPARSE" disk
	    if ($diskName =~ /^(.*)-(s?\d+)$/) {
		$parentDisk = $1 . "." . $ext;
	    }
	    # "FLAT" disk
	    if (($diskName =~ /^(.*)-flat$/) ||
		($diskName =~ /^(.*)-(f?\d+)$/)) {
		$parentDisk = $1 . "." . $ext;
	    }
	    # GSX 2.5 plain disk
            if ($diskExtension eq "dat") {
                if ($diskName =~ /^(.*)(\d+)$/) {
                    $parentDisk = $1 . ".pln";
                }
            }    
	    if (defined($parentDisk) && 
		fileExists($parentDisk) &&
		isCOWDisk($parentDisk)) {
		return 1;
	    }
	}
    }

    return 0;
}

#######################################################################
#
# Description:
#    Returns the size of the COW disk, all its members combined
#
# Returns:
#    Size of all the members combined in MB
#
#######################################################################
sub getCOWDiskSize($) {
    my ($diskname) = @_;

    my ($capacity, $size) = &VMware::VMServerd::GetDiskSize($diskname);
    if (defined($size)) {
       # If the COW disk is smaller than 1MB in size, serverd returns
       # the size as 0. We round it up to 1MB. 
       return ($size > 0) ? $size : 1;
    } else {
       &VMware::VMServerd::Warning("Could not get size of disk $diskname \n");
       return 0;
    }
}


#######################################################################
#
# Description:
#    Changes the owner of all COW disk member files
#
# Returns:
#
#######################################################################
sub ChangeOwnerCOWDisk($$) {
    my ($diskname, $owner) = @_;

    if (!defined($owner)) {
	return (0, "Invalid user $owner;");
    }

    my ($name, $passwd, $uid) = getpwnam($owner);

    if (!defined($uid)) {
	return (0, "Invalid user $owner;");
    }

    if (isDiskLocked($diskname)) {
	return (0, "Cannot change owner of disk $diskname. Disk is locked;");
    }

    my @files = getCOWDiskFiles($diskname);

    push (@files, $diskname);

    my $errstr = undef;
    foreach my $f (@files) {

	my ($fuid, $fgid) = (stat($f))[4,5];

	if (!chown($uid, $fgid, $f)) {
	    $errstr .= "Could not change owner of $f: $!;";
	    next;
	}
    }

    if (defined($errstr)) {
	return (0, $errstr);
    }

    return 1;
}

#######################################################################
#
# Description:
#    Changes the group of all COW disk member files
#
# Returns:
#
#######################################################################
sub ChangeGroupCOWDisk($$) {
    my ($name, $group) = @_;

    if (!defined($group)) {
	return (0, "ChangeGroup: invalid group;");
    }

    my $gid = getgrnam($group);
    
    if (!defined($gid)) {
	return (0, "ChangeGroup: invalid group;");
    }

    if (isDiskLocked($name)) {
	return (0, "Cannot change group on disk $name. Disk is locked;");
    }

    my @files = getCOWDiskFiles($name);

    push (@files, $name);

    my $errstr = undef;
    foreach my $f (@files) {

	my ($uid, $fgid) = (stat($f))[4,5];

	if ($gid != $fgid) {
	    if (!chown($uid, $gid, $f)) {
		$errstr .= "Could not change group of $f: $!;";
	    }
	}
    }

    if (defined($errstr)) {
	return (0, $errstr);
    }

    return 1;
}

#######################################################################
#
# Description:
#    Changes the access mode on all COW disk member files
#
# Returns:
#
#######################################################################
sub ChangeModeCOWDisk($$) {
    my ($name, $mode) = @_;

    if (!defined($mode)) {
	return (0, "Invalid file mode;");
    }

    if (isDiskLocked($name)) {
	return (0, "Cannot change access mode of disk $name. Disk is locked;");
    }

    my @files = getCOWDiskFiles($name);

    push (@files, $name);

    my $errstr = undef;
    foreach my $f (@files) {

	if (!chmod($mode, $f)) {
	    $errstr .= "Could not change mode on file $f;";
	    next;
	}

    }

    if (defined($errstr)) {
	return (0, $errstr);
    }

    return 1;
}

#######################################################################
#
# Description:
#   Moves the COW disk files to a new directory or file
#   The move _has_ to be within the same file system
#   If moving xyz.vmdk to abc.vmdk, then xyz-02.vmdk is
#   moved to abc-02.vmdk, etc.
#
# Returns:
#
#######################################################################
sub MoveCOWDisk($$) {
    my($name, $dest) = @_;

    my $fsep = ($^O =~ /linux/i) ? "/" : "\\";

    if (isDiskLocked($name)) {
	return (0, "Cannot remove disk $name. Disk is locked;");
    }

    my @files = &getCOWDiskFiles($name);

    my $srcDiskName  = internal_basename($name);
    $srcDiskName =~ s/^(.*)\..*$/$1/;

    if (-e $dest) {
	return (0, "Destination of Disk move ($dest) already exists;");
    }

    my $destDiskName = undef;
    my $destDiskExtn = undef;
    if (!-d $dest) {
	$destDiskName = internal_basename($dest);
	$destDiskName =~ s/^(.*)\..*$/$1/;
	$destDiskExtn = substr($dest, rindex($dest, ".")+1);
	if (!grep($_ eq $destDiskExtn, @validExtensions)) {
	    return (0, "Invalid extension $destDiskExtn for the destination disk;");
	}
    }

    my @commands;
    my $errstr = undef;
    my $destFile = undef;
    foreach my $f (@files) {
	if (-d $dest) {
	    $destFile = $dest . getFileSeparator() . $f;
	    if (-e $destFile) {
		$errstr .= "Destination file $destFile already exists;";
		next;
	    }
	    push(@commands, shell_string($f) . " " . shell_string($dest));
	}
	else {
	    $destFile = $dest;
	    if ($f ne $name) { # It is a COWD member
		my $memberExtn = substr($f, 
			length(internal_dirname($name) . $fsep . $srcDiskName));
		print STDERR "Intermediate memberExtn1 = $memberExtn \n";
		$memberExtn =~ s/(-\d+)\..*$/$1/;
		print STDERR "Intermediate memberExtn2 = $memberExtn \n";
		$destFile = internal_dirname($dest) . $fsep . 
		    $destDiskName . $memberExtn . "." . $destDiskExtn;
		print STDERR "Destination disk file = $destFile \n";
	    }
	    if (-e $destFile) {
		$errstr .= "Destination file $destFile already exists;";
		next;
	    }
	    push(@commands, shell_string($f) . " " . shell_string($destFile));
	}
    }

    if (!defined($errstr)) {
	for my $cmd (@commands) {
	    my ($ok, $err) = 
		&VMware::VMServerd::FileManager::FileManager_ExecuteProgram("mv", $cmd);

	    if (!$ok) {
		$errstr .= "$err;";
	    }
	}
    }
			    						     
    if (defined($errstr)) {
	return (0, $errstr);
    }

    return 1;
}



#######################################################################
#
# Description:
#    Gets all the COW disk member files
#
# Returns:
#    An array of all the member files, including the head disk file itself.
#
#######################################################################
sub getCOWDiskFiles($) {
    my ($name) = @_;

    my @files;
    
    push(@files, $name);
    push(@files, getCOWDiskMemberFiles($name));

#    my @redoLogs = getREDOLogs($name);
#    push(@files, @redoLogs);
#    my @lockFiles = getLockFiles(\@files);
#    push(@files, @lockFiles);

    return @files;
}

sub Disk_CreateCOWDisk($$$$$) {
    my ($config, $fileName, $capacityGB, $diskType, $virtualHWversion) = @_;


    if ( !defined($config) || !defined($fileName) ||
         !defined($capacityGB) || !defined($diskType) ||
         !defined($virtualHWversion) ) {
        return (0, "CreateCOWDisk: Invalid input");
    }
    if ($capacityGB <= 0) {
        return (0, "CreateCOWDisk: Invalid capacity value: $capacityGB GB");
    }
    if (fileExists($fileName)) {
        return (0, "CreateCOWDisk: Disk File $fileName already exists ");
    }
    if (($diskType ne "scsi") && ($diskType ne "ide")) {
        return (0, "CreateCOWDisk: Invalid disk type $diskType. " .
                   "Must be 'scsi' or 'ide'");
    }

    my ($ok, $err) = &VMware::VMServerd::CreateCOWDisk(
                         $config, $fileName, $capacityGB, $diskType, 
                         $virtualHWversion);
    return ($ok, $err);
}


sub CreateCOWDisk_Handler {
    my $in = shift;
    my $out = shift;

    &VMware::VMServerd::Log("CreateCOWDisk Handler called with");
    my $config     = $in->getValue(".config");
    my $fileName   = $in->getValue(".filename");
    my $capacityGB = $in->getValue(".capacityGB");
    my $diskType   = $in->getValue(".diskType");
    my $virtualHWversion   = $in->getValue(".virtualHWversion");

    my ($ok, $err) = Disk_CreateCOWDisk($config, $fileName, $capacityGB,
                                        $diskType, $virtualHWversion);
    if (!$ok) {
        &VMware::VMServerd::errorPost("$err");
        return 0;
    }

    return 1;
}

&VMware::VMServerd::addOperation( OPNAME => 'CreateCOWDisk',
                                  PERLFUNC => 'VMware::VMServerd::Disk::CreateCOWDisk_Handler',
                                  POLICY => 'authuser');

1;
