#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# EventLog.pm
# 
# Event-logging for all VMs
#

package VMware::VMServerd::EventLog;
use strict;
use VMware::DOMAccess;
use FileHandle;
use VMware::VMServerd qw(&Log &Warning);

my $var_log = &getEventLogDir();
my $locations_db = "/etc/vmware/locations";
my $eventLogTrimAt    = 40; # When the event log is this big, trim it.
my $eventLogStartWith = 15; # Trim it down to this many lines.
my $debug = 0;

my %logs;

my %powerSubjects = ( "on" => { "off" => "Power-off",
                                "suspended" => "Suspend" }, 
                      "off" => { "on" => "Power-on" },
                      "suspended" => { "on" => "Resume" } );

my %powerBodys = ( "on" => { "off" => "Powered-off",
                             "suspended" => "Suspended" }, 
                   "off" => { "on" => "Powered-on" },
                   "suspended" => { "on" => "Resumed" } );

my %dialogSubjects = ( "info" => "VMware Informational Message",
                       "warning" => "VMware Warning",
                       "error" => "VMware Error",
                       "question" => "VMware Question" );

sub debugTrace {
    my ($package, $filename, $line) = caller; 
    print STDERR "DEBUG: $package $filename $line\n" if ($debug);
}

# URL-encode data.  Taken from CGI.pm
sub escape {
    shift() if ref($_[0]) || $_[0] eq 'CGI';
    my $toencode = shift;
    return undef unless defined($toencode);
    $toencode=~s/([^a-zA-Z0-9_.-])/uc sprintf("%%%02x",ord($1))/eg;
    return $toencode;
}

sub new($$) {
    my $proto = shift;
    my $vmcfg = shift;
    my $class = ref($proto) || $proto;
    my $self = {};

    $self->{vmcfg} = $vmcfg;
    $self->{log} = [];

    if (defined($logs{vmcfg})) {
        die("Log for " . $vmcfg ." already exists\n");
    }

    bless($self, $class);

    readEventLogFile($self);

    $logs{$vmcfg} = $self;

    return($self);
}

sub readEventLogFile {
    my $self = shift;
    my $vmcfg = $self->{vmcfg};

    # Clear old events
    my @list = ();
    $self->{log} = \@list;

    # Read in previous event logs
    my $hash_cfg = escape($vmcfg);
    my $filename = $var_log . "/event-" . $hash_cfg . ".log";
    my $fh = new FileHandle;
    my $prev_lines;
    # store the log filename for convenient opening at future times
    $self->{logfilename} = $filename;

    if (open($fh, "<$filename")) {
        my $RS_sav = $/;
        $/ = undef;
        $_ = <$fh>; # slurp whole file
        $/ = $RS_sav;

        # Trim to $eventLogStartWith lines
        my @lines = split(/\n/, $_);
        if ($#lines > $eventLogStartWith) {
            Log( "Trimming event log for $vmcfg by " .
                 ($#lines - $eventLogStartWith) . " lines...\n");

            my $first = $lines[0];
            @lines = @lines[($#lines - $eventLogStartWith) .. $#lines];
            unshift(@lines, $first);

            print STDERR "Down to " . $#lines . " lines\n" if($debug);
            $prev_lines = join("\n", @lines) . "\n";
        } else {
            $prev_lines = $_;
        }
 
        # Translate the XML
        my $doc;
        eval {
            $doc = VMware::DOMAccess->newXML($prev_lines . "</vm>");
        };
        if(!($@)) {
            print STDERR "Adding XML events...\n" if ($debug);
            addXMLEvents($self, $doc);
            $doc->dispose();
        } else {
            Warning( "Improperly formatted XML in $filename: $@\n");
        }

        $fh->close();
    } else {
        # We are creating a new file
        my $doc = $self->toXML(0);
        my $tag = $doc->getValue("");
        $doc->dispose();

        $tag =~ s@</vm>$@@; #Turn the tag into an open tag
        $prev_lines = $tag . "\n";

        # Register the event log with installer
	registerEventLog($filename);
    }        

    # Open log for writing
    if (open($fh, ">$filename")) {
        autoflush $fh 1;

        # Write any previous lines
        $fh->print($prev_lines);
        close($fh);
    } else {
        Warning( "EventLog: could not open $filename for writing.\n");
    }
}

##
## checkLogName - Consistency check the log, make sure that the name in
##		  the log matches the vm we are looking for.
##

sub checkLogName($$) {
    my ($vmcfg, $eventLog) = @_;
    my $logcfg = $eventLog->{vmcfg};

    if( $^O eq "MSWin32" ) {
	$vmcfg = lc $vmcfg;
	$logcfg = lc $logcfg;
    }

    if ( $vmcfg eq $logcfg ) {
	return 1;
    }
    return 0;
}


sub addXMLEvents($@) {
    my $eventLog = shift;
    my $doc = shift;
    my $eventId = 0;
    my $subdoc;
    my $vmcfg = $doc->getAttribute("vm", "cfg");

    if (!checkLogName( $vmcfg , $eventLog )) {
        Warning( "Events in the loaded file for $vmcfg are for a different VM.  Ignoring them.\n");
        return undef;
    }

    while ($doc->hasElement("vm.event[$eventId]")) {
        $subdoc = $doc->getSubTree("vm.event[$eventId]");
        my $type = $subdoc->getAttribute(".", "type");
        my $time = $subdoc->getValue(".time");
        my $subject = $subdoc->getValue(".subject");
        my $body;

        if (! $subdoc->hasElement(".body.p[0]")) {
            $body = $subdoc->getValue(".body");
        } else {
            # body: paragraphs --> double new-lines
            $body = "";
            my $pnum = 0;
            while($subdoc->hasElement(".body.p[$pnum]")) {
                my $p = $subdoc->getValue(".body.p[$pnum]");
                if ($pnum > 0) {
                    $body .= "\n\n";
                }
                $body .= $p;
                $pnum++;
            }
        }

        # Optional
        my $id = $subdoc->getAttribute(".", "id");
        if ($id eq "") {
            undef $id;
        }
        my $user = undef;
        if ($subdoc->hasElement(".user")) {
            $user = $subdoc->getValue(".user");
        }

        my $event = VMware::VMServerd::EventLog::Event->new($vmcfg,
                                                            $type,
                                                            $time,
                                                            $subject,
                                                            $body,
                                                            $id,
                                                            $user);

        # Add the Choices
        my $i = 0;
        my $choice;
        while($subdoc->hasElement(".choice[$i]")) {
            $choice = $subdoc->getValue(".choice[$i]");
            $event->addChoice($choice, $i);
            $i++;
        };

        # Add it to the list
        _addEventToLog($eventLog, $event);

        $subdoc->dispose();
        $eventId++;
    }
    print STDERR "Added $eventId events.\n" if($debug);
}

sub toXML($$) {
    my $eventLog = shift;
    my $lastN = shift() - 1;

    # Create top level <vm cfg="foo">
    my $doc = VMware::DOMAccess->new("vm");
    $doc->setAttribute("vm", "cfg", $eventLog->{vmcfg});
    $doc->setValue("vm.safe_cfg", VMware::VMServerd::toSafeCfgString($eventLog->{vmcfg}));

    # Iterate over events, returning the most recent lastN
    my $eventRef = $eventLog->{log};
    my ($cfg, $subdoc);

    if ($#$eventRef < $lastN) {
        $lastN = $#$eventRef;
    }
    my $index = $#$eventRef - $lastN;

    foreach my $event (@$eventRef[$index..$#$eventRef]) {
        ($cfg, $subdoc) = $event->toXML();
	if (!checkLogName( $cfg , $eventLog )) {
            die("Event from a different VM in my event log!\n");
        }
        print STDERR $subdoc->prettyPrint(),"\n" if ($debug);
        $doc->addSubTree("vm", $subdoc);
        $subdoc->dispose();
    }

    return ($eventLog->{vmcfg}, $doc);
}

sub _addEventToLog {
    my $eventLog = shift;
    my $event = shift;

    # Add it to the internal list
    push(@{$eventLog->{log}}, $event);
    
    print STDERR "Adding event: ",$event->{subject}," -- ",$event->{body},"\n" if ($debug);
}

sub addEventToLog {
    my $eventLog = shift;
    my $event = shift;

    $eventLog->_addEventToLog($event);
    
    # Write it to the file
    my $filename = $eventLog->{logfilename};
    my $fh = new FileHandle;
    if (open($fh, ">>$filename")) {
        autoflush $fh, 1;
        my ($cfg, $doc) = $event->toXML();
	if (!checkLogName( $cfg , $eventLog )) {
            die("Event from a different VM in my event log!\n");
        }
   
        $fh->print($doc->getValue(""), "\n");
        $doc->dispose();
        close($fh);
    } else {
        Warning("EventLog: Could not open $filename for writing\n");
    }

    $eventLog->trimLog();
}

sub trimLog {
    my $eventLog = shift;

    my @list = @{$eventLog->{log}};

    if ( $#list >= $eventLogTrimAt ) {
        print STDERR "Re-reading event log (to trim it)...\n" if ($debug);
        $eventLog->readEventLogFile();
    }
}

##################################################################
### Helper functions (don't act on a specific EventLog object) ###
##################################################################

sub findLog($) {
    my $vmcfg = shift;

    if( $^O eq "MSWin32" ) {
	$vmcfg = lc $vmcfg;
    }
    return $logs{$vmcfg};
}

sub findOrCreateLog($) {
    my $vmcfg = shift;

    if( $^O eq "MSWin32" ) {
	$vmcfg = lc $vmcfg;
    }

    if (!defined($logs{$vmcfg})) {
      VMware::VMServerd::EventLog->new($vmcfg);
    }

    return $logs{$vmcfg};
}


#########################################################
### Callbacks (called from the C portion of vmserverd ###
#########################################################

sub powerCallback {
    my $vmcfg = shift;
    my $user = shift;
    my $oldState = shift;
    my $newState = shift;

    my $log = findOrCreateLog($vmcfg);

    print STDERR "powerCallback: $vmcfg $user $oldState $newState\n" if ($debug);

    my $subjecthashref = $powerSubjects{$oldState};
    my $subject = "VMware Power Operation";

    if (defined($subjecthashref) && ref($subjecthashref) eq "HASH" &&
        defined($$subjecthashref{$newState})) {
        $subject = $$subjecthashref{$newState};
    } else {
        warn("Subject of Event not defined for transition from $oldState to $newState\n");
        return 0; #XXX?
    }

    my $bodyhashref = $powerBodys{$oldState};
    my $body = "State transition from $oldState to $newState";
    if (defined($bodyhashref) && ref($bodyhashref) eq "HASH" &&
        defined($$bodyhashref{$newState})) {
        $body = "The virtual machine was " . $$bodyhashref{$newState}; # . " by " . $user;
    } else {
        warn("Body of Event not defined for transition from $oldState to $newState\n");
        return 0; #XXX?
    }

    if ($user eq "unknown") {
        undef $user;
    }

    my $event = VMware::VMServerd::EventLog::Event->new($vmcfg, "info", time(), $subject,
                                                        $body, undef, $user);

    addEventToLog($log, $event);
    return 1;
}

sub dialogCallback {
    my $vmcfg = shift;
    #my $user = shift;
    my $sequence = shift;
    my $type = shift;
    my $text = shift;

    my $log = findOrCreateLog($vmcfg);

    my $subject = $dialogSubjects{$type};
    if (!defined($subject)) {
        $subject = "VMware Question";
    }
    my $body = $text;

    my $event = VMware::VMServerd::EventLog::Event->new($vmcfg, $type, time(), $subject,
                                                        $body, $sequence);

    $event->addChoices(@_);

    addEventToLog($log, $event);
    return 1;
}

sub answerCallback {
    my $vmcfg = shift;
    my $user = shift;
    my $sequence = shift;
    my $text = shift;
    my $choiceNum = shift;
    my $choiceText = shift;

    my $log = findOrCreateLog($vmcfg);

    my $subject = "Question Answered \"$choiceText\" by $user";
    my $body = $text;

    my $event = VMware::VMServerd::EventLog::Event->new($vmcfg, "answer", time(), $subject,
                                                        $body, $sequence, $user);

    $event->addChoice($choiceText, $choiceNum);

    addEventToLog($log, $event);    
    return 1;
}

#########################################################
### External functions to query an event log          ###
#########################################################


sub VM_GetLastNEvents_Handler {
    my $in = shift;
    my $out = shift;

    my $cfg = $in->getValue(".cfg");
    my $lastN = $in->getValue(".lastn");

    # Does the user have read access?
    # XXX Should be using a symbolic constant rather than 4 to specify read access
    if( !VMware::VMServerd::accessVM($cfg, 4) ) {
	&VMware::VMServerd::errorPost("Permission denied for config file $cfg\n", "error");
	return(0);
    }

    my $doc = VM_GetLastNEvents($cfg, $lastN);

    if (!defined($doc)) {
        return 0;
    }

    $out->addSubTree("", $doc);
    $doc->dispose();
    return 1;
}

sub VM_GetLastNEvents {
    my $cfg = shift;
    my $lastN = shift;

    my $log = findOrCreateLog($cfg);

    if (!defined($log)) {
        #XXX ASSERT fail
        print STDERR "EventLog: unable to find a log for $cfg\n";
	&VMware::VMServerd::errorPost("EventLog: unable to find a log for $cfg\n", "error");
        return undef;
    }

    my ($cfg2, $doc) = $log->toXML($lastN);

    return $doc;
}

sub getEventLogDir() {
    if( $^O eq "MSWin32" ) {
	my $dir = VMware::VMServerd::W32UtilGetInstallPath() . "/VMServerdRoot/eventlog";
	if( ! -e $dir ) {
	    if( !mkdir($dir, 0744) ) {
		Warning("Could not create event log directory");
	    }
	}
	return($dir);
    } else {
	return("/var/log/vmware");
    }
}

# Register the event log with installer
sub registerEventLog($) {
    my($filename) = @_;
    if( $^O eq "MSWin32" ) {
	# NOT IMPLEMENTED
    } else {
	if (open(INSTALLDB, ">>$locations_db")) {
	    print INSTALLDB "file $filename\n";
            close INSTALLDB;
        }
    }
}

VMware::VMServerd::addOperation( OPNAME => 'VM_GetLastNEvents',
				 PERLFUNC => 'VMware::VMServerd::EventLog::VM_GetLastNEvents_Handler',
				 POLICY => 'authuser' );

######################################################################
package VMware::VMServerd::EventLog::Event;
######################################################################
use strict;
use VMware::DOMAccess;

sub new($$$$$) {
    my $proto = shift;
    my $class = ref($proto) || $proto;
    my $self = {};

    $self->{vmcfg} = shift;
    $self->{type} = shift;
    $self->{time} = shift;
    $self->{subject} = shift;
    $self->{body} = shift;

    #Optional
    $self->{id} = shift;
    $self->{user} = shift;

    bless($self, $class);
    return($self);
}
    
sub addChoice($$$) {
    my $event = shift;
    my $choice = shift;
    my $id = shift;

    my %choices;

    if (defined($event->{choices})) {
        %choices = %{$event->{choices}};
    }

    if (defined($choices{$id})) {
        die("VMServerd::EventLog: Duplicate choice added to event.\n");
    }
    
    $choices{$id} = $choice;

    $event->{choices} = \%choices;
}

sub addChoices($@) {
    my $event = shift;
    my $id = 0;
    while ($_ = shift) {
        $event->addChoice($_, $id);
        $id++;
    }
}

sub toXML($) {
    my $event = shift;

    my $cfg = $event->{vmcfg};

    #Add <event type="foo"> underneath vm
    my $doc = VMware::DOMAccess->new("event");
    $doc->setAttribute("event", "type", $event->{type});
    if (defined($event->{id})) {
        $doc->setAttribute("event", "id", $event->{id});
    }

    #Add <time> underneath event
    $doc->setValue("event.time", $event->{time});

    #Add <subject> underneath event
    $doc->setValue("event.subject", $event->{subject});

    #Add <body> underneath event
    my $body = $event->{body};

    # Double new-lines --> paragraphs
    my $pnum = 0;
    foreach my $p (split(/\n\n/, $body)) {
        $p =~ s@\n@ @g;         # Single new-lines --> space
        $doc->setValue("event.body.p[$pnum]", $p);
        $pnum++;
    }

    #Optional
    #Add <user> underneath event
    if (defined($event->{user})) {
        $doc->setValue("event.user", $event->{user});
    }

    #Add <choices>s underneath event
    if (defined($event->{choices})) {
        my %choices = %{$event->{choices}};
        foreach my $id (sort(keys(%choices))) {
            my $choice = $choices{$id};
            $doc->setValue("event.choice[$id]", $choice);            
            $doc->setAttribute("event.choice[$id]", "id", $id);
        }
    }

    #print STDERR "toXML($cfg): ", $doc->getValue(""), "\n"; #DEBUG

    #return confg, DOM-tree;
    return ($cfg, $doc);
}

1;

