#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# SecurityEdit.pm
# 
# allows editing of security parameters
#

package VMware::VMServerd::SecurityEdit;
  
use strict;
use Carp;
use Exporter   ();
use vars       qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);
BEGIN {
    # set the version for version checking
    $VERSION     = 1.00;
    @ISA         = qw(Exporter);
    @EXPORT_OK   = qw(%gSecurityOptions);
}

use vars @EXPORT_OK;

use VMware::DOMAccess;
use VMware::VMServerd qw(Warning
                         errorPost
                         Log);

use VMware::Management::Util qw(get_host_and_domain);
use VMware::Config;
use POSIX qw(setsid);

my $gPrefFile;
my $gMUICertDir;
my $gRUICertDir;
my $gMUICertName = "mui";
my $gRUICertName = "rui";
my $gOpenSSLBinary;
my $gIsWin32 = 0;
my $gLocationsDB = "/etc/vmware-mui/locations";

$gSecurityOptions{'muissl'} = {value=>"yes",
               msg=>"Encrypt management-ui sessions using SSL",
               startFn=>\&apacheEnableSSL,
               stopFn=>\&restart_httpd,
               restartFn=>\&restart_httpd,
               getValFn=>undef,
               valid=>"Y|N",
               name=>'muissl',
               };
$gSecurityOptions{'ruissl'} = {value=>"yes", 
               msg=>"Encrypt vmware-console connections using SSL",
               valid=>"Y|N",
               name=>'ruissl',
               startFn=>\&create_rui_certificate,
               stopFn=>\&no_action,
               getValFn=>undef,
               };


# OS specific initialization stuff
if ( $^O eq "MSWin32" ) {
   $gPrefFile = VMware::VMServerd::W32UtilGetCommonAppDataFilePath("config.ini");
   $gMUICertDir = $gRUICertDir = VMware::VMServerd::W32UtilGetInstallPath() . "/ssl";
   $gSecurityOptions{muissl}{startFn} = \&IISEnableSSL;
   $gSecurityOptions{muissl}{stopFn} = \&IISDisableSSL;
   $gSecurityOptions{muissl}{restartFn} = undef;
   $gSecurityOptions{muissl}{getValFn} = \&getFromIIS;
   #XXX for unknown reasons, trying to give a full path to openssl.exe fails.
   # (has something to do with running from the perl interpreter -- because the same
   # command will run from a standalone script.).  For now just assume that openssl.exe
   # will be installed in $gMUICertDir -- and thus a full path won't be needed.
   $gOpenSSLBinary = 'openssl.exe';
   $gIsWin32 = 1;
} else {
   $gPrefFile = "/etc/vmware/config";
   $gMUICertDir = "/etc/vmware-mui/ssl";
   $gRUICertDir = "/etc/vmware/ssl";
   $gOpenSSLBinary = VMware::VMServerd::GetLibPathName("libdir") .  "/bin/openssl";
}


sub IISEnableSSL($) {
   # Check if the certificate has been installed.  If not, install it. 
   my $pfxName = undef;

   if (!&VMware::VMServerd::IISCertificateExists()) {
      Log("Certificate does not exist, creating one.\n");
      if (!generateCert($gMUICertDir, $gMUICertName, "security.mui_certificate.")) {
         errorPost("Couldn't create management interface certificate.\n", 'error');
         return 0;
      }
      $pfxName = $gMUICertDir . '/' . $gMUICertName . '.pfx';
   }

   if (!&VMware::VMServerd::IISCertificateInstalled()) {
         if (!&VMware::VMServerd::IISInstallCertificate($pfxName)) {
            Log("Couldn't install management interface certificate.\n");
            return 0;
         }
   }

   #certificate is now installed, delete the .pfx file too
   unlink($pfxName) if (defined $pfxName);

   if (!&VMware::VMServerd::IISEnableSSL()) {
      #XXX this seems to fail when it shouldn't (mayhap fails when ssl already enabled)
      Log("Error while trying to enable SSL.\n");
   } 
   return 1;
}

sub IISDisableSSL($) {
   my $success = &VMware::VMServerd::IISDisableSSL();
   if (!$success) {
      Log("Error while trying to disable SSL.\n");
   }
   return 1;
}

sub getFromIIS() { 
   return (&VMware::VMServerd::IISIsSSLEnabled() ? "yes" : "no");
}

#
# FindHttpd()
#   Look at the MUI's locations DB to find out where httpd lives
#
#
sub FindHttpd() {
   my $installDir = undef;
   local *FD;

   if (open(FD, $gLocationsDB)) {
      while (<FD>) {
         if (/^answer INITSCRIPTSDIR (.*)$/) {
            $installDir = $1;
         } elsif (/^remove_answer INITSCRIPTSDIR$/) {
            $installDir = undef;
         }
      }
      close(FD);
   }

   if (!defined($installDir)) {
      Warning("Couldn't locate mui installation\n");
      return;
   }
   my $helper = $installDir . "/httpd.vmware";

   if (! -x $helper) {
      Warning("$helper, doesn't exist or isn't executable\n");
      return;
   }
   return $helper;
}



#
# apacheEnableSSL
#
# enables SSL for apache.  creates a certificate if one doesn't exist
#

sub apacheEnableSSL() 
{
   if (!certificateFileExists($gMUICertDir, $gMUICertName)) {
      if (!generateCert($gMUICertDir, $gMUICertName, "security.mui_certificate.")) {
         errorPost("Couldn't create management interface certificate.\n", 'error');
         return 0;
      }
   }

   return restart_httpd();
}

#
# restart_httpd()
# 
# restart httpd by forking a child who sleeps for a while then calls httpd.
# XXX Should check for existence of SSL (if enabbling).
#
sub restart_httpd() {
   my $helper = FindHttpd();
   
   #XXX Should use long running operation to do this? 
   Warning("restart_httpd: Restarting httpd\n");
   if (!defined($helper)) {
      errorPost("Cannot restart httpd without httpd.vmware.  Please manually restart httpd\n", 'warning');
      return(0);
   }

   # The double fork is needed to make sure that we don't leave
   # defunct httpd processes scattered around on the floor.
   # (caused by the fact that serverd doesn't ignore SIGCHLD signals -- and
   # therefore the child process will be kept around in a defunct state pending 
   # a possible future waitpid).

   my $pid = fork();
   if ($pid > 0) { 
      # Parent
      if (waitpid($pid, 0) == -1) {
         Warning("restart_httpd: waitpid() failed (non-fatal).  Reason $!\n");
      }
      return 1;
   }elsif ($pid == 0) {
      $pid = fork();
      if ($pid != 0) {
         # Middle Process, simply exits.

         if (!defined($pid)) {
            Warning("restart_httpd: failed to fork httpd. Reason $!\n");
         }
         exit (0);
      } elsif ($pid == 0) {
         # child process, actually run the command
        if (!chdir('/')) {
            Warning("restart_httpd: Couldn't chdir to /.  Reason $!\n");
        }
        umask 0;

        # XXX This call seems to cause the httpd restart to fail;
        # removing for now since things seem to work fine without it
        # VMware::VMServerd::ResetProcessState();

        # ResetProcessState doesn't close these 3 descriptors:
        close(STDIN);
        close(STDOUT);
        close(STDERR);

        if (!setsid()) {
          Warning("restart_httpd: setsid() failed.  Reason $!\n");
        }

        sleep(3);
        {
          exec($helper . ' restart');
        }
        Warning("restart_httpd: Failed to restart httpd.  Reason: $!\n");
        exit(0);
      }
   }else {
      Warning("restart_httpd: Could not restart httpd.  Reason: $!");
      return(0);
   }
}


#
# ActivateConfig_Handler
# 
# Input format:
#  .pref[i].name is the name of the config parameter
#  .pref[i].value is the value for the config param
#
#  Sets the specified keys in the global preferences file.  For each
#  preference, also runs associated call back.
sub ActivateConfig_Handler {
   my $in = shift;
   my $out = shift;
   my $key;
   my $rc;
   my $i = 0;

   #Warning("Security IN: Doing pretty print" .  $in->prettyPrint() . "\n");
   my $config = VMware::Config->new();

   #get existing values (if the pref file exists)
   $config->readin($gPrefFile);
    

   #XXX Need to error / sanity check values
   #XXX  if turning on on of the SSL enabled options, should check for cert first
   #     and create one if necessary
   while ($key = $in->getValue(".pref[$i].name")) {
      my $value = $in->getValue(".pref[$i].value");
      $config->set("security.host." . $key,$value);

      #now activate the changes
      my $fn = undef;
      if ($value =~ m/Y|T|1/i) {
         $fn = $gSecurityOptions{$key}{startFn};
         #Warning("Security: Turning $key on\n");
      } else {
         $fn = $gSecurityOptions{$key}{stopFn};
         #Warning("Security: Turning $key off\n");
      }

      #Warning("No fn defined!\n") if !defined($fn);
      if (defined($fn)) {
         my $retval = &$fn($gSecurityOptions{$key}{name});
         $out->setValue(".pref[$i].name", $in->getValue(".pref[$i].name"));
         $out->setValue(".pref[$i].value", $retval);
      }
      $i++;
   }

   #Warning("Security OUT: Doing pretty print" .  $out->prettyPrint() . "\n");
   if ( !$config->writeout($gPrefFile) ) {
      Log("Could not write the config file $gPrefFile:$!");
      return(0);
   }
   return (1);
}

#
# Get Config_Handler
#
# Input format:
#  .pattern : return only those keys in the config file matching pattern.
#
# Output format:
#  .pref[i].name is the name of the config parameter
#  .pref[i].value is the value for the config param
#
sub GetConfig_Handler {
   my $in = shift;
   my $out = shift;

   my $config = VMware::Config->new();
   if( !$config->readin($gPrefFile) ) {
      Log("Could not read the config file $gPrefFile.");
      return(0);
   }
   my $pattern = $in->getValue('.pattern');

   my @securityKeys = $config->list($pattern);
   my $i = 0;
   foreach my $key (@securityKeys) {
      $key =~ m/$pattern\.(.*)$/;
      my $keyName = $1;
      my $fn = undef;
      $out->setValue(".pref[$i].name", $keyName);
      # use the specified function to get the data, if no function exists
      # get the value from the config file. 
      $fn = $gSecurityOptions{$keyName}{getValFn};
      if (defined($fn)) {
         $out->setValue(".pref[$i].value", &$fn());
      } else {
         $out->setValue(".pref[$i].value", $config->get($key, ""));
      }
      $i++;
    }
    return (1);
}

#
# certificateFileExists
#
# Return true if the specified certificate file exists on disk
#
sub certificateFileExists($$) {
   my ($dir,$name) = @_;

   return (-e "$dir/$name.key" && -e "$dir/$name.crt");
}


#
# create_rui_certificate
# 
#  Create a certificate for use with the remote ui, if there isn't already one
# present.
sub create_rui_certificate($) {
   if (!certificateFileExists($gRUICertDir, $gRUICertName)) {
      if (!generateCert($gRUICertDir, $gRUICertName, "security.rui_certificate.", 0)) {
         errorPost("Couldn't create remote console certificate.\n", 'error');
         return 0;
      }
   }
   return 1;
}

#
# no_action -- somebody give me a better name
#  
#    This function does nothing and can serve as a start/stop function if
#    no action is necessary.
#
sub no_action() {
   return 1;
}

#
# getTmpFile
#
# return a file name that is likely to be unused in the current directory
sub getTmpFile() {
   my $i;

   #don't infinte loop: give up after 50 tries.
   for ($i = 0; $i < 50; $i++) {
     my $fname = "serverd-$i";
     if (! -e $fname) {
        return $fname;
     }
   }
}

# loadCertificateOptions
#
# load the certificate options from the config file,
# and place them in the $optRef hash.  (error checking,
# and supplying defaults along the way).
#

sub loadCertificateOptions($$$) {
   my ($prefix, $config, $optRef) = @_;
   my %certDefaults = 
      ( countryname => {value=>"US",  maxLen=>2},
        stateorprovincename => {value=>"California", maxLen=>64},
        localityname => {value=>"Palo Alto", maxLen =>64},
        organizationname => {value=>"VMware, Inc.", maxLen=>64},
        organizationalunitname => {value=>"VMware Management Interface", maxLen=>64},
        commonname => {value=>"Not Specified" , maxLen=>64},
        emailaddress => {value=>"ssl-certificates\@vmware.com", maxLen=>40},
        daysvalid => {value=>"5000", maxLen=>20},
      );


   my ($host, $domain) = get_host_and_domain();
   $certDefaults{commonname}{value} = $host . '.' . $domain;


# XXX  Set serial number to something vmware specific. this way we can
# detect with high probability whether or not a given certificate was created by
# us.

   foreach my $key (keys %certDefaults) {
      $optRef->{$key} = $config->get($prefix . '.' . $key, $certDefaults{$key}{'value'});
      $optRef->{$key} = substr($optRef->{$key}, 0, $certDefaults{$key}{'maxLen'});
      Warning("config->get() returned: " . $optRef->{$key} . "\n");
   }
}


# writeCertCnf
#
# Write out a certificate configuration file.  This file 
# will be used by openssl to create a X509 certificate for the 
# mui / rui.
#
# We tag all of our certificates with an encoded version of 
# "VMware Inc." in the unstructuredName field.  Not 100%
# foolproof, but it will let us know whether or not a given 
# certificate was generated by this script.
#
sub writeCertCnf($) {
   my ($optRef) = @_;
   local *TMP;
   my %gCertInfo;


   my $tmpfile = getTmpFile();
   if (!defined($tmpfile)) {
      Warning("Couldn't find a tmp file to use.\n");
      return undef;
   }

   if (!open(TMP, ">" . $tmpfile)) {
      Warning("Couldn't open $tmpfile for writing. Reason $!");
      return undef;
   }
   # Netscape chokes if we create a new certificate with all
   # the same fields as the old one, but a different public
   # key (it compares the subject fields, and then decides to
   # use the current certificate -- rather than the new one
   # with the updated public key.)  Thus we tag each
   # certificate with the current time.
   my $curTime = time();


   print TMP <<EOF;
# Conf file that the management-ui will use to generate SSL certificates.

[ req ]
default_bits		= 1024
default_keyfile 	= server.key
distinguished_name	= req_distinguished_name
default_days            = $optRef->{daysvalid}

#Don't encrypt the key
encrypt_key             = no
prompt                  = no

string_mask = nombstr

[ req_distinguished_name ]
countryName             = $optRef->{countryname}
stateOrProvinceName     = $optRef->{stateorprovincename}
localityName            = $optRef->{localityname}
0.organizationName      = $optRef->{organizationname}
organizationalUnitName  = $optRef->{organizationalunitname}
unstructuredName        = ($curTime),(564d7761726520496e632e)
commonName              = $optRef->{commonname}
emailAddress            = $optRef->{emailaddress}
EOF
   close(TMP);
   return($tmpfile);
}


#
#  generateCert
#
# Create a certificate in the specified with with the specified name, using
# values from the config file that have the specified prefix

# XXX break up into windows & linux specific portions
sub generateCert($$$) {
   my ($sslDir, $certName, $prefix) = @_; 
   my %options;

   #don't really care if the file doesn't exist -- the defaults 
   #specified to config->get() will suffice.
   my $config = VMware::Config->new();
   $config->readin($gPrefFile);

   #change to directory where we keep certificates. 
   #This may be the first certificate created, so create the directory if necessary.
   if (!-d $sslDir) {
      if (!mkdir($sslDir, 0700)) {
         Log("Couldn't create ssl certificate directory, $sslDir.  Reason $!\n");
         return undef;
      }
   }

   if (&VMware::VMServerd::LimitFileAccess($sslDir, 1) != 1) {
      Log("Couldn't limit permissions on $sslDir\n");
   }

   if (!chdir($sslDir)) {
      Log("Couldn't chdir to ssl certificate directory, $sslDir.  Reason $!\n");
      return undef;
   }

   #XXX hack to get around necessary installer changes I don't know how to make.
   if (($^O eq "MSWin32") && !( -e $gOpenSSLBinary)) {
      #XXX might be able to use full paths now that openssl.exe isn't dynamically linked
      system("copy ..\\" . $gOpenSSLBinary);
   }

   loadCertificateOptions($prefix, $config, \%options);
   my $tmpfile = writeCertCnf(\%options);
   if (!defined($tmpfile) || ! -e $tmpfile) {
      Log("Failed to create certificate configuration file.\n");
      return undef;
   }

   #XXX need to make sure that daysvalid has a valid value (ie in range >0 and < 10000)
   if (system($gOpenSSLBinary . ' req -new -x509 -keyout ' . $certName . 
              '.key -out ' .  $certName . '.crt -config ' . $tmpfile . 
              ' -days ' . $options{'daysvalid'})) {
      Log("'openssl req -x509 ...' failed to complete successfully.  Reason: $! $?\n");
      unlink($tmpfile);
      return(0);
   }
   unlink($tmpfile);

   if ($gIsWin32 && ($certName eq $gMUICertName)) {
      if (system($gOpenSSLBinary . ' pkcs12 -export -out ' . $certName .  
                 '.pfx -in ' .  $certName . '.crt -inkey ' .  $certName . 
                 '.key -name ' . $prefix .  ' -passout pass:testpassword')) {
         Log("'openssl pkcs12...' failed to complete successfully.  Reason: $! $?\n");
         return (0);
      }
      unlink($certName . '.crt');
      unlink($certName . '.key');	       
      #XXX doesn't do enough here.  (can't use generated cert immediately in its 
      # current form).
   } else {
      # Be paranoid.  This call shouldn't be necessary -- the directory is already 
      # readable only # by root / admin / __vmware_user__
      if (&VMware::VMServerd::LimitFileAccess($certName . '.key', 0) != 1) {
         errorPost("Wasn't able to set permissions on the certificate's private key.\n" .
                    "You should manually ensure that $sslDir/$certName.key is readable only by the.\n" .
                     "server's administrator\n", 'info');
      }
   }

   return(1);
}



#
# CreateCertificate_Handler
#
# Input format:
#   .certname = security.rui_certificate or security.mui_certificate
#   .pref[i].name is the name of the certificate parameter
#   .pref[i].value is the value for the cert param
#
# will create the MUI or the RUI certificate, as specified, using values
# from the config file.  If the specified service is already enabled, then
# restart the service.
sub CreateCertificate_Handler {
   my $in = shift;
   my $out = shift;
   my $rc;
   my $config = VMware::Config->new();
   my $i = 0;
   my $certDir;
   my $certName;
   my $certPrefix;

   Log($gMUICertDir . "\n");
   my $whichCert = $in->getValue(".certname");
   if ($whichCert =~ m/^management-ui$/) {
      ($certDir, $certName, $certPrefix) = ($gMUICertDir, $gMUICertName, "security.mui_certificate");
   } elsif ($whichCert =~ m/^remote-ui$/) {
      ($certDir, $certName, $certPrefix) = ($gRUICertDir, $gRUICertName, "security.rui_certificate");
   } else {
      Warning("Attempting to create a type of certificate I know nothing about\n");
      return (0);
   }

   #get existing values (if the pref file exists)
   $config->readin($gPrefFile);

   #XXX Need to error / sanity check values
   while (my $key = $in->getValue(".pref[$i].name")) {
      my $value = $in->getValue(".pref[$i].value");
      $config->set($certPrefix . '.' . $key, $value);
      $i++;
   }

   if ( !$config->writeout($gPrefFile) ) {
      Log("Could not write the config file $gPrefFile:$!");
      return(0);
   }

   $rc = generateCert($certDir, $certName, $certPrefix);

   if (!$rc) {
      errorPost("Failed to create a new certificate.  Disabling mui SSL support (NOT!)\n", 'error');
      #should only disable ssl if there doesn't exist a prev certificate.
   }

   #now restart the service (only if enabled)
   my $servName = $certName . 'ssl';
   if ($config->get('security.host.' . $servName, "") =~ m/y|t|1/i) {
      if ($gSecurityOptions{$servName}{restartFn} ne undef) {
         &{$gSecurityOptions{$servName}{restartFn}}();
   }
   }

   return ($rc);
}


&VMware::VMServerd::addOperation( OPNAME => 'Security_CreateCertificate',
				  PERLFUNC => 'VMware::VMServerd::SecurityEdit::CreateCertificate_Handler',
				  POLICY => 'rootonly');

&VMware::VMServerd::addOperation( OPNAME => 'Security_ActivateConfig',
				  PERLFUNC => 'VMware::VMServerd::SecurityEdit::ActivateConfig_Handler',
				  POLICY => 'rootonly');

&VMware::VMServerd::addOperation( OPNAME => 'Security_GetConfig',
				  PERLFUNC => 'VMware::VMServerd::SecurityEdit::GetConfig_Handler',
				  POLICY => 'rootonly');
1;

