/*
 * Copyright 2006.  VMware, Inc.  The use of this code is subject to 
 * your agreement to the terms of an applicable VMware license.
 */


#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include "ppport.h"

#include "vix.h"

static void TheCallback(VixHandle handle,
                        VixEventType eventType,
                        VixHandle moreEventInfo,
                        void *clientData);

/* Default "NULL" parameter, correct for Perl */
#define NULL_SV   &PL_sv_undef

/*-------------------------------------------------------------------
 *
 * TheCallback
 *
 *    All Vix callbacks will go through this function.
 *
 *-------------------------------------------------------------------
 */

void 
TheCallback(VixHandle handle,
            VixEventType eventType,
            VixHandle moreEventInfo,
            void *clientData)
{
   /* No uses.  Here as a useful stub only */
}

/*------------------------------------------------------------------
 *
 * Here is an implementation of FindItems, the enumerator for 
 * VMs on the connected machine
 *
 *------------------------------------------------------------------
 */

/*
 * XXX malloc() has issues in Win32 perl (at least ActivePerl), so
 * sleaze it and use an array for the results
 *
 * this will fail if more there are more  MAX_FIND_RESULTS results
 */

#ifdef WIN32
#define PERL_MALLOC_BUG_HACK
#define MAX_FIND_RESULTS 100
#endif

#ifdef PERL_MALLOC_BUG_HACK

typedef struct vmArray {
   int num;
   char loc[MAX_FIND_RESULTS][512];
} vmArray;

#else

typedef struct vmList {
   char *loc;
   struct vmList *next;
}  vmList;

#endif


static void
findvms_cb(VixHandle jobHandle, VixEventType evType, VixHandle propHandle,
           void *clientdata)
{
#ifdef PERL_MALLOC_BUG_HACK
   vmArray *results = (vmArray *) clientdata;
#else
   vmList **lst = (vmList **) clientdata;
   vmList *ne;
#endif
   char *loc;
   VixError err;

   if (VIX_EVENTTYPE_FIND_ITEM != evType) {
      return;
   }

   err = Vix_GetProperties(propHandle, VIX_PROPERTY_FOUND_ITEM_LOCATION, &loc, 
                           VIX_PROPERTY_NONE);

   if (VIX_OK == err) {
#ifdef PERL_MALLOC_BUG_HACK
      strcpy(results->loc[results->num++], loc);
      Vix_FreeBuffer(loc);
#else
      ne = (vmList *)malloc(sizeof(vmList));
      if (ne) {
         ne->loc = strdup(loc);
         ne->next = *lst;
         *lst = ne;
         Vix_FreeBuffer(loc);
      }
#endif
   }
}

static VixError
#ifdef PERL_MALLOC_BUG_HACK
doFindVMs(VixHandle hostHandle, int timeout, vmArray *vmlist)
#else
doFindVMs(VixHandle hostHandle, int timeout, vmList **vmlist)
#endif
{
   VixError err = VIX_OK;
   VixHandle jobHandle;

   jobHandle = VixHost_FindItems(hostHandle, VIX_FIND_RUNNING_VMS,
                                 VIX_INVALID_HANDLE, timeout,
                                 findvms_cb, vmlist);
   err = VixJob_Wait(jobHandle, VIX_PROPERTY_NONE);
   Vix_ReleaseHandle(jobHandle);

   return err;
}

/*-----------------------------------------------------------------
 *
 * Common property-collection code.  Handles mappings between Vix
 * types and Perl types.
 *
 *-----------------------------------------------------------------
 */

/* Presently, Vix_GetProperties and VixJob_Wait fit this prototype */
typedef VixError (*propertiesFunction)(VixHandle handle, VixPropertyID propID, ...);

static void
doProperty(VixHandle handle, SV* propertySV, SV* resultSV, propertiesFunction func)
{
   char *sarg;
   int   iarg;
   int property;
   VixError err;
   VixPropertyType propertyType;

   property = SvIV(propertySV);

   err = Vix_GetPropertyType(handle, property, &propertyType);
   if (VIX_OK != err) {
      die("Failed to get property type.\n");
   }
   switch (propertyType) {
   case VIX_PROPERTYTYPE_INTEGER:
   case VIX_PROPERTYTYPE_INT64:
   case VIX_PROPERTYTYPE_HANDLE:
   case VIX_PROPERTYTYPE_BOOL:
       err = func(handle, property, &iarg, VIX_PROPERTY_NONE);
       sv_setiv(resultSV, iarg);
       break;
   case VIX_PROPERTYTYPE_STRING:
       err = func(handle, property, &sarg, VIX_PROPERTY_NONE);
       sv_setpv(resultSV, sarg);
       Vix_FreeBuffer(sarg);
       break;
   default:
       die ("Unrecognized property type\n");
   }
}

/*******************************************************************/
/*******************************************************************/
/*******************************************************************/

/* MODULE vs. PACKAGE
 * 
 * This is a bootstrap hack.
 *
 * All functions in this file are in the VMware::Vix namespace.
 * On Windows, Perl creates a Vix.DLL to contain the functions
 * within said namespace.  However, Vix.DLL is already used as the
 * C implementations of all Vix functions.  So, this file (and
 * Makefile.PL and VixBinding.pm) contains some trickery to actually
 * build everything as the VMware::VixBinding module, even though
 * all the functions in this module are exported into the VMware::Vix
 * namespace.
 */


/*******************************************************************/

MODULE = VMware::VixBinding PACKAGE = VMware::Vix::API::API PREFIX = Vix_
PROTOTYPES: ENABLE

SV *
Vix_GetErrorText(err,language)
        VixError                       err
        const char *                    language
   PREINIT:
        char *txt;
   CODE:
        txt = (char *) Vix_GetErrorText(err, language);
        RETVAL = newSVpv(txt, 0);
   OUTPUT:
        RETVAL

void
Vix_ReleaseHandle(handle)
        VixHandle                      handle
   CODE:
        Vix_ReleaseHandle(handle);

VixHandleType
Vix_GetHandleType(handle)
        VixHandle                      handle

VixError
Vix_GetProperties(handle, ...)
        VixHandle                      handle
   PREINIT:
       int property;
       int i;
   CODE:
        if ((items % 2) != 0) {
           die("Vix_GetProperties must have an even number of arguments.\n");
        }
        for (i = 2; i < items; i+= 2) {
           doProperty(handle, ST(i - 1), ST(i), Vix_GetProperties);
        }
        // should always be NULL
        property = SvIV(ST(items-1)); /* last property */
        if (property != VIX_PROPERTY_NONE)
            die("Final argument to GetProperties() should be VIX_PROPERTY_NONE\n");
        RETVAL = Vix_GetProperties(handle, property);
   OUTPUT:
        RETVAL

VixError
Vix_GetPropertyType(handle,propertyID,propertyType)
      VixHandleType     handle
      VixPropertyID     propertyID
      VixPropertyType   &propertyType
   CODE:
      RETVAL = Vix_GetPropertyType(handle, propertyID, &propertyType);
   OUTPUT:
      propertyType
      RETVAL


void
Vix_PumpEvents(hostHandle,options)
        VixHandle                      hostHandle
        int                             options
   CODE:
      Vix_PumpEvents(hostHandle, (VixPumpEventsOptions)options);


####################################################
# Note: Vix_FreeBuffer is not implemented.        #
# The binding frees all buffers that are returned  #
# after wrapping them in a Perl SV, because of the #
# difference in memory management.                 #
####################################################


MODULE = VMware::VixBinding PACKAGE = VMware::Vix::API::Host PREFIX = VixHost_
PROTOTYPES: ENABLE

VixHandle
VixHost_Connect(apiVersion,hostType,hostName,hostPort,username,password,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        int                             apiVersion
        VixServiceProvider                    hostType
        const char *                    hostName
        int                             hostPort
        const char *                    username
        const char *                    password
        int                             options
        VixHandle                      propertyListHandle
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixHost_Connect(apiVersion, 
                              hostType, hostName, hostPort,
                              username, password,
                              options, 
                              propertyListHandle,
                              NULL, NULL);
   OUTPUT:
        RETVAL

void
VixHost_Disconnect(hostHandle)
        VixHandle                       hostHandle
   CODE:
        VixHost_Disconnect(hostHandle);


#
# This breaks the raw Perl<->C model used elsewhere, because of the pain
# of using Perl callbacks.  So rather than try to return a C type,
# this code stuffs a Perl array on the stack, with the first element
# being the error code.
#
void
VixHost_FindItems(hostHandle,searchType,searchCriteria,timeout,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      hostHandle
        VixFindItemType                searchType
        VixHandle                      searchCriteria
        int                            timeout
        SV *                           callbackProc
        SV *                           clientData
   PREINIT:
        VixError err;
#ifdef PERL_MALLOC_BUG_HACK
        vmArray results;
#else
        vmList *vmlist = NULL;
        vmList *l;
        vmList *t;
#endif
        int i;
   PPCODE:
#ifdef PERL_MALLOC_BUG_HACK
        results.num = 0;
        err = doFindVMs(hostHandle, timeout, &results);
        if (VIX_OK == err) {
           XPUSHs(sv_2mortal(newSVnv(err)));
           for (i = 0; i < results.num; i++) {
              XPUSHs(sv_2mortal(newSVpv(results.loc[i], 0)));
           }
        }
#else
        err = doFindVMs(hostHandle, timeout, &vmlist);
        if (VIX_OK == err) {
           l = vmlist;
           i = 0;
           XPUSHs(sv_2mortal(newSVnv(err)));
           i++;
           while (l) {
              XPUSHs(sv_2mortal(newSVpv(l->loc, 0)));
              t = l;
              l = l->next;
              free(t->loc);
              free(t);
              i++;
           }
        }
#endif

VixHandle 
VixHost_RegisterVM(hostHandle,vmxFilePathNameArg,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      hostHandle
        const char *                   vmxFilePathNameArg
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixHost_RegisterVM(hostHandle, vmxFilePathNameArg, NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle 
VixHost_UnregisterVM(hostHandle,vmxFilePathNameArg,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      hostHandle
        const char *                   vmxFilePathNameArg
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixHost_UnregisterVM(hostHandle, vmxFilePathNameArg, NULL, NULL);
   OUTPUT:
        RETVAL


#####################################################################
# VMware::Vix::API::VM
# 
# Per-VM functions
# - General functions (Open, Delete)
# - Power Ops (PowerOn, PowerOff, Reset, Suspend)
# - Snapshots
#####################################################################
MODULE = VMware::VixBinding PACKAGE = VMware::Vix::API::VM PREFIX = VixVM_
PROTOTYPES: ENABLE

##########################################################
# General Functions
###################
VixHandle 
VixVM_Open(hostHandle,vmxFilePathNameArg,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      hostHandle
        const char *                    vmxFilePathNameArg
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_Open(hostHandle, vmxFilePathNameArg, NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_Delete(vmHandle,deleteOptions,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                             deleteOptions
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_Delete(vmHandle, deleteOptions,
                              NULL, NULL);
   OUTPUT:
        RETVAL

##########################################################
# Power Ops
###########

VixHandle
VixVM_PowerOn(vmHandle,powerOnOptions,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                            powerOnOptions
        VixHandle                      propertyListHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_PowerOn(vmHandle, 
                                powerOnOptions, 
                                propertyListHandle, 
                                NULL, NULL);
     OUTPUT:
        RETVAL

VixHandle
VixVM_PowerOff(vmHandle,powerOffOptions,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        VixVMPowerOpOptions            powerOffOptions
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_PowerOff(vmHandle,
                                powerOffOptions,
                                NULL, NULL);
    OUTPUT:
        RETVAL

VixHandle
VixVM_Reset(vmHandle,powerOnOptions,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        VixVMPowerOpOptions            powerOnOptions
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_Reset(vmHandle, 
                              powerOnOptions,
                              NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_Suspend(vmHandle,powerOffOptions,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                       vmHandle
        VixVMPowerOpOptions             powerOffOptions
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_Suspend(vmHandle,
                                powerOffOptions,
                                NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_LoginInGuest(vmHandle,userName,password,options,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         userName
        char *                         password
        int                            options
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_LoginInGuest(vmHandle, userName,
                                         password, options,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_RunProgramInGuest(vmHandle,guestProgramName,commandLineArgs,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         guestProgramName
        char *                         commandLineArgs
        int                            options
        VixHandle                      propertyListHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_RunProgramInGuest(vmHandle, guestProgramName,
                                         commandLineArgs, options,
                                         propertyListHandle,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_CopyFileFromHostToGuest(vmHandle,hostPathName,guestPathName,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         hostPathName
        char *                         guestPathName
        int                            options
        VixHandle                      propertyListHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_CopyFileFromHostToGuest(vmHandle, hostPathName,
                                         guestPathName, options,
                                         propertyListHandle,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_CopyFileFromGuestToHost(vmHandle,guestPathName,hostPathName,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         guestPathName
        char *                         hostPathName
        int                            options
        VixHandle                      propertyListHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_CopyFileFromGuestToHost(vmHandle, guestPathName,
                                         hostPathName, options,
                                         propertyListHandle,
                                         NULL, NULL);
   OUTPUT:
        RETVAL


##########################################################
# Snapshot Functions
####################

VixError
VixVM_GetNumRootSnapshots(vmHandle,result)
        VixHandle                      vmHandle
        int                            &result
   CODE:
        RETVAL = VixVM_GetNumRootSnapshots(vmHandle, &result);
   OUTPUT:
        RETVAL
        result

VixError
VixVM_GetRootSnapshot(vmHandle,index,snapshotHandle)
        VixHandle                      vmHandle
        int                             index
        VixHandle                     &snapshotHandle
   CODE:
        RETVAL = VixVM_GetRootSnapshot(vmHandle, index,
                                           &snapshotHandle);
   OUTPUT:
        RETVAL
        snapshotHandle

VixHandle
VixVM_RemoveSnapshot(vmHandle,snapshotHandle,options,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        VixHandle                      snapshotHandle
        int                             options
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_RemoveSnapshot(vmHandle, snapshotHandle,
                                       options,
                                       NULL, NULL);
   OUTPUT:
        RETVAL
        
VixHandle
VixVM_RevertToSnapshot(vmHandle,snapshotHandle,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        VixHandle                      snapshotHandle
        int                             options
        VixHandle                      propertyListHandle
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_RevertToSnapshot(vmHandle, snapshotHandle,
                                         options, propertyListHandle,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_CreateSnapshot(vmHandle,name,description,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        const char *                    name
        const char *                    description
        int                             options
        VixHandle                      propertyListHandle
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_CreateSnapshot(vmHandle, name, description,
                                       options, propertyListHandle,
                                       NULL, NULL);
   OUTPUT:
        RETVAL

##########################################################
# General Functions
###################

VixHandle
VixVM_WaitForToolsInGuest(vmHandle,timeoutInSeconds,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                             timeoutInSeconds
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_WaitForToolsInGuest(vmHandle,
                                timeoutInSeconds,
                                NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle 
VixVM_UpgradeVirtualHardware(vmHandle,options,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                             options
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_UpgradeVirtualHardware(vmHandle, options,
                                      NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle 
VixVM_InstallTools(vmHandle,options,commandLineArgs,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                             options
        char *                          commandLineArgs
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_InstallTools(vmHandle, options, commandLineArgs,
                                     NULL, NULL);
   OUTPUT:
        RETVAL


MODULE = VMware::VixBinding PACKAGE = VMware::Vix::API::Job PREFIX = VixJob_
PROTOTYPES: ENABLE

VixError
VixJob_Wait(jobHandle, ...)
        VixHandle                       jobHandle
   PREINIT:
       int property;
       int i;
   CODE:
        if ((items % 2) != 0) {
           die("VixJob_Wait must have an even number of arguments.\n");
        }
        for (i = 2; i < items; i+= 2) {
           doProperty(jobHandle, ST(i - 1), ST(i), VixJob_Wait);
        }
        // should always be NULL
        property = SvIV(ST(items-1)); /* last property */
        if (property != VIX_PROPERTY_NONE)
            die("Final argument to Wait() should be VIX_PROPERTY_NONE\n");
        RETVAL = VixJob_Wait(jobHandle, property);
   OUTPUT:
        RETVAL

VixError 
VixJob_CheckCompletion(jobHandle,complete)
        VixHandle                      jobHandle
        Bool                           &complete
   OUTPUT:
        complete
        RETVAL

VixError
VixJob_GetError(jobHandle)
        VixHandle                      jobHandle



