#
# Copyright 2006.  VMware, Inc.  The use of this code is subject to 
# your agreement to the terms of an applicable VMware license.
#

package VMware::Vix::Simple;

# Simple Perl interface to Vix
# Currently exposes only those functions documented.
# Mission statement:
#	Hides all asynchronicity from caller.
#	Exposes only arguments that have documented values
#		(eg args that currently must be NULL are hidden)
#	Returns values as arrays whenever applicable, instead of
#		C-style reference

#use 5.00503;

# for debugging purposes
use lib qw( blib/lib blib/auto );

use strict;
use Carp;

use VMware::Vix::API::API;
use VMware::Vix::API::Job;
use VMware::Vix::API::VM;
use VMware::Vix::API::Host;

use VMware::Vix::API::Constants;

require Exporter;
require DynaLoader;
#use AutoLoader;

use vars qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);
@ISA = qw(Exporter DynaLoader);

# explcitly export things, to keep with the 'Simple' philosophy
@EXPORT = qw(
   GetErrorText
   ReleaseHandle
   GetHandleType

   GetProperties

   HostConnect
   HostDisconnect
   FindRunningVMs
   RegisterVM
   UnregisterVM

   VMOpen
   VMPowerOn
   VMPowerOff
   VMReset
   VMDelete
   VMSuspend
   VMWaitForToolsInGuest
   VMGetNumRootSnapshots
   VMGetRootSnapshot
   VMRemoveSnapshot
   VMRevertToSnapshot
   VMCreateSnapshot
   VMInstallTools
   VMUpgradeVirtualHardware
   VMLoginInGuest
   VMRunProgramInGuest
   VMCopyFileFromHostToGuest
   VMCopyFileFromGuestToHost
 );

$VERSION = '0.02';

# Preloaded methods go here.

=head1 NAME

VMware::Vix::Simple - Simple interface to VMware foundry

=head1 SYNOPSIS

    use VMWare::Vix::Simple;

=head1 DESCRIPTION

C<VMWare::Vix::Simple> provides a simple interface to the
VMware foundry library.  It hides the underlying asynchronous nature
will still providing the core functionality.

=head2 General Functions

=item B<HostConnect>

    ($err, $hostHandle) = HostConnect(VIX_API_VERSION, $hostType,
                                        $hostname, $hostport
                                        $username, $password,
                                        $options, $propertyListHandle);

Initializes the a remote host object; you must call this before calling 
any other Vix API. The host object will be used for all Vix
operations.  Returns an error code and the host handle. $hostType should
be VIX_SERVICEPROVIDER_VMWARE_SERVER. $options should be 0. 
$propertyListHandle should be VIX_INVALID_HANDLE.

=cut

# General functions

sub HostConnect($$$$$$$$) {
   my ($version, $hostType, $hostname, $hostport, $username, $password, $options, $propertyListHandle) = @_;
   my $err;
   my $hostHandle = VIX_INVALID_HANDLE;
   my $job = VIX_INVALID_HANDLE;

   $job = VMware::Vix::API::Host::Connect($version,
                     $hostType,
                     $hostname, $hostport,
                     $username, $password,
                     $options,
                     $propertyListHandle,
                     0, 0);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_JOB_RESULT_HANDLE,
                                      $hostHandle, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);
   return $err, $hostHandle;
};

=item B<HostDisconnect>

   HostDisconnect($hostHandle);

Shuts down a host handle returned by HostConnect.  The handle still needs
to be Released.

=cut

sub HostDisconnect($) {
   my ($host) = @_;

   VMware::Vix::API::Host::Disconnect($host);
}

=item B<FindRunningVMs>

    @vmlist = FindRunningVMs($hostHandle, $timeoutInSeconds);

Returns the the VMs running on the host associated with the hostHandle.
The first element in the list is an error code.

=cut


sub FindRunningVMs($$) {
   my ($hostHandle, $timeout) = @_;
   my @vms;

   return VMware::Vix::API::Host::FindItems($hostHandle, VIX_FIND_RUNNING_VMS,
                                          undef, $timeout, undef, undef);
}

=item B<RegisterVM>

    $err = RegisterVM($hostHandle, $vmxFilePath);

Register the VM with the host associated with the hostHandle.
VMs must be registered before they can be opened.

=cut


sub RegisterVM($$) {
   my ($hostHandle, $vmxFilePath) = @_;
   my $job;
   my $err;

   $job = VMware::Vix::API::Host::RegisterVM($hostHandle, $vmxFilePath,
                                             undef, undef);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);
   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
}

=item B<UnregisterVM>

    $err = UnregisterVM($hostHandle, $vmxFilePath);

Unregister the VM with the host associated with the hostHandle.
VMs must be registered before they can be opened.

=cut


sub UnregisterVM($$) {
   my ($hostHandle, $vmxFilePath) = @_;
   my $job;
   my $err;

   $job = VMware::Vix::API::Host::UnregisterVM($hostHandle, $vmxFilePath,
                                             undef, undef);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);
   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
}

=item B<ReleaseHandle>

    ReleaseHandle($handle);

Can be used on any handle when it is no longer needed.  Do not use a
handle once it has been Released.

=cut

sub ReleaseHandle($) {
   my ($handle) = @_;

   VMware::Vix::API::API::ReleaseHandle($handle);
}

=item B<GetHandleType>

    $type = GetHandleType($handle);

Returns the type of handle referred to by $handle.

=cut

sub GetHandleType($) {
   my ($handle) = @_;

   return VMware::Vix::API::API::GetHandleType($handle);
}

=item B<GetErrorText>

   $errstring = GetErrorText($err);

Returns the human-readable error string associated with an error code.

=cut

sub GetErrorText($) {
   my ($err) = @_;

   VMware::Vix::API::API::GetErrorText($err, "");
}

=item B<GetProperties>

   @properties = GetProperties($handle, prop1, prop2, ...);

Returns the values of the requested properties from the obejct
pointed to be $handle.  The first element in the list is an error code.

=cut

sub GetProperties {
   my $handle = shift @_;
   my $err;
   my $val = 0;
   my @retval;

   push @retval, VIX_OK;

   if (@_ < 1) {
      $retval[0] = VIX_E_FAIL;
      die ("GetProperties must have at least two arguments\n");
      return @retval;
   }
   foreach (@_) {
      my $property = $_;
      $err = VMware::Vix::API::API::GetProperties($handle, 
                                       $property, $val, VIX_PROPERTY_NONE);
      $retval[0] = $err;
      push @retval, $val;
      if ($err != VIX_OK) {
         last;
      }
   }
   return @retval;
}

# VM functions


=head2 VM Functions

=item B<VMOpen>

   ($err, $vmHandle) = VMOpen($hostHandle, $vmxPathName);

Opens the virtual machine in $vmxPathName on the host identified by 
$hostHandle.

=cut

sub VMOpen($$) {
   my ($host, $vmxPath) = @_;
   my $err;
   my $job;
   my $vm = VIX_INVALID_HANDLE;

   $job = VMware::Vix::API::VM::Open($host, $vmxPath, undef, 0);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_JOB_RESULT_HANDLE,
                                          $vm, VIX_PROPERTY_NONE);
   VMware::Vix::API::API::ReleaseHandle($job);

   return $err, $vm;
};

=item B<VMDelete>

   $err = VMDelete($vmHandle, $options);

Deletes the virtual machine $vmHandle returned by C<VMOpen>.
$options should be 0.

=cut

sub VMDelete($$) {
   my ($vm, $options) = @_;
   my $err;
   my $job;

   $job = VMware::Vix::API::VM::Delete($vm, $options, undef, 0);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
}

=item B<VMPowerOn>

   $err = VMPowerOn($vmHandle, $options, $propertyListHandle);

Powers on the VM. $options should be 0.
$propertyListHandle should be VIX_INVALID_HANDLE.

=cut

sub VMPowerOn($$$) {
   my ($vm, $options, $propertyListHandle) = @_;
   my $err;
   my $job;

# for now, make sure it starts the UI
   $job = VMware::Vix::API::VM::PowerOn($vm, $options | 0x0200,
                     $propertyListHandle, undef, 0);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
};

=item B<VMPowerOff>

   $err = VMPowerOff($vmHandle, $options);

Powers off the VM. $options should be 0.

=cut

sub VMPowerOff($$) {
   my ($vm, $options) = @_;
   my $err;
   my $job;

   $job = VMware::Vix::API::VM::PowerOff($vm, $options, undef, 0);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
};

=item B<VMReset>

   $err = VMReset($vmHandle, $options);

Restarts the VM. $options should be 0.

=cut

sub VMReset($$) {
   my ($vm, $options) = @_;
   my $err;
   my $job;

   $job = VMware::Vix::API::VM::Reset($vm, $options, undef, 0);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
};

=item B<VMSuspend>

   $err = VMSuspend($vmHandle, $options);

Suspends the VM. $options should be 0.

=cut

sub VMSuspend($$) {
   my ($vm, $options) = @_;
   my $err;
   my $job;

   $job = VMware::Vix::API::VM::Suspend($vm, $options, undef, 0);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
};

=item B<VMWaitForToolsInGuest>

   $err = VMWaitForToolsInGuest($vmHandle, $timeoutInSeconds);

This function will return when the VMware tools have successfully 
started in the guest or $timeoutInSeconds have elapsed.

=cut

sub VMWaitForToolsInGuest($$) {
   my ($vm, $timeoutInSeconds) = @_;
   my $err;
   my $job;

   $job = VMware::Vix::API::VM::WaitForToolsInGuest($vm, $timeoutInSeconds,
                                                     undef, 0);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
};

=item B<VMLoginInGuest>

   $err = VMLoginInGuest($vmHandle, $userName, $password, $options);

Validates a connection on the guest OS for further guest OS operations.
$options should be 0.

=cut

sub VMLoginInGuest($$$$) {
   my ($vm, $userName, $password, $options) = @_;
   my $err;
   my $job;

   $job = VMware::Vix::API::VM::LoginInGuest($vm, $userName, $password,
                                                     $options, undef, 0);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
};

=item B<VMRunProgramInGuest>

   $err = VMRunProgramInGuest($vmHandle, $guestProgramName, $commandLineArgs, $options, $propertyListHandle);

This will run $guestProgramName with $commandLineArgs.  If $options is
VIX_RUNPROGRAM_RETURN_IMMEDIATELY, it will not wait for the $guestProgramName
to complete before returning.  
$propertyListHandle should be VIX_INVALID_HANDLE.
LoginAsGuest() must be called first.

=cut

sub VMRunProgramInGuest($$$$$) {
   my ($vm, $guestProgramName, $commandLineArgs, $options, $propertyListHandle) = @_;
   my $err;
   my $job;

   $job = VMware::Vix::API::VM::RunProgramInGuest($vm, $guestProgramName,
                                                  $commandLineArgs,
                                                  $options, 
                                                  $propertyListHandle,
                                                  undef, 0);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
};

=item B<VMCopyFileFromHostToGuest>

   $err = VMCopyFileFromHostToGuest($vmHandle, $hostFilePath, $guestFilePath, $options, $propertyListHandle);

This will copy $hostFilePath $guestFilePath.  If $options is
VIX_RUNPROGRAM_RETURN_IMMEDIATELY, it will not wait for the copy
to complete before returning.  
$propertyListHandle should be VIX_INVALID_HANDLE.
LoginAsGuest() must be called first.

=cut

sub VMCopyFileFromHostToGuest($$$$$) {
   my ($vm, $hostFilePath, $guestFilePath, $options, $propertyListHandle) = @_;
   my $err;
   my $job;

   $job = VMware::Vix::API::VM::CopyFileFromHostToGuest($vm, $hostFilePath,
                                                  $guestFilePath,
                                                  $options,
                                                  $propertyListHandle,
                                                  undef, 0);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
};

=item B<VMCopyFileFromGuestToHost>

   $err = VMCopyFileFromGuestToHost($vmHandle, $guestFilePath, $hostFilePath, $options, $propertyListHandle);

This will copy $guestFilePath to $hostFilePath.  If $options is
VIX_RUNPROGRAM_RETURN_IMMEDIATELY, it will not wait for the copy
to complete before returning.  
$propertyListHandle should be VIX_INVALID_HANDLE.
LoginAsGuest() must be called first.

=cut

sub VMCopyFileFromGuestToHost($$$$$) {
   my ($vm, $guestFilePath, $hostFilePath, $options, $propertyListHandle) = @_;
   my $err;
   my $job;

   $job = VMware::Vix::API::VM::CopyFileFromGuestToHost($vm, $guestFilePath,
                                                  $hostFilePath,
                                                  $options, 
                                                  $propertyListHandle,
                                                  undef, 0);
   $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
};

=head2 VM Snapshot Functions

=item B<VMGetNumRootSnapshots>

   ($err, $numRootSnapshots) = VMGetNumRootSnapshots($vmHandle);

Returns the number of top-level snapshots in a VM.

=cut

sub VMGetNumRootSnapshots($) {
   my ($vm) = @_;
   my $num = 0;
   my $err = VMware::Vix::API::VM::GetNumRootSnapshots($vm, $num);

   return $err, $num;
}

=item B<VMGetRootSnapshot>

   ($err, $snapshotHandle) = VMGetRootSnapshot($vmHandle, $index);

Returns a handle to the $index snapshot (numbered from 0 to n-1).

=cut

sub VMGetRootSnapshot($$) {
   my ($vm, $index) = @_;
   my $snapshot = VIX_INVALID_HANDLE;
   my $err = VMware::Vix::API::VM::GetRootSnapshot($vm, $index, $snapshot);

   return $err, $snapshot;
}

=item B<VMRemoveSnapshot>

   $err = VMRemoveSnapshot($vmHandle, $snapshotHandle, $options);

Removes the snapshot from a VM.
$options should be 0.

=cut

sub VMRemoveSnapshot($$$) {
   my ($vm, $snapshot, $options) = @_;
   my $job = VMware::Vix::API::VM::RemoveSnapshot($vm, $snapshot, $options,
                                                     undef, 0);
   my $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
}

=item B<VMRevertToSnapshot>

   $err = VMRevertToSnapshot($vmHandle, $snapshotHandle, $options, $propertyListHandle);

Revert the state of the VM to the state when the snapshot was created.
$options should be 0.
$propertyListHandle should be VIX_INVALID_HANDLE.

=cut

sub VMRevertToSnapshot($$$$) {
   my ($vm, $snapshot, $options, $propertyListHandle) = @_;
   my $job = VMware::Vix::API::VM::RevertToSnapshot($vm, $snapshot, $options,
                                                $propertyListHandle,
                                                undef, 0);
   my $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
}

=item B<VMCreateSnapshot>

   ($err, $snapshotHandle) = VMCreateSnapshot($vmHandle, $name, $description, $options, $propertyListHandle);

Creates a new snapshot of the current VM state and stores it as $name
with description $description.
$options should be 0.
$propertyListHandle should be VIX_INVALID_HANDLE.
Returns an error code and the handle to the newly created snapshot.

=cut

sub VMCreateSnapshot($$$$$) {
   my ($vm, $name, $description, $options, $propertyListHandle) = @_;
   my $snapshot = VIX_INVALID_HANDLE;
   my $job = VMware::Vix::API::VM::CreateSnapshot($vm, $name, $description,
                                          $options, $propertyListHandle,
                                          undef, undef);
   my $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_JOB_RESULT_HANDLE,
                                          $snapshot, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err, $snapshot;
}

=item B<VMUpgradeVirtualHardware>

   $err = VMUpgradeVirtualHardware($vmHandle, $options);

Upgrade the virtual hadware of the VM.  The VM must be powered off to
perform this operation.
$options should be 0.

=cut

sub VMUpgradeVirtualHardware($$) {
   my ($vm, $options) = @_;
   my $job = VMware::Vix::API::VM::UpgradeVirtualHardware($vm, $options,
                                                          undef, undef);
   my $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
}

=item B<VMInstallTools>

   $err = VMInstallTools($vmHandle, $options, $commandLineArgs);

Installs the latest version of the tools onto the VM.
The VM must be powered on.
$options should be 0.
$commandLineArgs should be undef.

=cut

sub VMInstallTools($$$) {
   my ($vm, $options, $commandLineArgs) = @_;
   my $job = VMware::Vix::API::VM::InstallTools($vm, $options, 
                                                $commandLineArgs,
                                                undef, undef);
   my $err = VMware::Vix::API::Job::Wait($job, VIX_PROPERTY_NONE);

   VMware::Vix::API::API::ReleaseHandle($job);

   return $err;
}

# Autoload methods go after =cut, and are processed by the autosplit program.

1;
__END__

