# ---------------------------------------------------------------------------
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
# ---------------------------------------------------------------------------
package VMware::Management::Util;
# File: VMware/Management/Util.pm

use strict;

use MIME::Base64 ();
use URI::Escape ();
use VMware::DOMAccess ();

require Exporter;

use vars qw( @ISA @EXPORT );
@ISA = qw( Exporter );
@EXPORT = qw( trace
              return_string
              connect_ctrl_obj
              file_to_string
              string_to_file
              append_error
              hash_cookies
              hash_credentials
              append_vm
              append_newMui
              append_host
              append_agent
              append_user
              append_product
              append_build
              append_connected_users
              get_build
              get_host_and_domain
              get_cgi_hostname
	      error_page
	      use_vmdb
	      validate_session_non_vmdb
	      session_is_valid_non_vmdb
	      getProductName
              getDiskPath
	      jsStringLiteral
	      escapeMarkup
	    );

my $debug = $ENV{ "MOD_PERL" } ? $ENV{ "vmware_DEBUG_LEVEL" } : 4;

# %file is a hash whose keys are the filenames request by fileToString calls.
# If the scalar $file{ <file name> } is defined, then we do not attempt to re-
# read the file.
my %file;
# We cache the hostname and domain name for speed since these operations can be
# expensive (especially in windows)
my $gHost;
my $gDomain;

# Lists of common, useful values for virtual machine and host statistics
my @periods = ( 60, 300, 900 );
my @methods = ( "min", "avg", "max", "sum" );
my %methods = ( $methods[ 0 ] => "min",
                $methods[ 1 ] => "mean",
                $methods[ 2 ] => "max",
                $methods[ 3 ] => "sum" );

sub trace {
  my ($msg, $type, $fh, $package, $filename, $line) = @_;

  #file tmp change
  if ($^O ne "MSWin32") {
     if(! isTraceable()) {
        return 0;
     }
  } 

  #file tmp change
  if ($^O eq "MSWin32") {
     my $logname = $ENV{vmware_LOGFILE};
     open(FILE, ">> $logname");
  }

  if (!defined($fh)) {
     #file tmp change
     if ($^O eq "MSWin32") {
        $fh = *FILE;
     } else {
        $fh = *STDERR;
     }
  }

  my $level = $ENV{ "vmware_DEBUG_LEVEL" };
  if (! defined($package) || ! defined($filename) || ! defined($line)) {
      ($package, $filename, $line) = caller();
  }

  my ( $sec, $min, $hour, $mday, $mon, $year, $wday, $yday) = localtime( time() );
  my @days = ( "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" );
  my @months = ( "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" );
  my $localtime = $days[ $wday ]." ".
    $months[ $mon ]." ".
    ($mday < 10 ? "0$mday" : $mday)." ".
    ($hour < 10 ? "0$hour" : $hour).":".
    ($min < 10 ? "0$min" : $min).":".
    ($sec < 10 ? "0$sec" : $sec)." ".
    ($year + 1900);

  my $print = 0;

  if ( $type eq "error" ) {
    $print = 1;
  } elsif ( $debug > 0 && $type eq "warning" ) {
    $print = 1;
  } elsif ( $debug > 1 && $type eq "notice" ) {
    $print = 1;
  } elsif ( $debug > 2 ) {
    $print = 1;
  }

  if ( $print ) {
    print $fh
      "[".$localtime."] [vmware".( $type ? " $type" : "" )."] ".
      "$package line $line: $msg\n";
  }

  #file tmp change
  if ($^O eq "MSWin32") {
     close FILE;
  }
}

sub return_string {
  my $s;

  while (defined($_ = shift)) {
    if (ref($_)) {
      $s = "( ";
      if (ref($_) eq "ARRAY") {
        $s .= return_string(@{$_});
      } elsif (ref($_) eq "HASH") {
        $s .= return_string(%{$_});
      } else {
        $s .= return_string(${$_});
      }
      $s .= " ) ";
    } else {
      $s .= $_ . ":";
    }
  }
  return $s;
}

# connect_ctrl_obj ----------------------------------------------------------
#
# $ctrl_obj ... A control object such as Control::Server or Control::VM
#        $e ... The error
#        $s ... The error message
#
# Connect to the supplied control object. If there are problems, return false,
# the error and the error string. Otherwise return true.
#
# Example:
#
#   my $server = VMware::Control::Server::new( undef, 0, undef, undef );
#   my ( $server_ok, $err, $errstr ) = connect_ctrl_obj( $server );
#   unless( $server_ok ) {
#     append_error( $root, $err, $errstr );
#   }
#
# ---------------------------------------------------------------------------
sub connect_ctrl_obj {
  my $ctrl_obj = shift( );
  unless ( $ctrl_obj && $ctrl_obj->connect( ) ) {
    my ( $e, $s ) = $ctrl_obj->get_last_error( );
    trace( "$e: $s", "error" );
    return ( 0, $e, $s );
  }
  trace( "Connected to control object ($ctrl_obj).", "notice" );
  return ( 1, undef, undef );
}

# file_to_string ------------------------------------------------------------
#
#       $f ... The path to the file
#    $safe ... Errors are non-fatal, so continue.
# $nocache ... Don't cache this file.
#
# Given a file name, return a string of its contents. Cache file contents to
# prevent unecessary re-reads.
#
# ---------------------------------------------------------------------------
sub file_to_string {
  my ( $f, $safe, $nocache ) = @_;
  my $tmp = "";
  my $line = "";
  unless ( $file{ $f } ) {
    if ( $safe ) {
      open(FILE, "<$f") or trace("Can't open $f: $!\n", "warning");
    } else {
      open( FILE, "<$f" ) or die "Can't open $f: $!\n";
    }

    if ( $nocache ) {
      while ( $line = <FILE> ) {
        $tmp .= $line;
      }
    } elsif (!$file{ $f }) {
      while ( $line = <FILE> ) {
        $file{ $f } .= $line;
      }
    }
    close FILE;
  }
  return ($tmp ? $tmp : $file{ $f });
}

# string_to_file ------------------------------------------------------------
#
#    $s ... The string to write to the file
#    $f ... The path to the file
# $safe ... Errors are non-fatal, so continue.
#
# Given a string and a file name, write the string to the file.
#
# ---------------------------------------------------------------------------
sub string_to_file {
  my ( $s, $f, $safe ) = @_;
  if ( $safe ) {
    open(FILE, ">$f") or trace("Can't open $f: $!\n", "warning");
  } else {
    open( FILE, ">$f" ) or die "Can't open $f: $!\n";
  }
  print FILE $s;
  close FILE;
}

# append_error --------------------------------------------------------------
#
#  $doc ... XML::DOM document object
# $root ... XML::DOM node to which the error should be appended
#    $e ... Error code
#    $s ... Error message
#
# Append a standard error node to the node specified by $root.
#
# ---------------------------------------------------------------------------
sub append_error {
  my ( $doc, $root, $e, $s ) = @_;
  $root = $root->appendChild( $doc->createElement( "error" ) );
  $root->setAttribute( "code", $e ) if $e;
  $root->appendChild( $doc->createTextNode( $s ) );
}


# hash_cookies -------------------------------------------------------------
#
# Return HTTP_COOKIE environment variable as Perl hash.
#
# ---------------------------------------------------------------------------
sub hash_cookies {
  return map ( split( /=/, $_, 2 ), split( /; /, $ENV{'HTTP_COOKIE'} ) );
}

# hash_credentials ----------------------------------------------------------
#
# Return 'vmware.mui.kid' cookie as Perl hash with the keys:
#
#   l ... Base64-encoded username (login)
#   m ... Base64-encoded password
#
# ---------------------------------------------------------------------------
sub hash_credentials {
   if (use_vmdb()) {
      require VMware::Management::Session;

      return ("l" => VMware::Management::Session::get_username(),
	      "m" => VMware::Management::Session::get_password());
   }
   
   my %cookies = hash_cookies( );
   my @kid = split(":", MIME::Base64::decode_base64($cookies{"vmware.mui.kid"}));
   
   return ("l" => MIME::Base64::decode_base64($kid[ 0 ]),
	   "m" => MIME::Base64::decode_base64($kid[ 1 ]));
}

# catch_error ---------------------------------------------------------------
# catch_error expects a defined value for $_[0], otherwise it appends an error
# node to the current parent node.
# ---------------------------------------------------------------------------
sub catch_error {
  my ( $ret, $doc, $root, $vm ) = @_;
  unless( $ret ) {
    my ( $err, $errstr ) = $vm->get_last_error( );
    append_error( $doc, $root, $err, $errstr );
    return 1;
  }
  return undef;
}

# append_vm -----------------------------------------------------------------
#
#    $doc ... XML::DOM document object
#   $root ... XML::DOM node to which the VM should be appended
#    $cfg ... /path/to/vm/.cfg file
# $server ... VMware::Control::Server object
#
# Given a document object and a parent node, append an XML description of the
# VM $vm to the parent node.
#
# <vm cfg="_">
#   <displayName>
#     ---
#   </displayName>
#   <os>
#     ---
#   </os>
#   <state>
#     ---
#   </state>
# </vm>
#
# ---------------------------------------------------------------------------
sub append_vm {
  my $doc = shift( );
  my $root = shift( );
  my $cfg = shift( );
  my $server = shift( );
  my @params = @_;

  my ( $vm, $vm_ok, $err, $errstr, $ret );
  $vm = VMware::Control::VM::new( $server, $cfg );
  $root = $root->appendChild( $doc->createElement( "vm" ) );
  $root->setAttribute( "cfg", $cfg );

  ( $vm_ok, $err, $errstr ) = connect_ctrl_obj( $vm );
  unless ( $vm_ok ) {
    append_error ( $doc, $root, $err, $errstr );
  }

  if ($vm_ok) {
    my $pid = $vm->get("Status.pid");
    if (defined($pid)) {
      $root->setAttribute("pid", $pid);
    }
  }

  my $state = -1;
  if ($vm_ok) {
    $state = $vm->get( "Status.power" );
    if ( $state eq "off" ) {
      $state = 0;
    } elsif ( $state eq "on" ) {
      $state = 1;
    } elsif ( $state eq "suspended" ) {
      $state = 2;
    } elsif ( $state eq "stuck" ) {
      $state = 4;
      append_vm_question( $doc, $root, $vm );
    } else {
      ( $err, $errstr ) = $vm->get_last_error( );
      trace( "$err: $errstr", "error" );
      append_error( $doc, $root, $err, $errstr );
    }
  }

  my $display_name = $vm_ok ? $vm->get( "Config.displayName" ) : $cfg;
  my $node = $root->appendChild( $doc->createElement( "displayName" ) );
  $node->appendChild( $doc->createTextNode( $display_name or $cfg ) );

  # Append a file name that can be used in URLs by escaping everything but
  # '*', '@', '-', '_', '.', '/' and alphanumeric characters. We want to
  # duplicate JavaScript's escape function, but also encode '+'.
  $node = $root->appendChild( $doc->createElement( "safe_cfg" ) );
  $node->appendChild($doc->createTextNode(URI::Escape::uri_escape($cfg, "^a-zA-Z0-9\\*\\@\\-_\\.\\/")));

  $node = $root->appendChild( $doc->createElement( "os" ) );
  if ( $vm_ok ) {
    catch_error( ($ret = $vm->get( "Config.guestOS" )), $doc, $root, $vm );
    $node->appendChild( $doc->createTextNode( $ret or "--" ) );
  } else {
    $node->appendChild( $doc->createTextNode( "Unreachable" ) );
  }

  $node = $root->appendChild( $doc->createElement( "ip" ) );
  if ( $vm_ok ) {
    $ret = $vm->get( "Config.guestinfo.ip" );
    $node->appendChild( $doc->createTextNode( $ret or "Unknown" ) );
  } else {
    $node->appendChild( $doc->createTextNode( "Unreachable" ) );
  }

  $node = $root->appendChild( $doc->createElement( "state" ) );
  $node->appendChild( $doc->createTextNode( $state ) );

  if ( $vm_ok ) {
    my $resumed_session = $vm->get("Status.resume.repeatable.resumed_session");
    return undef unless !catch_error($resumed_session, $doc, $root, $vm);
    # XXX To avoid confusing TRUE w/true, etc.
    if ($resumed_session =~ /^(true|false)$/i) {
      $resumed_session =~ tr/A-Z/a-z/;
    }
    $node->setAttribute("resumed", $resumed_session);

    $ret = $vm->get("Config.resume.repeatable");
    if (! defined($ret)) {
      $ret = '';
    }
    # XXX To avoid confusing TRUE w/true, etc.
    if ($ret =~ /^(true|false)$/i) {
      $ret =~ tr/A-Z/a-z/;
    }
    $node = $root->appendChild($doc->createElement("resume.repeatable"));
    $node->appendChild($doc->createTextNode($ret));

    return undef unless
      !catch_error( append_guest_state( $doc, $root, $vm ), $doc, $root, $vm );
    return undef unless
      !catch_error( append_capabilities( $doc, $root, $vm ), $doc, $root, $vm );
  }

  return undef unless
    !catch_error( append_event_log( $doc, $root, $server, $cfg ), $doc, $root, $vm );

  return undef unless
    !catch_error( append_connected_users( $doc, $root, $server, $cfg ), $doc, $root, $vm );

  if ($vm_ok) {
    if (catch_error( append_devices( $doc, $root, $vm ), $doc, $root, $vm )) {
      return undef;
    }
  }

  # Accept any remaining args as specifying arbitrary configuration parameters
  # that should be included in the output document.
  for my $param (@params) {
    # XXX No good way to determine the default value.
    $ret = $vm->get("Config.$param");
    if (! defined($ret)) {
      $ret = '';
    }
    # XXX To avoid confusing TRUE w/true, etc.
    if ($ret =~ /^(true|false)$/i) {
      trace("> toLower: $ret", "debug");
      $ret =~ tr/A-Z/a-z/;
      trace("< toLower: $ret", "debug");
    }
    $node = $root->appendChild($doc->createElement("$param"));
    $node->appendChild($doc->createTextNode($ret));
  }

#   trace( "STATE -> $state", "debug" );
  if ( $state == 1 ) {
    my $productName = getProductName($server);

    if ($productName eq "ESX") {
      return undef unless
        !catch_error( append_esx_stats( $doc, $root, $server, $cfg ), $doc, $root, $vm );
    } elsif ($productName eq "GSX") {
      return undef unless
        !catch_error( append_cpu( $doc, $root, $vm ), $doc, $root, $vm );
      return undef unless
        !catch_error( append_ram( $doc, $root, $vm ), $doc, $root, $vm );
      return undef unless
        !catch_error( append_heartbeat( $doc, $root, $vm ), $doc, $root, $vm );
      return undef unless
        !catch_error( append_uptime( $doc, $root, $vm ), $doc, $root, $vm );
    }
  }

  $vm->disconnect( ) if $vm_ok;
}

sub append_guest_state {
  my ( $doc, $root, $vm ) = @_;
  my $state = $vm->get( "Status.guest.state" );
  return undef unless defined( $state );
  my $seen = $vm->get( "Status.guest.tools.seen" );
  return undef unless defined( $seen );
  my $capable = $vm->get( "Status.guest.tools.softPowerCapable" );
  return undef unless defined( $capable );
  my $pending = $vm->get( "Status.guest.tools.softPowerPending" );
  return undef unless defined( $pending );

  $root = $root->appendChild( $doc->createElement( "guest" ) );
  my $node = $root->appendChild( $doc->createElement( "state" ) );
  $node->appendChild( $doc->createTextNode( $state ) );

  $root = $root->appendChild( $doc->createElement( "tools" ) );
  $node = $root->appendChild( $doc->createElement( "seen" ) );
  $node->appendChild( $doc->createTextNode( $seen ) );

  $node = $root->appendChild( $doc->createElement( "softPowerCapable" ) );
  $node->appendChild( $doc->createTextNode( $capable ) );

  $node = $root->appendChild( $doc->createElement( "softPowerPending" ) );
  $node->appendChild( $doc->createTextNode( $pending ) );

  return 1;
}

sub append_devices {
  my ( $doc, $root, $vm ) = @_;
  my $devices_ref = $vm->get( "Status.devices" );
  return undef unless defined($devices_ref);
  my @devices = @{ $devices_ref };
  my @params = qw( deviceType
                   fileName
                   name
		   mode
                   startConnected
                   devName
                   connectionType
                   virtualDev
                   sharedBus
                   vnet
                   fileType );
  my $node;
  my $param;

  # Special-case RAM and VCPUs as devices...
  my $numvcpus = $vm->get("Config.numvcpus");
  if ($numvcpus && ($numvcpus > 1)) {
     $node = $root->appendChild( $doc->createElement( "device" ) );
     $node->setAttribute( "id", "CPU" );
     $param = $node->appendChild( $doc->createElement( "param" ) );
     $param->setAttribute( "name", "numVCPUs" );
     $param->appendChild( $doc->createTextNode( $numvcpus ) );
  }

  $node = $root->appendChild( $doc->createElement( "device" ) );
  $node->setAttribute( "id", "RAM" );
  $param = $node->appendChild( $doc->createElement( "param" ) );
  $param->setAttribute( "name", "memSize" );
  $param->appendChild( $doc->createTextNode( $vm->get("Config.memSize") ) );

  my $vm_is_nonpersistent = 1;

  foreach my $d ( @devices ) {
    my $deviceType = undef;
    my $deviceMode = undef;
    unless ( $d eq "vm" || $d eq "system" ) {
      my $node = $root->appendChild( $doc->createElement( "device" ) );
      $node->setAttribute( "id", $d );
      foreach my $p ( @params ) {
        # Special-case CD-ROMs so people don't hurt themseleves with logical
        # contradictions when they see a read-only device in persistent mode.
        if ($deviceType && $deviceType =~ /cdrom/ && $p eq "mode") {
          next;
        }
        my $ret = $vm->get( "Config.$d.$p" );
        unless ( $ret ) {
          next;
        }
        if ($p eq "mode") {
          $deviceMode = $ret;
        }
        if ($p eq "deviceType") {
          $deviceType = $ret;
        }
        my $param = $node->appendChild( $doc->createElement( "param" ) );
        $param->setAttribute( "name", $p );
        $param->appendChild( $doc->createTextNode( $ret ) );
      }
    }
    if (($d =~ /(scsi|ide)\d+:\d+/i) && $deviceType && ($deviceType !~ /cdrom/i)) {
      $vm_is_nonpersistent &&= ($deviceMode =~ /^nonpersistent$/i);
    }
  }

  if ($vm_is_nonpersistent) {
    $node = $root->appendChild($doc->createElement("nonpersistent"));
  }

  return 1;
}

# append_samples ------------------------------------------------------------
#
# Generic routine for appending sample nodes to stat nodes.
#
# <parent>
#   <sample period="_" units="_">---</sample>
#   ---
# </parent>
# ---------------------------------------------------------------------------
sub append_samples {
  my ( $doc, $root, $method, $stat, $info ) = @_;

  foreach my $i ( @periods ) {
    my $interval = $$info{ "interval" } ? $i / $$info{ "interval" } : 0;

    my $node = $root->appendChild( $doc->createElement( "sample" ) );
    my %usage = %{ $$stat{ $methods{ $method } } };

    $node->setAttribute( "interval",
                         int( ( $interval * $$info{ "interval" } ) / 60 ) );
    $node->setAttribute( "units", "minutes" );
    $node->appendChild( $doc->createTextNode( $usage{ $interval } eq "-" or
      $usage{ $interval } eq "" ? "0" : $usage{ $interval } ) );
  }
}

# append_cpu ----------------------------------------------------------------
#
#    $doc ... XML::DOM document object
#   $root ... XML::DOM node to which the CPU stats should be appended
#    $obj ... vm or host object for which stats should be collected
#
# Given a VMware Control object and the requisite DOM objects, append CPU
# usage nodes to the supplied parent node.
#
# <stat name="cpu" units="_">
#   <method>
#     <sample period="_" units="_">
#       ---
#     </sample>
#     ...
#   </method>
#   ...
# </stat>
#
# ---------------------------------------------------------------------------
sub append_cpu {
#   trace("APPEND_CPU", "debug");
  my ( $doc, $root, $obj, $opt ) = @_;
  my ( $stat, $stat_obj, $stat_obj_ok, $err, $errstr );

  # Factor $stat for the correct object
  if ( ref( $obj ) eq "VMware::Control::Server" ) {
    $stat = "Status.stats.System.CPUUsage";
    $stat_obj = VMware::Control::VM::new( $obj, ($obj->enumerate( ))[ 0 ] );
    ( $stat_obj_ok, $err, $errstr ) = connect_ctrl_obj( $stat_obj );
    unless ( $stat_obj_ok ) {
      return undef;
    }
  } elsif ( defined( $opt ) && $opt eq "Server") {
    trace( "Fire when ready.", "debug" );
    $stat = "Status.stats.System.CPUUsage";
    $stat_obj = $obj;
  } else {
    $stat = "Status.stats.VM.CPUUsage";
    $stat_obj = $obj;
  }

#   trace("STAT -> $stat", "debug");

  my $stat_info = "$stat.info";
#   trace("GET STAT INFO -> $stat.info", "debug");
  $stat_info = $stat_obj->get( $stat_info );
  return undef unless defined($stat_info);
  my %info = %{ $stat_info };

  my @intervals;
  foreach my $interval ( @periods ) {
    $intervals[ scalar( @intervals ) ] = $interval / $info{ "interval" };
  }

  $stat = $stat_obj->get( $stat, @intervals );
#   trace("GET STAT -> $stat", "debug");
  return undef unless defined( $stat );
  my %stat = %{ $stat };

  $root = $root->appendChild( $doc->createElement( "stat" ) );
  $root->setAttribute( "name", "cpu" );
  $root->setAttribute( "units", $info{ "units" } );

  foreach my $method ( @methods ) {
    # Don't emit the sum for CPU usage. It doesn't make sense.
    if ($method eq "sum") {
      next;
    }
    # Grab the current min, avg or max CPU.
    my $cpu = \%{ $stat{ $methods{ $method } } };
    # Normalize CPU where 499 stands for 49.9%.
    foreach my $period ( @periods ) {
      unless ( !$$cpu{ $period } ) {
        $$cpu{ $period } = int( ( $$cpu{ $period } / 10 ) );
        if ( $$cpu{ $period } > 100 ) {
          $$cpu{ $period } = 100;
        }
      }

#       trace("CPU for $period -> $$cpu{ $period }", "debug");
    }

    my $node = $root->appendChild( $doc->createElement( "usage" ) );
    $node->setAttribute( "type", $method );

    append_samples( $doc, $node, $method, \%stat, \%info );
  }
  return 1;
}

# append_ram ----------------------------------------------------------------
#
#    $doc ... XML::DOM document object
#   $root ... XML::DOM node to which the RAM stats should be appended
#    $obj ... vm or host object for which stats should be collected
#
# Given a VMware Control object and the requisite DOM objects, append RAM
# usage nodes to the supplied parent node.
#
# <stat name="ram" units="_">
#   <method>
#     <sample period="_" units="_">
#       ---
#     </sample>
#     ...
#   </method>
#   ...
# </stat>
#
# ---------------------------------------------------------------------------
sub append_ram {
  my ( $doc, $root, $obj, $opt ) = @_;
  my ( $stat, $stat_obj, $stat_obj_ok, $err, $errstr );

  # Factor $stat for the correct object
  if ( ref( $obj ) eq "VMware::Control::Server" ) {
    $stat = "Status.stats.System.RAM";
    $stat_obj = VMware::Control::VM::new( $obj, ($obj->enumerate( ))[ 0 ] );
    ( $stat_obj_ok, $err, $errstr ) = connect_ctrl_obj( $stat_obj );
    unless ( $stat_obj_ok ) {
      return undef;
    }
  } elsif ( defined($opt) && $opt eq "Server") {
    $stat = "Status.stats.System.RAM";
    $stat_obj = $obj;
  } else {
    $stat = "Status.stats.VM.RAM";
    $stat_obj = $obj;
  }

  my $stat_info = "$stat.info";
  $stat_info = $stat_obj->get( $stat_info );
  return undef unless defined($stat_info);
  my %info = %{ $stat_info };

  my @intervals;
  foreach my $interval ( @periods ) {
    $intervals[ scalar( @intervals ) ] = $interval / $info{ "interval" };
  }

  $stat = $stat_obj->get( $stat, @intervals );
  return undef unless defined( $stat );
  my %stat = %{ $stat };

  $root = $root->appendChild( $doc->createElement( "stat" ) );
  $root->setAttribute( "name", "ram" );
  $root->setAttribute( "units", "percent" );

  foreach my $method ( @methods ) {
    # Don't emit the sum for RAM usage. It doesn't make sense.
    if ($method eq "sum") {
      next;
    }
    # Grab the current min, avg or max RAM.
    my $ram = \%{ $stat{ $methods{ $method } } };
    # Normalize RAM as a percentage of the limit.
    foreach my $period ( @intervals ) {
      unless ( !$$ram{ $period } ) {
        $$ram{ $period } = int( ( $$ram{ $period } / $info{ "limit" }  ) * 100);
        if ( $$ram{ $period } > 100 ) {
          $$ram{ $period } = 100;
        }
      }
    }

    my $node = $root->appendChild( $doc->createElement( "usage" ) );
    $node->setAttribute( "type", $method );

    append_samples( $doc, $node, $method, \%stat, \%info );
  }

  return 1;
}

sub append_esx_stats {
  trace( "append_esx_stats", "debug" );
  my ( $doc, $root, $server, $cfg ) = @_;
  my ( $r, $n, $p, $di, $do );
  my ( $df, $stat, @stats );

  $p = new XML::DOM::Parser;
  $di = $p->parse( "<exec></exec>" );
  $r = $di->getDocumentElement( );
  $n = $r->appendChild( $di->createElement( "op" ) );
  $n->appendChild( $di->createTextNode( "Stats_GetVM" ) );

  $r = $r->appendChild( $di->createElement( "in" ) );
  $n = $r->appendChild( $di->createElement( "cfg" ) );
  $n->appendChild( $di->createTextNode( $cfg ) );

  $do = $server->exec( $di->toString( ) );
  $di->dispose( );

  return undef unless defined( $do );
  $do = VMware::DOMAccess->newXML( $do );

  return undef unless defined( $do );
  my $rc = $do->getValue( "exec.returncode" );
  if ( $rc == 1 && $do->getValue( "exec.out" ) ) {
    $df = $do->getDocument( )->createDocumentFragment( );

    @stats = $do->get( "exec.out" )->getElementsByTagName( "stat" );
    foreach $stat ( @stats ) {
      $n = $df->appendChild( $stat );
    }

    @stats = $do->get( "exec.out" )->getElementsByTagName( "uptime" );
    foreach $stat ( @stats ) {
      $n = $df->appendChild( $stat );
    }

    $df->setOwnerDocument( $doc );
    $root->appendChild( $df );
    $df->dispose( );

  } else {
    trace( "exec.returncode is $rc", "error" );
    trace( $do->getDocument( )->toString( ), "error" );
  }
  $do->dispose( );
  return 1;
}

sub append_connected_users {
  trace( "append_connected_users", "debug" );
  my ( $doc, $root, $server, $cfg ) = @_;
  my ( $r, $n, $p, $di, $do );

  $p = new XML::DOM::Parser;
  $di = $p->parse( "<exec></exec>" );
  $r = $di->getDocumentElement( );
  $n = $r->appendChild( $di->createElement( "op" ) );
  $n->appendChild( $di->createTextNode( "VM_Who" ) );

  $r = $r->appendChild( $di->createElement( "in" ) );
  $n = $r->appendChild( $di->createElement( "cfg" ) );
  $n->appendChild( $di->createTextNode( $cfg ) );

  $do = $server->exec( $di->toString( ) );
  $di->dispose( );

  return undef unless defined( $do );
  $do = VMware::DOMAccess->newXML( $do );

  return undef unless defined( $do );
  my $rc = $do->getValue( "exec.returncode" );
  if ( $rc == 1 && $do->getValue( "exec.out" ) ) {
    my @connections = $do->get( "exec.out" )->getElementsByTagName( "connection" );
    my $df = $do->getDocument( )->createDocumentFragment( );
    foreach my $connection ( @connections ) {
      $n = $df->appendChild( $connection );
      my @nodes = $n->getElementsByTagName( "connect" );
      foreach $n ( @nodes ) {
        $n->getFirstChild( )->setNodeValue(
          scalar( localtime( $n->getFirstChild( )->getNodeValue( ) ) ) );
      }
      @nodes = $n->getElementsByTagName( "idle" );
      foreach $n ( @nodes ) {
        $n->getFirstChild( )->setNodeValue(
          scalar( localtime( $n->getFirstChild( )->getNodeValue( ) ) ) );
      }
    }
    $df->setOwnerDocument( $doc );
    $root->appendChild( $df );
    $df->dispose( );
  } else {
    trace( "exec.returncode is $rc", "error" );
    trace( $do->getDocument( )->toString( ), "error" );
  }
  $do->dispose( );
  return 1;
}

sub append_event_log {
  trace( "append_event_log", "debug" );
  my ( $doc, $root, $server, $cfg ) = @_;
  my ( $r, $n, $p, $di, $do );

  $p = new XML::DOM::Parser;
  $di = $p->parse( "<exec></exec>" );
  $r = $di->getDocumentElement( );
  $n = $r->appendChild( $di->createElement( "op" ) );
  $n->appendChild( $di->createTextNode( "VM_GetLastNEvents" ) );

  $r = $r->appendChild( $di->createElement( "in" ) );
  $n = $r->appendChild( $di->createElement( "cfg" ) );
  $n->appendChild( $di->createTextNode( $cfg ) );

  $n = $r->appendChild( $di->createElement( "lastn" ) );
  $n->appendChild( $di->createTextNode( $ENV{ "vmware_EVENTLOG_LASTN" } or "10" ) );

  $do = $server->exec( $di->toString( ) );
  $di->dispose( );

  return undef unless defined( $do );
  $do = VMware::DOMAccess->newXML( $do );

  return undef unless defined( $do );
  my $rc = $do->getValue( "exec.returncode" );
  if ( $rc == 1 && $do->getValue( "exec.out" ) ) {
    my @events = $do->get( "exec.out.vm" )->getElementsByTagName( "event" );
    my $df = $do->getDocument( )->createDocumentFragment( );
    my @reorder;
    my $i = 0;

    foreach my $event ( @events ) {
      $i++;
      $reorder[ scalar( @events ) - $i ] = $event;
    }
    foreach my $event ( @reorder ) {
      $n = $df->appendChild( $event );
      my @nodes = $n->getElementsByTagName( "time" );
      foreach $n ( @nodes ) {
        $n->getFirstChild( )->setNodeValue(
          scalar( localtime( $n->getFirstChild( )->getNodeValue( ) ) ) );
      }
    }
    $df->setOwnerDocument( $doc );
    $root->appendChild( $df );
    $df->dispose( );
  } else {
    trace( "exec.returncode is $rc", "error" );
    trace( $do->getDocument( )->toString( ), "error" );
  }
  $do->dispose( );

  return 1;
}

sub get_host_and_domain {
  my ($host, $domain);

  if (defined($gHost) && defined($gDomain)) {
      return ($gHost, $gDomain);
  }

  if ( $^O eq "MSWin32" ) {
    my $name = `hostname`;
    $host = $name;
    $domain = "";

    if ( $name =~ /\./ ) {
      ( $host = $name ) =~ s/([^.]*)\..*/$1/ ;
      ( $domain = $name ) =~ s/[^.]*\.(.*)/$1/ ;
    }

    if ( ! $domain ) {
      my ( $name, $aliases, @rest ) = gethostbyname( "localhost" );
      if ( $name =~ /\./ ) {
        ( $host = $name ) =~ s/([^.]*)\..*/$1/ ;
        ( $domain = $name ) =~ s/[^.]*\.(.*)/$1/ ;
      }

      if ( ! $domain ) {
        my @names = split( /\s+/, $aliases );
        my $alias;
        foreach $alias ( @names ) {
          if ( $alias =~ /\./ ) {
            ( $host = $alias ) =~ s/([^.]*)\..*/$1/ ;
            ( $domain = $alias ) =~ s/[^.]*\.(.*)/$1/ ;
            last;
          }
        }
      }
    }

    if ( ! $domain ) {
      my @ipinfo = `ipconfig`;

      my $entry;
      foreach $entry ( @ipinfo ) {
        if ( $entry =~ / : / ) {
          my ( $field, $value, @rest ) = split( / : /, $entry );
          if ( $field =~ /DNS Suffix/i ) {
            if ( $value =~ /\./ ) {
              ( $domain = $value ) =~ s/^\s*([^\s]*)\s*$/$1/ ;
              last;
            }
          }
        }
      }
    }
  } else {
    $host = `hostname -s`;
    $domain = `hostname -d`;
  }
  chomp($host);
  chomp($domain);

  $gHost = $host;
  $gDomain = $domain;
  return( $host, $domain );
}

# This function uses the cgi object to find the url used to visit the MUI
# and extracts the hostname part of it  (be it IP address, windows networking name
# or whatever)
sub get_cgi_hostname {
  my $cgi = shift;
  my $self_url = $cgi->url();
  $self_url =~ m|^(\w+://)?([^:/]+)|;
  $self_url = $2;
  return $self_url;
}

sub append_newMui {
  my ( $doc, $root, $muiLayout ) = @_;

  $root = $root->appendChild( $doc->createElement( "mui" ) );
  $root->setAttribute( "layout", $muiLayout );
}

sub append_host {
  my ( $doc, $root ) = @_;
  my $os = $^O;
  my ( $host, $domain ) = get_host_and_domain();

  $root = $root->appendChild( $doc->createElement( "host" ) );
  $root->setAttribute( "domain", $domain );
  $root->setAttribute("os", $os);
  $root->appendChild( $doc->createTextNode( $host ) );
}

sub append_agent {
  my ( $doc, $root ) = @_;
  my ($agent, $version, $os);
  my $ua = $ENV{HTTP_USER_AGENT};
  $ua =~ tr/A-Z/a-z/;
  trace("------------- $ua", "debug");

  if (index($ua, 'mozilla') != -1 && index($ua, 'compatible') == -1) {
    $agent = "nn";
  } elsif (index($ua, 'msie') != -1) {
    $agent = "ie";
  }
  trace("------------- $agent", "debug");

  $version = $ua;
  if ($agent eq "nn") {
    $version =~ s:^mozilla/(\d+?\.\d+?) .*$:$1:;
  } elsif ($agent eq "ie") {
    $version =~ s:^.*? msie (\d+?\.\d+?); .*$:$1:;
  }
  trace("------------- $version", "debug");

  $os   = 'win' if (index($ua, 'win') != -1 || index($ua, '16bit') != -1);
  $os ||= 'x11' if (index($ua, 'x11') != -1 ||
                    index($ua, 'sunos') != -1 ||
                    index($ua, 'irix') != -1 ||
                    index($ua, 'hp-ux') != -1 ||
                    index($ua, 'sco') != -1 ||
                    index($ua, 'unix_sv') != -1 ||
                    index($ua, 'unix_system_v') != -1 ||
                    index($ua, 'ncr') != -1 ||
                    index($ua, 'reliantunix') != -1 ||
                    index($ua, 'dec') != -1 ||
                    index($ua, 'osf1') != -1 ||
                    index($ua, 'dec_alpha') != -1 ||
                    index($ua, 'alphaserver') != -1 ||
                    index($ua, 'ultrix') != -1 ||
                    index($ua, 'alphastation') != -1 ||
                    index($ua, 'sinix') != -1 ||
                    index($ua, 'aix') != -1 ||
                    index($ua, 'inux') != -1 ||
                    index($ua, 'bsd') != -1);
  $os ||= 'os2' if (index($ua, 'os/2') != -1 ||
                    index($ua, 'ibm-webexplorer') != -1);
  $os ||= 'mac' if (index($ua, 'mac') != -1);
  trace("------------- $os", "debug");

  $root = $root->appendChild($doc->createElement("agent"));
  $root->setAttribute("os", $os);
  $root->setAttribute("version", $version);
  $root->appendChild($doc->createTextNode($agent));
}

sub append_user {
  my ( $doc, $root, $username ) = @_;
  $root = $root->appendChild( $doc->createElement( "username" ) );
  $root->appendChild( $doc->createTextNode( $username ) );
}

sub append_adminuser {
  my ($doc, $root, $server) = @_;
  my $in_str = "<exec><op>VMServerd_IsAdminUser</op><in></in></exec>";
  my $out_str = $server->exec($in_str);
  my $out = VMware::DOMAccess->newXML($out_str);
  my $adminuser = $out->getValue('.out.adminuser');
  $out->dispose();

  $root = $root->appendChild( $doc->createElement( "adminuser" ) );
  $root->appendChild( $doc->createTextNode( $adminuser ) );

  return 1;
}

sub get_build {

  return '108231';
}

sub append_build {
  my ( $doc, $root ) = @_;

  my $build = get_build();

  $root = $root->appendChild( $doc->createElement( "build" ) );
  $root->appendChild( $doc->createTextNode( $build ) );
}

sub append_product {
  my ($doc, $root, $server) = @_;
  my ($p, $product) = getProductInfo($server);
  $root = $root->appendChild($doc->createElement("product"));
  $root->setAttribute("id", $p);
  $root->appendChild($doc->createTextNode($product));
  return 1;
}

sub getProductName($) {
  my ($server) = @_;
  my ($p, undef) = getProductInfo($server);
  return $p;
}

sub getProductInfo($) {
  my ($server) = @_;
  my ($p, $product);

  if ($server && $server->is_connected()) {
    my $in_str = "<exec><op>VMServerd_GetProductInfo</op><in></in></exec>";
    my $out_str = $server->exec($in_str);
    if (!defined($out_str)) {
      my ($err, $errstr) = $server->get_last_error();
      trace("Server error $err: $errstr.", "error");
    } else {
      my $out = VMware::DOMAccess->newXML($out_str);
      $product = $out->getValue('.out.product');
      $p = $out->getAttribute('.product', 'id');
      $out->dispose();
    }
  }

  if (($p ne "ESX") && ($p ne "GSX")) {
    trace("Unknown product in Util::getProductInfo - '$p'\n", "error");
    # XXX Hard-code correct values in case of server error
    $product = "VMware GSX Server Version 3.0";
    $p = "GSX";
  }

  return ($p, $product);
}

# append_vm_question --------------------------------------------------------
#
# <question id="_">
#   ---
# </question>
sub append_vm_question {
  my ( $doc, $root, $vm ) = @_;
  my $id = $vm->get( "Status.question.current" );
  return undef unless ( defined( $id ) );
  my $limit = $vm->get( "Status.question.$id.choices.number" );
  return undef unless ( defined( $id ) );
  my @answers;
  for ( my $i = 0; $i < $limit; $i++ ) {
    $answers[ $i ] = $vm->get( "Status.question.$id.choices.$i" );
  }

  my $node = $root->appendChild( $doc->createElement( "question" ) );
  $node->setAttribute( "id",  $id );
  $node->appendChild( $doc->createTextNode( $vm->get( "Status.question.$id.text" ) ) );

  for ( my $i = 0; $i < scalar( @answers ); $i++ ) {
    $node = $root->appendChild( $doc->createElement( "answer" ) );
    $node->setAttribute( "ref",  $id );
    $node->setAttribute( "id",  $i );
    $node->appendChild( $doc->createTextNode( $vm->get( "Status.question.$id.choices.$i" ) ) );
  }

  return 1;
}

# append_vm_uptime ----------------------------------------------------------
#
# <uptime>
#   <days>---</days>
#   <hours>---</hours>
#   <minutes>---</minutes>
#   <seconds>---</seconds>
# </uptime>
# ---------------------------------------------------------------------------
sub append_uptime {
  my( $doc, $root, $obj, $opt ) = @_;
  my ( $stat, $stat_obj, $stat_obj_ok, $err, $errstr );

  # Factor $stat for the correct object
  if ( ref( $obj ) eq "VMware::Control::Server" ) {
    $stat = "Status.stats.System.Uptime";
    $stat_obj = VMware::Control::VM::new( $obj, ($obj->enumerate( ))[ 0 ] );
    ( $stat_obj_ok, $err, $errstr ) = connect_ctrl_obj( $stat_obj );
    return undef unless $stat_obj_ok;
  } elsif ( defined( $opt ) && $opt eq "Server") {
    $stat = "Status.stats.System.Uptime";
    $stat_obj = $obj;
  } else {
    $stat = "Status.stats.VM.Uptime";
    $stat_obj = $obj;
  }

  my $stat_info = "$stat.info";
  $stat_info = $stat_obj->get( $stat_info );
  return undef unless defined( $stat_info );
  my %info = %{ $stat_info };

  $stat = $stat_obj->get( $stat, 1 );
  return undef unless defined( $stat );
  my %stat = %{ $stat };
  %stat = %{ $stat{ $methods{ "min" } } };
  my ( $s, $m, $h, $d ) = calculate_uptime( $stat{ 1 } );

  $root = $root->appendChild( $doc->createElement( "uptime" ) );
  my $node = $root->appendChild( $doc->createElement( "days" ) );
  $node->appendChild( $doc->createTextNode( $d or "0" ) );
  $node = $root->appendChild( $doc->createElement( "hours" ) );
  $node->appendChild( $doc->createTextNode( $h or "0" ) );
  $node = $root->appendChild( $doc->createElement( "minutes" ) );
  $node->appendChild( $doc->createTextNode( $m or "0" ) );
  $node = $root->appendChild( $doc->createElement( "seconds" ) );
  $node->appendChild( $doc->createTextNode( $s or "0" ) );

  return 1;
}

sub calculate_uptime {
  my $uptime = shift( );
  my $s = $uptime % 60; $uptime /= 60;
  my $m = $uptime % 60; $uptime /= 60;
  my $h = $uptime % 24; $uptime /= 24;
  my $d = int( $uptime );
  return ( $s >= 1 ? $s : 0,
           $m >= 1 ? $m : 0,
           $h >= 1 ? $h : 0,
           $d >= 1 ? $d : 0 );
}

# append_ram ----------------------------------------------------------------
#
#    $doc ... XML::DOM document object
#   $root ... XML::DOM node to which the RAM stats should be appended
#    $obj ... vm or host object for which stats should be collected
#
# Given a VMware Control object and the requisite DOM objects, append hearbeat
# info nodes to the supplied parent node.
#
# <stat name="heartbeat" units="_">
#   <method>
#     <sample period="_" units="_">
#       ---
#     </sample>
#     ...
#   </method>
#   ...
# </stat>
#
# ---------------------------------------------------------------------------
sub append_heartbeat {
  my ( $doc, $root, $obj, $opt ) = @_;
  my ( $stat, $stat_obj, $stat_obj_ok, $err, $errstr );

  # Factor $stat for the correct object
  if ( ref( $obj ) eq "VMware::Control::Server" ) {
    $stat = "Status.stats.System.Heartbeat";
    $stat_obj = VMware::Control::VM::new( $obj, ($obj->enumerate( ))[ 0 ] );
    ( $stat_obj_ok, $err, $errstr ) = connect_ctrl_obj( $stat_obj );
    return undef unless $stat_obj_ok;
  } elsif ( defined( $opt ) && $opt eq "Server") {
    $stat = "Status.stats.System.Heartbeat";
    $stat_obj = $obj;
  } else {
    $stat = "Status.stats.VM.Heartbeat";
    $stat_obj = $obj;
  }

  my $stat_info = "$stat.info";
  $stat_info = $stat_obj->get( $stat_info );
  return undef unless defined( $stat_info );
  my %info = %{ $stat_info };

  my @intervals;
  foreach my $interval ( @periods ) {
    $intervals[ scalar( @intervals ) ] = $interval / $info{ "interval" };
  }

  $stat = $stat_obj->get( $stat, @intervals );
  return undef unless defined( $stat );
  my %stat = %{ $stat };

  $root = $root->appendChild( $doc->createElement( "stat" ) );
  $root->setAttribute( "name", "heartbeat" );
  $root->setAttribute( "units", "percent" );

  foreach my $method ( @methods ) {
    # Grab the current min, avg or max heartbeat.
    my $hb = \%{ $stat{ $methods{ $method } } };
    # Normalize heartbeat as a percentage of the limit.
    foreach my $period ( @intervals ) {
      # &! The aggregator is not passing back a limit. It should be safe to assume
      # that heartbeats will arrive at least 1 time per second on average.
      if ($method eq "sum") {
        $info{ "limit" } = $period;
      } else {
        $info{ "limit" } = 1;
      }

      unless ( !$$hb{ $period } ) {
        $$hb{ $period } = int( ( $$hb{ $period } / $info{ "limit" }  ) * 100);
        # Heartbeat is computed once every five seconds
        $$hb{ $period } = int($$hb{ $period } / 5);
        if ( $$hb{ $period } > 100 ) {
          $$hb{ $period } = 100;
        }
      }
    }

    my $node = $root->appendChild( $doc->createElement( "usage" ) );
    $node->setAttribute( "type", $method );

    append_samples( $doc, $node, $method, \%stat, \%info );
  }

  return 1;
}

# append_samplecount ----------------------------------------------------------------
#
#    $doc ... XML::DOM document object
#   $root ... XML::DOM node to which the samples count should be appended
#    $obj ... vm or host object for which stats should be collected
#
# Given a VMware Control object and the requisite DOM objects, append sample count
# info nodes to the supplied parent node.
#
# <stat name="sample_count">
#   <usage type="expected">
#      <sample interval=" ">
#          ---
#       </sample>
#       ---
#   </usage>
#   <usage type="actual">
#       <sample interval=" "> ---</sample>
#        ---
#   </usage>
# </stat>
#
# ---------------------------------------------------------------------------
sub append_samplecount {
  my( $doc, $root, $obj, $opt ) = @_;
  my ( $stat, $stat_obj, $stat_obj_ok, $err, $errstr );

  # Factor $stat for the correct object. We always use uptime as a reference
  if ( ref( $obj ) eq "VMware::Control::Server" ) {
    $stat = "Status.stats.System.CPUUsage";
    $stat_obj = VMware::Control::VM::new( $obj, ($obj->enumerate( ))[ 0 ] );
    ( $stat_obj_ok, $err, $errstr ) = connect_ctrl_obj( $stat_obj );
    return undef unless $stat_obj_ok;
  } elsif ( defined( $opt ) && $opt eq "Server") {
    $stat = "Status.stats.System.CPUUsage";
    $stat_obj = $obj;
  } else {
    $stat = "Status.stats.VM.CPUUsage";
    $stat_obj = $obj;
  }

  my $stat_info = "$stat.info";
  $stat_info = $stat_obj->get( $stat_info );
  return undef unless defined( $stat_info );
  my %info = %{ $stat_info };

  my ( $sum, $avg, $samples, $stat_ref, $child_node, $period );
  my %stat;

  $root = $root->appendChild( $doc->createElement( "sample_count" ) );
  my $node = $root->appendChild( $doc->createElement( "actual" ) );

  foreach $period ( @periods ) {
    $stat_ref = $stat_obj->get( $stat, $period );
    return undef unless defined( $stat_ref );
    %stat = %{ $stat_ref };
    my $sum = $stat{$methods{"sum"}}->{$period};
    my $avg = $stat{$methods{"avg"}}->{$period};
    my $samples = ($avg > 0) ? int ($sum / $avg) : 0;
    $child_node = $node->appendChild( $doc->createElement("sample" ) );
    $child_node->setAttribute( "interval", $info{ "interval" } ?
                               $period / $info{ "interval" } : 0 );
    $child_node->appendChild( $doc->createTextNode( $samples ) );
  }

  $node = $root->appendChild( $doc->createElement( "expected") );
  foreach $period (@periods) {
    $child_node = $node->appendChild( $doc->createElement("sample" ) );
    $child_node->setAttribute( "interval", $info{ "interval" } ?
                               $period / $info{ "interval" } : 0 );
    $child_node->appendChild( $doc->createTextNode( $period ) );
  }

  return 1;

}

sub append_capabilities {
  my ( $doc, $root, $vm ) = @_;
  my $ret = $vm->get( "Status.capabilities" );
  return undef unless defined( $ret );

  $root = $root->appendChild( $doc->createElement( "capabilities" ) );
  $root->setAttribute( "mode", $ret );
  my $node;

  trace( "$ret -> Read: ".($ret & 4).", Write: ".($ret & 2).", Execute: ".($ret & 1), "debug" );

  if ( $ret & 4 ) {
    $node = $root->appendChild( $doc->createElement( "read" ) );
  }
  if ( $ret & 2 ) {
    $node = $root->appendChild( $doc->createElement( "write" ) );
  }
  if ( $ret & 1 ) {
    $node = $root->appendChild( $doc->createElement( "execute" ) );
  }

  return 1;
}

sub rfc_1123_time {
  my ( $sec, $min, $hour, $mday, $mon, $year, $wday, $yday) = gmtime($_[ 0 ] or time());
  my @days = ( "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" );
  my @months = ( "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" );
  return $days[ $wday ].", ".
    ($mday < 10 ? "0$mday" : $mday)." ".
    $months[ $mon ]." ".
    ($year + 1900)." ".
    ($hour < 10 ? "0$hour" : $hour).":".
    ($min < 10 ? "0$min" : $min).":".
    ($sec < 10 ? "0$sec" : $sec)." GMT";
}

my $traceInitialized = 0;
my $logOK = 0;

sub isTraceable {
  if ($^O eq "MSWin32") {
    if (!$traceInitialized) {
      if (defined($ENV{vmware_LOGFILE})) {
        $logOK = open(STDERR, ">>$ENV{vmware_LOGFILE}") ;
      } else {
        $logOK = 0;
      }
      $traceInitialized = 1;
    }
    return $logOK;
  } else {
    return 1;
  }
}

sub tmpdir {
  my $dir = "";

  if ($^O eq "MSWin32") {
    for ($ENV{TMPDIR}, $ENV{TEMP}, $ENV{TMP}, "c:\\temp", "c:\\", "\\") {
      if ($_ && -e $_ && -d $_ && -w $_) {
        $dir = $_;
        last;
      }
    }
  } else {
    for ($ENV{TMPDIR}, "/tmp/") {
      if ($_ && -e $_ && -d $_ && -w $_) {
        $dir = $_;
        last;
      }
    }
  }

  if ($dir =~ /\/$/ || $dir =~ /\\$/) {
    return $dir;
  } else {
    return $dir . ($^O eq "MSWin32" ? "\\" : "/");
  }
}


# ---------------------------------------------------------------------------
#
# error_page --
#
#      Returns an error_message to the MUI.
#      A MUI Dialog box is opened with the exception message shown, 'Ok'
#      is the only option to close the dialog box.
#
# Prototype:
#
#      string error_page(string $errMsg, string $pkg, int $ln)
#
# Arguments:
#
#      $errMsg  ... exception message
#      $pkg     ... package name exception was thrown from (optional)
#      $ln      ... line number execption was thrown on (optional)
#
# Results:
#
#      An html page with embedded JavaScript is returned to be further
#      processed by the MUI. If no values for $pkg or $ln are provided,
#      then they will not appear in the error box. The optional values
#      are useful for debugging purposes only.
#
# ---------------------------------------------------------------------------

sub error_page($$$) {
   my ($errMsg, $pkg, $ln) = @_;
   my $level = $ENV{ "vmware_DEBUG_LEVEL" };

   # Send error to logfile
   # trace("Originator: $pkg {Line: $ln} $errMsg", "debug");
   # ($msg, $type, $fh, $package, $filename, $line)
   trace("$errMsg", "debug", undef, $pkg, undef, $ln);

   # Mimic JS escape function
   $errMsg = URI::Escape::uri_escape($errMsg, "^a-zA-Z0-9\\*\\@\\-_\\.\\/");

   # Test to see if page is being called within the MUI. If so, then display
   # the error dialog box from the MUI. If not, then dump contents to
   # page. Code is helpful if you are debugging modules outside of the MUI.
   my $errorHtml = "<html><head><script>";
   $errorHtml .= "if(parent.errorMsg) {";
   $errorHtml .= "parent.errorMsg('$errMsg');";
   $errorHtml .= "} else {";
   $errorHtml .= "document.writeln('$errMsg');";
   $errorHtml .= "}";
   $errorHtml .= "</script><head><body></body></html>\n";

   return ( $errorHtml,
            "Pragma"        => "no-cache",
            "Cache-Control" => "no-cache, must-revalidate",
            "Content-Type"  => "text/html; charset=utf-8" );
}


sub use_vmdb() {
   # Whether to use VMDB or not

   my $useVmdb = !defined($ENV{"vmware_NO_VMDB"});
   return $useVmdb;
}

############################################################
##
## validate_session_non_vmdb
##   Given a valid session id, extend its expiration time.
##   If no session id is given, a new session is created.
##   This is the non-vmdb version.
##
## Returns:
##   New session cookie.
##
## Side effects:
##
############################################################s
sub validate_session_non_vmdb {
  my $l = shift( );
  my $sc = shift( );
  my $sec = time( );

  if ( $sc ) {
    # Sid already exists.
    # sid:epoch_seconds:epoch_expire_seconds
    my @sid = split( ":", MIME::Base64::decode_base64( $sc ) );
    # Bump the expiration up.
    $sid[ 2 ] = ( 60 * $ENV{ "vmware_SESSION_LENGTH" } ) + $sec;
    $sc = "vmware.mui.sid=".MIME::Base64::encode_base64( join( ":", @sid ), "" );
    chomp($sc);
  } else {
    # Sid doesn't exist, generate it.
    $sc = "vmware.mui.sid=";
    # base64-encoded username, pid, epoch_seconds
    my $ran = int( rand( 10000000 ) );
    my $sid = MIME::Base64::encode_base64( $l.$$.$ran,"" );
    # sid:epoch_seconds:epoch_expire_seconds
    $sc .= MIME::Base64::encode_base64( "$sid:$sec:".
                                        ( 60 * $ENV{ "vmware_SESSION_LENGTH" } + $sec ), "" );
    chomp($sc);
  }
  return "$sc;path=/;";
}

############################################################
##
## session_is_valid_non_vmdb
##
##   Check if the given session is valid. Given a session ID (sid)
##   cookie value, decode its parts and make sure that it has not expired.
##   This is the non-vmdb version.
##
## Returns:
##   TRUE if the session is valid. FALSE otherwise.
##
## Side effects:
##
############################################################
sub session_is_valid_non_vmdb {
  my $sc = shift(); return 0 unless $sc;
  my %credentials = &VMware::Management::Util::hash_credentials();
  if (! $credentials{"l"} || ! defined($credentials{"l"})) {
    return 0;
  }

  # sid:epoch_seconds:epoch_expire_seconds
  my @sid = split( ":", MIME::Base64::decode_base64( $sc ) );

  # The session is valid if the time-out mechanism is disabled or
  # the epoch_expire_seconds value is less than the current epoch
  # seconds value as returned by time.
  if ($ENV{ "vmware_SESSION_LENGTH" } != 0 &&
    ($ENV{ "vmware_SESSION_LENGTH" } == -1 ||
    $sid[ 2 ] > time())) {
    return 1;
  }

  return 0;
}


# ---------------------------------------------------------------------------
#
# hashQueryString--
#
#      Given a formatted string such as ?key=value, return its hash
#      representation, or empty hash if the string is not properly formatted.
#
# ---------------------------------------------------------------------------

sub hashQueryString {
   if (shift() =~ /^\?(.*)/) {
      my %hash;

      for (split('&', $1)) {
         s/\+/ /g;
         my ($key, $val) = split('=', $_);
         $key = URI::Escape::uri_unescape($key);
         $val = URI::Escape::uri_unescape($val);
         if (exists($hash{$key})) {
            $hash{$key} .= "," . $val;
         } else {
            $hash{$key} = $val;
         }
      }

      return %hash;
   }

   return ();
}


# ---------------------------------------------------------------------------
#
# getDiskPath --
#
#      Get the fully qualified disk path if it is currently a relative path
#
# ---------------------------------------------------------------------------

sub getDiskPath {
  my $loc = shift();
  my $vmPath = shift();
  my $sep = ($^O eq "MSWin32" ? "\\" : "/");

  my $trnCtx = &VMware::Management::Session::get_temp_context();
  my $ctx = &VMware::Management::Session::get_temp_context();

  $trnCtx->SetCurrentPath($vmPath);
  $ctx->SetCurrentPath($vmPath);

  if ($loc !~ /^(.*)\Q$sep\E([^\Q$sep\E]*)$/) {
    # Relative path -- prepend the config directory
    $trnCtx->BeginTransaction();
    my $cmdPath = $trnCtx->NewArrayIndex("/host2/#_mui/util/file/cmd/##/");
    $cmdPath = $trnCtx->GetAbsPath($cmdPath);
    $trnCtx->SetString("${cmdPath}op", "parse");
    my $opPath = "${cmdPath}op/parse/";
    $trnCtx->SetString("${opPath}in/filename", $ctx->GetString("cfgPath"));
    my ($ok, $msg) = VMware::Management::VmdbUtil::vmdbEndCmd($trnCtx, $cmdPath);
    if (!$ok) {
      return undef;
    }

    my $dirname = $trnCtx->GetString("${opPath}out/dirname");
    $loc = $dirname . $sep . $loc;
    $trnCtx->LocalArrayUnset($cmdPath, 1);
  }

  return $loc;
}


# ---------------------------------------------------------------------------
#
# jsStringLiteral--
#
#      Make $str safe for inclusion in an HTML/JavaScript application by
#      scrubbing backslashes, quotes, new lines, and markup delimeters.
#
#      If $escapeMarkup is specified as 0, markup will not be scrubbed.
#
# ---------------------------------------------------------------------------

my %JS_UNICODE_SEQUENCES = ('&' => '\u0026', '<' => '\u003C', '>' => '\u003E');
my $JS_UNICODE_CHARS = join("", keys %JS_UNICODE_SEQUENCES);

sub jsStringLiteral {
   my $str = shift();

   # Escape markup delimeters by default.
   my $escapeMarkup = shift();
   if (! defined($escapeMarkup)) {
      $escapeMarkup = 1;
   }

   # Duplicate backslashes.
   $str =~ s/\\/\\\\/g;
   # Escape quotes with backslashes.
   $str =~ s/("|')/\\$1/g;
   # Replace new line variants with \n.
   $str =~ s/(\r\n|\r|\n)/\\n/g;

   if ($escapeMarkup) {
      # Use unicode escape sequences rather than HTML entities to prevent
      # markup injection without losing the original data.
      $str =~ s/([$JS_UNICODE_CHARS])/$JS_UNICODE_SEQUENCES{$1}/g;
   }

   return qq{"$str"};
}


# ---------------------------------------------------------------------------
#
# escapeMarkup--
#
#      Make the specified string ($_[0]) safe for inclusion in an HTML
#      application by scrubbing markup delimeters.
#
#      Note: This modifies the specified string inplace.
#
# ---------------------------------------------------------------------------

my %HTML_ENTITIES = ('&' => '&amp;', '<' => '&lt;', '>' => '&gt;');
my $HTML_CHARS = join("", keys %HTML_ENTITIES);

sub escapeMarkup {
   $_[0] =~ s/([$HTML_CHARS])/$HTML_ENTITIES{$1}/g;
}


1;
