#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# FileInfo.pm : Operations related to gathering file information
#
#

package VMware::VMServerd::FileInfo;
use strict;

use VMware::VMServerd qw(&Warning 
			 &Panic
			 );
use VMware::ExtHelpers qw(&internal_dirname 
                          &internal_basename
			  &shell_string
			  &isRootDir
			  &isAbsolutePath
			  &getSizeMB);
use VMware::HConfig::Host qw(&sameFileSystem);

use VMware::VMServerd::AuthPolicy qw(&BEGIN_PRIVILEGED
				     &END_PRIVILEGED);

use Exporter   ();
use vars       qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

BEGIN {
    if ($VMware::VMServerd::PRODUCT =~ /ESX/i) {
        require VMware::VMServerd::ESXHostInfo;
    }

    @EXPORT = qw(&fileExists
		 &isFile
		 &isDirectory
		 &isReadable
		 &isWritable
		 &isExecutable
		 &getFileSize
		 &getMode
		 &isSameFS
		 &canDeleteFile
		 &isVmfsFile
		 &getFileInfo
		 &getVmfsFileInfo
		 &getExt2FileInfo
		 &removeVmfsFile
		 &isVmfsMountPoint
		 &enableVmfsFileCaching
		 &disableVmfsFileCaching
		 );

    # set the version for version checking
    $VERSION     = 1.00;
    @ISA         = qw(Exporter);
}

my %gCachedFileInfo;
my $gCachedVmfsFiles = undef;
my $gVmfsFileCachingEnabled = 0;

my $gStartTime = undef;
sub startTimer()
{
    $gStartTime = &VMware::VMServerd::GetRealTime();
}

sub stopTimer($) {
    my ($msg) = @_;

    my $elapsedTime = &VMware::VMServerd::GetRealTime() - $gStartTime;
    $gStartTime = undef;
    Warning($msg . ": $elapsedTime ms \n");
}


#########################################################################
#
# fileExists: Implements the "-e filename" test.
#
# Returns: 1 if file exists, 0 otherwise.
#
#########################################################################
sub fileExists($) {
    my ($filename) = @_;

    if (isVmfsFile($filename)) {
	my $fileInfo = getVmfsFileInfo($filename);

	if (!defined($fileInfo)) {
	    return 0;
	}

	return 1;
    }

    return (-e $filename);
}

#########################################################################
#
# isFile: Implements the "-f filename" test.
#
# Returns: 1 if it is a file, 0 otherwise
#
#########################################################################
sub isFile($) {
    my ($filename) = @_;

    if (isVmfsFile($filename)) {
	my $fileInfo = getVmfsFileInfo($filename);

	if (!defined($fileInfo)) {
	    return 0;
	}

	return 1;
    }

    return (-f $filename);
}

#########################################################################
#
# isDirectory: Implements the "-d filename" test.
#
# Returns: 1 if directory, 0 otherwise
#
#########################################################################
sub isDirectory($) {
    my ($filename) = @_;

    if (isVmfsFile($filename)) {
	return 0;
    }

    return (-d $filename);
}

#########################################################################
#
# isReadable: Implements the "-r filename" test.
#
# Returns: 1 if readable, 0 otherwise.
#
#########################################################################
sub isReadable($) {
    my ($filename) = @_;

    my $bits = &VMware::VMServerd::GetAccessBits($filename);
    return ($bits & 0x4);
}

#########################################################################
#
# isWritable: Implements the "-w filename" test.
#
# Returns: 1 if writable, 0 otherwise.
#
#########################################################################
sub isWritable($) {
    my ($filename) = @_;

    my $bits = &VMware::VMServerd::GetAccessBits($filename);
    return ($bits & 0x2);
}
    
#########################################################################
#
# isExecutable: Implements the "-x filename" test.
#
# Returns: 1 if executable, 0 otherwise.
#
#########################################################################
sub isExecutable($) {
    my ($filename) = @_;

    my $bits = &VMware::VMServerd::GetAccessBits($filename);
    return ($bits & 0x1);
}
    

#########################################################################
#
# getFileSize: Implements "-s filename".
#
# Returns: Size in MB
#
#########################################################################
sub getFileSize($) {
    my ($filename) = @_;

    if (isVmfsFile($filename)) {
	my $vmfs = getVmfs($filename);

	if (!$vmfs) {
	    return 0;
	}
    
	my ($name, $size, $mode, $uid, $gid, $attr, $mtimeStr) =
	    getVmfsFile($vmfs, $filename);

	return $size;
    }

    return ( (-s $filename) / (1024*1024) );
}

#########################################################################
#
# getMode: Gets the access mode for the file/directory
#
# Returns: Integer representing the mode
#
#########################################################################
sub getMode($) {
    my ($filename) = @_;

    if (isVmfsFile($filename)) {
	my $vmfs = getVmfs($filename);

	if (!$vmfs) {
	    return 0;
	}
    
	my ($name, $size, $mode, $uid, $gid, $attr, $mtimeStr) =
	    getVmfsFile($vmfs, $filename);

	return $mode;
    }

    return (stat($filename))[2];
}

#########################################################################
#
# isSameFS: Checks if the given files are on the same File System
#
# Returns: 1 if they are, 0 otherwise
#
#########################################################################
sub isSameFS($$) {
    my ($file1, $file2) = @_;

    my $file1IsVmfs = isVmfsFile($file1);
    my $file2IsVmfs = isVmfsFile($file2);

    if ($file1IsVmfs && $file2IsVmfs) {
	my $vmfs1 = getVmfs($file1);
	my $vmfs2 = getVmfs($file2);
	
	return ($vmfs1 == $vmfs2) ? 1 : 0;
    }

    if ( ($file1IsVmfs && !$file2IsVmfs) ||
	 (!$file1IsVmfs && $file2IsVmfs) ) {
	return 0;
    }

    return &VMware::HConfig::Host::sameFileSystem($file1, $file2);
}

#########################################################################
#
# canDeleteFile: Checks to see if the user (current euid) can delete 
# the given file. 
#
# Returns: 1 the user has right to delete the file, 0 otherwise.
#
#########################################################################
sub canDeleteFile($) {
    my ($path) = @_;
    
    if (!isAbsolutePath($path)) {
	return 0;
    }

    if ($^O =~ /linux/i) {
	my $euid = &VMware::VMServerd::GetEUID();
	# Root can delete anything
	if ($euid == 0) {
	    return 1;
	}

	my $parentDir = internal_dirname($path);
	if ( !isWritable($parentDir) ||
	     !isExecutable($parentDir) ) {
	    return 0;
	}

	# If the parent directory has a sticky bit set you need to own this file
	if (-k $parentDir) {
	    my $fileInfo = getFileInfo($path, 0);
	    if ($fileInfo->{uid} != $euid) {
		return 0;
	    }
	}
	
	# If you have reached this far, you must be authentic...
	return 1;
    }
    else {
	Panic("canDeleteFile not implemented for Windows ");
    }

}

#########################################################################
#
# isVmfsFile: Checks if given file is a VMFS file
#
# Returns: 1 if VMFS file, 0 otherwise
#
#########################################################################
sub isVmfsFile($) {
    my ($f) = @_;

    # We have to canonicalize the path to resolve symlinks,
    # since we rely on string manipulation to get the dirname
    # and determine if that directory is a VMFS mount point.
    # Note that we have to use the FileManager version, since it
    # handles the case where $f refers to a non-existent file in an
    # existing directory (possible in case of Move/Rename operation)
    my $cname = &VMware::VMServerd::FileManager::FileManager_CanonicalPath($f);

    if (!defined($cname)) {
	return 0;
    }

    my $parentDir = internal_dirname($cname);

    return (isVmfsMountPoint($parentDir));
}


#########################################################################
#
# getFileInfo: Generic wrapper around VMFS and EXT2 specific get functions
#
# Returns: Reference to a hash representing the file information
#
#########################################################################
sub getFileInfo($$) {
    my ($filename, $dirOnly) = @_;
    
    if (isVmfsFile($filename)) {
	if ($dirOnly) {
	    return undef;
	}
	my $info = getVmfsFileInfo($filename);
	return $info;
    }

    my $info2 = getExt2FileInfo($filename, $dirOnly);
    return $info2;
}

#########################################################################
#
# getVmfsFileInfo: VMFS specific file info function
#
# Returns: Reference to a hash representing the file information
#
#########################################################################
sub getVmfsFileInfo($) {
    my ($file) = @_;

    my $hash = undef; 
    if (defined($hash)) {
	return $hash;
    }

    my $vmfs = getVmfs($file);

    if (!$vmfs) {
	return undef;
    }
    
    my ($name, $size, $mode, $uid, $gid, $attr, $mtimeStr) = getVmfsFile($vmfs, $file);

    if (!defined($name)) {
	return undef;
    }

    $hash = {};

    $hash->{name} = $name;
    $hash->{type} = "file";
    $hash->{size} = $size;
    $hash->{uid} = $uid;
    $hash->{gid} = $gid;
    # Note that $mode is an integer representing the access permissions
    $hash->{mode} = $mode;
    $hash->{attr} = $attr;

    # Get permissions
    my $permissions = "";
    my $bits = &VMware::VMServerd::GetAccessBits($file);
    if ($bits & 0x04) {
	$permissions .= "r";
    }
    if ($bits & 0x02) {
	$permissions .= "w";
    }
    if ($bits & 0x01) {
	$permissions .= "x";
    }
    $hash->{permissions} = $permissions;
    
    $hash->{mtime} =  $mtimeStr;

    if (isConfigFile($file, 1)) {
	$hash->{config} = 1;
    }

    if (&VMware::VMServerd::Disk::hasValidDiskExtension($file)) {
	if ($hash->{attr} =~ /cow disk/i) {
	    $hash->{virtualdisk} = 1;
	} elsif ($hash->{attr} =~ /disk/i) {
	    $hash->{vmfsdisk} = 1;
	}
    }
    
    # No need to check for vmfsMountPoint, since a VMFS file cannot be one.
    
    $hash->{fstype} = "vmfs";

    return $hash;
}


#########################################################################
#
# getExt2FileInfo: non-VMFS file info function
#
# Returns: Reference to a hash representing the file information
#
#########################################################################
sub getExt2FileInfo($$) {
    my ($f, $dirOnly) = @_;
    my $hash = undef;

    my $filename = $f;

    $hash = getCachedFileInfo($f);
    if (defined($hash)) {
	return $hash;
    }

    $hash = {};
    my @fileStats = stat($f);
    if (!defined(@fileStats)) {
	&VMware::VMServerd::Warning("Could not stat file $f. Reason: $!");
    }

    # Add name and absolute path to the hash
    $hash->{name} = internal_basename($f);
    $hash->{abspath} = $f;

    # Add type to the hash
    my $canonicalPath = $f;
    if (-l $f) {
	$canonicalPath = &VMware::VMServerd::GetCanonicalPath($f);
	if (!defined($canonicalPath) || (! -e $canonicalPath)) {
	    $canonicalPath = "";
	}

	if (-f $canonicalPath) {
	    $hash->{type} = "file";
	#} elsif (-d $canonicalPath) {
	} elsif (-d _) {
	    $hash->{type} = "directory";
	}
	else {
	    $hash->{type} = "other";
	}
	$hash->{realname} = $canonicalPath;
    #} elsif (-f $f) {
    } elsif (-f _) {
	$hash->{type} = "file";
    #} elsif (-d $f) {
    } elsif (-d _) {
	$hash->{type} = "directory";
    } else {
	$hash->{type} = "other";
    }

    # If we are looking for directories only, any further work is 
    # a waste if this is not a directory
    if ($dirOnly && ($hash->{type} ne "directory")) {
	return undef;
    }

    # Handle dangling symlinks
    if ((-l $f) && ($canonicalPath eq "")) {
	return $hash;
    }

    # Rest of the data is gathered for what it points to in case of symlink
    $f = $canonicalPath;
    @fileStats = stat($f);
    if (!defined(@fileStats)) {
	&VMware::VMServerd::Warning("Could not stat file $f. Reason: $!");
	return $hash;
    }
    # Get permissions
    my $permissions = "";
    my $bits = &VMware::VMServerd::GetAccessBits($f);
    if ($bits & 0x04) {
	$permissions .= "r";
    }
    if ($bits & 0x02) {
	$permissions .= "w";
    }
    if ($bits & 0x01) {
	$permissions .= "x";
    }
    $hash->{permissions} = $permissions;

    # Size and modif time
    my $size  = $fileStats[7];
    my $mtime = $fileStats[9];

    # mode, uid and gid 
    $hash->{mode} = $fileStats[2];
    $hash->{uid} = $fileStats[4];
    $hash->{gid} = $fileStats[5];

    $hash->{size} = getSizeMB($size);
    my $modifTime = scalar(localtime($mtime));
    $hash->{mtime} = $modifTime;
    $hash->{mtimeRaw} = $mtime;

    if (isConfigFile($f, 0)) {
	$hash->{config} = 1;
    }

    if (isVmfsMountPoint($f)) {
	$hash->{vmfsMountPoint} = 1;
    }
    
    if (&VMware::VMServerd::Disk::isCOWDisk($f)) {
	$hash->{virtualdisk} = 1;
	$hash->{size} = &VMware::VMServerd::Disk::getCOWDiskSize($f);
    }

    # No need to check for isVmfsDisk, since we are not in VMFS file system

    if ($hash->{vmfsMountPoint}) {
	# stat on VMFS mount point is very expensive. Cache the information.
	$hash->{fstype} = "ext2";
	addCachedFileInfo($filename, $hash);
    }
    return $hash;
}


sub removeVmfsFile($) {
    my ($filename) = @_;

    my $fileInfo = getVmfsFileInfo($filename);
    if (defined($fileInfo->{vmfsdisk})) {
	# XXX TODO make sure no VM is using this disk.
    }

    if (defined($fileInfo->{config})) {
	# XXX TODO make sure this VM is not running.
    }

    if (!canDeleteFile($filename)) {
	return (0, "You do not have permission to delete $filename; ");
    }

    my $vmfs = getVmfs($filename);

    BEGIN_PRIVILEGED();
    $vmfs->removeFile(internal_basename($filename));
    END_PRIVILEGED();
    
    return 1;
}

#########################################################################
#
# isVmfsMountPoint: Checks if the given directory is a VMFS mount point
#
# Returns: 1 if it is a mount point, 0 otherwise
#
#########################################################################
sub isVmfsMountPoint($) {
    my ($dirname) = @_;

    if ($VMware::VMServerd::PRODUCT !~ /ESX/i) {
        return 0;
    }

    if (! -d $dirname) {
	return 0;
    }

    my $realpath = &VMware::VMServerd::GetCanonicalPath($dirname);
    my $parentdir = &VMware::ExtHelpers::internal_dirname($realpath);

    my $gHost = &VMware::VMServerd::ESXHostInfo::getESXHost();

    my %DirMap = %{ $gHost->getDirMapping() };

    my $mt_point;
    foreach $mt_point (keys(%DirMap)) {
	my $type = $DirMap{$mt_point}->[1];
	if (($type =~ /vmfs/i) && ($mt_point eq $parentdir)) {
	    return 1;
	}
    }

    return 0;
}

#########################################################################
#
# isConfigFile: Checks if the given file is a config file
#
# Returns: 1 if it is a config file, 0 otherwise
#
#########################################################################
sub isConfigFile($$) {
    my ($filename, $vmfsFile) = @_;
    my @validExtensions = qw(cfg vmx);

    # If the file is in VMFS, it is a file. Save one stat.
    if (!$vmfsFile) {
	if (! (-f $filename && -r _) ) {
	    return 0;  
	}
    }

    # If the file has the right extension, it probably is a config file
    my $basename = internal_basename($filename);
    my @parts = split(/\./, $basename);

    if (scalar(@parts) < 2) {
	return 0;
    }
    my $fileExtension = $parts[$#parts];
    
    if (!grep($_ eq $fileExtension, @validExtensions)) {
	return 0;
    }

    # Looks like a valid extension. Let us look inside the file.
    my $cfg = new VMware::Config;

    my $retval = $cfg->readin($filename);

    if (!defined($retval)) {
	return 0;
    }

    return 1;
}

#########################################################################
#########>>>            Internal Helper functions            <<<#########
#########################################################################


#########################################################################
#
# getVmfs: Returns a handle to an object of class MFSFileSystem. 
#
#########################################################################
sub getVmfs($) {
    my ($file) = @_;

    # XXX Passing extra arg '1' is a workaround for PR21800. See FileManager.pm.
    $file = &VMware::VMServerd::FileManager::FileManager_CanonicalPath($file, 1);

    my $parentDir = internal_dirname($file);
    my $vmfsName  = internal_basename($parentDir);

    # Note that Lookup of MFS file system as well as files requires
    # root permission, because vmkfstools works only if euid is root.
    # Therefore, we have to revert impersonation for a short while
    BEGIN_PRIVILEGED();
    my $vmfs = &VMware::VMServerd::ESXHostInfo::getESXHost()->getMFSLookup($vmfsName);
    END_PRIVILEGED();

    return $vmfs;
}

#########################################################################
#
# getCachedFileInfo: Get the cached information about files if the 
#                    file does not have a newer modification time.
#
# Returns: reference to hash as returned by getFileInfo
#
#########################################################################
sub getCachedFileInfo($) {
    my ($filename) = @_;

    my $hash = $gCachedFileInfo{$filename};

    if (!defined($hash)) {
	return undef;
    }

    my $mtime;

    # We do not cache VMFS files here
    if ($hash->{fstype} eq "vmfs") {
	$gCachedFileInfo{$filename} = undef;
	return undef;
    }

    my @fileStats = stat($filename);
    if (!defined(@fileStats)) {
	$gCachedFileInfo{$filename} = undef;
	return undef;
    }
    $mtime = $fileStats[9];

    if ($mtime > $hash->{mtimeRaw}) {
	$gCachedFileInfo{$filename} = undef;
	$hash = undef;
    }

    return $hash;
}

#########################################################################
#
# addCachedFileInfo: Add the given file information to the cache
#
#########################################################################
sub addCachedFileInfo($$) {
    my ($filename, $hash) = @_;

    $gCachedFileInfo{$filename} = $hash;
}

#########################################################################
#
# enableVmfsFileCaching: Enable caching of VMFS files. Note that
# it is the responsibility of the user of this cache to ensure
# caching is enabled at the begining of a exec request and disabled
# at the end, otherwise you risk returning stale data.
#
#########################################################################
sub enableVmfsFileCaching()
{
    if (defined($gCachedVmfsFiles) || ($gVmfsFileCachingEnabled == 1)) {
	Panic("VMFS File Info Caching being enabled when already enabled");
    }

    $gVmfsFileCachingEnabled = 1;
    $gCachedVmfsFiles = {};
}

#########################################################################
#
# disbleVmfsFileCaching: Disable caching of VMFS files. 
#
#########################################################################
sub disableVmfsFileCaching()
{
    if (!defined($gCachedVmfsFiles) || ($gVmfsFileCachingEnabled != 1)) {
        Panic("VMFS File Info Caching being disabled without being enabled");
    }

    $gCachedVmfsFiles = undef;
    $gVmfsFileCachingEnabled = 0;
}

#########################################################################
#
# getVmfsFile: Returns information about a VMFS file, either obtained
# from the cache or going directly to MFSFileSystem class.
#
#########################################################################
sub getVmfsFile($$) {
    my ($vmfs, $file) = @_;

    my $fileBaseName = internal_basename($file);

    my $allFiles;
    if (!defined($gCachedVmfsFiles) ||
	!defined($gCachedVmfsFiles->{$vmfs})) {
	# Note that Lookup of MFS file system as well as files requires
	# root permission, because vmkfstools works only if euid is root.
	# Therefore, we have to revert impersonation for a short while

	BEGIN_PRIVILEGED();
	$allFiles = $vmfs->getAllFiles();
	END_PRIVILEGED();
	
	if ($gVmfsFileCachingEnabled) {
	    $gCachedVmfsFiles->{$vmfs} = $allFiles;
	}

    }
    else {
	$allFiles = $gCachedVmfsFiles->{$vmfs};
    }
    
    my $arr = $allFiles->{$fileBaseName};

    if (!defined($arr)) {
	return undef;
    }

    my ($size, $mode, $uid, $gid, $attr, $mtimeStr) = @{ $arr };

    return ($fileBaseName, $size, $mode, $uid, $gid, $attr, $mtimeStr);
}

1;
