#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# VMConfigDefaults.pm
# 
# Returns a ESX VM's config state.
#

package VMware::VMServerd::VMConfigDefaults;

use strict;
use Carp;
use VMware::DOMAccess;
use VMware::VMServerd;



# Gets the defaults values to be used for guest OSes.
#
# Input format:
#   No input args
#
# Output format:
#   .guests.guest[].os := the name of the guest OS
#   .guests.guest[].displayName := the display name for the guest OS
#   .guests.guest[].scsi := flag indicating if the VM should use SCSI
#   .guests.guest[].size := size of virtual disk for VM
#   .guests.guest[].colorDepth := color depth for VM
#   .guests.guest[].memory.minimum := minimum memory needed by VM
#   .guests.guest[].memory.maximum := maximum memory needed by VM (XXX not used)
#   .guests.guest[].memory.recommended := recommended memory for VM
#   .guests.guest[].memory.warning := should the user be warned about the lack of memory?
#
sub GetAll_Handler {
    my $in = shift;
    my $out = shift;

    $out->setValue(".guests", "");
    
    my @oslist = VMware::VMServerd::GetGuestOsList();

    foreach my $os (@oslist) {
	my ($name, $scsi, $size, $colorDepth) = VMware::VMServerd::GetGuestDefaults($os);
	# XXX The size we get is in X000 MB (defined in defaultsInc.pl)
	# Convert it to GB.
	$size /= 1000;

	my ($min, $rec, $max, $warn) = VMware::VMServerd::GetGuestMemoryDefaults($os);

	my $doc = VMware::DOMAccess->new("guest");
	
	$doc->setValue(".os", $os);
	$doc->setValue(".displayName", $name);
	$doc->setValue(".scsi", $scsi);
	$doc->setValue(".size", $size);
	$doc->setValue(".colorDepth", $colorDepth);
	
	$doc->setValue(".memory.minimum", $min);
	$doc->setValue(".memory.maximum", $max);
	$doc->setValue(".memory.recommended", $rec);
	$doc->setValue(".memory.warning", $warn);

	$out->addSubTree(".guests", $doc);
	$doc->dispose();
    }
    
    return(1);
}


# Gets the config and virtual HW version to create a new VM with
#
# Input format:
#   No input args
#
# Output format:
#   .configVersion.variable := Name of the config version variable
#   .configVersion.value    := Value of the config version variable
#   .virtualHWVersion.variable := Name of the virtual HW version variable
#   .virtualHWVersion.varuable := Value of the virtual HW version variable
#
sub GetConfigVersion_Handler {
  my $in = shift;
  my $out = shift;

  my ($cvVar, $cvVal, $hvVar, $hvVal) = &VMware::VMServerd::GetConfigVersion();

  $out->setValue(".configVersion.variable", $cvVar);
  $out->setValue(".configVersion.value", $cvVal);
  $out->setValue(".virtualHWVersion.variable", $hvVar);
  $out->setValue(".virtualHWVersion.value", $hvVal);

  return (1);
}

&VMware::VMServerd::addOperation( OPNAME => 'VMConfigDefaults_GetAll',
				  PERLFUNC => 'VMware::VMServerd::VMConfigDefaults::GetAll_Handler',
				  POLICY => 'authuser');

&VMware::VMServerd::addOperation( OPNAME => 'VMConfigDefaults_GetConfigVersion',
				  PERLFUNC => 'VMware::VMServerd::VMConfigDefaults::GetConfigVersion_Handler',
				  POLICY => 'authuser');

1;
