#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# VMDeleteInfo.pm
# 
# Returns information about what will be deleted when the VM is removed.
#

# It doesn't really make sense to split this package from the other because 
# they aren't useful by themselves.  However, we're sticking to convention
# because a dependency could be mode on this module in the future.

package VMware::VMServerd::VMDeleteInfo;

use strict;
use VMware::DOMAccess;
use VMware::Config;
use VMware::VMServerd::VMList;
use VMware::ExtHelpers qw(&internal_dirname 
			  &internal_basename
			  &isAbsolutePath
			  &getFileSeparator
			 );
use VMware::VMServerd qw($VMSTATE_ERROR 
			 $VMSTATE_OFF 
			 $VMSTATE_ON 
			 $VMSTATE_SUSPEND 
			 $VMSTATE_STUCK 
			 %VMSTATE_NAMES
			 );
use VMware::VMServerd::Disk;
                     


use Exporter   ();
use vars       qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

BEGIN {
    # set the version for version checking
    $VERSION     = 1.00;
    @ISA         = qw(Exporter);
    @EXPORT_OK   = qw(&canDeleteVM 
		      &getDisksForVM 
		      &buildDiskUsageHash 
		      &getDiskUsage
		      &getFilesToDelete
		      &getDeleteTimestamp
		      );
}

# Gets information about what will happen if a VM is deleted.  This
# function should be called in preparation for the deletion of a VM.
#
# Input format:
#   .config := the config file to be deleted
#
# Output format:
#   .disks.disk[].name := name of the disk file
#   .disks.disk[].virtual_name := the name as seen by the VM
#   .disks.disk[].present := flag indicating if this VM is using the disk
#   .disks.disk[].usage := usage count of the disk (1 if only this VM is using it)
#   .disks.disk[].vms.vm[] := names of VMs using the disk
#   .files.file[] := the name of files to be deleted
#   .timestamp := the timestamp for this delete operation
#   .disks.disk[]@usage := usage count of the disk (1 if only this VM is using it)
#
sub DeleteVMInfo_Handler {
    my $in = shift;
    my $out = shift;

    &VMware::VMServerd::errorReset();

    my $config = $in->getValue('.config');

    my ($rc, $err) = canDeleteVM($config);
    if( !$rc ) {
	&VMware::VMServerd::errorPost($err);
	return(0);
    }

    if( !query_deleteVMInfo($out, $config) ) {
	&VMware::VMServerd::errorPost("query of VM failed.");
	return(0);
    }

    return(1);
}


# Gets information about what will happen if a VM is deleted.  This
# function should be called in preparation for the deletion of a VM.
#
# Input format:
#   .config := the config file to be deleted
#
# Output format:
#   .cfgName := the config file name of the config file
#   .safe_cfg := the "safe" version of the config file name of the config file
#   .displayName := the display name of the config file
#   .disks.disk[].name := name of the disk file
#   .disks.disk[].virtual_name := the name as seen by the VM
#   .disks.disk[].present := flag indicating if this VM is using the disk
#   .disks.disk[].usage := usage count of the disk (1 if only this VM is using it)
#   .disks.disk[].vms.vm[] := names of VMs using the disk
#   .files.file[] := the name of files to be deleted
#   .timestamp := the timestamp for this delete operation
#   .disks.disk[]@usage := usage count of the disk (1 if only this VM is using it)
#
sub query_deleteVMInfo {
    my($out, $config) = @_;
    my $timestamp = 0;

    my ($rc, $err) = canDeleteVM($config);
    if( !$rc ) {
	&VMware::VMServerd::errorPost($err);
	return(0);
    }

    $out->setValue('.cfgName', $config);
    $out->setValue('.safe_cfg', VMware::VMServerd::toSafeCfgString($config));
    my($displayName) = getConfigInfo($config);
    if( !defined($displayName) ) {
	$out->setValue('.displayName', '');
    } else {
	$out->setValue('.displayName', $displayName);
    }

    my @diskList = getDisksForVM($config);
    my $usageHash = buildDiskUsageHash();
    my $diskObj;
    foreach $diskObj (@diskList) {
	my $present = $diskObj->{PRESENT};
	my $name = $diskObj->{NAME};
	my $virtual_name = $diskObj->{VIRTUAL_NAME};
        my $canonicalName = $diskObj->{CANONICAL_NAME};

	my @usage = getDiskUsage($usageHash, $name, $config);

	my $node = VMware::DOMAccess->new('disk');
	$node->setValue('.present', $present);
	$node->setValue('.name', $canonicalName);
	$node->setValue('.virtual_name', $virtual_name);
	$node->setValue('.usage', $#usage + 1);
	$node->setAttribute('.', 'usage', $#usage + 1);

	# Add each of the usage
	my $vm_config;
	foreach $vm_config (@usage) {
	    my $usage_node = VMware::DOMAccess->new('vm');

	    $usage_node->setValue('.', $vm_config);

	    $node->addSubTree('.vms', $usage_node);
	    $usage_node->dispose();
	}

	$out->addSubTree('.disks', $node);
	$node->dispose();
    }

    # Build the list of files to be deleted
    my @deleteFiles = getFilesToDelete($config);
    my $file;
    foreach $file (@deleteFiles) {
	my $node = VMware::DOMAccess->new('file');
	$node->setValue('.', $file);
	$out->addSubTree('.files', $node);
	$node->dispose();
    }
    
    # Find the timestamp
    my @diskNameList = map { $_->{NAME} } @diskList;
    $timestamp = getDeleteTimestamp($usageHash, $config, \@diskNameList);
    if( $timestamp == 0 ) {
	&VMware::VMServerd::errorPost("Timestamp $timestamp is not valid.");
    }
    $out->setValue('.timestamp', $timestamp);
    return(1);
}


sub canDeleteVM {
    my($config) = @_;

    # Make sure this VM is actually registered
    my @vms = VMware::VMServerd::VMList::ListAllVMs();

    $config = &VMware::VMServerd::canonicalize($config);
    for (my $i=0; $i <= $#vms; $i++) {
        $vms[$i] = &VMware::VMServerd::canonicalize($vms[$i]);
    }
    if (! fileInList($config, @vms)) {
	my $err = "Virtual Machine $config is no longer registered.";
	return(0, $err);
    }

    my $regbymui = &VMware::VMServerd::GetOnlyRegisteredByMui($config);
    if (!$regbymui) {
        my $err = "Virtual Machine $config is being managed by others.";
        return(0, $err);
    }

    my $mode = &VMware::VMServerd::GetAccessBits($config);
    if (($mode & 2) != 2) { # No write permission
        return (0, "You do not have permission to delete this VM");
    }
      
    # Make sure the VM is not running
    my $state = VMware::VMServerd::getVMState($config);
    if( $state == $VMSTATE_STUCK ) {
	$state = $VMSTATE_ON;
    }
    
    if( $state == $VMSTATE_ON ) {
	my $err = "Virtual machine \"$config\" is currently " . $VMSTATE_NAMES{$state} .
          ". The virtual machine must be powered off before it can be deleted.";
	return(0, $err);
    }
 
    return(1);
}

# Returns a array of hashes containing information about SCSI and IDE
# devices
#
# ->[]->{PRESENT} := 0 if the disk is not present. 1 if it is
# ->[]->{NAME} := the name of the device.
# ->[]->{VIRTUAL_NAME} := the virtual name for the device relative as seen by the VM.
# ->[]->{DEVICETYPE} := the type of device.
sub getDisksForVM {
    my($config) = @_;
    my @diskList;
    my @buslogic;

    my $cfg = new VMware::Config;
    if( !$cfg->readin($config) ) {
	&VMware::VMServerd::errorAppend("File $config could not be read.");
	return(undef);
    }

    my $vname;

    my @scsi_devices = $cfg->device_list("scsi", '\d:\d+', 1);
    my @ide_devices = $cfg->device_list("ide", '\d:\d+', 1);
    my @device_list = (@scsi_devices, @ide_devices);
    
    foreach my $vname (@device_list) {
	my $present = $cfg->get_bool("$vname.present", undef);
	if( !defined($present) ) {
	    &VMware::VMServerd::errorAppend('$vname in file $config does not have a present attribute set.');
	    next;
	}

	if ($vname =~ /scsi(\d)[^\.]/) {
	    my $bl = $cfg->get("$vname.virtualDev", "useDefault");
	    if ($bl ne "vmxbuslogic") {
		$buslogic[$1] = "buslogic";
	    }
	}	
	    
        # XXX GSX/ESX-ism
	my $nameString;
        $nameString = "name" if (&VMware::VMServerd::isESX());
        $nameString = "fileName" if (&VMware::VMServerd::isGSX());

	my $name = $cfg->get("$vname.$nameString");
	if( !defined($name) ) {
 	    &VMware::VMServerd::errorAppend("Config file $config has a disk " .
                      "$vname, however, no name is specified for it. ".
                      "Config file may be malformed.");

	    next;
	}
	
	if (!isAbsolutePath($name)) {
	  # VM is using relative paths
	  $name = internal_dirname($config) . getFileSeparator() . $name;
	}

	my $deviceType = $cfg->get("$vname.deviceType", "useDefault");
	if ($deviceType eq "useDefault") {
	    if ( ($vname =~ /scsi(\d):\d+/) &&
	         ($buslogic[$1] eq "vmxbuslogic") ) {
	        $deviceType = "vmfs";
	    }
	    else {
	        # Look at devices/disk.c for more details
	        $deviceType = ($vname =~ /scsi/) ? "scsi-hardDisk" : "ata-hardDisk";
	    }
	}
  	if ( !defined($deviceType) ) {
	    &VMware::VMServerd::errorAppend("$vname in file $config does not have a device Type attribute set.");
	    next;
	}

	my $diskInfo = { PRESENT => $present,
		     NAME => $name,
		     VIRTUAL_NAME => $vname,
		     DEVICETYPE => $deviceType };

        if (! diskIsDeletable($diskInfo)) {
	    next;
 	}

	my $cname = &VMware::VMServerd::Disk::getCanonicalName($name, $config);
	if ( !defined($cname)) {
	  &VMware::VMServerd::errorAppend("Cannot get canonical name for $name");
	  next;
	}
	$diskInfo->{CANONICAL_NAME} = $cname;

        if ( ($deviceType ne "vmfs") && 
	     (! &VMware::VMServerd::Disk::diskIsDeletableByUser($cname)) ) {
	    next;
 	}

	push( @diskList, $diskInfo );
    }

    return(@diskList);
}

# Prune out the obvious non-candidates
sub diskIsDeletable($) {
    my ($diskInfo) = @_;

    my $name = $diskInfo->{NAME};
    my $deviceType = $diskInfo->{DEVICETYPE};

    if (($name =~ /cdrom/i) ||
   	($name =~ /\/dev\//)) {
	return 0;    
    }

    if ($deviceType =~ /cdrom/i) {
	return 0;
    }

    # It seems to be a legitimate disk. See if the user can delete it
    if (! &VMware::VMServerd::Disk::diskIsDeletableByUser($name)) {
      return 0;
    }

    return 1;
}


# Builds a hash indicating what disks are being used by which VMs
# Returns a reference to this hash.
#
sub buildDiskUsageHash() {
    my %usageHash;

    # Get the list of all the registered VMs
    my @vms = VMware::VMServerd::VMList::ListAllVMs();

    my $config;
    foreach $config (@vms) {
	my @diskList = getDisksForVM($config);
	my $diskObj;
	foreach $diskObj (@diskList) {
	    my $name = $diskObj->{NAME};

	    my $canonicalName = $diskObj->{CANONICAL_NAME};
	    if( !defined($canonicalName) ) {
		&VMware::VMServerd::errorAppend("Could not get Canonical Name for $name");
		next;
	    }

	    # XXX TODO handle REDO usage counts. This code can be
            # switched on later.
	    # my $cname = $canonicalName;
	    # while (defined ($cname)) {
	    #    if ( !defined($usageHash{$cname})) {
	    #        $usageHash{$cname} = [];
    	    #    }
	    #
	    #    if ( !grep($_ eq $config, @{$usageHash{$cname}}) ) {
	    #        push (@{$usageHash{$cname}}, $config);
	    #    }
	    #    $cname = &VMware::VMServerd::Disk::getREDOParent($cname);
	    # }
            if( !defined($usageHash{$canonicalName}) ) {
	        $usageHash{$canonicalName} = [];
            }
    
            # Ignore duplicates within a config file
            if( grep($_ eq $config, @{$usageHash{$canonicalName}}) ) {
	        next;
            }

	    push( @{$usageHash{$canonicalName}}, $config);
	}
    }

    return(\%usageHash);
}


# Gets the list of VMs that are using a disk.
#
# Input:
#     A reference to the usage hash built by buildDiskUsageHash().
#     A VMFS disk file name.  The disk name accepts both the vmhba and volume name.
#
# Output:
#     An array of config file names
#  
sub getDiskUsage($$$) {
    my($refUsageHash, $vdiskname, $config) = @_;
    my @result;

    # Find the unique canonical name for this disk.
    my $canonicalName = &VMware::VMServerd::Disk::getCanonicalName($vdiskname,
                                                                   $config);
    if( !defined($canonicalName) ) {
	&VMware::VMServerd::errorAppend("Could not get canonical name " .
                                        "for $vdiskname");
	return undef;
    }

    if( defined($refUsageHash->{$canonicalName}) ) {
	@result = @{ $refUsageHash->{$canonicalName} };
    }
    return(@result);
}

# Returns the display name of this VM.
sub getConfigInfo($) {
    my($config) = @_;

    my $cfg = new VMware::Config;
    if( !$cfg->readin($config) ) {
	&VMware::VMServerd::errorAppend("File $config could not be read.");
	return(undef);
    }
    return($cfg->get('displayName'));
}

sub fileInList($@) {
    my ($file, @files) = @_;

    if ($^O =~ /MSWin32/i) {
        return (grep(lc($_) eq lc($file), @files));
    }
    else {
        return (grep($_ eq $file, @files));
    }
}

# Build the list of files to be deleted 
sub getFilesToDelete($) {
    my($config) = @_;
    my @cFilesToDelete = ('nvram', 'vmware.log');
    my @extens = ('.std', '.vmss', '.nvram', '.log', '.vmx.bak', '.cfg.bak', '.png');
    my @files;

    my $fileSeparator = ($^O eq "MSWin32") ? "\\" : "\/";
 
    my $dirname = internal_dirname($config);

    my $file;
    foreach $file (@cFilesToDelete) {
	$file = $dirname . $fileSeparator . $file;
	if( -e $file && -f $file ) {
            if (! fileInList($file, @files)) {
	        push(@files, $file);
            }
	}
    }

    # Look for extensions
    if( $config =~ /^(.+)\.cfg$/ ) {
      my $base = $1;
      foreach my $ext (@extens) {
	my $otherFile = $base . $ext;
	if( -e $otherFile && -f _ ) {
            if (! fileInList($otherFile, @files)) {
	        push(@files, $otherFile);
            }
	}
      }
    }

    if( $config =~ /^(.+)\.vmx$/ ) {
      my $base = $1;
      foreach my $ext (@extens) {
	my $otherFile = $base . $ext;
	if( -e $otherFile && -f _ ) {
            if (! fileInList($otherFile, @files)) {
	        push(@files, $otherFile);
            }
	}
      }
    }

    my $cfg = new VMware::Config;
    if (! $cfg->readin($config)) {
      &VMware::VMServerd::errorAppend("Could not read Config file in getFilesToDelete ");
    }
    else {
      my $nvramName = $cfg->get("nvram", undef);
      if (defined ($nvramName)) {
	if (!grep($_ eq $nvramName, @files) && 
	    (-e $nvramName && -f $nvramName)) {
          if (! fileInList($nvramName, @files)) {
	    push(@files, $nvramName);
          }
	}
      }
      my $logName = $cfg->get("log.fileName", undef);
      if (defined ($logName)) {
	if (!grep($_ eq $logName, @files) &&
	    (-e $logName && -f $logName)) {
          if (! fileInList($logName, @files)) {
	    push (@files, $logName);
          }
	}
      }
    }
	

    return(@files);
}


sub getDeleteTimestamp($$$) {
    my($refUsageHash, $config, $diskNameList) = @_;
    my @vms = ($config);
    my $timestamp = 0;

    my @vm_configs;

    # Build a list of unique VMs that reference the disks of this VM 
    if( defined($diskNameList) ) {
	my @diskNames = @$diskNameList;
	my $diskname;
	foreach $diskname (@diskNames) {
	    @vm_configs = getDiskUsage($refUsageHash, $diskname, $config);
	    
	    foreach my $cfg (@vm_configs) {
		if( !grep($_ eq $cfg, @vms) ) {
		    push(@vms, $cfg);
                }
	    }
	}
    }
	
    foreach my $cfg (@vms) {
	my @stats = stat($cfg);
	if( !defined($stats[9]) ) {
	    &VMware::VMServerd::errorPost("Could not stat() file $cfg to get timestamp.");
	    next;
	}
	my $mtime = $stats[9];
	if( $mtime > $timestamp ) {
	    $timestamp = $mtime;
	}
    }

    return($timestamp);
}


&VMware::VMServerd::addOperation( OPNAME => 'VMDeleteInfo_DeleteVMInfo',
				  PERLFUNC => 'VMware::VMServerd::VMDeleteInfo::DeleteVMInfo_Handler',
				  POLICY => 'authuser');

1;
