#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# VMList.pm
# 
#   Perl wrapper around serverd registry code in C
#

package VMware::VMServerd::VMList;

use strict;
use Carp;

use VMware::VMServerd::AuthPolicy qw(&BEGIN_PRIVILEGED
				     &END_PRIVILEGED);

use Exporter   ();
use vars       qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

BEGIN {
    # set the version for version checking
    $VERSION     = 1.00;
    @ISA         = qw(Exporter);
    @EXPORT_OK   = qw(&becomeRootAndRegister
		      &becomeRootAndUnRegister
		      );
}

############################################################
##
## Enumerate --
##
##   Return a list of registered VMs that the currently 
##   impersonated user has read access to. 
##
## Returns:
##   An array of registered VM config file names.
##
############################################################

sub Enumerate {
  my @vms = &VMware::VMServerd::Enumerate();
  return @vms;
}

############################################################
##
## ListAllVMs --
##
##   Return a list of all registered VMs.
##
## Returns:
##   An array of registered VM config file names.
##
############################################################

sub ListAllVMs {
  my @vms = &VMware::VMServerd::ListAllVMs();
  return @vms;
}

############################################################
##
## becomeRootAndRegister --
##
##   Revert impersonation, if any, and register the given VM.
##
## Returns:
##   Success or error message.
##
############################################################

sub becomeRootAndRegister($) {
    my ($config) = @_;

    BEGIN_PRIVILEGED();
    my ($ok, $err) = &VMware::VMServerd::Register($config);
    END_PRIVILEGED();

    return ($ok, $err);
}


############################################################
##
## becomeRootAndUnRegister --
##
##   Revert impersonation, if any, and unregister the given VM.
##
## Returns:
##   Success or error message.
##
############################################################

sub becomeRootAndUnRegister($) {
    my ($config) = @_;

    BEGIN_PRIVILEGED();
    my ($ok, $err) = &VMware::VMServerd::Unregister($config);
    END_PRIVILEGED();

    return ($ok, $err);
}

############################################################
##
## Register_Handler --
##   exec handler for VM registration
##
## Input format:
##   .config := the config file to be registered.
## 
## Returns:
##   Success or error message.
##
############################################################

sub Register_Handler($$) {
    my $in  = shift();
    my $out = shift();

    &VMware::VMServerd::errorReset();

    my $config = $in->getValue('.config');
    if (!defined($config)) {
      &VMware::VMServerd::errorPost("Register: No config name specified ");
      return 0;
    }

    my ($ok, $err) = &becomeRootAndRegister($config);

    if (!$ok) {
      &VMware::VMServerd::errorPost($err);
    }

    return $ok;
}

############################################################
##
## UnRegister_Handler --
##   exec handler for VM un-registration
##
## Input format:
##   .config := the config file to be unregistered.
## 
## Returns:
##   Success or error message.
##
############################################################

sub UnRegister_Handler($$) {
    my $in  = shift();
    my $out = shift();

    &VMware::VMServerd::errorReset();

    my $config = $in->getValue('.config');
    if (!defined($config)) {
      &VMware::VMServerd::errorPost("Register: No config name specified ");
      return 0;
    }

    my ($ok, $err) = &becomeRootAndUnRegister($config);

    if (!$ok) {
      &VMware::VMServerd::errorPost($err);
    }

    return $ok;
   
}


&VMware::VMServerd::addOperation( OPNAME => 'VMRegister',
				  PERLFUNC => 'VMware::VMServerd::VMList::Register_Handler',
				  POLICY => 'authuser');

&VMware::VMServerd::addOperation( OPNAME => 'VMUnRegister',
				  PERLFUNC => 'VMware::VMServerd::VMList::UnRegister_Handler',
				  POLICY => 'authuser');


1;
