#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################
#
# ExtHelpers.pm
#
# External helper functions.  These should be common to all Perl that VMware
# does, eventually.
#
package VMware::ExtHelpers;
use strict;
use Carp qw(croak cluck);
use Exporter   ();
use vars       qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

BEGIN {
    # set the version for version checking
    $VERSION     = 1.00;
    @ISA         = qw(Exporter);
    @EXPORT_OK   = qw(&LOG 
		      &Warning 
		      &Panic 
                      &round
		      &p_number 
		      &mktmpfile 
		      &System 
		      &System2
                      &SystemWithReset
		      &Debug 
		      $gDEBUG
		      &internal_basename
		      &internal_dirname
		      &internal_echo
		      &getSizeMB
		      &isRootDir
		      &isAbsolutePath
                      &dir_remove_trailing_slashes
                      &absolute_path
                      &remove_dots_from_path
		      &getFileSeparator
		      &getFileSepRegEx
		      &safe_move
		      &assertTainted
		      @BINPATH
		      @gWarnings
		      &DebugDump
		      &getWarnings
		      &clearWarnings
		      $gWarningsQuiet
		      $gPanicCallback
		      &shell_string
		      &unshell_string
		      &intersection_string
		      );
}

use vars qw( @BINPATH $gDEBUG @gWarnings $gWarningsQuiet $gPanicCallback);

# This is a flag to turn on/off debugging.  Debug is off with 0.
$gDEBUG = 
    (defined($ENV{'VMWARE_HCONFIG_DEBUG'} && $ENV{'VMWARE_HCONFIG_DEBUG'} =~ /^[0-9]+$/) ? 
    $ENV{'VMWARE_HCONFIG_DEBUG'} : 0);

# The paths that we use to check for files
@BINPATH = ( '/bin', '/usr/bin', '/usr/X11R6/bin', '/usr/local/bin', '/sbin', '/usr/sbin',
             '/usr/lib/vmware/bin' );

@gWarnings = ();

# Quick hacks to get some features working
$gWarningsQuiet = 0;        # Flag to turn off warnings
$gPanicCallback = undef;    # Callback function that executes when Panic occurs

# Absoluate path to mv.  Stored here as a variable local to this package.
my $mvPath = undef;

# Takes a hash containing info about a command to be resolved and its arguments
#
# Consider the following data structure, an hash where the key is a string describing
# a command and the value contains the command name to be resolved by internal_which()
# and command line arguments to supply the command.
#
# ( 'mv' => ['mv', '-f'],
#   'cp' => ['cp', '-r', '-p', '-a'],
#   'lilo' => ['lilo'] )
#
sub build_helpers(%) {
    my %helperInfo = @_;
    my @helperNames = keys(%helperInfo);

    my %helpers;
    my $name;
    foreach $name (@helperNames) {
	my @args = @{ $helperInfo{$name} };
	my $arg0 = shift(@args);

	$helpers{$name} = internal_which($arg0);

	if( ! $helpers{$name} ) {
#	    Warning("Helper $name could not be found.");
	    next;
	}

	$helpers{$name} = shell_string($helpers{$name});
	while( $#args > -1 ) {
	    my $arg = shift(@args); 
	    $helpers{$name} .= ' ' . shell_string($arg);
	}
    }
    return(%helpers);
}

# Finds and resolves helpers.  
sub verify_helpers(@) {
    my @names = @_;
    my $count = -1;
    my %helpers;

    while( $#names > -1 ) {
	my $name = shift(@names);
	my $abspath = internal_which($name);

	if( $abspath eq '' ) {
#	    Warning("Could not locate external helper '$name'.");
	    next;
	} 

	$count++;
	$helpers{$name} = $abspath;
    }

    return($count, %helpers);
}

# Takes a list of files and makes sure that they all exist and are
# readable.  Returns the number that exist.
sub verify_files(@) {
    my @files = @_;
    my $count = -1;
    my @vfiles;

    while( $#files > -1 ) {
	my $f = shift(@files);
        if( file_name_exist($f) ) {
	    push(@vfiles, $f);
	    $count++;
	} else {
#	    Warning("Could not locate file '$f'.");
	    next;
	} 
    }

    return($count, @vfiles);
}

# Takes a list of files and makes sure that they all exist and
# are readable and accessible.  Returns the number that exist.
sub verify_dirs(@) {
    my @dirs = @_;
    my $count = -1;
    my @vdirs;

    while( $#dirs > -1 ) {
	my $d = shift(@dirs);
        if( dir_name_exist($d) ) {
	    push(@vdirs, $d);
	    $count++;
	} else {
#	    Warning("Could not locate dir '$d'.");
	    next;
	} 
    }

    return($count, @vdirs);
}

# Gets all the helper names that have been initialized
#sub get_helper_names() {
#    return(keys(%gExtHelpers));
#}

# Gets a fully qualified external helper program
# Returns undef if the helper does not exist.
#sub get_helper($) {
#    my $name = shift;
#    return($gExtHelpers{$name});
#}

# Contrary to a popular belief, 'which' is not always a shell builtin command.
# So we can not trust it to determine the location of other binaries.
# Moreover, SuSE 6.1's 'which' is unable to handle program names beginning with
# a '/'...
#
# Return value is the complete path if found, or '' if not found
sub internal_which {
    my $bin = shift;
    if (substr($bin, 0, 1) eq '/') {
	# Absolute name
	if ((-f $bin) && (-x $bin)) {
	    return $bin;
	}
    } else {
	# Relative name
	my @paths = @BINPATH;
	my $path;
	
	if (index($bin, '/') == -1) {
	    # There is no other '/' in the name
	    foreach $path (@paths) {
		my $fullbin;
		
		$fullbin = $path . '/' . $bin;
		if ((-f $fullbin) && (-x $fullbin)) {
		    return $fullbin;
		}
	    }
	}
    }
    
    return '';
}


# Check if a file name exists and we can read it
sub file_name_exist {
  my $file = shift;
  return (-f $file && -r $file)
}

# Check if a dir name exists and we can access it
sub dir_name_exist {
  my $dir = shift;
  return (-d $dir && -x $dir && -r $dir)
}

# Remove trailing slashes in a dir path
sub dir_remove_trailing_slashes {
  my $path = shift;
  my $fileSeparator = ($^O eq "MSWin32") ? '\\' : '/';

  for(;;) {
    my $len;
    my $pos;

    $len = length($path);
    if ($len < 2) {
      # Could be '/' or any other character. Ok.
      return $path;
    }

    $pos = rindex($path, $fileSeparator);
    if ($pos != $len - 1) {
      # No trailing slash
      return $path;
    }

    # Remove the trailing slash
    $path = substr($path, 0, $len - 1)
  }
}

# Emulate a simplified dirname program
sub internal_dirname {
  my $path = shift;
  my $pos;
  my $fileSeparator = ($^O eq "MSWin32") ? '\\' : '/';


  $path = dir_remove_trailing_slashes($path);

  $pos = rindex($path, $fileSeparator);
  if ($pos == -1) {
    # No slash
    return '.';
  }

  if ($pos == 0) {
    # The only slash is at the beginning
    return $fileSeparator;
  }

  return substr($path, 0, $pos);
}

# Emulate a simplified basename program
sub internal_basename {
    my $path = shift;
    my $pos;
    my $fileSeparator = ($^O eq "MSWin32") ? '\\' : '/';
    
    $path = dir_remove_trailing_slashes($path);
    
    $pos = rindex($path, $fileSeparator);
    if ($pos == -1) {
	# No slash
	return $path;
    }
    
    return substr($path, $pos + 1);
}

# Emulate a simplified echo program
sub internal_echo {
   my $path = shift;
   my $text = shift;

   local *FD;
   if (!open(FD, "> $path")) {
      return 0;
   }
   print FD $text;
   close(FD);
   return 1;
}

# Returns a given size string by converting it to MegaBytes. 
# The size string could be a Big Integer (i.e., larger than 2GB)
# Useful for getting size of large COW and VMFS disks
sub getSizeMB($) {
    my ($size) = @_;

    my $sizeMB = int(($size - 1) / (1024*1024)) + 1;      # Size, rounded _up_ to the next MB
    
    if ($sizeMB >= 0) {
	return sprintf("%lu", $sizeMB);
    }

    # $size is greated then 2^31
    my $bigint = Math::BigInt->new(sprintf("%lu", $size));
    my $bigintMB = $bigint->bdiv(1024*1024);
    $bigintMB =~ /(\d+)/;
    
    return $1;
}

# Returns 1 if root directory, 0 otherwise
sub isRootDir($) {
    my ($path) = @_;

    if (!defined($path)) {
	return 0;
    }

    if ($^O =~ /linux/i) {
	if ($path =~ /^\/$/) {
	    return 1;
	}
	return 0;
    }

    if ($^O =~ /MSWin32/i) {
	if ( ($path =~ /^\\$/) ||
	     ($path =~ /^[A-Za-z]:\\$/) ) {
	    return 1;
	}
	return 0;
    }

    Panic("Unknown OS not supported ");
    return 0;
}

# Returns 1 if it is an absolute path, 0 otherwise
sub isAbsolutePath($) {
    my ($path) = @_;

    if ($^O =~ /linux/i) {
	if ($path =~ /^\//) {
	    return 1;
	}
	return 0;
    }

    if ($^O =~ /MSWin32/i) {
	if ( ($path =~ /^\\/) ||
	     ($path =~ /^[A-Za-z]:\\/) ) {
	    return 1;
	}
	return 0;
    }

    Panic("Unknown OS not supported ");
    return 0;
}


# Converts relative path to absolute path, if necessary.
sub absolute_path($$) {
    my ($dname, $relativeRoot) = @_;

    # If the path is already absolute, keep it that way
    if (($dname =~ /^\//) || ($dname =~ /^\\/)) {
      return $dname;
    }
    if (($^O eq "MSWin32") && ($dname =~ /^[a-zA-Z]:/)) {
      return $dname;
    }

    my $dirname = dir_remove_trailing_slashes($relativeRoot);
    my $basename= $dname;
    my $fileSeparator = ($^O eq "MSWin32") ? '\\' : '/';
    
    return $dirname . $fileSeparator . $basename;
}

# Removes '..' and '.' from directory path name. The '..'s that attempt
# to take the path beyond the root directory are ignored.
# So /foo/bar/../stuff becomes /foo/stuff, and
# \uno\dos\..\..\..\tres becomes \tres
sub remove_dots_from_path($) {
    my ($path) = @_;

    my $osIsWindows = $^O eq "MSWin32";
    my @dirs = $osIsWindows ? split(/\\/, $path) : split(/\//, $path);
    my @newdirs;
    foreach my $d (@dirs) {
        next if ($d eq "\.");
        if ($d eq "\.\.") {
	  next if (scalar(@newdirs) <= 1);
	  pop (@newdirs);
	  next;
	}
	push (@newdirs, $d);
      }
    my $val = ord('\\');
    my $newpath = $osIsWindows ? join('\\', @newdirs) : join('/', @newdirs);

    return $newpath;
}

# Returns the file separator depending on the OS
sub getFileSeparator() {
    if ($^O =~ /linux/i) {
	return "/";
    }

    if ($^O =~ /mswin32/i) {
	return "\\";
    }

    return undef;
}

# Returns the file separator regular expression
sub getFileSepRegEx() {
    if ($^O =~ /linux/i) {
	return "/";
    }

    if ($^O =~ /mswin32/i) {
	return "\\\\";
    }

    return undef;
}

# Moves a file, preserving its mode
sub safe_move {
    my($src, $dest) = @_;

    # Get the mode of the destination file if it exists.
    my $mode = undef;
    my @stats = stat($dest);
    if( defined($stats[2]) ) {
	$mode = $stats[2] & 07777;
    }

    if( !defined($mvPath) || $mvPath eq '' ) {
	# Initialize the absolute path to mv
	$mvPath = internal_which('mv');
	if( $mvPath eq '' ) {
	    Warning("mv utility could not be found in path.");
	    return(0);
	}
    }

    # Change the mode of the source file into that of the destination.
    if( defined($mode) && !chmod($mode, $src) ) {
	Warning("Could not chmod file $src to mode " . sprintf("%o", $mode) . ".  Reason: $!.");
	return(0);
    }

    if( System(shell_string($mvPath) . ' ' . shell_string('-f') . ' ' . 
	       shell_string($src) . ' ' . shell_string($dest)) ) {
	return(0);
    }
    return(1);
}

# Round a number to the nearest integer
sub round($) {
   my $num = shift;
   return int($num + .5);
}

# Pretty print a number
sub p_number($) {
    my ($num) = @_;
    my $neg = "";
    my $str = "";

    my $count = 0;

    $num = int($num);
    if( $num < 0 ) {
	$neg = "-";
	$num = -$num;
    } elsif( $num == 0 ) {
	return("0");
    }

    while( $num > 0 ) {
	if( $count % 3 == 0 && $count != 0 ) {
	    $str = "," . $str;
	}

	$str = sprintf("%d", ($num % 10)) . $str;
	if( $num > 9 ) {
	    $num = ($num - ($num % 10))  / 10;
	} else {
	    last;
	}
	$count++;
    }
    $str = $neg . $str;

    return($str);
}

# Quick hack
sub mktmpfile() {
    local *FD;
    my $tmpbase = "/tmp/exthelpers.tmpfile." . $$;
    my $tmpfile;

    if( ! (-w "/tmp" && -d "/tmp") ) {
	Warning("Could not create file in /tmp directory: $!\n");
	return(undef);
    }

    $tmpfile = $tmpbase;
    my $i = 0;
    while( -e $tmpfile ) {
	$tmpfile = "$tmpbase-$i";
	$i++;
    }

    # Very restrictive umask to prevent other users from seeing what's
    # in the file.
    my $mask = umask();
    umask(00177);

    if( !open(FD, "> $tmpfile") ) {
	umask($mask);
	return("");
    }

    close(FD);

    umask($mask);

    return($tmpfile);
}

sub System($) {
    my ($cmd) = @_;
    LOG(0, $cmd);
    return( system($cmd . " > /dev/null 2>&1") );
}

sub System2($) {
    my ($cmd) = @_;
    LOG(0, $cmd);
    return( system($cmd) );
}


#####################################################################
#
# SystemWithReset --
#
#      Runs the system in a seperate process where the process state
#      is cleaned
# 
# Results
#      Returns error code for the program
#
#####################################################################

sub SystemWithReset($) {
    my @commands = @_;
    my $pid = fork();
    my $rc = 0;
    if( !defined($pid) ) {
        return(127);
    }

    if( $pid == 0 ) {
        # Child
        VMware::HConfig::ResetProcessState();
        exec(@commands);
        exit(127);
    }

    # Parent
    if( waitpid($pid, 0) == -1 ) {
        return(-1);
    }

    # Set the error value
    $! = $? >> 8;
    return($?);
}


sub LOG {
    my $level = shift;
    my @error = @_;
    my ($package, $filename, $line, $subroutine) = caller(1);
    print STDERR $subroutine . '[' . $line . ']: ' . "@_\n";
}

sub Debug {
    my $level = shift;
    my @message = @_;

    if( $gDEBUG ) {
	if( $gDEBUG >= $level ) {
	    my ($package, $filename, $line, $subroutine) = caller(1);
	    $filename = internal_basename($filename);
	    my $str = $filename . ' [' . ($line) . '] ' . $subroutine . ' -- ' . "@message\n";
	    print STDERR $str;
	}
    }
}

sub Warning {
    my @error = @_;
    my ($package, $filename, $line, $subroutine) = caller(1);
    $filename = internal_basename($filename);
    my $str = 'Warning in ' . $filename . ' [' . ($line) . '] ' . $subroutine . ' -- ' . "@error\n";

    push(@gWarnings, @error);
    if( !$gWarningsQuiet ) {
	print STDERR $str;
    }
}

sub Panic {
    my @error = @_;
    my ($package, $filename, $line, $subroutine) = caller(1);
    $filename = internal_basename($filename);
    my $str = 'Error in ' . $filename . ' [' . ($line) . '] ' . $subroutine . ' -- ' . "@error\n";
    if( defined($gPanicCallback) ) {
	&{$gPanicCallback}($str);
    }
    croak($str . 'Panic: Terminating.' . "\n");
    exit(1);
}

sub getWarnings() {
    return(@gWarnings);
}

sub clearWarnings() {
    undef(@gWarnings);
}


# Debug function to test for taintedness of an expression
sub assertTainted {
    my $exp = join('',@_);

    my $taint = ! eval {
	my $junk = join('',@_), kill 0;
	1;
    };

    if( $taint ) {
	Panic("Expression '$exp' is tainted");
    }
}

sub indent($) {
    my($level) = @_;
    if( !defined($level) ) {
	return("");
    }
    my $INDENT = " ";
    my $s = "";
    while($level-- > 0) {
	$s .= $INDENT;
    }
    return($s);
}

# Convert a string to its equivalent shell representation
sub shell_string {
  my $single_quoted = shift;

  $single_quoted =~ s/'/'"'"'/g;
  # This comment is a fix for emacs's broken syntax-highlighting code --hpreg
  return '\'' . $single_quoted . '\'';
}

# Unshell a converted string
sub unshell_string {
  my $single_quoted = shift;

  $single_quoted =~ s/^\'(.*)\'$/$1/;
  return($single_quoted);
}

# Takes in two arrays of strings and produces the intersection of the two.
sub intersection_string {
    my($a, $b) = @_;
    my @new;

    foreach my $s (@$a) {
	if (grep($s eq $_, @$b)) {
	    push(@new, $s);
	}
    }
    return(@new);
}

sub DebugDump {
    my ($data, $indent) = @_;
    if( !defined($indent) ) {
	$indent = 0;
    }
    my $s = indent($indent);

    my $d = $data;
    if( ref($d) ) {
	if( UNIVERSAL::isa($d, 'HASH') ) {
	    $s .= indent($indent) . "(REF-Ha ";
	    my($key, $value);
	    my $first = 1;
	    while( (($key, $value) = each %$d) ) {
		if( $first ) {
		    $s .= DebugDump($key, 0) . " => " . DebugDump($value, 0);
		    $first = 0;
		} else {
		    $s .= ",\n";
		    $s .= DebugDump($key, $indent + 8) . " => " . DebugDump($value, 0);
		}
	    }
	    $s .= "\n" . indent($indent) . ")";
	} elsif( UNIVERSAL::isa($d, 'ARRAY') ) {
	    $s .= indent($indent) . "(REF-Ar ";
	    my $first = 1;
	    foreach my $e (@$d) {
		if( $first ) {
		    $s .= DebugDump($e, 0);
		    $first = 0;
		} else {
		    $s .= ",\n";
		    $s .= DebugDump($e, $indent + 8);
		}
	    }
	    $s .= "\n" . indent($indent) . ")";
	} elsif( UNIVERSAL::isa($d, 'SCALAR') ) {
	    $s .= "(REF-Sc " . ($$d) . ")";
	} else {
	    $s .= "(REF-" . ref($d) . " " . DebugDump($$d) . ")";
	}
    } else {
	$s .= "(Sc " . $d . ")";
    }

    return($s);
}

1;

