#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################
#
# Host.pm
# 
# A class for gathering and manipulating information about the
# host. This is a class common to GSX and ESX. SSHost.pm implements
# ESX-specific class.

package VMware::HConfig::Host;
use strict; 

use VMware::HConfig::CommonConstant qw(c_getHelper);
use VMware::ExtHelpers qw(&Warning &internal_dirname &shell_string);

use Exporter   ();
use vars       qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

BEGIN {
    # set the version for version checking
    $VERSION     = 1.00;
    @ISA         = qw(Exporter);
    @EXPORT      = qw(&sameFileSystem
		      );

}


# Get information about the file system that the given path resides in
# Information returned is:
#   fsName: Name of the device 
#   fsCapacity: Total size of the file system in MB
#   fsUsedSpace: Used space in MB
#   fsFreeSpace: Available space in MB
#   fsMountPoint: mount point for the file system
sub getFileSystemInfo($) {
    my ($path) = @_;
    my $hash = undef;
    my $line;

    my $command = c_getHelper('df') . " " . shell_string($path);

    if ( !open(SRC, "$command |") ) {
	Warning("Command " . c_getHelper('df') . " failed. Reason: $!");
	return undef;
    }

    while ($line = <SRC>) {
	chomp($line);
	if ($line !~ /Filesystem/i) {
	    $hash = {};
	    $line =~ /^(.*?)\s+(\d+)\s+(\d+)\s+(\d+)\s+(.*?)\s+(.*?)/;
	    $hash->{fsName} = $1;
	    $hash->{fsCapacity} = int($2) / 1024;
	    $hash->{fsUsedSpace} = int($3) / 1024;
	    $hash->{fsFreeSpace} = int($4) / 1024;
	    $hash->{fsMountPoint}  = $6;
	}
    }
    close(SRC);
    
    return $hash;
}


# Check to see if the given two files/directories are on the
# same file system. Returns 1 if they are, 0 if not and 
# -1 in case of error.
sub sameFileSystem($$) {
   my ($src, $dest) = @_;
   my $line;
   my $srcFS = undef;
   my $destFS = undef;

   if ( (!(defined($src) && defined($dest))) ) {
       return -1;
   }

   my $destFile = (-e $dest) ? $dest : internal_dirname($dest);

   if (! (-e $src && -e $destFile) ) {
       return -1;
   }

   my $srcFSInfo = getFileSystemInfo($src);
   my $destFSInfo = getFileSystemInfo($destFile);

   if (defined($srcFSInfo) && defined($destFSInfo)) {
      return ($srcFSInfo->{fsName} eq $destFSInfo->{fsName});
   }

   return -1;
}

1;
