#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# AuthPolicy.pm
# 
#   Module that implements the vmserverd handler execution policy 
#   We have three policies:
#     authuser:    The handler function is run as the authenticated user
#     setuid_root: The handler function is run as superuser, irrespective of who the authenticated user is
#     rootonly   : The handler function is run as superuser, only if authenticated as superuser
#
#   authuser and rootonly are implemented for Linux and Windows
#   setuid_root is implemented for Linux only (and it just verifies euid is root).
#

package VMware::VMServerd::AuthPolicy;
use strict;
use VMware::VMServerd qw(&Log
			 &Warning
			 &Panic
			 &GetUID
			 &GetEUID
			 &GetGID
			 &GetEGID
			 &GetAccessBits
			 );
			 

use Exporter   ();
use vars       qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

BEGIN {
    # set the version for version checking
    $VERSION     = 1.00;
    @ISA         = qw(Exporter);
    @EXPORT_OK   = qw(&doPolicy
		      &undoPolicy
		      &BEGIN_PRIVILEGED
		      &END_PRIVILEGED
		      &isVMReadable
		      &isVMExecutable
		      &isVMWritable
		      );
}

my %gDoPolicyHandlers = (
    'authuser' => [\&doPolicyAuthuser, \&undoPolicyAuthuser],
    'setuid_root' => [\&doPolicySetuidRoot, \&undoPolicySetuidRoot],
    'rootonly' => [\&doPolicyRootOnly, \&undoPolicyRootOnly],
);

my $gDefaultPolicy = "authuser";

# The do policy dispatch function
sub doPolicy($$) {
    my ($policy, $username) = @_;

    if( !defined($policy) ) {
	&VMware::VMServerd::Warning("doPolicy: No policy specified. Using " . $gDefaultPolicy . " \n");
	$policy = $gDefaultPolicy;
    }
    my $policyHandler = $gDoPolicyHandlers{$policy};
    if( !defined($policyHandler) ) {
	&VMware::VMServerd::Warning("doPolicy: Unknown policy '$policy' encountered.\n");
	return(0);
    }

    &VMware::VMServerd::Log("doPolicy: Using access control policy $policy\n");

    my $func = $gDoPolicyHandlers{$policy}->[0];
    return &$func($username);
}

# The undo policy dispatch function
sub undoPolicy($$) {
    my ($policy, $username) = @_;

    if( !defined($policy) ) {
	&VMware::VMServerd::Warning("undoPolicy: No policy specified. Using " . $gDefaultPolicy . " \n");
	$policy = $gDefaultPolicy;
    }
    my $policyHandler = $gDoPolicyHandlers{$policy};
    if( !defined($policyHandler) ) {
	&VMware::VMServerd::Warning("undoPolicy: Unknown policy '$policy' encountered.\n");
	return(0);
    }

    &VMware::VMServerd::Log("undoPolicy: Was using access control policy $policy\n");

    my $func = $gDoPolicyHandlers{$policy}->[1];
    return &$func($username);
}

# authuser policy
#
# Impersonate the authenticated user
sub doPolicyAuthuser($) {
    my ($username) = @_;

    &VMware::VMServerd::GoUser();

    # Note that the Perl special variables, "$>", "$)" and "$(" do not get 
    # modified in call to GoUser(). You should not rely on them in Perl code,
    # instead use GetEUID(), GetEGID() and GetGID() C functions respectively.

    # Also, the Perl %ENV hash is not modified by the above call either. So,
    # do not rely on $ENV{HOME} etc., use the GetEnvVariable() function in C.

    return(1);
}


# authuser policy
#
# Revert back from impersonating the authenticated user
sub undoPolicyAuthuser($) {
    my ($username) = @_;

    &VMware::VMServerd::RevertUser();

    return(1);
}

# root only policy
#
# The handler can run only if the authenticated user is root/Administrator
sub doPolicyRootOnly($) {
    my ($username) = @_;

    &VMware::VMServerd::GoUser();
    if (!&VMware::VMServerd::IsAdminUser()) {
       &VMware::VMServerd::RevertUser();
       return 0;
    }
    return 1;
}

# root only policy
#
sub undoPolicyRootOnly($) {
    my ($username) = @_;
    &VMware::VMServerd::RevertUser();
}

# setuid root policy
#
# Run the handler as super user
sub doPolicySetuidRoot($) {
    my ($username) = @_;

    # This policy is not used on Windows
    if ( $^O eq "MSWin32" ) {
        if (! &VMware::VMServerd::IsAdminUser()) {
            &VMware::VMServerd::Panic("doPolicySetuidRoot: vmserverd running as non-root not implemented");
            return(0);
        }
    }

    # Nothing to be done, since vmserverd runs as root. 
    if ($^O eq "linux") {
	if (&VMware::VMServerd::GetEUID() != 0) {
	    &VMware::VMServerd::Panic("doPolicySetuidRoot: vmserverd running as non-root not implemented");
	    return (0);
	}
    }

    return(1);
}

# setuid root policy
#
# Revert back from running as super user
sub undoPolicySetuidRoot($) {
    my ($username) = @_;

    # This policy is not used on Windows
    if ( $^O eq "MSWin32" ) {
        if (! &VMware::VMServerd::IsAdminUser()) {
            &VMware::VMServerd::Panic("undoPolicySetuidRoot: vmserverd running as non-root not implemented");
            return(0);
        }
    }

    # Nothing to be done, since vmserverd runs as root. 
    if ($^O eq "linux") {
	if (&VMware::VMServerd::GetEUID() != 0) {
	    &VMware::VMServerd::Panic("undoPolicySetuidRoot: vmserverd running as non-root not implemented");
	    return (0);
	}
    }

    return(1);
}

my $gUserReverted = undef;

sub IN_PRIVILEGED_SECTION() {
    return (defined($gUserReverted));
}

sub BEGIN_PRIVILEGED()
{
    if (IN_PRIVILEGED_SECTION()) {
	&VMware::VMServerd::Panic("Nested PRIVILEGED sections");
    }

    $gUserReverted = &VMware::VMServerd::RevertUser();

    if (!IN_PRIVILEGED_SECTION()) {
	&VMware::VMServerd::Panic("RevertUser returned undefined value");
    }
}

sub END_PRIVILEGED() {
    if (!IN_PRIVILEGED_SECTION()) {
	&VMware::VMServerd::Panic("END_PRIVILEGED called outside of PRIVILEGED section");
    }

    if ($gUserReverted) {
	&VMware::VMServerd::GoUser();
    }
    $gUserReverted = undef;

    # The following is redundant, but symmetry is good. Also, helps keep
    # IN_RPIVILEDGED_SECTION honest.
    if (IN_PRIVILEGED_SECTION()) {   
	&VMware::VMServerd::Panic("Still inside priviledged section");
    }
}

# Class 2 access
sub isVMReadable($) {
    my ($config) = @_;

    my $bits = &VMware::VMServerd::GetAccessBits($config);
    return (($bits & 0x4) == 0x4);
}

# Class 1 access
sub isVMExecutable($) {
    my ($config) = @_;

    my $bits = &VMware::VMServerd::GetAccessBits($config);
    return (($bits & 0x5) == 0x5);
}

# Class 0 access
sub isVMWritable($) {
    my ($config) = @_;

    my $bits = &VMware::VMServerd::GetAccessBits($config);
    return (($bits & 0x7) == 0x7);
}

1;



