#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################
#
# FileManager.pm
#
# VMServerd wrappers for file and directory management operations
#
package VMware::VMServerd::FileManager;

use strict;
use VMware::DOMAccess;
use VMware::Config;
use VMware::ExtHelpers qw(&getWarnings
			  &clearWarnings
			  &System
			  &System2
                          &SystemWithReset
			  &shell_string
			  &Debug
			  &isRootDir
			  &isAbsolutePath
			  &internal_basename
			  &internal_dirname
			  &dir_remove_trailing_slashes
			  &shell_string
			  &getSizeMB
			  &getFileSeparator
			  &getFileSepRegEx
			  );

use VMware::VMServerd qw(&errorReset
			 &haveError
			 &errorAppend
			 &errorPost
			 &Log
			 &Panic
			 &Warning);

use VMware::VMServerd::TaskManager;
use VMware::VMServerd::FileInfo qw(&fileExists
				   &isFile
				   &isDirectory
				   &getFileSize
				   &getMode
				   &isSameFS
				   &isVmfsFile
				   &getVmfsFileInfo
				   &getExt2FileInfo
				   &removeVmfsFile
				   &isVmfsMountPoint
				   &enableVmfsFileCaching
				   &disableVmfsFileCaching
				   );
use VMware::VMServerd::Disk;

use File::Find;
use File::Copy;

#############################################################
# Globals specific to this serverd module.
#############################################################

use Exporter   ();
use vars       qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

sub S_IRUSR     () { 00400 }    # owner  has  read    permission
sub S_IWUSR     () { 00200 }    # owner  has  write   permission
sub S_IXUSR     () { 00100 }    # owner  has  execute permission
sub S_IRGRP     () { 00040 }    # group  has  read    permission
sub S_IWGRP     () { 00020 }    # group  has  write   permission
sub S_IXGRP     () { 00010 }    # group  has  execute permission
sub S_IROTH     () { 00004 }    # others have read    permission
sub S_IWOTH     () { 00002 }    # others have write   permission
sub S_IXOTH     () { 00001 }    # others have execute permission

BEGIN {
    if ($VMware::VMServerd::PRODUCT =~ /ESX/i) {
        require VMware::VMServerd::ESXHostInfo;
    }

     # set the version for version checking
    $VERSION     = 1.00;
    @ISA	 = qw(Exporter);

    @EXPORT = qw(&setLargeFileSizeThreshold
		 &FileManager_FileExists
		 &FileManager_ExecuteProgram
		 &FileManager_CanonicalPath
		 );

}

my %gSystemCommands = ( "mv"      => "/bin/mv",
			"cp"      => "/bin/cp",
                        "df"      => "/bin/df"
			);

my $PKG_FULL_NAME;
my $PKG_NAME;
my $LARGE_FILE_SIZE_THRESHOLD = 10;   # In MB
my $gRecursiveDeleteStatus;
my $gListDirTimeout = 15;  # seconds
my $gListDirMaxNumFiles = 256;

####################################################################################
#
# External API functions
#
####################################################################################

sub setLargeFileSizeThreshold($) {
    my ($size) = @_;

    if ($size > 0) {
	$LARGE_FILE_SIZE_THRESHOLD = $size;
    }
}

sub FileManager_ListDir($$) {
    my ($directory, $dirOnly) = @_;

    my $startTime = time();
    my $diskFreeSpace = 0;

    if (!defined($directory)) {
	return (0, "No directory specified in ListDir");
    }

    my $tdir = resolveTildeUser($directory);

    if (!defined($tdir)) {
	return (0, "\"$directory\" is not a valid home directory.");
    }

    $directory = $tdir;
    if (!isAbsolutePath($directory)) {
	return (0, "Directory $directory is not an absolute path");
    }

    if (! -d $directory) {
	return (0, "Directory $directory does not exist or is not accessible");
    }

    my $retfiles = [];

    my $isVmfs = isVmfsMountPoint($directory);

    if ($isVmfs && $dirOnly) {
	return (1, "", FileManager_CanonicalPath($directory), $retfiles);
    }

    # Determine free and used space on disk
    if (!$isVmfs) {
        my $command = $gSystemCommands{"df"} . ' -P -k ' . shell_string($directory);
        local *FD;
        if (!open(FD, "$command |") ) {
           return (0, "Could not determine free disk space in $directory. Reason: $!");
        }

        while (<FD>) {
           chomp();
           if (!/^Filesystem\s*/) {
              if (/\S+\s*\d+\s*\d+\s*(\d+)\s*/) {
                # We got the information we were looking for
                $diskFreeSpace = $1 * 1024; # Standardize on bytes.
              } else {
                 return (0, "Failed to parse output of df");
              }
           }
        }
    } else {
       if (&VMware::VMServerd::isESX()) {
           # determine free space on vmfs. The output format for this is in blocks though
           my $dirpart;
           if ($directory =~ /^\/vmfs\/(\S+)/) {
               $dirpart = $1;
           } else {
               return (0, "Directory not in proper format for vmfs drive: $directory\n");
           }
           my $vmfs = VMware::VMServerd::ESXHostInfo::getESXHost()->getMFSLookup($dirpart);
           if (!defined($vmfs)) {
               return (0, "Could not lookup vmfs information for $directory");
           }

           $diskFreeSpace = $vmfs->getFreeCapacity() * 1024 * 1024;
       }
   }

    if( !opendir(DIR, $directory) ) {
	return (0, "Could not open directory $directory for reading");
    }
    my @allfiles = readdir(DIR);
    closedir DIR;

    # @filesRelativeNames is relative path names, relative to $directory
    # @files, below, is absolute path names

    my @filesRelativeNames = cleanupFilesList(@allfiles);

    $directory = dir_remove_trailing_slashes($directory);

    my @files;
    for (my $i=0; $i <= $#filesRelativeNames; $i++) {
	my $file = (isRootDir($directory) ? $directory : ($directory . getFileSeparator())) .
	     $filesRelativeNames[$i];
        if (!&VMware::VMServerd::Disk::isCOWDiskMember($file)) {
            push(@files, $file);
        }
    }

    my $numTotalDir = 0;
    my $numTotalFiles = 0;

    if ($isVmfs) {
        $numTotalDir = 0;
        $numTotalFiles = scalar(@files);
    } else {
        for (my $i=0; $i < scalar(@files); $i++) {
            if ( -d $files[$i] ) {
                $numTotalDir++;
            } else {
                $numTotalFiles++;
            }
        }
    }
    if ($isVmfs) {
	enableVmfsFileCaching();
    }
    my $numFilesToShow = (scalar(@files) > $gListDirMaxNumFiles) ?
                         $gListDirMaxNumFiles : scalar(@files);
    for (my $i=0; $i < $numFilesToShow; $i++) {
	my $f = $files[$i];
	my $hash = {};

	if ($isVmfs) {
	    $hash = getVmfsFileInfo($f);
	} else {
	    $hash = getExt2FileInfo($f, $dirOnly);
	}

        if (defined($hash)) {
            push (@$retfiles, $hash);
        }

        if (time() > ($startTime + $gListDirTimeout)) {
            last;
        }

    }
    if ($isVmfs) {
	disableVmfsFileCaching();
    }

    return (1, "", $numTotalDir, $numTotalFiles, $diskFreeSpace,
            FileManager_CanonicalPath($directory), $retfiles);
}

sub FileManager_ListDirTree($) {
    my ($directory) = @_;

    if (!defined($directory)) {
	return (0, "No directory specified in ListDir");
    }

    my $tdir = resolveTildeUser($directory);

    if (!defined($tdir)) {
	return (0, "\"$directory\" is not a valid home directory.");
    }

    $directory = $tdir;

    if (!isAbsolutePath($directory)) {
	return (0, "Directory $directory is not an absolute path");
    }

    my $done = 0;
    my @ret;
    my @dirComponents  = undef;

    @dirComponents = split(getFileSepRegEx(), $directory);

    if (scalar(@dirComponents) == 0) {
	# Possible on linux for root directory ("/"). Not a problem on windows.
	push (@dirComponents, "");
    }

    my $dirstr = undef;
    my $dirsref;
    my $prevdir = undef;
    my $rethash = {};
    for (my $i=0; $i < scalar(@dirComponents); $i++) {
	my $ok;
	my $err;
	my $relativeRoot;

	# First component (root dir) gets special treatment, because the file separator comes _after_
	# the component string ("blank" in case of Linux, "<drive>:" in case of Windows), i.e.,
	# "/" for linux and "<drive>:\" for windows.
	# For all other directories, file separator comes before appending the next component. This
	# way we avoid having a trailing fileSeparator.
	if ($i == 0) {
	    $dirstr = $dirComponents[0] . getFileSeparator();
	} else {
	    $dirstr .= getFileSeparator() . $dirComponents[$i];
	}

	my $n1; # ignored
        my $n2; # ignored
        my $n3; # ignored
        ($ok, $err, $n1, $n2, $n3, $relativeRoot, $dirsref)
          = FileManager_ListDir($dirstr, 1);
	if (!$ok) {
	    return (0, $err);
	}

	if (isRootDir($dirstr)) {
	    $rethash->{name} = $dirstr;
	    $rethash->{abspath} = $dirstr;
	    $rethash->{type} = "directory";
	    $rethash->{dirs} = $dirsref;
	}

	if ($i == 0) {
	    # Prepare the $dirstr for subsequent append of fileseparator and next component.
	    $dirstr = $dirComponents[0];
	}

	if (defined($prevdir)) {
	    $prevdir->{dirs} = $dirsref;
	}

	for my $d (@$dirsref) {
	    if (defined($dirComponents[$i+1]) && ($d->{name} eq $dirComponents[$i+1])) {
		$prevdir = $d;
		last;
	    }
	}

    }

    return (1, "", $rethash);
}

sub FileManager_MakeDir($$$$@) {
    my ($owner_mode, $group_mode, $world_mode, $makeParents, @dirs) = @_;

    my ($ok, $err, $mode) = octalMode($owner_mode, $group_mode, $world_mode);
    if (!$ok) {
	return (0, $err);
    }

    if (!defined($mode)) {
	$mode = 0777;    # The mode of the created directory is (mode & ~umask)
    }

    my $errStr = undef;
    for my $d (@dirs) {
	if (!isAbsolutePath($d)) {
	    $errStr .= "Directory $d is not absolute path;";
	    next;
	}

	($ok, $err) = createDir($d, $mode, $makeParents);
	if (!$ok) {
	    $errStr .= $err;
	}
    }

    if (defined($errStr)) {
	return (0, $errStr);
    }

    return 1;
}

sub FileManager_RemoveDir(@) {
    my (@dirs) = @_;

    my $errstr = undef;
    for my $d (@dirs) {
	if (!isAbsolutePath($d)) {
	    $errstr .= "Directory $d is not absolute path;";
	    next;
	}

	my $ok = doRemoveDir($d);
	if (!$ok) {
	    if( !opendir(DIR, $d) ) {
		$errstr .= "Directory $d to delete is not valid;";
		next;
	    }
	    my @allfiles = readdir(DIR);
	    closedir DIR;

	    my @files = grep (!/^\.\.?$/, @allfiles);
	    if (scalar(@files) > 0) {
		$errstr .= "Cannot delete directory $d: Not empty;";
		next;
	    }

	    $errstr .= "Cannot delete directory $d: $!;";
	}
    }

    if (defined($errstr)) {
	return (0, $errstr);
    }

    return 1;
}

sub FileManager_GetParentDir($) {
    my ($dir) = @_;

    if (!defined($dir)) {
	return (0, "Directory not specified");
    }

    if (!isAbsolutePath($dir)) {
	return (0, "Directory path is not absolute");
    }

    if (! -d $dir) {
	return (0, "Directory $dir does not exist or is inaccessible");
    }

    if (isRootDir($dir)) {
	return (1, "", $dir);
    }

    $dir = dir_remove_trailing_slashes($dir);

    my $pos = rindex($dir, getFileSeparator());

    if ($pos == -1) {
	return (0, "Invalid directory $dir");
    }

    my $len = ($pos == 0) ? 1 : $pos;
    my $parentDir = substr($dir, 0, $len);

    return (1, "", $parentDir);
}

sub FileManager_GetHomeDir() {
    my $homedir = &VMware::VMServerd::GetEnvVariable("HOME");

    # XXX TODO Make this function work on windows as well
    if ($^O =~ /MSWin32/i) {
	return (0, "GetHomeDir not implemented on Windows");
    }

    if (!defined($homedir) || ($homedir eq "")) {
	&VMware::VMServerd::Warning("FileManager_GetHomeDir: No home directory for the user. Returning \"/\".");
	return (1, "", "/");
    }

    return (1, "", $homedir);
}


sub FileManager_FileExists($) {
    my ($filename) = @_;

    if (!isAbsolutePath($filename)) {
	return (0, "File path $filename is not absolute");
    }

    my ($exists, $dir) = (0, 0);
    my ($user, $ur, $uw, $ux, $group, $gr, $gw, $gx, $or, $ow, $ox) =
      ('', 0, 0, 0, '', 0, 0, 0, 0, 0, 0);

    my $mode = undef;

    if (isVmfsFile($filename)) {
	my $hash = getVmfsFileInfo($filename);
	if (!defined($hash)) {
	    return (1, "", 0);
	}

	$user = getpwuid($hash->{uid});
	$group = getgrgid($hash->{gid});
	$dir = 0;

	$mode = $hash->{mode};
    }
    else {
	if (! -e $filename) {
	    return (1, "", 0);
	}

	$dir = (-d $filename) ? 1 : 0;

	my ($d1, $d2, $d3, $uid, $gid);
	($d1, $d2, $mode, $d3, $uid, $gid) = stat($filename);

	$user = getpwuid($uid);
	$group = getgrgid($gid);
    }

    $ur = ($mode & S_IRUSR) ? 1 : 0;
    $uw = ($mode & S_IWUSR) ? 1 : 0;
    $ux = ($mode & S_IXUSR) ? 1 : 0;

    $gr = ($mode & S_IRGRP) ? 1 : 0;
    $gw = ($mode & S_IWGRP) ? 1 : 0;
    $gx = ($mode & S_IXGRP) ? 1 : 0;

    $or = ($mode & S_IROTH) ? 1 : 0;
    $ow = ($mode & S_IWOTH) ? 1 : 0;
    $ox = ($mode & S_IXOTH) ? 1 : 0;

    return (1, "", 1, $dir,
	    $user, $ur, $uw, $ux, $group, $gr, $gw, $gx, $or, $ow, $ox);
}

sub FileManager_FileCopyOrMove($@) {
    my ($move, @files) = @_;
    my $operation = $move ? "mv" : "cp";

    &VMware::VMServerd::FileInfo::startTimer();

    if (scalar(@files) < 2) {
	my $n = scalar(@files);
	return (0, "Too few arguments: $n");
    }

    my $dest = pop(@files);
    my $tpath = resolveTildeUser($dest);

    if (!defined($tpath)) {
	return (0, "\"$dest\" is not a valid home directory.");
    }
    $dest = $tpath;

    if (!isAbsolutePath($dest)) {
	return (0, "Path $dest is not absolute path");
    }

    my $destCanonical = FileManager_CanonicalPath($dest);

    if (!defined($destCanonical)) {
	return (0, "Invalid destination file/directory $dest ");
    }

    my $multipleFiles = 0;
    if (scalar(@files) > 1) {
	$multipleFiles = 1;
    }

    if ($multipleFiles) {
	if (! isDirectory($destCanonical)) {
	    return (0, "Destination for " . $move ? "move" : "copy" .
		       " of multiple files is not a directory: $dest");
	}
    }
    &VMware::VMServerd::FileInfo::stopTimer("==== Initial part of Move: ");
    &VMware::VMServerd::FileInfo::startTimer();

    my $f;
    my $errstr = undef;
    my @largeFiles;
    for $f (@files) {
	if (!isAbsolutePath($f)) {
	    $errstr .= "File $f is not absolute path;";
	    next;
	}

	if (!fileExists($f)) {
	    $errstr .= "File $f does not exist or is unreadable;";
	    next;
	}

	my $filename = internal_basename($f);

	my $destFile = $dest;
	if (isDirectory($destCanonical)) {
	    if (fileExists($destCanonical . getFileSeparator() . $filename)) {
		if ($move) {
		    $errstr .= "File/directory $filename already exists in directory $dest;";
		    next;
		} else {
		    $destFile = alternateFileName($destCanonical, $filename);
		}
	    }
	} else {
	    if (isFile($destCanonical)) {
		if ($move) {
		    $errstr .= "File " . internal_basename($dest) .
			" already exists in directory " . internal_dirname($dest) . ";";
		    next;
		} else {
		    my $dir = internal_dirname($destCanonical);
		    $destFile = alternateFileName($dir, $filename);
		}
	    }
	}

	my $doNow = 1;
	if (&VMware::VMServerd::Disk::isCOWDisk($f)) {
	    if (&VMware::VMServerd::Disk::isDiskLocked($f)) {
		$errstr .= "Cannot " . ($move ? "move" : "copy") . " disk $f. It is locked.;";
	    }
	    else {
		if ($move && (isSameFS($f, $dest) == 1) ) {
		    &VMware::VMServerd::Disk::MoveCOWDisk($f, $dest);
		} else {
		    my @members = &VMware::VMServerd::Disk::getCOWDiskFiles($f);
		    push(@largeFiles, @members);
		}
	    }
	    $doNow = 0;
	}
	elsif ( isDirectory($f) || isLargeFile($f)) {
	    # Make it a long running op if we are copying or moving/copying across FS
	    if ( !$move || (isSameFS($f, $dest) != 1) ) {
		push(@largeFiles, $f);
		$doNow = 0;
	    }
	    # else $doNow is 1
	}

	if ($doNow) {
	    my $ok;
	    if ($move) {
	      $ok = File::Copy::move($f, $destFile);
            }
	    else {
		my $execPath = $gSystemCommands{"cp"};
                # File::Copy::copy does not preserve access permissions.
                $ok = (SystemWithReset($execPath . " -p " . shell_string($f) . " "
                                       . shell_string($destFile)) == 0);
            }
	    if (!$ok) {
		$errstr .= ($move ? "Move" : "Copy") . " of $f failed: $!;"; #$err;";
	    }
	}
    }

    my $ok;
    my $err;
    my $taskid = undef;
    if (scalar(@largeFiles) > 0) {
	push(@largeFiles, $dest);
	($ok, $err, $taskid) = largeFileCopyOrMove($move, @largeFiles);
	if (!$ok) {
	    $errstr .= "$err;";
	}
    }

    if (defined($errstr)) {
	return (0, $errstr);
    }

    &VMware::VMServerd::FileInfo::stopTimer("=== Final part of Move: ");
    return (1, "", $taskid);
}


sub largeFileCopyOrMove($@) {
    my ($move, @args) = @_;

    my $dest = $args[$#args];

    if (!isAbsolutePath($dest)) {
	return (0, "Path $dest is not absolute path");
    }

    my $destCanonical = FileManager_CanonicalPath($dest);

    if (!defined($destCanonical)) {
	return (0, "Invalid destination file/directory $dest ");
    }

    my $multipleFiles = 0;
    if (scalar(@args ) > 2) {
	$multipleFiles = 1;
    }

    if ($multipleFiles) {
	if (! isDirectory($destCanonical)) {
	    return (0, "Destination for " . $move ? "move" : "copy" .
		       " of multiple files is not a directory: $dest");
	}
    }

    my $destFile;
    if (isDirectory($destCanonical)) {
	if (scalar(@args) == 2) {
	    $destFile = $destCanonical . getFileSeparator() . internal_basename($args[0]);
	    if (fileExists($destFile)) {
		if ($move) {
		    return (0, "FileMove: Destination $destFile already exists;");
		}
		$destFile = alternateFileName($dest, internal_basename($args[0]));
	    }
	    # For one file/directory copy/move, the destination file/directory is the target.
	    # in other words, we don't do "cp /file /dir" but rather "cp /file /dir/file"
	    pop(@args);
	    push(@args, $destFile);
	}
	else {
	    my @existFiles;
	    for (my $i=0; $i < $#args; $i++) {
		$destFile = $destCanonical . getFileSeparator() . internal_basename($args[$i]);
		if (fileExists($destFile)) {
		    push(@existFiles, internal_basename($args[$i]));
		}
	    }
	    #
	    # XXX Unfortunately, we have a limitation in the way long running operations are implemented.
	    # XXX If you want to do "cp /tmp/a /tmp/b /tmp/Dir" where a and b are large files and
	    # XXX /tmp/Dir/a and /tmp/Dir/b exist, you really want to do two commands:
	    # XXX "cp /tmp/a /tmp/Dir/copy_of_a" and "cp /tmp/b /tmp/Dir/copy_of_b". However, LRO can
	    # XXX only handle single command at a time. So we bail out.
	    #
	    if (scalar(@existFiles) > 0) {
		my $errstr = "The following files/directories already exist in destination directory $dest: ";
		my $firstFile = 1;
		for my $f (@existFiles) {
		    $errstr .= ($firstFile ? "" : ", ") . $f;
		    $firstFile = 0;
		}
		$errstr .= ". ";
		if ($move) {
		    $errstr .= "Cannot move the given files.; ";
		}
		else {
		    $errstr .= "Try to copy them one at a time.; ";
		}
		return (0, $errstr);
	    }
	}
    }
    else {
	my $src = $args[0];
	if (isFile($destCanonical)) {
	    if ($move) {
		return(0, "File " . internal_basename($dest) .
		       " already exists in directory " . internal_dirname($dest) . ";");
	    } else {
		my $dir = internal_dirname($destCanonical);
		$destFile = alternateFileName($dir, $src);
		pop(@args);
		push(@args, $destFile);
	    }
	}
    }

    my ($ok, $err, $taskid) =
	TaskManager_NewTask(\&largeFileCopyOrMoveUndo, \&largeFileCopyOrMoveStatus, undef);

    if (!$ok) {
	return (0, $err);
    }

    my @taskargs;
    push(@taskargs, $move);
    push(@taskargs, @args);

    TaskManager_SetArgs($taskid, \@taskargs);

    my $execPath = $move ? $gSystemCommands{"mv"} : $gSystemCommands{"cp"};

    my @argv = @args;
    $move || unshift(@argv, "-pr");

    ($ok, $err) = TaskManager_StartTask($taskid, $execPath, @argv);

    if (!$ok) {
	TaskManager_DeleteTask($taskid);
	return (0, $err);
    }

    return (1, "", $taskid);
}

sub largeFileCopyOrMoveUndo(@) {
    my ($taskid, $args) = @_;

    my @files = @$args;

    my $move = shift(@files);
    if ($move) {
	# XXX Undo not yet supported for move
	# It is a little tricky. We need to actually move back files that have been
	# moved. And this could be another long running operation...
	return 1;
    }

    my $dest = $files[$#files];

    if (-f $dest) {
	unlink($dest);
	return 1;
    }

    for (my $i=0; $i < $#files; $i++) {
	my $destFile = $dest . getFileSeparator() . internal_basename($files[$i]);
	unlink ($destFile);
    }

    return 1;
}

# returns size in bytes of a directory or file
# returns undef if size cannot be obtained
sub getInodeSizeMB($) {
    my ($inode) = @_;

    if (isFile($inode)) {
	return getFileSize($inode);
    }
    elsif (isDirectory($inode)) {
	return getDirSize($inode);
    }

    return 0;
}

sub largeFileCopyOrMoveStatus($@) {
    my ($taskid, $args) = @_;

    my @files = @$args;

    my $move = shift(@files);

    my $dest = pop(@files);

    my @destFiles;

    # For one file/directory copy/move, the destination file/directory is the target.
    # in other words, we don't do "cp /file /dir" but rather "cp /file /dir/file"
    if ((scalar(@files) > 1) && isDirectory($dest)) {
	for my $f (@files) {
	    push(@destFiles, $dest . getFileSeparator() . internal_basename($f));
	}
    }
    else {
	push(@destFiles, $dest);
    }

    my $srcSizeMB = 0;
    for my $f (@files) {
	my $sizeMB = getInodeSizeMB($f);
	if (!$sizeMB) {
	    # possible in case of move. The file may have been moved completely.
	    my $destFile = isDirectory($dest) ?
		($dest . getFileSeparator() . internal_basename($f)) : $dest;
	    $sizeMB = getInodeSizeMB($destFile);
	}
	$srcSizeMB += $sizeMB;
    }

    my $destSizeMB = 0;
    for my $f (@destFiles) {
	my $sizeMB = getInodeSizeMB($f);
	$destSizeMB += $sizeMB;
    }

    my $percent = int(($destSizeMB * 100) / $srcSizeMB);

    return (1, "", $percent);
}


sub FileManager_FileRemove($@) {
    my $recursive = shift();
    my (@files) = @_;

    if (scalar(@files) < 1) {
	my $n = scalar(@files);
	return (0, "Too few arguments: $n");
    }

    my $f;
    my $errstr = undef;
    for $f (@files) {
	if (!isAbsolutePath($f)) {
	    $errstr .= "Path $f is not absolute path;";
	    next;
	}

	# We do not use 'isVmfsFile' since it checks for VMFS-ness of
	# the file that $f points to, in case of $f being a symlink.
	# For file remove, we want to not resolve symlinks.
	if (isVmfsMountPoint(internal_dirname($f))) {
	    my ($ok, $err) = removeVmfsFile($f);
	    if (!$ok) {
		$errstr .= $err;
	    }
	    next;
	}

	if (&VMware::VMServerd::Disk::isCOWDisk($f)) {
	    my ($ok, $err) = &VMware::VMServerd::Disk::RemoveCOWDisk($f);
	    if (!$ok) {
		$errstr .= $err;
	    }
	    next;
	}

	if (-l $f  || isFile($f)) {
	    if (!unlink($f)) {
		$errstr .= "Could not remove file $f: $!;";
	    }
	} elsif (isDirectory($f)) {
          if (!$recursive) {
	      $errstr .= "File $f is a directory. Not removed;";
          } else {
            my ($ok, $err) = FileManager_RemoveDir($f);
	    if (!$ok) {
              $errstr .= $err;
	    }
	    next;
          }
	}
    }

    if (defined($errstr)) {
	return (0, $errstr);
    }

    return 1;
}

sub FileManager_FileMakeLink($$) {
    my ($filename, $link) = @_;

    if (! -e $filename) {
	return (0, "File $filename does not exist or is not accessible");
    }

    if (-e $link) {
	return (0, "File $link already exists");
    }

    my $ok = eval {symlink($filename, $link); };

    if ($@ ne "") {
	return (0, "Symbolic links not supported on this platform ($^O)");
    }

    if (!$ok) {
	return (0, "Could not create symlink to file $filename: $!");
    }

    return 1;
}

sub FileManager_ChangeOwner($@) {
    my ($owner, @files) = @_;

    if (!defined($owner)) {
	return (0, "ChangeOwner: invalid user $owner");
    }

    my ($name, $passwd, $uid) = getpwnam($owner);

    if (!defined($uid)) {
	return (0, "ChangeOwner: invalid user $owner");
    }

    my $errstr = undef;
    for my $f (@files) {
	if (!isAbsolutePath($f)) {
	    $errstr .= "ChangeOwner: $f is not absolute path;";
	    next;
	}

	if (&VMware::VMServerd::Disk::isCOWDisk($f)) {
	    my ($ok, $err) = &VMware::VMServerd::Disk::ChangeOwnerCOWDisk($f, $owner);
	    if (!$ok) {
		$errstr .= $err;
	    }
	    next;
	}

	if (!chown($uid, -1, $f)) {
	    $errstr .= "ChangeOwner of $f failed: $!;";
	    next;
	}
    }

    if (defined($errstr)) {
	return (0, $errstr);
    }

    return 1;
}

sub FileManager_ChangeGroup($@) {
    my ($group, @files) = @_;

    if (!defined($group)) {
	return (0, "ChangeGroup: invalid group");
    }

    my $gid = getgrnam($group);

    if (!defined($gid)) {
	return (0, "ChangeGroup: invalid group");
    }

    my $errstr = undef;
    for my $f (@files) {
	if (!isAbsolutePath($f)) {
	    $errstr .= "ChangeGroup: $f is not absolute path;";
	    next;
	}

	if (&VMware::VMServerd::Disk::isCOWDisk($f)) {
	    my ($ok, $err) = &VMware::VMServerd::Disk::ChangeGroupCOWDisk($f, $group);
	    if (!$ok) {
		$errstr .= $err;
	    }
	    next;
	}

	if (!chown(-1, $gid, $f)) {
	    $errstr .= "ChangeGroup of $f failed: $!;";
	    next;
	}

    }

    if (defined($errstr)) {
	return (0, $errstr);
    }

    return 1;
}

sub FileManager_ChangeMode($$$$$$$$$@) {
    my ($ur, $uw, $ux, $gr, $gw, $gx, $or, $ow, $ox, @files) = @_;

    my $errstr = undef;
    for my $f (@files) {
	if (!isAbsolutePath($f)) {
	    $errstr .= "ChangeMode: $f is not absolute path;";
	    next;
	}

	my ($um, $gm, $om) = (0, 0, 0);

	my $file_mode = getMode($f);

	if ($ur == -1) {
	    $um += ($file_mode & S_IRUSR) ? 4 : 0;
	} else {
	    $um += $ur ? 4 : 0;
	}
	if ($uw == -1) {
	    $um += ($file_mode & S_IWUSR) ? 2 : 0;
	} else {
	    $um += $uw ? 2 : 0;
	}
	if ($ux == -1) {
	    $um += ($file_mode & S_IXUSR) ? 1 : 0;
	} else {
	    $um += $ux ? 1 : 0;
	}

	if ($gr == -1) {
	    $gm += ($file_mode & S_IRGRP) ? 4 : 0;
	} else {
	    $gm += $gr ? 4 : 0;
	}
	if ($gw == -1) {
	    $gm += ($file_mode & S_IWGRP) ? 2 : 0;
	} else {
	    $gm += $gw ? 2 : 0;
	}
	if ($gx == -1) {
	    $gm += ($file_mode & S_IXGRP) ? 1 : 0;
	} else {
	    $gm += $gx ? 1 : 0;
	}

	if ($or == -1) {
	    $om += ($file_mode & S_IROTH) ? 4 : 0;
	} else {
	    $om += $or ? 4 : 0;
	}
	if ($ow == -1) {
	    $om += ($file_mode & S_IWOTH) ? 2 : 0;
	} else {
	    $om += $ow ? 2 : 0;
	}
	if ($ox == -1) {
	    $om += ($file_mode & S_IXOTH) ? 1 : 0;
	} else {
	    $om += $ox ? 1 : 0;
	}

	my $mode = oct("0" . $um . $gm . $om);

	if (&VMware::VMServerd::Disk::isCOWDisk($f)) {
	    my ($ok, $err) = &VMware::VMServerd::Disk::ChangeModeCOWDisk($f, $mode);
	    if (!$ok) {
		$errstr .= $err;
	    }
	    next;
	}
	if (!chmod($mode, $f)) {
	    $errstr .= "Could not change mode on file $f;";
	    next;
	}
    }

    if (defined($errstr)) {
	return (0, $errstr);
    }

    return 1;
}

sub FileManager_ChangeAttribs($$$$$$$$$$$@) {
    my ($owner, $group, $ur, $uw, $ux, $gr, $gw, $gx, $or, $ow, $ox,
        @files) = @_;

    my $errstr = undef;

    my ($ok, $err);

    if (defined($owner) && $owner ne '') {
        ($ok, $err) = FileManager_ChangeOwner($owner, @files);

        if (!$ok) {
            $errstr .= $err;
        }
    }

    if (defined($group) && $group ne '') {
        ($ok, $err) = FileManager_ChangeGroup($group, @files);

        if (!$ok) {
            $errstr .= $err;
        }
    }

    ($ok, $err) = FileManager_ChangeMode($ur, $uw, $ux, $gr, $gw, $gx,
                                         $or, $ow, $ox, @files);

    if (!$ok) {
	$errstr .= $err;
    }

    if (defined($errstr)) {
	return (0, $errstr);
    }

    return 1;

}


sub FileManager_ExecuteProgram($$) {
    my ($operation, $args) = @_;

    # XXX TODO Make this function work on windows as well
    if ($^O =~ /MSWin32/i) {
	return (0, "FileManager_ExecuteProgram not implemented for Windows");
    }

    my $command = $gSystemCommands{$operation};

    if (!defined($command)) {
	return (0, "Internal Error: Invalid operation in FileManager_ExecuteProgram");
    }

    $command .= " " . $args;

    my $retval = System2($command);

    if ($retval == 0) {
	return 1;
    }

    return (0, $!);
}

############################################################
#
# Internal helper functions
#
############################################################

sub octalMode($$$) {
    my ($owner_mode, $group_mode, $world_mode) = @_;

    if ($owner_mode eq "" && $group_mode eq "" && $world_mode eq "") {
	return (1, "", undef);
    }

    my $om = 0;
    if ($owner_mode =~ /^(r{0,1})(w{0,1})(x{0,1})$/) {
	$om = ($1 eq "r")*4 + ($2 eq "w")*2 + ($3 eq "x")*1;
    } else {
	return (0, "Invalid mode $owner_mode for user");
    }

    my $gm = 0;
    if ($group_mode =~ /^(r{0,1})(w{0,1})(x{0,1})$/) {
	$gm = ($1 eq "r")*4 + ($2 eq "w")*2 + ($3 eq "x")*1;
    } else {
	return (0, "Invalid mode $group_mode for group");
    }

    my $wm = 0;
    if ($world_mode =~ /^(r{0,1})(w{0,1})(x{0,1})$/) {
	$wm = ($1 eq "r")*4 + ($2 eq "w")*2 + ($3 eq "x")*1;
    } else {
	return (0, "Invalid mode $world_mode for others");
    }

    my $mode = "0" . $om . $gm . $wm;

    return (1, "", $mode);
}

# Resolves paths of the form ~user to the user's home directory
# e.g., ~foo to /home/foo and ~foo/bar to /home/foo/bar
sub resolveTildeUser($) {
    my ($path) = @_;

    if ($^O =~ /linux/i) {
	if ($path =~ /^~([^\/]*)(.*)$/) {
	    my $homedir;

	    if ($1 ne "") {
		$homedir = (getpwnam($1))[7];
	    } else {
		$homedir = FileManager_GetHomeDir();
	    }

	    if (defined($homedir) && (-e $homedir)) {
		return $homedir . $2;
	    }
	    else {
		return undef;
	    }
	}
    }

    return $path;
}

sub cbDelete {
    if (!-l && -d _) {
	unless (rmdir($File::Find::name)) {
	    &VMware::VMServerd::Warning("rmdir($File::Find::name) $!");
	    $gRecursiveDeleteStatus = 0;
	}
    } else {
	unless (unlink($File::Find::name)) {
	    &VMware::VMServerd::Warning("unlink($File::Find::name) $!");
	    $gRecursiveDeleteStatus = 0;
	}
    }
}

sub doRemoveDir {
    $gRecursiveDeleteStatus = 1;
    finddepth(\&cbDelete, @_);
    return $gRecursiveDeleteStatus;
}

# Global used to accumulate the size of directory
my $gDirSize;
sub cbAccumulateSize {
    my $filename = $File::Find::name;

    if (!isFile($filename)) {
	return;
    }

    my $size = int(getFileSize($filename) + 0.5);
    $gDirSize += $size;
}

sub getDirSize($) {
    my ($dirname) = @_;

    if (!defined($dirname) || ($dirname eq "")) {
	return 0;
    }

    # Recursively call cbAccumulateSize() on each sub dir/files of $dirname
    # and accumulate the sum in the global $gDirSize
    $gDirSize = 0;
    find(\&cbAccumulateSize, $dirname);
    my $size = $gDirSize;
    $gDirSize = 0;

    return $size;
}

sub isLargeFile($) {
    my ($file) = @_;

    my $size = getFileSize($file); # Size in MB

    return ($size > $LARGE_FILE_SIZE_THRESHOLD);
}

# XXX $isVmfsFile is a workaround for PR 21800
sub FileManager_CanonicalPath($) {
    my ($path, $isVmfsFile) = @_;

    my $cpath = VMware::VMServerd::GetCanonicalPath($path);
    if (defined($cpath)) {
	# XXX THIS is a hack until PR 21800 is fixed.
	if ($isVmfsFile) {
	    if (&VMware::ExtHelpers::internal_basename($path) ne
		&VMware::ExtHelpers::internal_basename($cpath)) {
		$cpath .= '/' . &VMware::ExtHelpers::internal_basename($path);
	    }
	}
	return $cpath;
    }

    # On Linux, we may have $path be a symlink pointing to a
    # path to a non-existent file in an existing directory.
    # I.e., $path may point to /.../Dir/File, where /.../Dir
    # exists while /.../Dir/File does not. We need to do the
    # 'parent directory exists' check below with resolved name
    # and not $path as it comes.
    if ($^O =~ /linux/i) {
	if (-l $path) {
	    $path = readlink($path);
	    if (!defined($path)) {
		return undef;
	    }
	}
    }

    # File does not exist, it could be target of a move. Canonicalize the
    # parent directory.
    my $parent = internal_dirname($path);
    if (! -d $parent) {
	&VMware::VMServerd::Warning("Invalid path $path. ");
	return undef;
    }
    return VMware::VMServerd::GetCanonicalPath($parent) .
	getFileSeparator() . internal_basename($path);
}

sub createDir($$$) {
    my ($path, $mode, $makeParents) = @_;

    if (-d $path) {
	return (0, "Creation of directory $path failed: File exists");
    }

    if (!$makeParents) {
	my $ok = mkdir($path, $mode);
	if (!$ok) {
	    return (0, "Creation of directory $path failed: $!");
	}
	return 1;
    }

    my @dirComponents = split(getFileSepRegEx(), $path);
    my $dirstr = $dirComponents[0] . getFileSeparator();
    if (! -d $dirstr) {
	return (0, "Root directory does not exist");
    }

    for (my $i = 1; $i < scalar(@dirComponents); $i++) {
	$dirstr .= $dirComponents[$i];

	if (! -d $dirstr) {
	    my $ok = mkdir($dirstr, $mode);
	    if (!$ok) {
		return (0, "Creation of directory $dirstr failed: $!");
	    }
	}
    }

    return 1;
}

sub cleanupFilesList(@) {
    my (@allfiles) = @_;

    # Ignore '.' and '..'
    my @filesRelativeNames = grep (!/^\.\.?$/, @allfiles);

    # Ignore files and directories starting with a .
    @filesRelativeNames = grep (!/^\..*$/, @filesRelativeNames);

    # Remove lost+found as well...
    @filesRelativeNames = grep (!/^lost\+found$/, @filesRelativeNames);

    # And finally, sort it.
    @filesRelativeNames = sort(@filesRelativeNames);

    return @filesRelativeNames;
}

sub alternateFileName($$) {
    my ($dir, $filename) = @_;

    # Make sure filename does not have the fileseparator in it.
    if ($filename =~ getFileSepRegEx()) {
	Panic("copyOfPrefix called with a file that is not basename");
	return undef;
    }

    my $newFileName = $dir . getFileSeparator() . "copy_of_" . $filename;
    if (! -e $newFileName) {
	return $newFileName;
    }

    my $MAX_INDEX = 0x7FFFFFFF;
    for (my $index = 2; $index < $MAX_INDEX; $index++) {
	$newFileName = $dir . getFileSeparator() . "copy_" . $index . "_of_" . $filename;
	if (! -e $newFileName) {
	    return $newFileName;
	}
    }

    Panic("Not Implemented: Too many copies of the file exist");
}

sub getDirWithoutTrailingSlash($) {
    my ($directory) = @_;
    my $dirWithoutTrailingSlash = undef;

    $directory = dir_remove_trailing_slashes($directory);
    if (isRootDir($directory)) {
	$dirWithoutTrailingSlash = $directory;
    }
    else {
	$dirWithoutTrailingSlash = $directory;
	$directory = $directory. getFileSeparator();
    }

    return ($directory, $dirWithoutTrailingSlash);
}

############################################################
#
# VMServerd handlers
#
############################################################

#  External API
#  Each entry is: Operation Name, Auth. Policy
#

my $API = [
		["ListDir", "authuser"],
		["ListDirTree", "authuser"],
		["MakeDir", "authuser"],
		["RemoveDir", "authuser"],
		["GetParentDir", "authuser"],
		["GetHomeDir", "authuser"],
		["FileExists", "authuser"],
		["FileCopy", "authuser"],
		["FileMove", "authuser"],
		["FileRemove", "authuser"],
		["FileMakeLink", "authuser"],
		["ChangeOwner", "authuser"],
		["ChangeGroup", "authuser"],
		["ChangeMode", "authuser"],
		["ChangeAttribs", "authuser"]
	  ];

sub reportError($) {
    my ($err) = @_;

    my ($package, $filename, $line, $callerSub) = caller(1);

    my @words = split("::", $callerSub);

    my $operation = $words[$#words];

    $operation =~ s/(.*)_Handler/$1/;

    # multiple error strings are separated by ';'
    my @errors = split(/;/, $err);

    # remove leading nulls. split only removes trailing nulls.
    while (scalar(@errors) > 0 && $errors[0] eq "") {
	shift(@errors);
    }

    #my $e = shift(@errors);
    #errorPost($PKG_NAME . "_" . $operation . ": " . $e, "error");

    while (scalar(@errors) > 0) {
	my $e = shift(@errors);
	if ($e =~ /^\s+$/) {
	    next;
	}
	errorPost($e, "error");
    }
}

####################################################################################
#
# ListDir:
#   List a given directory
#
# Input format:
#   <in>
#      <directory> absolute path of the directory to be listed </directory>
#   </in>
#
# Note:
#      All the files and directories below are relative paths, relative to relativeRootDir
#      If an entity is a symlink, then "realname" gives the file or directory it points to
#      and all the properties (permissions, size, ...) are for the realname.
#      A symlink will be in <files> if the realname is a file; likewise for directory.
#
#      Optional elements are indicated by an asterix next to the element name.
#
# Output format:
#   <out>
#      <relativeRootDir> absolute path of the directory being listed </relativeRootDir>
#      <files>
#	 <file>
#	    <name> file name </name>	     ;; if symlink, then name of the symlink
#	    <realname>* name the symlink points to </realname>
#	    <permissions>
#	       <r/>* <w/>* <x/>*
#	    </permissions>
#	    <size units="MB"> size </size>       ;; size in MB (rounded _up_ to next MB)
#	    <mtime> modif time </mtime>	  ;; in format "Thu Sep  6 18:29:23 2001"
#	    <config />*			   ;; is it a config file, or
#	    <disk type="tttt" />*		 ;; is it a disk. tttt could be "virtual" or "vmfs"
#	 </file>
#	 ...
#      </files>
#      <directories>
#	 <directory>
#	    <name> directory name </name>
#	    <realname>* name the symlink points to </realname>
#	    <permissions>
#	       <r/>* <w/>* <x/>*
#	    </permissions>
#	    <mtime> modif time </mtime>
#	    <vmfsMountPoint />*
#	 </directory>
#	 ...
#      </directories>
#      <dirInfo>				  ;; Info about the directory being listed
#	 <vmfsMountPoint />*
#      </dirInfo>
#   </out>
#
####################################################################################
sub ListDir_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my $directory = $in->getValue('.directory');

    # Returns an array of hash references each for a file (or dir) in the directory
    # if $directory is undef, the listing is for user's home directory

    my ($ok, $err, $numTotalDir, $numTotalFiles, $diskFreeSpace,
        $dirRoot, $filesref) =
        FileManager_ListDir($directory, 0);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    my @files = @$filesref;

    $out->setValue('.relativeRootDir', $dirRoot);
    $out->setValue('.numTotalDir', $numTotalDir);
    $out->setValue('.numTotalFiles', $numTotalFiles);
    $out->setValue('.diskAvailSpace', $diskFreeSpace);
    my $i = 0;
    for my $f (@files) {
	# If it is anything other than a file or directory (e.g., named pipe,
	# device, etc.) it will be treated as a file.
	my $key = ($f->{type} eq "directory") ? "directory" : "file";
	$i++;
	my $fileNode = VMware::DOMAccess->new($key);

	$fileNode->setValue('.name', $f->{name});
	if (defined($f->{realname})) {
	    $fileNode->setValue('.realname', $f->{realname});
	}

	$fileNode->setValue('.permissions', "");
	if (defined($f->{permissions})) {  # Undefined for dangling symlinks
	    if ($f->{permissions} =~ /r/) {
		$fileNode->setValue('.permissions.r', "");
	    }
	    if ($f->{permissions} =~ /w/) {
		$fileNode->setValue('.permissions.w', "");
	    }
	    if ($f->{permissions} =~ /x/) {
		$fileNode->setValue('.permissions.x', "");
	    }
	}

	if ($f->{type} eq "file") {
	    $fileNode->setValue('.size', $f->{size});
	    $fileNode->setAttribute('.size', "units", "MB");
	}

	if (defined($f->{mtime})) {
	    $fileNode->setValue('.mtime', $f->{mtime});
	}

	if (defined($f->{config})) {
	    $fileNode->setValue('.config', "");
	} elsif (defined($f->{virtualdisk})) {
	    $fileNode->setValue('.disk', "");
	    $fileNode->setAttribute('.disk', "type", "virtual");
	} elsif (defined($f->{vmfsdisk})) {
	    $fileNode->setValue('.disk', "");
	    $fileNode->setAttribute('.disk', "type", "vmfs");
	} elsif (defined($f->{vmfsMountPoint})) {
	    $fileNode->setValue('.vmfsMountPoint', "");
	}

	if ($f->{type} eq "file") {
	    $out->addSubTree('.files', $fileNode);
	} else {
	    $out->addSubTree('.directories', $fileNode);
	}

	$fileNode->dispose();
    }

    if (isVmfsMountPoint($directory)) {
	$out->setValue('.dirInfo.vmfsMountPoint', "");
    }

    return 1;
}

sub appendDirs($$$) {
    my ($out, $keystr, $aref) = @_;

    my @arr = @{ $aref };

    my $index = 0;
    for my $d (@arr) {
	my $key = $keystr . "[" . $index . "]";
	$index++;
	$out->setAttribute($key, "name", $d->{name});
	$out->setAttribute($key, "abspath", $d->{abspath});
	if (defined($d->{realname})) {
	    $out->setAttribute($key, "realname", $d->{realname});
	}

	if (defined($d->{dirs})) {
	    appendDirs($out, $key . ".directory", $d->{dirs});
	}
    }
}

####################################################################################
#
# ListDirTree:
#   List a directory tree. Given a directory it lists the subdirectories of directories
#   at each level in the path.
#
# Sample execution times on Arcadia (seconds):
# / 0.15
# /usr 0.28
# /usr/bin 0.32
# /usr/bin/X11 0.32
# /exit14/home/dilip/work/esx10/bora/apps/vmserverd/perlroot/VMware/VMServerd 2.45 (0.11 w/o XML generation)
#
# Input format:
#   <in>
#      <directory> absolute path of the directory  </directory>
#   </in>
#
# Note:
#      Optional attributes are indicated by an asterix next to the name.
#
# Output format: (shown by giving example of /usr/bin, with /foo and /tmp as directories under /)
#   <out>
#      <directory name = "/" abspath = "/">
#	 <directory name = "usr" abspath="/usr" >
#	    <directory name="bin" abspath="/usr/bin" realname* = "symlink path">   ;; if this is a symlink
#	       <directory name="a" abspath="/usr/bin/a" />
#	       <directory name="b" abspath="/usr/bin/b" />
#	    <directory>
#	 </directory>
#	 <directory name = "foo" abspath="/usr/foo" realname* = "symlink path" />  ;; if this is a symlink
#	 <directory name = "tmp" abspath="/usr/tmp" />
#      </directory>
#   </directory>
#  </out>
#
####################################################################################
sub ListDirTree_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my $directory = $in->getValue('.directory');

    # Returns an array of hash references. Each hash identifies the directory and if it is
    # a symlink - in which case the directory it points to.
    my ($ok, $err, $href) =  FileManager_ListDirTree($directory);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    my $keystr = ".directory";

    $out->setAttribute($keystr, "name", $href->{name});
    $out->setAttribute($keystr, "abspath", $href->{abspath});
    if (defined($href->{realname})) {
	$out->setAttribute($keystr, "realname", $href->{realname});
    }

    appendDirs($out, $keystr . ".directory", $href->{dirs});

    return 1;
}

####################################################################################
#
# MakeDir:
#   Create a directory
#
# Input format:
#   <in>
#      <directories>
#	 <directory> absolute path name of the directory </directory>
#	 ...
#      </directories>
#      <owner_mode> rwx </owner_mode>
#      <group_mode> rwx </group_mode>
#      <other_mode> rwx </other_mode>
#      <makeParents> 1 | 0 </makeParents>  ;; Create intermediate directories if non-existent
#   </in>
#
# Output format:
#
####################################################################################
sub MakeDir_Handler($$) {
    my ($in, $out) = @_;

    errorReset();
    my @tags = $in->listElementNames(".directories");

    my @dirs;
    for my $t (@tags) {
	push(@dirs, $in->getValue(".directories." . $t));
    }

    my $owner_mode = $in->getValue(".owner_mode");
    my $group_mode = $in->getValue(".group_mode");
    my $world_mode = $in->getValue(".world_mode");
    my $makeParents = $in->getValue(".makeParents");

    my ($ok, $err) = FileManager_MakeDir($owner_mode, $group_mode, $world_mode, $makeParents, @dirs);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    return 1;
}

####################################################################################
#
# RemoveDir:
#   Remove a directory
#
# Input format:
#   <in>
#      <directories>
#	 <directory> absolute path name of the directory </directory>
#	 ...
#      </directories>
#   </in>
#
# Output format:
#
####################################################################################
sub RemoveDir_Handler($$) {
    my ($in, $out) = @_;

    errorReset();
    my @tags = $in->listElementNames(".directories");

    my @dirs;
    for my $t (@tags) {
	push(@dirs, $in->getValue(".directories." . $t));
    }

    my ($ok, $err) = FileManager_RemoveDir(@dirs);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    return 1;
}


####################################################################################
#
# GetParentDir:
#   Get parent directory of a given directory
#
# Input format:
#   <in>
#      <directory> name of directory </directory>
#   </in>
#
# Output format:
#
#   <out>
#      <directory> parent directory </directory>
#   </out>
#
####################################################################################
sub GetParentDir_Handler($$) {
    my ($in, $out) = @_;

    errorReset();
    my $directory = $in->getValue(".directory");

    my ($ok, $err, $parentDir) = FileManager_GetParentDir($directory);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    $out->setValue(".directory", $parentDir);

    return 1;
}


####################################################################################
#
# GetHomeDir:
#   Get home directory of the user
#
# Input format:
#
# Output format:
#   <out>
#      <directory> home directory </directory>
#   </out>
#
####################################################################################
sub GetHomeDir_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my ($ok, $err, $homeDir) = FileManager_GetHomeDir();

    if (!$ok) {
	reportError($err);
	return 0;
    }

    $out->setValue(".directory", $homeDir);

    return 1;
}

####################################################################################
#
# FileExists:
#   Check to see if a file exists
#
# Input format:
#   <in>
#      <file> absolute path of the file </file>
#   </in>
#
# Output format:
#   <out>
#      <exists> 0 | 1 </exists>
#      <directory 0 | 1 </directory>
#      <user> username </user>
#      <read-user> 0 | 1 </read-user>
#      <write-user> 0 | 1 </write-user>
#      <execute-user> 0 | 1 </execute-user>
#      <group> groupname </group>
#      <read-group> 0 | 1 </read-group>
#      <write-group> 0 | 1 </write-group>
#      <execute-group> 0 | 1 </execute-group>
#      <read-other> 0 | 1 </read-other>
#      <write-other> 0 | 1 </write-other>
#      <execute-other> 0 | 1 </execute-other>
#   </out>
#
####################################################################################
sub FileExists_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my $filename = $in->getValue('.file');

    if (!defined($filename)) {
	reportError("No input argument \'file\'");
	return 0;
    }

    my ($ok, $err, $exists, $dir,
	$user, $ur, $uw, $ux, $group, $gr, $gw, $gx,
	$or, $ow, $ox) = FileManager_FileExists($filename);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    $out->setValue('.exists', $exists);

    if ($exists) {
	$out->setValue('.directory', $dir);
	$out->setValue('.user', $user);
	$out->setValue('.read-user', $ur);
	$out->setValue('.write-user', $uw);
	$out->setValue('.execute-user', $ux);
	$out->setValue('.group', $group);
	$out->setValue('.read-group', $gr);
	$out->setValue('.write-group', $gw);
	$out->setValue('.execute-group', $gx);
	$out->setValue('.read-other', $or);
	$out->setValue('.write-other', $ow);
	$out->setValue('.execute-other', $ox);
    }

    return 1;
}

####################################################################################
#
# FileCopy:
#   Copy one or more files
#
# Input format:
#   <in>
#      <files>
#	 <file> absolute path of the file </file>
#	 ...
#	 <file> absolute path of the file </file>   ;; Last element is the destination
#      </files>
#   </in>
#
# Output format:
#
####################################################################################
sub FileCopy_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my @tags = $in->listElementNames(".files");

    my @files;
    for my $t (@tags) {
	push(@files, $in->getValue(".files." . $t));
    }

    my ($ok, $err, $taskid) = FileManager_FileCopyOrMove(0, @files);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    if (defined($taskid)) {
	$out->setValue(".taskid", $taskid);
    }
    return 1;
}

####################################################################################
#
# FileMove:
#   Move one or more files
#
# Input format:
#   <in>
#      <files>
#	 <file> absolute path of the file </file>
#	 ...
#	 <file> absolute path of the file </file>   ;; Last element is the destination
#      </files>
#   </in>
#
# Output format:
#
####################################################################################
sub FileMove_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my @tags = $in->listElementNames(".files");

    my @files;
    for my $t (@tags) {
	push(@files, $in->getValue(".files." . $t));
    }

    my ($ok, $err, $taskid) = FileManager_FileCopyOrMove(1, @files);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    if (defined($taskid)) {
	$out->setValue(".taskid", $taskid);
    }

    return 1;
}

####################################################################################
#
# FileRemove:
#   Remove one or more files
#
# Input format:
#   <in>
#      <files>
#	 <file> absolute path of the file </file>
#	 ...
#      </files>
#   </in>
#
# Output format:
#
####################################################################################
sub FileRemove_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my @tags = $in->listElementNames(".files");

    my @files;
    for my $t (@tags) {
	push(@files, $in->getValue(".files." . $t));
    }

    my $recursive = 0;

    if ($in->getValue(".recursive")) {
      $recursive = 1;
    }

    my ($ok, $err) = FileManager_FileRemove($recursive, @files);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    return 1;
}

####################################################################################
#
# FileMakeLink:
#   Create a symbolic link
#
# Input format:
#   <in>
#      <file> absolute path of existing file </file>
#      <link> name of the link to create </link>
#   </in>
#
# Output format:
#
####################################################################################
sub FileMakeLink_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my $filename = $in->getValue('.file');
    my $link = $in->getValue('.link');

    if (!defined($filename)) {
	reportError("No input argument \'file\'");
	return 0;
    }

    if (!defined($link)) {
	reportError("No input argument \'link\'");
	return 0;
    }

    my ($ok, $err, $exists, $dir, $r, $w, $x) = FileManager_FileMakeLink($filename, $link);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    return 1;
}

####################################################################################
#
# ChangeOwner:
#   Change owner of one of more files
#
# Input format:
#   <in>
#      <files>
#	 <file> absolute path of existing file </file>
#	 ...
#      </files>
#      <owner> owner </owner>
#   </in>
#
# Output format:
#
####################################################################################
sub ChangeOwner_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my @tags = $in->listElementNames(".files");

    my @files;
    for my $t (@tags) {
	push(@files, $in->getValue(".files." . $t));
    }

    my $owner = $in->getValue(".owner");

    my ($ok, $err) = FileManager_ChangeOwner($owner, @files);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    return 1;
}

####################################################################################
#
# ChangeGroup:
#   Change group of one of more files
#
# Input format:
#   <in>
#      <files>
#	 <file> absolute path of existing file </file>
#	 ...
#      </files>
#      <group> group </owner>
#   </in>
#
# Output format:
#
####################################################################################
sub ChangeGroup_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my @tags = $in->listElementNames(".files");

    my @files;
    for my $t (@tags) {
	push(@files, $in->getValue(".files." . $t));
    }

    my $group = $in->getValue(".group");

    my ($ok, $err) = FileManager_ChangeGroup($group, @files);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    return 1;
}

####################################################################################
#
# ChangeMode:
#   Change access mode of one of more files
#
# Input format:
#   <in>
#      <files>
#	 <file> absolute path of existing file </file>
#	 ...
#      </files>
#
#      <read-user> -1 | 0 | 1 </read-user>
#      <write-user> -1 | 0 | 1 </write-user>
#      <execute-user> -1 | 0 | 1 </execute-user>
#
#      <read-group> -1 | 0 | 1 </read-group>
#      <write-group> -1 | 0 | 1 </write-group>
#      <execute-group> -1 | 0 | 1 </execute-group>
#
#      <read-other> -1 | 0 | 1 </read-other>
#      <write-other> -1 | 0 | 1 </write-other>
#      <execute-other> -1 | 0 | 1 </execute-other>
#   </in>
#
# Output format:
#
####################################################################################
sub ChangeMode_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my @tags = $in->listElementNames(".files");

    my @files;
    for my $t (@tags) {
	push(@files, $in->getValue(".files." . $t));
    }

    my $ur = $in->getValue(".read-user");
    my $uw = $in->getValue(".write-user");
    my $ux = $in->getValue(".execute-user");

    my $gr = $in->getValue(".read-group");
    my $gw = $in->getValue(".write-group");
    my $gx = $in->getValue(".execute-group");

    my $or = $in->getValue(".read-other");
    my $ow = $in->getValue(".write-other");
    my $ox = $in->getValue(".execute-other");

    my ($ok, $err) = FileManager_ChangeMode($ur, $uw, $ux, $gr, $gw, $gx,
					    $or, $ow, $ox, @files);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    return 1;
}

####################################################################################
#
# ChangeAttribs:
#   Change owner, group and mode of one or more files
#
# Input format:
#   <in>
#      <files>
#	 <file> absolute path of existing file </file>
#	 ...
#      </files>
#
#      <read-user> -1 | 0 | 1 </read-user>
#      <write-user> -1 | 0 | 1 </write-user>
#      <execute-user> -1 | 0 | 1 </execute-user>
#
#      <read-group> -1 | 0 | 1 </read-group>
#      <write-group> -1 | 0 | 1 </write-group>
#      <execute-group> -1 | 0 | 1 </execute-group>
#
#      <read-other> -1 | 0 | 1 </read-other>
#      <write-other> -1 | 0 | 1 </write-other>
#      <execute-other> -1 | 0 | 1 </execute-other>
#
#      <group> group </group>
#      <owner> owner </owner>
#   </in>
#
# Output format:
#
####################################################################################
sub ChangeAttribs_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my @tags = $in->listElementNames(".files");

    my @files;
    for my $t (@tags) {
	push(@files, $in->getValue(".files." . $t));
    }

    my $ur = $in->getValue(".read-user");
    my $uw = $in->getValue(".write-user");
    my $ux = $in->getValue(".execute-user");

    my $gr = $in->getValue(".read-group");
    my $gw = $in->getValue(".write-group");
    my $gx = $in->getValue(".execute-group");

    my $or = $in->getValue(".read-other");
    my $ow = $in->getValue(".write-other");
    my $ox = $in->getValue(".execute-other");

    my $group = $in->getValue(".group");
    my $owner = $in->getValue(".owner");

    my ($ok, $err) = FileManager_ChangeAttribs($owner, $group,
					       $ur, $uw, $ux, $gr, $gw, $gx,
					       $or, $ow, $ox, @files);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    return 1;
}

#############################################################
# Define operations and the Perl functions to be executed
# when a request for such an operation is received.
# Permissions information is also indicated here.
#############################################################

my $i;
my @array = @$API;
for ($i=0; $i <= $#array; $i++) {
    my $arr = $API->[$i];

    $PKG_FULL_NAME = __PACKAGE__;
    my @components = split(/::/, $PKG_FULL_NAME);
    $PKG_NAME = $components[$#components];

    my $opname  = $PKG_NAME . "_" . $arr->[0];
    my $handler = $PKG_FULL_NAME . "::" . $arr->[0] . "_Handler";
    my $policy  = $arr->[1];

    &VMware::VMServerd::addOperation( OPNAME => $opname,
				      PERLFUNC => $handler,
				      POLICY => $policy);
}

1;
