#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# HTTP.pm
# 
# HTTP interface to vm serverd.
#

package VMware::VMServerd::HTTP;

use strict;
use HTTP::Daemon;
use HTTP::Status;
use URI::URL;
use CGI;

if ( $^O ne "MSWin32" ) {
   require Authen::PAM;
}

use MIME::Base64;
use Carp;
use VMware::DOMAccess;
use VMware::VMServerd qw(%gConfig);

################################
###
###  Init
###
################################

my %opsTable = ("/info" => \&info_handler,
                "/exec" => \&exec_handler
               );

my $daemon;
  
my $username;
my $password;

my $execEntryFormDoc;

$daemon = HTTP::Daemon->new( LocalPort => 8223, Reuse => 1, Timeout => 0 );
print "HTTP: Connect to " . $daemon->url() . "\n";


################################
###
###  HTTP
###
################################

if ( $^O eq "MSWin32" ) {
  VMware::VMServerd::Warning("Unable to register callback on Windows\n");
} else {
  VMware::VMServerd::RegisterPollCallback(VMware::VMServerd::POLL_FD(),
                                        $daemon->fileno(),
                                        \&accept_http_connection);
}

sub accept_http_connection {
  my $conn = $daemon->accept();

  if (defined($conn)) {
    $conn->timeout(0.1);
    VMware::VMServerd::RegisterPollCallback(VMware::VMServerd::POLL_FD,
                                            $conn->fileno(),
                                            \&get_conn_request, $conn);
  }

  return 1;
}

sub get_conn_request {
  my($id, $conn) = @_;

 BLOCK: {
    my $req = $conn->get_request();

    if (!defined($req)) {
      last BLOCK;
    }
    
    print $req->as_string();
    
    if (!authenticate($req->header('Authorization'))) {
      $conn->send_error(HTTP::Status::RC_UNAUTHORIZED());
      last BLOCK;
    }
    
    my $url = URI::URL->new($req->url());
    my $handler = $opsTable{$url->path()};
    
    if (!defined($handler)) {
      if ($url->path() eq "/") {
        $conn->send_redirect("/info");
      } else {
        $conn->send_error(HTTP::Status::RC_NOT_FOUND());
      }
      last BLOCK;
    }
    
    my $queryStr;
    my $content = "";
    my $method = $req->method();
    
    if ($method eq "GET") {
      $queryStr = $url->equery();
    } elsif ($method eq "POST") {
      $queryStr = $req->content();
    } else {
      $queryStr = $url->equery();
      $content = $req->content();
    }
    
    my $query = new CGI((defined($queryStr)) ? $queryStr : "");
    my $resp = HTTP::Response->new();
      
    eval '$handler->($req, $resp, $url->path(), $query, $content)';
    
    if ($@) {
      $conn->send_error(HTTP::Status::RC_INTERNAL_SERVER_ERROR(), $@);
      last BLOCK;
    }

    $conn->send_response($resp);
  }

  VMware::VMServerd::UnregisterPollCallback($id);
  undef $conn;
  return 1;
} 


################################
###
###  Authentication
###
################################

sub authenticate {
  my($key) = @_;

  if (!defined($key)) {
    return 0;
  }
  
  if (!($key =~ /^Basic\s(\S+)$/)) {
    die "NOT_IMPLEMENTED::: $key";
  }

  if ( $^O eq "MSWin32" ) {
    return 1;
  }

  ($username, $password) = split(/:/, MIME::Base64::decode_base64($1));
  my $pamh = new Authen::PAM("passwd", $username, \&my_conv_func);
  my $retval = $pamh->pam_authenticate();
  undef $pamh;
  
  if ($retval != Authen::PAM::PAM_SUCCESS()) {
    undef $username;
    undef $password;
    return 0;
  }
  
  return 1;
}

sub my_conv_func {
  my @res;

  if ( $^O eq "MSWin32" ) {
    @res = [];
    return @res;
  }

  while ( @_ ) {
    my $code = shift;
    my $msg = shift;
    my $ans = "";
    
    $ans = $username if ($code == Authen::PAM::PAM_PROMPT_ECHO_ON() );
    $ans = $password if ($code == Authen::PAM::PAM_PROMPT_ECHO_OFF() );
    
    push @res, Authen::PAM::PAM_SUCCESS();
    push @res, $ans;
  }
  push @res, Authen::PAM::PAM_SUCCESS();
  return @res;
}


################################
###
###  Handlers
###
################################

sub info_handler {
  my($req, $resp, $path, $query, $content) = @_;

  my $str = "<h1>VMware Serverd</h1>\n";

  $str = $str . htmlVar(\%gConfig, "VMware::VMServerd::gConfig");

  $resp->header("Content-type", "text/html");
  $resp->content($str);
  $resp->code(HTTP::Status::RC_OK());
}

sub exec_handler {
  my($req, $resp, $path, $query, $content) = @_;

  my $op = $query->param("op");

  if (!defined($op)) {
    $resp->header("Content-type", "text/html");
    my $body = execForm(xmlBlock("exec", 
                                 xmlBlock("op", opInput()) 
                                 . xmlBlock("in", inInput(""))));
    $resp->content($body);
    $resp->code(HTTP::Status::RC_OK());
    
  } else {
    my $in = $query->param("in");
    $in = "" if !defined($in);

    my $str = "<exec><op>$op</op><in>$in</in></exec>";
    
    $str = VMware::VMServerd::handleExecRequest($username, $str);
    
    my $doc = VMware::DOMAccess->newXML($str);
    $str = $doc->prettyPrint();
    $doc->dispose();

    $resp->header("Content-type", "text/plain");
    $resp->content($str);
    $resp->code(HTTP::Status::RC_OK());
  }
}


################################
###
###  HTML ouput
###
################################

sub htmlVar {
  my($var, $name) = @_;

  my $str = "";
  
  if (defined($name)) {
    $str = $str . "$name ";
  }

  $str = $str . "=> ";
  
  if (!defined($var)) {
    return $str . htmlEscape("<undef>\n");
  }
  
  if (!(ref $var)) {
    return $str . htmlEscape("$var\n");
  }
  
  if (UNIVERSAL::isa($var, 'HASH')) {
    return $str . htmlHash($var);
  }
  
}

sub htmlEscape {
  my($str) = @_;
  $str =~ s/</&lt;/g;
  return $str;
}

sub htmlHash {
  my($var) = @_;

  my $str = "";
  
  $str = $str . "<menu>\n";
  foreach my $key (sort(keys(%{$var}))) {
    $str = $str . "<li>" . htmlVar($var->{$key}, $key) . "\n";
  }
  $str = $str . "</menu>\n";
  
  return $str;
}

sub execForm {
  my($str) = @_;
  $str = $str . "<input type=reset value=\"Reset\">\n";
  $str = $str . "<input type=submit value=\"Submit\">\n";
  return "<html><body><form action=\"/exec\">$str</form></body></html>\n";
}

sub xmlBlock {
  my($tag, $str) = @_;
  return "&lt;$tag&gt;<blockquote>$str</blockquote>&lt;/$tag&gt;<p>\n";
}

sub opInput {
  my $str = "<select name=op size=1>\n";

  foreach my $op (sort(keys(%{$gConfig{OPERATION_TABLE}}))) {
    $str = $str . "<option>$op\n";
  }

  $str = $str . "</select>\n";
}

sub inInput {
  my($str) = @_;
  return "<textarea name=in cols=80 rows=16>$str</textarea>\n";
}


1;
