#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# Info.pm
# 
# Generic info about VMs.
#

package VMware::VMServerd::Info;
  
use strict;
use Carp;
use VMware::DOMAccess;
use VMware::VMServerd qw($VMSTATE_ERROR 
			 $VMSTATE_OFF
			 $VMSTATE_ON 
			 $VMSTATE_SUSPEND 
			 $VMSTATE_STUCK
			 %VMSTATE_NAMES);
use VMware::VMServerd::VMList;
use VMware::Control::VM;
use VMware::Config;
use VMware::Management::Util qw( file_to_string
                                 string_to_file
                                 hash_credentials );

if (VMware::VMServerd::product() eq "ESX") {
    require VMware::VMServerd::ESXHostInfo;
} else {
    require VMware::VMServerd::GSXHostInfo;
}

my @periods = ( 60, 300, 900 );
my @methods = ( "min", "avg", "max", "sum" );
my %methods = ( $methods[ 0 ] => "min", $methods[ 1 ] => "mean",
                $methods[ 2 ] => "max", $methods[ 3 ] => "sum", );
my $doc;

sub VM_Info_Handler {
    my $in = shift;
    my $out = shift;
    my $hostStats;
    my $root;

    $out->setValue(".server", "");
    $root = $out->get(".server");
    $doc = $out->getDocument();

    &VMware::Management::Util::append_host($doc, $root);
    append_product($root);

    if (VMware::VMServerd::product() eq "ESX") {
      VMware::VMServerd::Stats::appendSystem($out);
      VMware::VMServerd::Stats::appendConsole($out);
    }

    # Get the list of registered VMs that the user has access to
    my @vms = VMware::VMServerd::VMList::Enumerate();

    my $system_vm;
    my $num_samples = $periods[0];
    my $max_samples = 0;

    foreach my $cfg (@vms) {
      my $vm = VMware::VMServerd::findVM($cfg);
      append_vm($out, $cfg);

      # We want to find a reasonable vm to query for system stats.
      # "Reasonable" means: is running, has a full set of samples(60) or has 
      # been running the longest
      # We are inferring the number of samples by taking the ratio of the 
      # sum and the average
      if ((VMware::VMServerd::product() eq "GSX") && ($max_samples < $num_samples) &&
          $vm && $vm->is_connected()) {
        my $statname = "Status.stats.vm.uptime";
        my $statref = $vm->get($statname, $num_samples);
        if (defined($statref)) {
          my %stat = %$statref;
          my $sum = $stat{$methods{"sum"}}->{$num_samples};
          my $avg = $stat{$methods{"avg"}}->{$num_samples};
          my $samples = ($avg > 0) ? int($sum / $avg) : 0;
          if ($samples > $max_samples) {
            $max_samples = $samples;
            $system_vm = $vm;
          }
        }
      }
    }

    # If we didnt find a reasonable vm, we wont attach system stats
    if ((VMware::VMServerd::product() eq "GSX") && $system_vm && $system_vm->is_connected()) {
      append_stats($root, $system_vm);
    }

    append_system($out);

    return 1;
}


# Used to support the VMControl_ServerVMQuery() function
sub VMQuery {
    my ($username, $vmList, $queries) = @_;
    my $i = 0;
    my @results;

    # Set the per request data.  Needed to get accessVM to work.
    &VMware::VMServerd::setRequestData('USERNAME', $username);

    $i = 0;
    foreach my $cfg (@$vmList) {
	my $config;
	my $vm;
	if (!defined($cfg) || !VMware::VMServerd::accessVM($cfg, 4)) {
	    # Skip this VM
	    foreach (1..scalar(@$queries)) {
		$results[$i++] = undef;
	    }
	    next;
	}

	# First try to see if the VM is connected.  Then try getting the information
	# from the config file.
	$vm = VMware::VMServerd::findVM($cfg);
	if (!defined($vm) || !$vm->is_connected()) {
	    undef($vm);

	    if (-e $cfg) {
		$config = new VMware::Config;
		$config->readin($cfg);
	    }
	}

	if (!defined($vm) && !defined($config)) {
	    # Skip this VM
	    foreach (1..scalar(@$queries)) {
		$results[$i++] = undef;
	    }
	    next;
	}

	foreach my $query (@$queries) {
	    my $val;
	    if ($query) {
		if (defined($vm)) {
		    $val = $vm->get($query);
		} else {
		    # No running VMX process for this VM
		    if ($query =~ /^[Cc]onfig\.(\S+)$/) {
			# Config. queries
			$val = $config->get($1);
		    } elsif ($query =~ /^[Ss]tatus.(\S+)$/) {
			# Status. queries
			my $key = $1;
			if ($key eq 'power') {
			    my $state = VMware::VMServerd::getVMState($cfg);
			    if ($state != $VMSTATE_ERROR) {
				$val = $VMSTATE_NAMES{$state};
			    }
			} elsif ($key eq 'id') {
			    $val = -1;
			}
		    } else {
			&VMware::VMServerd::Warning("Info.pm: VMQuery received invalid query $query " .
						    "for config file $cfg.\n");
		    }
		}
	    }
	    $results[$i++] = $val;
	} # foreach my $query (@$queries)
    }

    # Clear the per request data.  Needed to get accessVM to work.
    &VMware::VMServerd::setRequestData('USERNAME', undef);

    return(\@results);
}


sub append_product {
  my ( $root ) = @_;
  my $product = "$VMware::VMServerd::PRODUCT Version $VMware::VMServerd::VERSION";
  my $p = VMware::VMServerd::product();

  $root = $root->appendChild( $doc->createElement( "product" ) );
  $root->setAttribute( "id", $p );
  $root->appendChild( $doc->createTextNode( $product ) );
}

# append_stats --------------------------------------------------------------
#
# <stats>
#   <cpu>
#     ---
#   </cpu>
#   ...
# </stats>
# ---------------------------------------------------------------------------
sub append_stats {
  my ( $root, $server ) = @_;
  &VMware::Management::Util::append_uptime( $doc, $root, $server, "Server" );
  &VMware::Management::Util::append_cpu( $doc, $root, $server, "Server" );
  &VMware::Management::Util::append_ram( $doc, $root, $server, "Server" );
}

# append_vm -----------------------------------------------------------------
#
#    $doc ... XML::DOM document object
#   $root ... XML::DOM node to which the VM should be appended
#    $cfg ... /path/to/vm/.cfg file
# $server ... VMware::Control::Server object
#
# Given a document object and a parent node, append an XML description of the
# VM $vm to the parent node.
#
# <vm cfg="_">
#   <safe_cfg>---</safe_cfg>
#   <displayName>
#     ---
#   </displayName>
#   <os>
#     ---
#   </os>
#   <state>
#     ---
#   </state>
# </vm>
#
# ---------------------------------------------------------------------------
sub append_vm {
  my $out = shift( );
  my $cfg = shift( );
  
  my $display_name;
  my $guest_os;
  my $state;
  my $error;
  my $vm_username;

  my ( $vm, $d, $root, $err, $errstr, $connected );
  
  $vm = VMware::VMServerd::findVM($cfg);
  $connected = defined($vm) && $vm->is_connected();

  $d = VMware::DOMAccess->new("vm");
  $d->setValue('.safe_cfg', VMware::VMServerd::toSafeCfgString($cfg));
  $root = $d->get(".");
  my $doc = $d->getDocument();

  $root->setAttribute( "cfg", $cfg );

  ($state, $error) = VMware::VMServerd::getVMState($cfg);
  if ($state == $VMSTATE_ERROR) {
    &VMware::Management::Util::append_error( $doc, $root, undef, $error );
  } elsif ($state == $VMSTATE_STUCK) {
    &VMware::Management::Util::append_vm_question( $doc, $root, $vm );
  }

  if ($connected) {
    my $pid = $vm->get("Status.pid");
    if (defined($pid)) { 
      $root->setAttribute("pid", $pid);
    }
    # trace( "Connected to vm object for $cfg.", "notice" );
    $vm_username = $vm->get( "Status.username" );

    &VMware::Management::Util::append_guest_state( $doc, $root, $vm );
    append_caps($doc, $root, $cfg);
    
    $display_name = $vm->get( "Config.displayName" );
    $guest_os = $vm->get( "Config.guestOS" );
  } else {
    my $config = new VMware::Config;

    if (-e $cfg) {
      if ($config->readin($cfg)) {
        $display_name = $config->get("displayName");
        $guest_os = $config->get("guestOS");
      } else {
        &VMware::Management::Util::append_error( $doc, $root, undef,
                                                 "Syntax error in config file");
      }

      append_caps($doc, $root, $cfg);
      
    } else {
      &VMware::Management::Util::append_error( $doc, $root, undef,
                                               "Config file does not exist");
    }
  }

  my $node = $root->appendChild( $doc->createElement( "displayName" ) );
  $node->appendChild( $doc->createTextNode( $display_name or $cfg ) );

  $node = $root->appendChild( $doc->createElement( "username" ) );
  $node->appendChild( $doc->createTextNode( $vm_username or "--" ) );

  $node = $root->appendChild( $doc->createElement( "os" ) );
  $node->appendChild( $doc->createTextNode( $guest_os or "--" ) );

  $node = $root->appendChild( $doc->createElement( "state" ) );
  $node->appendChild( $doc->createTextNode( $state ) );

  # append_event_log( $doc, $root, $server, $cfg );
  # append_connected_users( $doc, $root, $server, $cfg );

  if ($connected) {
    my $ret = $vm->get("Status.resume.repeatable.resumed_session");
    if (! defined($ret)) {
      $ret = "false";
    }
    # XXX To avoid confusing TRUE w/true, etc.
    if ($ret =~ /^(true|false)$/i) {
      $ret =~ tr/A-Z/a-z/;
    }
    $node->setAttribute("resumed", $ret);

    $ret = $vm->get("Config.resume.repeatable");
    if (! defined($ret)) {
      $ret = '';
    }
    # XXX To avoid confusing TRUE w/true, etc.
    if ($ret =~ /^(true|false)$/i) {
      $ret =~ tr/A-Z/a-z/;
    }
    $node = $root->appendChild($doc->createElement("resume.repeatable"));
    $node->appendChild($doc->createTextNode($ret));

    &VMware::Management::Util::append_devices( $doc, $root, $vm );
  }

  if ( $state == $VMSTATE_ON ) {
    if (VMware::VMServerd::product() eq "ESX") {
      &VMware::VMServerd::Stats::appendVM($d, $vm);
    } else {
      &VMware::Management::Util::append_cpu( $doc, $root, $vm );
      &VMware::Management::Util::append_ram( $doc, $root, $vm );
      &VMware::Management::Util::append_heartbeat( $doc, $root, $vm );
      &VMware::Management::Util::append_uptime( $doc, $root, $vm );
      &VMware::Management::Util::append_samplecount( $doc, $root, $vm );
    }
  }

  $out->addSubTree(".server", $d);
  $d->dispose();
}

# append_system -----------------------------------------------------------------
#
# Given a DOMAccess object, append the CPU info for the VM to the node.
#
# <system>
#   <cpu>number of CPUs</cpu>
#   <ram>amount of RAM on system</ram>
# </system>
#
# ---------------------------------------------------------------------------
sub append_system {
  my $out = shift;
  
  my $cpu;
  my $memMB;

  if (VMware::VMServerd::product() eq "ESX") {
      my $host = VMware::VMServerd::ESXHostInfo::getESXHost();
      my $memKB = $host->getTotalMemSize();
      $cpu = $host->getNumCPUs();
      $memMB = ($memKB - ($memKB % 1024)) / 1024;
  } else {
      my $memory = &VMware::VMServerd::GSXHostInfo::getMemoryOnHost();
      $cpu = &VMware::VMServerd::GSXHostInfo::getCPUOnHost();
      $memMB = $memory->{TOTAL};
  }

  $out->setValue(".system.cpu", $cpu);
  $out->setValue(".system.ram", $memMB);
}

sub append_caps {
  my($doc, $root, $cfg) = @_;

  ##
  ## set capabilities
  ##

  $root = $root->appendChild( $doc->createElement( "capabilities" ) );
  my $mode;
  my $node;

  $mode = &VMware::VMServerd::GetAccessBits($cfg);
  if (( $mode & 4 ) == 4 ) {
    $node = $root->appendChild( $doc->createElement( "read" ) );
  }

  if (( $mode & 2 ) == 2 ) {
    $node = $root->appendChild( $doc->createElement( "write" ) );
  }

  if (( $mode & 1 ) == 1 ) {
    $node = $root->appendChild( $doc->createElement( "execute" ) );
  }

  $root->setAttribute( "mode", $mode );
}


VMware::VMServerd::addOperation( OPNAME => 'VM_INFO',
				 PERLFUNC => 'VMware::VMServerd::Info::VM_Info_Handler',
                                 POLICY => "authuser" );

1;
