#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# Mac.pm
# 
# MAC address database.
#

package VMware::VMServerd::Mac;
  
use strict;
use Carp;
use VMware::DOMAccess;
use VMware::VMServerd qw(&Log &Warning);
use VMware::Control::VM;
use VMware::VMServerd::VMList;

my %reserve; # config => MAC reservations
my $reservation_expire = 60; #Seconds before a reservation expires

sub clear_reservation($) {
    my $cfg= shift;

    undef $reserve{$cfg};
}

sub add_reservation($$) {
    my $cfg = shift;
    my $mac = shift;
    
    my $time;
    my @macs;

    if ($reserve{$cfg}) {
        ($time, @macs) = @{$reserve{$cfg}};
    }

    $time = time();

    push(@macs, $mac);
    unshift(@macs, $time);

    $reserve{$cfg} = \@macs;
}

#Returns the cfg that has this MAC reserved, if any
sub is_reserved($) {
    my $mac = shift;

    foreach my $cfg (keys(%reserve)) {
        my $ref = $reserve{$cfg};
        if($ref) {
            my ($time, @macs) = @{$ref};
            
            if (time() - $time > $reservation_expire) {
                clear_reservation($cfg);
            } elsif (grep($_ eq $mac, @macs)) {
                return $cfg;
            }
        }
    }
    return undef;
}

sub search_for_mac($$) {
    my $exclude_cfg = shift;
    my $target_mac = shift;

    my $canon_exclude_cfg = &VMware::VMServerd::GetCanonicalPath($exclude_cfg);

    # Get the list of all registered VMs.
    my @vms = VMware::VMServerd::VMList::ListAllVMs();
    for (my $i=0; $i <= $#vms; $i++) {
        $vms[$i] = &VMware::VMServerd::canonicalize($vms[$i]);
    }
    foreach my $cfg (@vms) {

        if ($cfg eq $canon_exclude_cfg) {
            next;
        }

        my $vm = VMware::VMServerd::findVM($cfg);
        if ($vm && $vm->is_connected()) {
            # Make sure it's not the same VM
            # XXX this shouldn't be necessary, now that we are comparing canonically above
            my $cfg2 = $vm->get("Config.filename");
            if ($cfg2 eq $exclude_cfg) {
                next;
            }
            
            # Ask VM for it's MAC address(es)
            my $dev_ref = $vm->get("Status.devices");
            my @devices = ();
            if (!defined($dev_ref)) {
                &VMware::VMServerd::Log("couldn't get devices from VMX: $cfg\n");
            } else {
                @devices = @{$dev_ref};
            }
                
            for my $device (@devices) {
                # XXX I don't think this works
                #     We need to read the VM-assigned MAC
                if ( $device =~ m/^ethernet/ ) {
                    my $mac = $vm->get("Config." . $device . ".address");
                    if (defined($mac) && ($mac =~ /^(\s*)(\S+)(\s*)$/)) {
			$mac = $2;
			if (lc($mac) eq lc($target_mac)) {
			    VMware::VMServerd::Log("MAC addresses matched: $cfg\n");
			    return $cfg;
			}
		    }
                    $mac = $vm->get("Config." . $device . ".generatedAddress");
                    if (defined($mac) && ($mac =~ /^(\s*)(\S+)(\s*)$/)) {
                        $mac = $2;
                        if (lc($mac) eq lc($target_mac)) {
                            VMware::VMServerd::Log("MAC addresses matched: $cfg\n");
                            return $cfg;
                        }
                    }	
                }
            }
        } else {
            # Pull out of .cfg or .std file
	    my $std = VMware::VMServerd::getVMSuspendFile($cfg);

	    if ($std) {
		# Pull out of suspend-to-disk file
		# XXX call Beng's code
		  my @macs = VMware::VMServerd::GetMACFromCPT($std);
		  foreach my $mac (@macs) {
		      if (defined($mac) && ($mac =~ /^(\s*)(\S+)(\s*)$/)) { 
			  $mac = $2;
			  if (lc($mac) eq lc($target_mac)) {
			      VMware::VMServerd::Log("MAC addresses matched: $cfg\n");
			      return $cfg;
			  }
		      }
                }
            } else {
                # Pull out of .cfg file
                my $config = new VMware::Config;
                
                if (-e $cfg) {
                    if ($config->readin($cfg)) {
                        # XXX we should use device_list()
                        for my $i (0..10) {
                            my $mac = $config->get("ethernet" . $i .".address");
			      if (defined($mac) && ($mac =~ /^(\s*)(\S+)(\s*)$/)) {
				  $mac = $2;
				  if (lc($mac) eq lc($target_mac)) {
				      VMware::VMServerd::Log("MAC addresses matched: $cfg\n");
				      return $cfg;
				  }
			      }
                        }
                    }
                    
                    for my $i (0..10) {
                      my $mac = $config->get("ethernet" . $i .".generatedAddress");
                      if (defined($mac) && ($mac =~ /^(\s*)(\S+)(\s*)$/)) {
                        $mac = $2;
                        if (lc($mac) eq lc($target_mac)) {
                          VMware::VMServerd::Log("MAC addresses matched: $cfg\n");
                          return $cfg;
                        }
                      }
                    }
                }
            }

        }
    }
    return undef;
}

sub ReserveMAC($$) {
    my $cfg = shift;
    my $mac = shift;

    my $ret = request_mac($cfg, $mac);

    if (!defined($ret)) {
        return "CONFLICT";
    }
    return $ret;
}

sub request_mac($$) {
    my $cfg = shift;
    my $mac = shift;

    my $canon_cfg = &VMware::VMServerd::GetCanonicalPath($cfg);
    $cfg = $canon_cfg if defined($canon_cfg);

    my $other_cfg = search_for_mac($cfg, $mac);

    if (defined($other_cfg)) {
        #This MAC is in use by another VM
        #The VM could be powered on, suspended, or have this MAC
        #  in it's .cfg file
        &VMware::VMServerd::Log( "MAC Address conflict: $cfg denied use of $mac because $other_cfg is using it.\n");
        return undef;
    }

    $other_cfg = is_reserved($mac);

    # Both $other_cfg and $cfg are canonicalized, so direct comparison is OK.
    if($other_cfg && ($other_cfg ne $cfg) ) {
        #This MAC is reserved by another VM
        #That is, the VM is just about to use it
        #XXX this case should never happen since the vmnet module should
        #    catch it first
        &VMware::VMServerd::Log( "MAC address reservation conflict between $cfg and $other_cfg\n");
        return undef;
    }

    #OK, go ahead and reserve it
    add_reservation($cfg, $mac);

    return $mac;
}

1;
