#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# Stats.pm
# 
# VM and host statistics.
#

package VMware::VMServerd::Stats;
use strict;
use Carp;
use VMware::DOMAccess;
use VMware::Control::VM;
use VMware::Management::Util qw( trace
                                 file_to_string
                                 string_to_file
                                 hash_credentials );
use VMware::VMServerd qw(Warning);

my %stats;         # Hash where stats are kept

# Flag to turn on/off writing stats to /tmp/stats files.
my $gDebug = 0;

# samplePeriod = number of seconds between samples
# sampleRate = number of samples per minute
#  therefore: 60%samplePeriod == 0
my $samplePeriod = 10;
my $sampleRate = 6;

# Time when statistics gathering began.  
#
# XXX NOT IMPLEMENTED
my $start_time;    # not currently used

my $nonce = 0;     # Counter indicating when a stat was collected
my $consoleID;     # The world ID of the console

# Hash pointing counting the number of warnings for a proc file.  If above
# 10 warnings, stop printing the warnings.
my %WarningsCount;

# XXX Old stats are not cleaned up if a world goes away.  Use time field
#     to determine when to perform cleanup.

# XXX Calculation of Max, Min, Mean should make use of the time field

# XXX If serverd handles a long request, samples will be taken at irregular
#     intervals.

# Stats that are currently collected
#
# Per VM (console included):
#   vm.cpuTime, vm.cpuUsage, vm.cpuActiveShares, vm.cpuShares, vm.cpuActivePercent
#   vm.memSize, vm.memUsage, vm.memActivePercent, vm.memActive,
#   vm.netTotalSize, vm.netSize, vm.netUsage
#
# System:
#   system.cpuUsage, system.memUsage, system.uptime
#
# See the sample functions below for detail about what each statistic means.

################################
###
###  XML Output routines
###
################################


sub GetVM_Handler {
  my $in = shift;
  my $out = shift;

  my $cfg = $in->getValue(".cfg");
  if (!defined($cfg)) {
    return 0;
  }

  my $vm = VMware::VMServerd::findVM($cfg);
  if (defined($vm) && $vm->is_connected()) {
    my $doc = $out->getSubTree(".");
    appendVM($doc, $vm);
    $doc->dispose();    
  }

  return 1;
}

# Gets all available stats information for a particular VM.
#
# Input format:
#   .worldID := the world ID for which to retrieve the stats
#   .stat := the name of the statistic to retrieve
#   .current_time := the current nonce
# 
# Output format:
#   .samples.sample[] := the value of the sample.  Includes an attribute indicating 
#                        the time/nonce at which the sample was taken
#
# XXX This function is slow and should not be called in a production environment.
#
sub GetStats_Handler {
  my $in = shift;
  my $out = shift;

  my $count = 0;

  my $worldID = $in->getValue('.worldID');
  my $stat = $in->getValue('.stat');

  if (!isStatGroup($worldID)) {
      &VMware::VMServerd::errorAppend("No stats available for world $worldID");
      return(0);
  }

  if (!isStat($worldID, $stat)) {
      &VMware::VMServerd::errorAppend("Could not find '$stat' stat for world $worldID");
      return(0);
  }

  my @samples = @{ $stats{$worldID}{$stat}{'samples'} };
  my @times = @{ $stats{$worldID}{$stat}{'times'} };

  if( $#samples != $#times ) {
      &VMware::VMServerd::errorAppend("Number of samples (" . ($#samples + 1) . ") and times (" . 
				      ($#times + 1) . ") does not match");
      return(0);
  }

  my $out_count;
  for( $out_count = 0, $count = $#samples; $count >= 0; $count--, $out_count++ ) {
    $out->setValue(".samples.sample[$out_count]", $samples[$count]);
    $out->setAttribute(".samples.sample[$out_count]", 'time', $times[$count]);
  }

  $out->setValue(".current_time", $nonce);

  return(1);
}


# Gets information about all the stats that are available for a VM.
#
# Input format:
#   .world := a world name for which to query the stats
# 
# Output format:
#   .stat.name := the name of the statistic
#   count (attribute) := the number of samples for this statistic
#   last (attribute) := the last nonce value for this statistic (if there
#                   are any samples)
sub GetStatInfo_Handler {
  my $in = shift;
  my $out = shift;

  my $count = 0;

  my $worldID = $in->getValue('.worldID');

  if (!isStatGroup($worldID)) {
      &VMware::VMServerd::errorAppend("No stats available for world $worldID");
      return(0);
  }

  my @stats = sort(keys(%{ $stats{$worldID} }));

  my $stat;
  foreach $stat (@stats) {
      my $num = getNumSamples($worldID, $stat);
      my ($time, $sample) = getLastSample($worldID, $stat);

      $out->setValue(".stat[$count]", $stat);
      $out->setAttribute(".stat[$count]", 'count', $num);
      if( defined($time) ) {
	  $out->setAttribute(".stat[$count]", 'last', $time);
      }
      $count++;
  }

  return(1);
}


# Gets information about all the stats that are available.
#
# Input format:
#   No input args.
# 
# Output format:
#   .current_time := the current nonce
#   .worlds.world[].worldID := the world ID for the statistic.  also (SYSTEM)
#   .worlds.world[].stats := the stats information for this world
#   
#   See GetStatInfo_Handler for more information about the stats that
#   are added.
#
sub GetAllStatInfo_Handler {
  my $in = shift;
  my $out = shift;

  my $count = 0;

  # Ignore input args.
  $out->setValue(".current_time", $nonce);

  my @worlds = sort(keys(%stats));

  my $worldID;
  foreach $worldID (@worlds) {
      $out->setValue(".worlds.world[$count].worldID", $worldID);

      my $doc_in = VMware::DOMAccess->new('in');
      $doc_in->setValue('.worldID', $worldID);

      $out->setValue(".worlds.world[$count].stats", '');
      my $node = $out->getSubTree(".worlds.world[$count].stats");
      if( !GetStatInfo_Handler( $doc_in, $node ) ) {
	  &VMware::VMServerd::errorAppend("GetAllStatInfo_Handler failed for world $worldID.");
	  return(0);
      }
      $node->dispose();

      $doc_in->dispose();

      $count++;
  }

  return(1);
}


sub appendSystem {
  my($out) = @_;

  my $doc = $out->getSubTree(".server");
  
  appendStat($doc, "SYSTEM", "cpu", "percent", "system.cpuUsage");
  appendStat($doc, "SYSTEM", "ram", "percent", "system.memUsage");

  appendSystemUptime($doc);
  
  $doc->dispose();
}

sub appendSystemUptime {
  my($out) = @_;

  # XXX If stats are requested before the SYSTEM stats have a chance to be 
  # initialized, we'll have an temporarily erroneous uptime of 0. -jhu
  my($time, $s) = (0, 0);
  if (isStatGroup("SYSTEM")) {
    ($time, $s) = getLastSample("SYSTEM", "system.uptime");
  }

  $out->setValue(".uptime.days", int($s / (24*3600)));
  $out->setValue(".uptime.hours", ($s / 3600) % 24);
  $out->setValue(".uptime.minutes", ($s / 60) % 60);
  $out->setValue(".uptime.seconds", $s % 60);
}

sub appendConsole {
  my($out) = @_;

  if (!defined($consoleID)) {
    return;
  }
  
  my $doc = VMware::DOMAccess->new("console");

  appendStat($doc, $consoleID, "cpu", "percent", "vm.cpuUsage");
  appendStat($doc, $consoleID, "ram", "percent", "vm.memUsage");  
  
  $out->addSubTree(".server", $doc);
  $doc->dispose();
}

sub appendVM {
  my($out, $vm) = @_;

  if (!defined($vm) || !$vm->is_connected()) {
    return;
  }
  
  my($vmid) = $vm->get("Status.id");

  $out->setAttribute(".", "id", $vmid);

  appendStat($out, $vmid, "cpu", "percent", "vm.cpuUsage");
  appendStat($out, $vmid, "ram", "percent", "vm.memUsage");

  my $root = $out->get(".");
  my $doc = $out->get(".")->getOwnerDocument();
  
  &VMware::Management::Util::append_heartbeat( $doc, $root, $vm );
  &VMware::Management::Util::append_uptime( $doc, $root, $vm);
}


sub appendStat {
  my($out, $vmid, $name, $units, $stat) = @_;
  
  my $doc = VMware::DOMAccess->new("stat");

  $doc->setAttribute(".", "name", $name);
  $doc->setAttribute(".", "units", $units);

  appendUsage($doc, $vmid, $stat);

  $out->addSubTree(".", $doc);
  $doc->dispose();
}

##
## min max mean
## 1, 5, 15 minute output
## assumes stat is a percent kept in seconds
##

sub appendUsage {
  my($out, $vmid, $stat) = @_;

  my @ops = (["min", \&getMin],
             ["avg", \&getMean],
             ["max", \&getMax]
            );

  foreach my $op (@ops) {
    my ($type, $func) = @{$op};
    my $doc = VMware::DOMAccess->new("usage");
    $doc->setAttribute(".", "type", $type, 1);

    foreach my $interval (1, 5, 15) {
      appendSample($doc, $vmid, $stat, $func, $interval);
    }
    
    $out->addSubTree(".", $doc);
    $doc->dispose();
  }
}

# interval in minutes
# stat is assumed to be in seconds and a percentage

sub appendSample {
  my($out, $vmid, $stat, $func, $interval) = @_;

  # If the stats for the VM has not been initialized, we won't produce a
  # stats node for the VM.
  if (!isStatGroup($vmid)) {
    return;
  }

  my $doc = VMware::DOMAccess->new("sample");

  my $sample = $func->($vmid, $stat, $interval*$sampleRate);
  if( defined($sample) ) {
    $doc->setValue(".", int(100 * $sample), 1);
    $doc->setAttribute(".", "interval", $interval, 1);
    $doc->setAttribute(".", "units", "minutes", 1);
  
    $out->addSubTree(".", $doc);
  }

  $doc->dispose();
}


################################
###
###  ESX Specific Routines
###
################################

my $rootDir = "/proc/vmware";
my %totalMem = ( 'console' => 0, 'vmkernel' => 0 );
my $numCPUs;

# ESX Init VM

sub initVM {
  my($vmid) = @_;
  initStat($vmid, "AGE", 1);

  initStat($vmid, "vm.cpuTime", 15);
  initStat($vmid, "vm.cpuUsage", 15);
  initStat($vmid, "vm.cpuActiveShares", 15);
  initStat($vmid, "vm.cpuShares", 15);
  initStat($vmid, "vm.cpuActivePercent", 15);

  initStat($vmid, "vm.memSize", 15);
  initStat($vmid, "vm.memUsage", 15);
  initStat($vmid, "vm.memActivePercent", 15);
  initStat($vmid, "vm.memActive", 15);

  initStat($vmid, "vm.netTotalSize", 15);
  initStat($vmid, "vm.netSize", 15);
  initStat($vmid, "vm.netUsage", 15);
}

# ESX periodic routines

sub incNonce() {
    $nonce++;
}

sub getNonce() {
    return($nonce);
}

sub StatsCallback {
  if (!isStatGroup("SYSTEM")) {
    initStat("SYSTEM", "system.cpuUsage", 15);
    initStat("SYSTEM", "system.memUsage", 15);
    initStat("SYSTEM", "system.uptime", 15);
  }

  # Perform this check after the initialization of the SYSTEM stats
  # so that we don't get uninitalized data errors when the vmkernel
  # is not loaded.
  if (!is_vmkernel_running()) {
    return(0);
  }

  sampleSystem();

  if (!defined($consoleID)) {
    foreach my $worldID (getProcWorlds()) {
       my $file = getCPUFile($worldID);
       next if !(-f $file);

       my $stats = parseKeyValueStatusFile($file);
       if ($stats->{'name'} eq 'console') {
          $consoleID = $worldID;
          last;
       }
    }
   
    if( defined($consoleID) ) {
	initVM($consoleID);
    }
  }

  # Sample each world
  my $worldID;
  foreach $worldID (getProcWorlds()) {
      if (!isStatGroup($worldID)) {
	  &initVM($worldID);
      }
      &sampleVM($worldID);
  }
  
  cleanupOldVMs();

  # Console is special since it has to look elsewhere for its memory info
  # The CPU information was already sample in the previous loop
  sampleConsoleMemUsage($consoleID);
  
  &incNonce();

  return 1;
}

sub sampleSystem {
  sampleSystemCPUUsage();
  sampleSystemMemUsage();
}

my $lastTotal;
my $lastUsed;

# The amount of time that has elapsed in the last time interval
my $lastSampleInterval;

sub getProcWorlds {
  opendir(VMS, "/proc/vmware/vm/") || die "can't open /proc/vmware/vm/";
  my @worlds = grep(!/^\.\.?/, readdir(VMS)); 
  closedir(VMS);
  return @worlds;
}

# Adds the following samples:
# system.cpuUsage := total CPU usage
# system.uptime := total number of seconds the system has been up
sub sampleSystemCPUUsage {
  my $uptime = getProc("/proc/vmware/uptime");
  if (!defined($uptime)) {
    $uptime = 0;
  } else {
    chop($uptime);
  }

  my($time, $lastUptime) = getLastSample("SYSTEM", "system.uptime");
  
  if ($lastUptime) {
    $lastSampleInterval = $uptime - $lastUptime;
  }
  addSample("SYSTEM", "system.uptime", $uptime);

  my $total = $uptime;
  my $used = 0;

  my $str = getProc("/proc/vmware/sched/idle");
  my @lines = split(/\n/, $str);
  shift(@lines);
  foreach my $line (@lines) {
    $line =~ s/^\s*//;
    my @fields = split(/\s+/, $line);
    $used += $fields[2];
  }

  if (defined($lastTotal) && ($total - $lastTotal)) {
    my $v = (1.0 * ($used - $lastUsed)) / (($total - $lastTotal) * getNumCPUs());
    if ($v > 1.0) {
      $v = 1.0;
    }
    addSample("SYSTEM", "system.cpuUsage", $v);  
  }

  $lastTotal = $total;
  $lastUsed = $used;  
}

# Adds the following samples:
# system.memUsage := total Mem usage
sub sampleSystemMemUsage {
  my $total = getTotalMem();
  my $avail = getAvailVMKMem();
  my $v = ($total > 0) ? ($total - $avail) / $total : 0;
  addSample("SYSTEM", "system.memUsage", $v);  
}

sub sampleVM {
  my($vmid) = @_;
  return if ($vmid == -1); 
  sampleCPUUsage($vmid);
  sampleMemUsage($vmid);
  sampleNetUsage($vmid);
  addSample($vmid, "AGE", 1);
}


# Adds the following samples:
#  vm.cpuTime: the usedsec field in cpu/status
#  vm.cpuUsage: the percentage used in the last period
#  vm.cpuActiveShares: the number of shares that are active for the VM
#  vm.cpuShares := the number of shares that are active for the VM
#  vm.cpuActivePercent: the percentage that the VM is active
sub sampleCPUUsage {
  my($vmid) = @_;
  my($newTime, $oldTime, $v, $time);

  my $file = getCPUFile($vmid);
  if( ! -f $file ) {
      return;
  }

  my $stats = parseKeyValueStatusFile($file);

  if (!defined($stats->{'vm'})) {
    return;
  }

  # vm.cpuTime
  $newTime = $stats->{'usedsec'};
  ($time, $oldTime) = getLastSample($vmid, "vm.cpuTime");
  addSample($vmid, "vm.cpuTime", $newTime);

  # vm.cpuUsage
  if (($time != getNonce()) && defined($lastSampleInterval)) {
    $v = (1.0 * ($newTime - $oldTime)) / $lastSampleInterval;
    if ($v > 1.0) {
      $v = 1.0;
    }

    addSample($vmid, "vm.cpuUsage", $v);
  }

  # vm.cpuActiveShares
  # XXX No longer reported by status proc node
  addSample($vmid, "vm.cpuActiveShares", 0);
  
  # vm.cpuShares
  addSample($vmid, "vm.cpuShares", $stats->{'shares'} / 100);

  # vm.cpuActivePercent
  # XXX No longer reported by status proc node
  addSample($vmid, "vm.cpuActivePercent", 0);
}

# Adds the following samples:
#  vm.memSize: the size field in mem/status
#  vm.memUsage: the percentage used over the total amount of memory
#  vm.memActivePercent: the percent of memory touched by the VM
#  vm.memActive: the amount of memory touched by the VM
sub sampleMemUsage {
  my($vmid) = @_;
  my($v);

  my $file = getMemFile($vmid);
  if( ! -f $file ) {
      return;
  }

  my $stats = parseKeyValueStatusFile($file);

  if( !defined($stats->{'vm'}) ) {
      return;
  }

  my $size = $stats->{'size'};
  my $active = $stats->{'active'};
  my $total = getTotalMem() * 1024.0;

  # vm.memSize
  $v = ($size / 1024) / 100;
  addSample($vmid, "vm.memSize", $v);

  # vm.memUsage
  $v = ($total > 0) ? $size / $total : 0;
  addSample($vmid, "vm.memUsage", $v);

  # vm.memActivePercent
  $v = ($size > 0) ? $active / $size : 0;
  addSample($vmid, "vm.memActivePercent", $v);

  # vm.memActive
  $v = ($active / 1024) / 100;
  addSample($vmid, "vm.memActive", $v);
}

# Adds the following samples:
#  vm.netTotalSize: the total number of kilobits transfered by the VM
#  vm.netSize: the number of kilobits transfered by the VM
#  vm.netUsage: the percentage of allocated bandwidth used over
sub sampleNetUsage {
  my($vmid) = @_;
  my($v);

  my $file = getNetFile($vmid);
  if( !defined($file) || ! -f $file ) {
      return;
  }

  my ($rc, $forward, $avg_bandwidth, $peak_bandwidth, $burst_size, $period_peak,
      $max_queued_packets, $packets_sent, $bytes_sent, $packetsDropped,
      $bytesDropped, $avgBucket_TokensLeft, $avgBucket_TokensTotal,
      $avgBucket_TokensNum, $avgBucket_TokensPeriod, $peakBucket_TokensLeft,
      $peakBucket_TokensTotal, $peakBucket_TokensNum, $peakBucket_TokensPeriod) =
	parseNfshaperFile($file);

  # vm.netSize (in kbps)
  if (defined($lastSampleInterval)) {
    my ($time, $oldSize) = getLastSample($vmid, "vm.netTotalSize");
    $v = ($bytes_sent - $oldSize);
    if( $v < 0 ) {
	# When a filter is detached and then attached, the bytes sent drops to zero.
	$v = $bytes_sent;
    }
    $v = $v / $lastSampleInterval;
    # Multiply by 8 to get from bytes to bits
    $v = $v * 8;
    addSample($vmid, "vm.netSize", $v / 1000 / 100);  # kbps

    # vm.netUsage
    $v = $v / $avg_bandwidth;
    if( $v > 1.0 ) {
	$v = 1.0;
    }
    addSample($vmid, "vm.netUsage", $v);
  }

  # vm.netTotalSize (in kbytes)
  addSample($vmid, "vm.netTotalSize", $bytes_sent);
}

sub sampleConsoleMemUsage {
  my($vmid) = @_;
  my $size = getTotalConsoleMem();
  addSample($vmid, "vm.memSize", $size / 100);

  my $total = getTotalMem();
  my $v = ($total > 0) ? $size / $total : 0;
  addSample($vmid, "vm.memUsage", $v);
}

### ESX?? accessor function

sub getNumCPUs {
  if (defined($numCPUs)) {
    return $numCPUs;
  }

  local(*FD);
  if (!open(FD, "/proc/vmware/sched/ncpus")) {
    Warning("Cannot open /proc/vmware/sched/ncpus. Reason: $!");
    return 1;
  } else {
     while (<FD>) {
        if (/(\d+)\s+physical/) {
           $numCPUs = $1;
        }
     }
    close(FD);
    return $numCPUs;
  }
}

sub getTotalVMKMem {
  if (($totalMem{'vmkernel'} <= 0) && is_vmkernel_running()) {
    my($str) = getProc("/proc/vmware/mem");
    if(defined($str)) {
      foreach (split('\n', $str)) {
        if (/^Machine\s*memory\s*free:\s*(\d+)\s*Mbytes\/\s*(\d+)\s*Mbytes/) {
          $totalMem{'vmkernel'} = $2;
          last;
        }
      }
    }
  }
  return $totalMem{'vmkernel'};
}

sub getAvailVMKMem {
  my($str) = getProc("/proc/vmware/mem");
  if( !defined($str) ) {
      return(0);
  }

  foreach (split('\n', $str)) {
     if (/^Machine\s*memory\s*free:\s*(\d+)\s*Mbytes\/\s*(\d+)\s*Mbytes/) {
        return $1;
     }
  }
  return 0;
}

sub getTotalConsoleMem {
  if ($totalMem{'console'} <= 0) {
    $totalMem{'console'} = getProcField("/proc/meminfo", 1, 1) / (1024*1024);
  }
  return $totalMem{'console'};
}

sub getTotalMem {
  return getTotalConsoleMem() + getTotalVMKMem();
}

sub getCPUFile {
  my($vmid) = @_;
  return("$rootDir/vm/$vmid/cpu/status");
}

sub getMemFile {
  my($vmid) = @_;
  return("$rootDir/vm/$vmid/mem/status");
}

sub getNetFile {
  my($vmid) = @_;
  local(*FD);
  if( !opendir(FD, "/proc/vmware/filters/xmit") ) {
      return(undef);
  }
  my(@files) = grep(/^nfshaper\.[0-9]+\.$vmid/, readdir(FD));
  closedir(FD);
  if( $#files > -1 ) {
      return('/proc/vmware/filters/xmit/' . $files[0]);
  } else {
      return(undef);
  }
}
    

### ESX proc stuff

sub is_vmkernel_running() {
  # The filters directory appears pretty late in the VMkernel initialization
  # sequence.
  return(-e "/proc/vmware/filters");
}

sub getProcField {
  my($fileName, $field, $line) = @_;
  my $buf = getProc($fileName);
  if(!defined($buf)) {
      die "Could not open proc file $fileName";
  }
  return getField($buf, $field, $line);
}

sub getField {
  my($buf, $field, $line) = @_;

  if (defined($line)) {
    my @lines = split(/\n/, $buf);
    if (!defined($lines[$line])) {
      die "Could not parse line $line field $field.";
    }
    $buf = $lines[$line];
  }

  # chop leading whitespace
  $buf =~ s/^\s*//;

  my @fields = split(/\s+/, $buf);
  if (!defined($fields[$field])) {
    die "XXX";
  }
  
  return $fields[$field];
}

sub getProc {
  my($fileName) = @_;
  my($ret, $buf);

  $ret = open(VMSTATUS, $fileName);
  if (!$ret) {
    # Only print warnings for a file 10 times.  Otherwise, our logs will be filled with useless noise.
    if( !defined($WarningsCount{$fileName}) ) {
      $WarningsCount{$fileName} = 1;
    } elsif( $WarningsCount{$fileName} > 9 ) {
      return;
    } else {
      $WarningsCount{$fileName} += 1;
    }
    # This happens more often than one would think so we cannot just
    # simply die here.
    Warning("Stats::getProc: Could not open file $fileName.  Reason: $!\n");
    return(undef);
  }
  
  $ret = read(VMSTATUS, $buf, 4096);
  if (!$ret) {
    die "XXX";
  }
  
  close(VMSTATUS);

  # Refresh the warning count
  $WarningsCount{$fileName} = 0;

  return $buf;
}

if (VMware::VMServerd::product() eq "ESX") {
   # Definitely a source of memory hog, and since we are not using it this
   # stats module in the new MUI, silence it as a workaround -- vui
   #
   #VMware::VMServerd::RegisterPollCallback(VMware::VMServerd::POLL_PERIODIC,
   #                                      $samplePeriod*1000, \&StatsCallback);

  VMware::VMServerd::addOperation( OPNAME => 'Stats_GetVM',
                                   PERLFUNC => 'VMware::VMServerd::Stats::GetVM_Handler',
				   POLICY => 'authuser' );

  if( $gDebug ) {
    # Only register this function if we are not in debug mode
    VMware::VMServerd::addOperation( OPNAME => 'Stats_GetStats',
				     PERLFUNC => 'VMware::VMServerd::Stats::GetStats_Handler',
				     POLICY => 'authuser' );
  }

  VMware::VMServerd::addOperation( OPNAME => 'Stats_GetStatInfo',
                                   PERLFUNC => 'VMware::VMServerd::Stats::GetStatInfo_Handler',
				   POLICY => 'authuser' );

  VMware::VMServerd::addOperation( OPNAME => 'Stats_GetAllStatInfo',
                                   PERLFUNC => 'VMware::VMServerd::Stats::GetAllStatInfo_Handler',
				   POLICY => 'authuser' );

}



################################
###
###  Generic Stats Stuff
###
################################

#yuck, clean this up
sub cleanupOldVMs {
  my $oldage = 15 * $sampleRate; # one hour
  foreach my $vm (keys(%stats)) {
    if (($vm ne "SYSTEM")
        && ($stats{$vm}{AGE}{'times'}[0] < ($nonce-$oldage))) {
      delete($stats{$vm});
    }
  }
}

sub addSample {
  my ($vm, $stat, $sample) = @_;
  unshift(@{$stats{$vm}{$stat}{'samples'}}, $sample);
  unshift(@{$stats{$vm}{$stat}{'times'}}, $nonce);
  splice(@{$stats{$vm}{$stat}{'samples'}}, $stats{$vm}{$stat}{'history'} + 1);
  splice(@{$stats{$vm}{$stat}{'times'}}, $stats{$vm}{$stat}{'history'} + 1);

  DebugWrite($vm, $stat, $sample);
}

sub getLastSample {
  my ($vm, $stat) = @_;
  if (!isStat($vm, $stat)) {
      return (undef, undef);
  }
  if( !defined($stats{$vm}{$stat}{'samples'}[0]) || !defined($stats{$vm}{$stat}{'times'}[0]) ) {
      return(undef, undef);
  }
  return($stats{$vm}{$stat}{'times'}[0], $stats{$vm}{$stat}{'samples'}[0]);
}

sub getNumSamples {
  my ($vm, $stat) = @_;
  if (!isStat($vm, $stat)) {
      return 0;
  }
  my @samples = @{$stats{$vm}{$stat}{'samples'}};
  return($#samples + 1);
}

sub getMin {
  my ($vm, $stat, $period) = @_;
  my($min) = 0;

  if (!isStat($vm, $stat)) {
      return 0;
  }
  foreach my $i (0..$period) {
    if (defined($stats{$vm}{$stat}{'samples'}[$i])
        && (!defined($min) || ($stats{$vm}{$stat}{'samples'}[$i] < $min))) {
      $min = $stats{$vm}{$stat}{'samples'}[$i];
    }
  }
  return $min;
}

sub getMax {
  my ($vm, $stat, $period) = @_;
  my($max) = 0;

  if (!isStat($vm, $stat)) {
      return 0;
  }
  foreach my $i (0..$period) {
    if (defined($stats{$vm}{$stat}{'samples'}[$i])
        && (!defined($max) || ($stats{$vm}{$stat}{'samples'}[$i] > $max))) {
      $max = $stats{$vm}{$stat}{'samples'}[$i];
    }
  }
  return $max;
}

sub getMean {
  my ($vm, $stat, $period) = @_;
  my $sum = 0;
  my $num = 0;

  if (!isStat($vm, $stat)) {
      return 0;
  }
  foreach my $i (0..$period) {
    if (defined($stats{$vm}{$stat}{'samples'}[$i])) {
      $sum += $stats{$vm}{$stat}{'samples'}[$i];
      $num++;
    }
  }
  if (!$num) {
    return 0;
  }
  return ($sum/$num);
}

sub initStat {
  my ($vm, $stat, $history) = @_;

  if (!defined($history)) {
    $history = 15;
  }

  # history is given in minutes so convert to samples
  $history = $history*$sampleRate;
  
  $stats{$vm}{$stat}{'samples'} = [];
  $stats{$vm}{$stat}{'times'} = [];
  $stats{$vm}{$stat}{'history'} = $history;
  
  foreach my $i (1..$history) {
    addSample($vm, $stat, 0);
  }

  DebugInit($vm, $stat);
}

# Does this group of statistics exist?
sub isStatGroup {
  my ($vm) = @_;
  return( defined($stats{$vm}) );
}

# Is this a defined stat?
sub isStat {
  my ($vm, $stat) = @_;
  return( defined($stats{$vm}) && defined($stats{$vm}{$stat}) );
}

################################
###
###  Debugging
###
################################
sub DebugInit {
    my($vm, $stat) = @_;

    if( !$gDebug ) {
	return;
    }

    my $file = "/tmp/stats-$vm-$stat";

    local(*DFD);
    if( !open(DFD, "> $file") ) {
	print STDERR "Could not open file $file for writing.  Reason: $!\n";
	return;
    }

    close(DFD);
}

sub DebugWrite {
    my($vm, $stat, $sample) = @_;

    if( !$gDebug ) {
	return;
    }

    my $file = "/tmp/stats-$vm-$stat";

    local(*DFD);
    if( !open(DFD, ">> $file") ) {
	print STDERR "Could not open file $file for append.  Reason: $!\n";
	return;
    }

    print DFD "$nonce: $sample\n";

    close(DFD);
}



################################
###
###  Parsing routines
###
###  XXX These should not be in this file.  They were taken from
###  VMResources.pm.  However, we do not want to introduce a 
###  dependency on VMResources in Stats.pm as it is a common
###  VMServerd module.  When Stats does finally get split into
###  an ESX Server and GSX Server portion, the routines can be
###  removed.  -jhu
################################

# Parses a key-value status file into a hash
# * Keys taken from the header line
# * Values taken from the data line
sub parseKeyValueStatusFile($) {
    my($file) = @_;
    my @keys = ();
    my @values = ();

    local(*FD);
    if( !open(FD, "< $file") ) {
	Warning("Could not get status information because could not open file $file.  Reason: $!");
	return(undef);
    }

    while(<FD>) {
       # Ignore obj debug information for each line
       # The debug info is demarcated by a '|' character on a line
       s/\|.*//g;
       
       # Replace '/' with ' ' and split line into an array
       s^/^ ^g;
       if (!@keys) {
          @keys = split;
       } else {
          @values = split;
       }
    }

    close(FD);

    # Perl idiom for building a hash from key/value lists
    my %stats = ();
    @stats{@keys} = @values;

    return \%stats;
}

# Parses the nfshaper file returning the following variables in a list:
#
# ($rc, $forward, $avg_bandwidth, $peak_bandwidth, $burst_size, $period_peak,
#  $max_queued_packets, $packets_sent, $bytes_sent, $packetsDropped,
#  $bytesDropped, $avgBucket_TokensLeft, $avgBucket_TokensTotal,
#  $avgBucket_TokensNum, $avgBucket_TokensPeriod, $peakBucket_TokensLeft,
#  $peakBucket_TokensTotal, $peakBucket_TokensNum, $peakBucket_TokensPeriod);

sub parseNfshaperFile($) {
    my($file) = @_;
    my $cNumMatches = 11;

    my ($forward, $avg_bandwidth, $peak_bandwidth, $burst_size, $period_peak,
	$max_queued_packets, $packets_sent, $bytes_sent, $packetsDropped,
	$bytesDropped, $avgBucket_TokensLeft, $avgBucket_TokensTotal,
	$avgBucket_TokensNum, $avgBucket_TokensPeriod, $peakBucket_TokensLeft,
	$peakBucket_TokensTotal, $peakBucket_TokensNum, $peakBucket_TokensPeriod);

    if( ! -f $file ) {
	Warning("Could not find file $file.");
	return(0);
    }

    local(*FD);
    if( !open(FD, "< $file") ) {
	Warning("Could not open file $file.  Reason: $!");
	return(0);
    }

    my $matchCount = 0;
    while(<FD>) {
	if( /forwards: (\S+)/ ) {
	    $forward = $1;
	} elsif( /([0-9]+) bpsAverage/ ) {
	    $avg_bandwidth = $1;
	    $matchCount++;
	} elsif( /([0-9]+) bpsPeak/ ) {
	    $peak_bandwidth = $1;
	    $matchCount++;
	} elsif( /([0-9]+) burstBytes/ ) {
	    $burst_size = $1;
	    $matchCount++;
	} elsif( /([0-9]+) periodPeak/ ) {
	    $period_peak = $1;
	    $matchCount++;
	} elsif( /([0-9]+) maxQueuedPackets/ ) {
	    $max_queued_packets = $1;
	    $matchCount++;
	} elsif( /([0-9]+) packetsSent/ ) {
	    $packets_sent = $1;
	    $matchCount++;
	} elsif( /([0-9]+) bytesSent/ ) {
	    $bytes_sent = $1;
	    $matchCount++;
	} elsif( /([0-9]+) packetsDropped/ ) {
	    $packetsDropped = $1;
	    $matchCount++;
	} elsif( /([0-9]+) bytesDropped/ ) {
	    $bytesDropped = $1;
	    $matchCount++;
	} elsif( /([0-9]+) \/\s*([0-9]+) avgTokens\s*\(\s*([0-9]+) \/\s*([0-9]+) ms\)/ ) {
	    $avgBucket_TokensLeft = $1;
	    $avgBucket_TokensTotal = $2;
	    $avgBucket_TokensNum = $3;
	    $avgBucket_TokensPeriod = $4;
	    $matchCount++;
	} elsif( /([0-9]+) \/\s*([0-9]+) peakTokens\s*\(\s*([0-9]+) \/\s*([0-9]+) ms\)/ ) {
	    $peakBucket_TokensLeft = $1;
	    $peakBucket_TokensTotal = $2;
	    $peakBucket_TokensNum = $3;
	    $peakBucket_TokensPeriod = $4;
	    $matchCount++;
	}
    }

    close(FD);

    if( $matchCount < $cNumMatches ) {
	Warning("Did not find all the information expected from nfshaper file $file.  " .
		"Expected $cNumMatches fact(s) but got $matchCount fact(s).");
    }

    if( $matchCount > 0 ) {
	return (1,
		$forward, 
		$avg_bandwidth, 
		$peak_bandwidth, 
		$burst_size, 
		$period_peak,
		$max_queued_packets, 
		$packets_sent, 
		$bytes_sent, 
		$packetsDropped,
		$bytesDropped, 
		$avgBucket_TokensLeft, 
		$avgBucket_TokensTotal,
		$avgBucket_TokensNum, 
		$avgBucket_TokensPeriod, 
		$peakBucket_TokensLeft,
		$peakBucket_TokensTotal, 
		$peakBucket_TokensNum, 
		$peakBucket_TokensPeriod);
    } else {
	return(0);
    }
}



#
# perlAPI $vm->get() interface support functions
# 

#######################
# GetStatsList()
#
# This Function returns a comma delmited list of all the
# statistics for a given vm.  
#
#######################

sub GetStatsList($) {
  my $worldID = shift;


  if (!isStatGroup($worldID)) {
      Warning("No stats available for world $worldID");
      return undef;
  }
  my @statList = sort(grep(!/AGE/,keys(%{ $stats{$worldID} })));

  for(my $i=0; $i < @statList; $i++) {
     $statList[$i] =~ s/^[^.]*\.(.*)/$1/;
  }
  return join(',',@statList); 
}


1;
