#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################
#
# TaskManager.pm
# 
# Managing long running tasks
# 
package VMware::VMServerd::TaskManager;

use strict;

use VMware::VMServerd qw(&errorReset
			 &haveError 
			 &errorAppend 
			 &errorPost 
			 &Log
			 &Panic
			 &Warning);

#############################################################
# Globals specific to this serverd module.
#############################################################

use Exporter   ();
use vars       qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

BEGIN {
    # set the version for version checking
    $VERSION     = 1.00;
    @ISA         = qw(Exporter);
    @EXPORT      = qw(&TaskManager_NewTask
		      &TaskManager_DeleteTask
		      &TaskManager_SetArgs
		      &TaskManager_StartTask
		      &TaskManager_StopTask
		      &TaskManager_StatusOfTask
		      &TaskManager_GetTaskStdout
		      &TaskManager_GetTaskStderr
		      &TaskManager_RedirectTaskPipe
		      &TaskManager_RedirectTaskPipeFile
		      &TaskManager_GetTaskFlags
		      &TaskManager_SetTaskFlags

		      TASK_FLAG_SETSID
		      );
    #@EXPORT_OK   =
    #%EXPORT_TAGS = ( tag => [set of symbols] )
    
}

# XXX We really should derive these constants from the C files. -jhu
sub TASK_FLAG_SETSID { 1 }

my $PKG_FULL_NAME;
my $PKG_NAME;

my %gTasks;

sub TaskManager_NewTask($$$) {
    my ($undoFunc, $statusFunc, $cleanupFunc) = @_;

    my ($ok, $err, $taskid) = &VMware::VMServerd::NewTask();

    if (!$ok) {
	return (0, $err);
    }

    $gTasks{$taskid} = {UNDOFUNC => $undoFunc,
			STATUSFUNC => $statusFunc,
			CLEANUPFUNC => $cleanupFunc
			};

    return (1, "", $taskid);
}

sub TaskManager_DeleteTask($) {
    my ($taskid) = @_;

    my ($ok, $err) = &VMware::VMServerd::DeleteTask($taskid);

    if (!$ok) {
	return (0, $err);
    }

    $gTasks{$taskid} = undef;

    return 1;
}

sub TaskManager_SetArgs($$) {
    my ($taskid, $args) = @_;

    $gTasks{$taskid}->{ARGS} = $args;
}

sub TaskManager_StartTask($$@) {
    my ($taskid, $execPath, @args) = @_;

    if (!defined($gTasks{$taskid})) {
	return (0, "Invalid task id");
    }

    # Add an empty string so XS can detect the end of argument list
    push(@args, "");

    if (!-x $execPath) {
	return (0, "Executable $execPath does not exists or is not executable;");
    }

    my ($ok, $err) = &VMware::VMServerd::StartTask($taskid, $execPath, @args);

    if (!$ok) {
	return (0, $err);
    }

    return 1;
}

sub TaskManager_StopTask($) {
    my ($taskid) = @_;

    if (!defined($gTasks{$taskid})) {
	return (0, "Invalid task id");
    }
    
    my ($ok, $err) = &VMware::VMServerd::StopTask($taskid);

    if (!$ok) {
	return (0, $err);
    }

    my $args = $gTasks{$taskid}->{ARGS};
    my $undoFunc = $gTasks{$taskid}->{UNDOFUNC};

    if (!defined($undoFunc)) {
	return 1;
    }

    ($ok, $err) = &$undoFunc($taskid, $args);

    if (!$ok) {
	return (0, $err);
    }

    return 1;
}

sub TaskManager_StatusOfTask($) {
    my ($taskid) = @_;

    if (!defined($gTasks{$taskid})) {
	return (0, "Taskid is invalid");
    }
    
    my ($ok, $err, $status, $info, $taskStderr) = &VMware::VMServerd::StatusOfTask($taskid);

    if (!$ok) {
	if ($err =~ /invalid/i) {
	    $gTasks{$taskid} = undef;
	}
	return (0, $err);
    }

    if ($status eq "running") {
	my $args = $gTasks{$taskid}->{ARGS};
	my $statusFunc = $gTasks{$taskid}->{STATUSFUNC};
	my ($ok, $err, $percent);
	if (defined($statusFunc)) {
	    ($ok, $err, $percent) = &$statusFunc($taskid, $args);
	    if (!defined($percent)) {
		# Possible if the child process just started running and 
		# hasn't printed anything to output yet.
		$percent = 0;
	    }
	} else {
	    ($ok, $err, $percent) = (1, 0, 0);
	}
	return (1, "", $status, $percent, undef);
    }

    if ($status eq "start_failed") {
	return (1, "", $status, $info, undef);
    }

    if ($status eq "cleanup") {
        return (1, "", $status);
    }

    if (($status eq "exited") || ($status eq "interrupted") || ($status eq "aborted")) {
	return (1, "", $status, $info, $taskStderr);
    }

    if ($status eq "unknown") {
	return (0, "Unknown status returned by task manager");
    }

    return (0, "Task Status unknown");
}

sub TaskManager_GetTaskStdout($) {
    my ($taskid) = @_;

    if (!defined($gTasks{$taskid})) {
	return (0, "Taskid is invalid");
    }

    my $taskStdout = &VMware::VMServerd::GetTaskStdout($taskid);
    return (1, "", $taskStdout);
}

sub TaskManager_GetTaskStderr($) {
    my ($taskid) = @_;

    if (!defined($gTasks{$taskid})) {
	return (0, "Taskid is invalid");
    }

    my $taskStderr = &VMware::VMServerd::GetTaskStderr($taskid);
    return (1, "", $taskStderr);
}

sub TaskManager_RedirectTaskPipe($$$) {
    my ($taskid, $srcPipe, $dstPipe) = @_;

    if (!defined($gTasks{$taskid})) {
	return (0, "Taskid is invalid");
    }

    return &VMware::VMServerd::RedirectTaskPipe($taskid, $srcPipe, $dstPipe);
}

sub TaskManager_RedirectTaskPipeFile {
    my ($taskid, $srcPipe, $filename, $inDirection, $mode) = @_;

    if (!defined($gTasks{$taskid})) {
	return (0, "Taskid is invalid");
    }

    return &VMware::VMServerd::RedirectTaskPipeFile($taskid, $srcPipe, $filename, $inDirection, $mode);
}

sub TaskManager_GetTaskFlags {
    my ($taskid) = @_;

    if (!defined($gTasks{$taskid})) {
	return (0, "Taskid is invalid");
    }

    return &VMware::VMServerd::GetTaskFlags($taskid);
}

sub TaskManager_SetTaskFlags {
    my ($taskid, $flags) = @_;

    if (!defined($gTasks{$taskid})) {
	return (0, "Taskid is invalid");
    }

    return &VMware::VMServerd::SetTaskFlags($taskid, $flags);
}

sub TaskManager_CleanupTask {
    my($taskid) = @_;
   
    if (!defined($gTasks{$taskid})) {
	return 0;
    }
    
    my $args = $gTasks{$taskid}->{ARGS};
    my $cleanupFunc = $gTasks{$taskid}->{CLEANUPFUNC};

    if ($cleanupFunc) {
        my ($ok, $err) =  &$cleanupFunc($taskid, $args);
	return !$ok ? 0 : 1;
    }

    return 1;
}

############################################################
#
# VMServerd handlers
#
############################################################

#  External API
#  Each entry is: Operation Name, Auth. Policy
#

my $API = [
	        ["StatusOfTask", "authuser"],
	        ["StopTask", "authuser"],
	        ["DeleteTask", "authuser"]
	  ];

sub reportError($) {
    my ($err) = @_;

    my ($package, $filename, $line, $callerSub) = caller(1);

    my @words = split("::", $callerSub);
    
    my $operation = $words[$#words];

    $operation =~ s/(.*)_Handler/$1/;

    # multiple error strings are separated by ';'
    my @errors = split(/;/, $err);

    # remove leading nulls. split only removes trailing nulls.
    while (scalar(@errors) > 0 && $errors[0] eq "") {
	shift(@errors);
    }

    my $e = shift(@errors);
    errorPost($PKG_NAME . "_" . $operation . ": " . $e, "error");

    while (scalar(@errors) > 0) {
	$e = shift(@errors);
	if ($e =~ /^\s+$/) {
	    next;
	}
	errorPost($e, "error");
    }
}

####################################################################################
#
# StatusOfTask:
#   Get the status of a task
#
# Input format:
#   <in>
#      <taskid> ID of the task </taskid>
#   </in>
#
# Output format: 
#   <out>
#      <status> running | succeeded | failed </status>
#      <percent> number between 0,...,99 </percent>    ;; present if status == running
#      <error>  Error message </error>                 ;; if status == failed
#      <info> additional info </info>                  ;; optional for all status cases
#   </out>
#
####################################################################################
sub StatusOfTask_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my $taskid = $in->getValue(".taskid");

    if (!defined($taskid) || ($taskid eq "")) {
	reportError("Taskid not specified");
	return 0;
    }

    my ($ok, $err, $status, $info, $taskerr) = TaskManager_StatusOfTask($taskid);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    $out->setValue(".status", $status);

    if ($status eq "running") {
	$out->setValue(".percent", $info);
    } 
    elsif ($status eq "exited") {
	$out->setValue(".exitstatus", $info);
	if (defined($taskerr)) {
	    $out->setValue(".taskStderr", $taskerr);
	}
    }
    elsif ($status eq "interrupted") {
	$out->setValue(".interruptsignal", $info);
	if (defined($taskerr)) {
	    $out->setValue(".taskStderr", $taskerr);
	}
    }
    elsif ($status eq "aborted") {
	if (defined($taskerr)) {
	    $out->setValue(".taskStderr", $taskerr);
	}
    }
    elsif ($status eq "start_failed") {
	$out->setValue(".error", $info);
    }

    return 1;
}

####################################################################################
#
# StopTask:
#   Stop a running task
#
# Input format:
#   <in>
#      <taskid> ID of the task </taskid>
#   </in>
#
# Output format: 
#
####################################################################################
sub StopTask_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my $taskid = $in->getValue(".taskid");

    if (!defined($taskid) || ($taskid eq "")) {
	reportError("Taskid not specified");
	return 0;
    }

    my ($ok, $err) = TaskManager_StopTask($taskid);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    return 1;
}


####################################################################################
#
# DeleteTask:
#   Delete a task entry
#
# Input format:
#   <in>
#      <taskid> ID of the task </taskid>
#   </in>
#
# Output format: 
#
####################################################################################
sub DeleteTask_Handler($$) {
    my ($in, $out) = @_;

    errorReset();

    my $taskid = $in->getValue(".taskid");

    if (!defined($taskid) || ($taskid eq "")) {
	reportError("Taskid not specified");
	return 0;
    }

    my ($ok, $err) = TaskManager_DeleteTask($taskid);

    if (!$ok) {
	reportError($err);
	return 0;
    }

    return 1;
}

#############################################################
# Define operations and the Perl functions to be executed
# when a request for such an operation is received.
# Permissions information is also indicated here.
#############################################################

my $i;
my @array = @$API;
for ($i=0; $i <= $#array; $i++) {
    my $arr = $API->[$i];

    $PKG_FULL_NAME = __PACKAGE__;
    my @components = split(/::/, $PKG_FULL_NAME);
    $PKG_NAME = $components[$#components];

    my $opname  = $PKG_NAME . "_" . $arr->[0];
    my $handler = $PKG_FULL_NAME . "::" . $arr->[0] . "_Handler";
    my $policy  = $arr->[1];

    &VMware::VMServerd::addOperation( OPNAME => $opname,
				      PERLFUNC => $handler,
				      POLICY => $policy);
}


1;
