#!/usr/bin/perl -w

#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# VMDeleteEdit.pm
# 
# Deletes a VM
#

package VMware::VMServerd::VMDeleteEdit;

use strict;
use VMware::ExtHelpers qw(&getWarnings &clearWarnings &internal_dirname);
use VMware::VMServerd::VMDeleteInfo qw(&canDeleteVM 
                                       &getDisksForVM 
                                       &getFilesToDelete
                                       &buildDiskUsageHash 
                                       &getDiskUsage 
                                       &getDeleteTimestamp);

		      
use VMware::VMServerd qw(&Warning);
use VMware::VMServerd::Disk;


# Delete a VM.  This function will delete a VM and the disks for the VM.
# See the DeleteVMInfo_Handler in VMDeleteInfo.pm for more information 
# about the delete preparation.
#
# This function only deletes a VMFS disk if the VM is the last one to use it.
#
# Input format:
#   .config := the config file to be deleted
#   .timestamp := a timestamp marking
#   .disks.disk[] := a list of VMFS disks that are not to be deleted
#
# Output format:
#   No output.
#
sub DeleteVM_Handler {
    my $in = shift;
    my $out = shift;
    my $disk;

    &VMware::VMServerd::errorReset();

    my @diskDelete;          # List of disks to delete

    my $config = $in->getValue('.config');
    my $timestamp = $in->getValue('.timestamp');

    if( $timestamp !~ /^[0-9]+$/ ) {
	&VMware::VMServerd::errorPost("Timestamp $timestamp is not valid. " .
                                      "It must be a number greater than 0.");
	return(0);
    }
    if( $timestamp == '0' ) {
	&VMware::VMServerd::errorPost("Cannot delete VM. Timestamp is invalid.");
	return(0);
    }

    # Make sure VM is not running and the validity of the timestamp
    my ($rc, $err) = canDeleteVM($config);
    if( !$rc ) {
	&VMware::VMServerd::errorPost("Cannot delete VM.  $err");
	return(0);
    }

    # Verify the timestamp.  The timestamp is the latest modification
    # time of the config file that shares disks with the VM to be deleted.
    my $usageHash = buildDiskUsageHash();
    my @diskList = getDisksForVM($config);

    my @diskNameList = map { $_->{CANONICAL_NAME} } @diskList;
    my $curTimestamp = getDeleteTimestamp($usageHash, $config, \@diskNameList);
    if( $curTimestamp == 0 ) {
	&VMware::VMServerd::errorPost('Unable to calculate timestamp for deletion.');
	return(0);
    }

    if( $curTimestamp != $timestamp ) {
	Warning("DeleteVM_Handler: Old delete VM timestamp $timestamp detected.  The current " . 
		"delete timestamp is $curTimestamp");
	&VMware::VMServerd::errorPost("The VM or one of the files that it shares its disks with was modified.  " .
				      "If you are sure that this is ok, you may try again.");
	return(0);
    }

    # Build the list of disks to save
    my @disk_keys = $in->listElementNames('.disks');
    my $key;
    foreach $key (@disk_keys) {
	$disk = $in->getValue('.disks.' . $key);
	if( !grep($_ eq $disk, @diskDelete) ) {
	    push(@diskDelete, $disk);
	}
    }
    @diskDelete = VMware::VMServerd::Disk::getUniqCanonicalDiskList(\@diskDelete, $config);

    # Delete the disks
    foreach $disk (@diskNameList) {
	# Only delete disks if they are not being used by another VM and 
        # the user has specified that he wishes to delete the disk.
	my @usage = getDiskUsage($usageHash, $disk, $config);
	if( $#usage > 0 ) {
	    next;
	}

	if( grep($_ eq $disk, @diskDelete) ) {

	    if( !VMware::VMServerd::Disk::RemoveDisk($disk) ) {
		appendWarnings();
		&VMware::VMServerd::errorPost("Failed to delete disk $disk.");
	      }
        }
    }

    # Delete the files
    my @deleteFiles = getFilesToDelete($config);
    my $file;
    foreach $file (@deleteFiles) {
	if( -d $file ) {
	    &VMware::VMServerd::errorAppend("Did not delete $file. " .
                                            "It is a directory.");
	    next;
	}
	if( !unlink($file) ) {
	    &VMware::VMServerd::errorAppend("Could not delete $file. ".
                                            "Reason: $!.");
	}
    }

    # Now Unregister the VM
    &VMware::VMServerd::VMList::becomeRootAndUnRegister($config);

    # Now delete the config file
    if( -e $config && -f $config ) {
        if (!unlink($config)) {
	    &VMware::VMServerd::errorAppend("Could not delete $file. " . 
					"Reason: $!.");
        }
    }

    removeDirectoryIfEmpty(internal_dirname($config));

    if (&VMware::VMServerd::haveError()) {
        &VMware::VMServerd::errorPost("Delete VM was partially successful. ");
    }

    return(1);
}

sub removeDirectoryIfEmpty($) {
    my ($dir) = @_;

    return 0 if (! -d $dir);
    return 0 if (!opendir(DIRECTORY, $dir));
    my @files = readdir(DIRECTORY);
    closedir(DIRECTORY);

    my @filesToDelete = grep(!/^\.\.$|^\.$/, @files);
    if (scalar(@filesToDelete) == 0) {
        # The directory is empty
        if (! rmdir($dir)) {
	    return 0;
	}
	return 1;
    }
    # The directory is not empty
    return 0;
}

# Appends warnings to be passed back to the user
sub appendWarnings() {
    my @errors = &getWarnings();
    my $error;
    foreach $error (@errors) {
	&VMware::VMServerd::errorPost($error);
    }
    clearWarnings();
}


&VMware::VMServerd::addOperation( OPNAME => 'VMDeleteEdit_DeleteVM',
				  PERLFUNC => 'VMware::VMServerd::VMDeleteEdit::DeleteVM_Handler',
				  POLICY => 'authuser');

1;
