/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

#ifndef _X86CPUID_H_
#define _X86CPUID_H_

/* http://www.sandpile.org/ia32/cpuid.htm */

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_VMX
#define INCLUDE_ALLOW_VMMEXT
#define INCLUDE_ALLOW_VMKERNEL
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_VMNIXMOD
#define INCLUDE_ALLOW_DISTRIBUTE
#define INCLUDE_ALLOW_VMK_MODULE
#define INCLUDE_ALLOW_VMCORE
#define INCLUDE_ALLOW_VMMON
#include "includeCheck.h"

/*
 * FLAGDEF can be defined to process the CPUID information provided in
 * the following macros.  The first parameter is the bit location of
 * the feature.  The second parameter is the name of the feature.
 *
 * The third and fourth parameters are the default CPUID behavior for
 * power-on, guest view, and migration tests (cpt/rsm & vmotion).
 * Parameter 3 is the default, while parameter 4 is the default if LM
 * is disabled.
 *
 *
 * When adding a new feature bit, be sure to consider its purpose.
 * The following list of mask characters is provided in order of
 * likely use.
 *
 * H - Important to the guest and cannot change during migration.
 * 0 - Hide from the guest, because we don't support it or we don't
 *     want the guest to know that it exists.
 * T - Require host CPUID bit to be set (1) for power-on (i.e. TSC)
 *
 *     (Think twice before using these last three mask types)
 *
 * 1 - Force the guest to always see & don't compare for migration --
 *     only APIC as of today; it is controlled by software and we
 *     know how to toggle it
 * F - Require host CPUID bit to NOT be set (0) for power-on
 * X - Ignore this bit for all tests
 *
 * Table to explain mask character meanings:
 *
 *                         X   0   1   F   T   H
 * ---------------------------------------------
 * Req'd val for power-on  -   -   -   0   1   -
 * Value guest sees        *   0   1   *   *   *
 * Checked for migration?  N   N   N   Y   Y   Y
 *
 * * - initial host's power-on CPUID value
 *
 */

/*
 * CPUID Leaf 0x1, Register EDX (the original feature field)
 */

#define CPUID_DATA_COMMON_ID1EDX                             \
FLAGDEF(0,  CPUID_FEATURE_COMMON_ID1EDX_FPU,       'H', 'H') \
FLAGDEF(1,  CPUID_FEATURE_COMMON_ID1EDX_VME,       'H', 'H') \
FLAGDEF(2,  CPUID_FEATURE_COMMON_ID1EDX_DBGE,      'H', 'H') \
FLAGDEF(3,  CPUID_FEATURE_COMMON_ID1EDX_PGSZE,     'H', 'H') \
FLAGDEF(4,  CPUID_FEATURE_COMMON_ID1EDX_TSC,       'T', 'T') \
FLAGDEF(5,  CPUID_FEATURE_COMMON_ID1EDX_MSR,       'H', 'H') \
FLAGDEF(6,  CPUID_FEATURE_COMMON_ID1EDX_PAE,       'H', 'H') \
FLAGDEF(7,  CPUID_FEATURE_COMMON_ID1EDX_MCK,       'H', 'H') \
FLAGDEF(8,  CPUID_FEATURE_COMMON_ID1EDX_CPMX,      'H', 'H') \
FLAGDEF(9,  CPUID_FEATURE_COMMON_ID1EDX_APIC,      '1', '1') \
FLAGDEF(11, CPUID_FEATURE_COMMON_ID1EDX_SEP,       'H', 'H') \
FLAGDEF(12, CPUID_FEATURE_COMMON_ID1EDX_MTTR,      'H', 'H') \
FLAGDEF(13, CPUID_FEATURE_COMMON_ID1EDX_PGE,       'H', 'H') \
FLAGDEF(14, CPUID_FEATURE_COMMON_ID1EDX_MCA,       'H', 'H') \
FLAGDEF(15, CPUID_FEATURE_COMMON_ID1EDX_CMOV,      'H', 'H') \
FLAGDEF(16, CPUID_FEATURE_COMMON_ID1EDX_PAT,       'H', 'H') \
FLAGDEF(17, CPUID_FEATURE_COMMON_ID1EDX_36PG,      'H', 'H') \
FLAGDEF(19, CPUID_FEATURE_COMMON_ID1EDX_CLFL,      'H', 'H') \
FLAGDEF(23, CPUID_FEATURE_COMMON_ID1EDX_MMX,       'H', 'H') \
FLAGDEF(24, CPUID_FEATURE_COMMON_ID1EDX_FXSAVE,    'H', 'H') \
FLAGDEF(25, CPUID_FEATURE_COMMON_ID1EDX_SSE,       'H', 'H') \
FLAGDEF(26, CPUID_FEATURE_COMMON_ID1EDX_SSE2,      'H', 'H') \
FLAGDEF(28, CPUID_FEATURE_COMMON_ID1EDX_HT,        '0', '0')

#define CPUID_DATA_INTEL_ID1EDX                              \
FLAGDEF(18, CPUID_FEATURE_INTEL_ID1EDX_PSN,        'H', 'H') \
FLAGDEF(21, CPUID_FEATURE_INTEL_ID1EDX_DTES,       'H', 'H') \
FLAGDEF(22, CPUID_FEATURE_INTEL_ID1EDX_ACPI,       'H', 'H') \
FLAGDEF(27, CPUID_FEATURE_INTEL_ID1EDX_SS,         'H', 'H') \
FLAGDEF(29, CPUID_FEATURE_INTEL_ID1EDX_TM,         '0', '0') \
FLAGDEF(30, CPUID_FEATURE_INTEL_ID1EDX_IA64,       '0', '0') \
FLAGDEF(31, CPUID_FEATURE_INTEL_ID1EDX_PBE,        '0', '0')

#define CPUID_DATA_AMD_ID1EDX

/*
 * CPUID Leaf 0x1, Register ECX (Intel has started using this for features)
 */

#define CPUID_DATA_COMMON_ID1ECX                             \
FLAGDEF(0,  CPUID_FEATURE_COMMON_ID1ECX_SSE3,      'H', 'H') \
FLAGDEF(9,  CPUID_FEATURE_COMMON_ID1ECX_SSE4,      'H', 'H') \
FLAGDEF(13, CPUID_FEATURE_COMMON_ID1ECX_CMPX16,    'H', '0')

#define CPUID_DATA_INTEL_ID1ECX                              \
FLAGDEF(3,  CPUID_FEATURE_INTEL_ID1ECX_MWAIT,      '0', '0') \
FLAGDEF(4,  CPUID_FEATURE_INTEL_ID1ECX_DSCPL,      'H', 'H') \
FLAGDEF(5,  CPUID_FEATURE_INTEL_ID1ECX_VMX,        '0', '0') \
FLAGDEF(7,  CPUID_FEATURE_INTEL_ID1ECX_EST,        '0', '0') \
FLAGDEF(8,  CPUID_FEATURE_INTEL_ID1ECX_TM2,        '0', '0') \
FLAGDEF(10, CPUID_FEATURE_INTEL_ID1ECX_HTCACHE,    '0', '0') \
FLAGDEF(14, CPUID_FEATURE_INTEL_ID1ECX_xPPR,       '0', '0') \
FLAGDEF(14, CPUID_FEATURE_INTEL_ID1ECX_NDA16,      '0', '0')

#define CPUID_DATA_AMD_ID1ECX

/*
 * CPUID Leaf 0x80000001, Register EDX (AMD has always stored their
 * proprietary features here, AMD64 has forced Intel to start putting
 * feature bits here as well)
 */

#define CPUID_DATA_COMMON_ID81EDX                            \
FLAGDEF(11, CPUID_FEATURE_COMMON_ID81EDX_SYSC,     'H', 'H') \
FLAGDEF(20, CPUID_FEATURE_COMMON_ID81EDX_NX,       'H', 'H') \
FLAGDEF(29, CPUID_FEATURE_COMMON_ID81EDX_LM,       'H', '0')

#define CPUID_DATA_INTEL_ID81EDX

#define CPUID_DATA_AMD_ID81EDX                               \
FLAGDEF(0,  CPUID_FEATURE_AMD_ID81EDX_FPU,         'H', 'H') \
FLAGDEF(1,  CPUID_FEATURE_AMD_ID81EDX_VME,         'H', 'H') \
FLAGDEF(2,  CPUID_FEATURE_AMD_ID81EDX_DBGE,        'H', 'H') \
FLAGDEF(3,  CPUID_FEATURE_AMD_ID81EDX_PGSZE,       'H', 'H') \
FLAGDEF(4,  CPUID_FEATURE_AMD_ID81EDX_TSC,         'T', 'T') \
FLAGDEF(5,  CPUID_FEATURE_AMD_ID81EDX_MSR,         'H', 'H') \
FLAGDEF(6,  CPUID_FEATURE_AMD_ID81EDX_PAE,         'H', 'H') \
FLAGDEF(7,  CPUID_FEATURE_AMD_ID81EDX_MCK,         'H', 'H') \
FLAGDEF(8,  CPUID_FEATURE_AMD_ID81EDX_CPMX,        'H', 'H') \
FLAGDEF(9,  CPUID_FEATURE_AMD_ID81EDX_APIC,        '1', '1') \
FLAGDEF(12, CPUID_FEATURE_AMD_ID81EDX_MTTR,        'H', 'H') \
FLAGDEF(13, CPUID_FEATURE_AMD_ID81EDX_PGE,         'H', 'H') \
FLAGDEF(14, CPUID_FEATURE_AMD_ID81EDX_MCA,         'H', 'H') \
FLAGDEF(15, CPUID_FEATURE_AMD_ID81EDX_CMOV,        'H', 'H') \
FLAGDEF(16, CPUID_FEATURE_AMD_ID81EDX_PAT,         'H', 'H') \
FLAGDEF(17, CPUID_FEATURE_AMD_ID81EDX_36PG,        'H', 'H') \
FLAGDEF(22, CPUID_FEATURE_AMD_ID81EDX_MMXEXT,      'H', 'H') \
FLAGDEF(23, CPUID_FEATURE_AMD_ID81EDX_MMX,         'H', 'H') \
FLAGDEF(24, CPUID_FEATURE_AMD_ID81EDX_FXSAVE,      'H', 'H') \
FLAGDEF(25, CPUID_FEATURE_AMD_ID81EDX_FFXSR ,      'H', 'H') \
FLAGDEF(27, CPUID_FEATURE_AMD_ID81EDX_RDTSCP,      'H', 'H') \
FLAGDEF(30, CPUID_FEATURE_AMD_ID81EDX_3DNOWPLUS,   'H', 'H') \
FLAGDEF(31, CPUID_FEATURE_AMD_ID81EDX_3DNOW,       'H', 'H')

/*
 * CPUID Leaf 0x80000001, Register ECX (AMD is starting to put new
 * feature bits in here as they ran out of space in EDX)
 */

#define CPUID_DATA_COMMON_ID81ECX                            \
FLAGDEF(0,  CPUID_FEATURE_COMMON_ID81ECX_LAHF,     'H', '0')

#define CPUID_DATA_INTEL_ID81ECX

#define CPUID_DATA_AMD_ID81ECX                               \
FLAGDEF(1,  CPUID_FEATURE_AMD_ID81ECX_CMPLEGACY,   '0', '0') \
FLAGDEF(2,  CPUID_FEATURE_AMD_ID81ECX_SVM,         '0', '0') \
FLAGDEF(3,  CPUID_FEATURE_AMD_ID81ECX_EXTAPICSPC,  'H', 'H') \
FLAGDEF(4,  CPUID_FEATURE_AMD_ID81ECX_CR8AVAIL,    '0', '0') \
FLAGDEF(8,  CPUID_FEATURE_AMD_ID81ECX_3DNPREFETCH, 'H', 'H')


/* Define all flag values as an enum */

enum {
#define FLAGDEF(bit, name, a, b)  name = (1 << bit),
   CPUID_DATA_COMMON_ID1EDX
   CPUID_DATA_COMMON_ID1ECX
   CPUID_DATA_COMMON_ID81EDX
   CPUID_DATA_COMMON_ID81ECX
   CPUID_DATA_INTEL_ID1EDX
   CPUID_DATA_INTEL_ID1ECX
   CPUID_DATA_INTEL_ID81EDX
   CPUID_DATA_INTEL_ID81ECX
   CPUID_DATA_AMD_ID1EDX
   CPUID_DATA_AMD_ID1ECX
   CPUID_DATA_AMD_ID81EDX
   CPUID_DATA_AMD_ID81ECX
#undef FLAGDEF
};


/*
 * Other constants and macros.
 */
#define CPUID_INTEL_VENDOR_STRING       "GenuntelineI"
#define CPUID_AMD_VENDOR_STRING         "AuthcAMDenti"
#define CPUID_CYRIX_VENDOR_STRING       "CyriteadxIns"
#define CPUID_INTEL_VENDOR_STRING_FIXED "GenuineIntel"
#define CPUID_AMD_VENDOR_STRING_FIXED   "AuthenticAMD"
#define CPUID_CYRIX_VENDOR_STRING_FIXED "CyrixInstead"

#define CPUID_STEPPING_MASK  0xf
#define CPUID_STEPPING(_eax) ((_eax) & CPUID_STEPPING_MASK)

/* Effective Intel CPU Families */
#define CPUID_FAMILY_486     4
#define CPUID_FAMILY_P5      5
#define CPUID_FAMILY_P6      6
#define CPUID_FAMILY_P4      15

/* Effective AMD CPU Families */
#define CPUID_FAMILY_5x86    4
#define CPUID_FAMILY_K5      5
#define CPUID_FAMILY_K6      5
#define CPUID_FAMILY_K7      6
#define CPUID_FAMILY_OPTERON 15
#define CPUID_FAMILY_K8L     16

#define CPUID_FAMILY_EXTENDED 15

#define CPUID_FAMILY(_eax)   (((_eax) >> 8) & 0xf)
#define CPUID_EXTENDED_FAMILY(_eax)   (((_eax) >> 20) & 0xff)
#define CPUID_EFFECTIVE_FAMILY(_eax)                       \
	   (CPUID_FAMILY(_eax) +                           \
	    (CPUID_FAMILY(_eax) == CPUID_FAMILY_EXTENDED   \
		? CPUID_EXTENDED_FAMILY(_eax) : 0))

/*
 * Notice that CPUID families for Intel and AMD overlap. The following macros
 * should only be used AFTER the manufacturer has been established (through
 * the use of CPUID standard function 0).
 */
#define CPUID_FAMILY_IS_486(_eax) (CPUID_FAMILY(_eax) == CPUID_FAMILY_486)
#define CPUID_FAMILY_IS_P5(_eax)  (CPUID_FAMILY(_eax) == CPUID_FAMILY_P5)
#define CPUID_FAMILY_IS_P6(_eax)  (CPUID_FAMILY(_eax) == CPUID_FAMILY_P6)
#define CPUID_FAMILY_IS_PENTIUM4(_eax)  \
                     (CPUID_EFFECTIVE_FAMILY(_eax) == CPUID_FAMILY_P4)

#define CPUID_FAMILY_IS_OPTERON(_eax) \
                     (CPUID_EFFECTIVE_FAMILY(_eax) == CPUID_FAMILY_OPTERON)

#define CPUID_FAMILY_IS_K8L(_eax) \
                     (CPUID_EFFECTIVE_FAMILY(_eax) == CPUID_FAMILY_K8L)

#define CPUID_MODEL_PPRO       1
#define CPUID_MODEL_PII_3      3
#define CPUID_MODEL_PII_5      5
#define CPUID_MODEL_CELERON_6  6
#define CPUID_MODEL_PIII_7     7
#define CPUID_MODEL_PIII_8     8
#define CPUID_MODEL_PM         9
#define CPUID_MODEL_PIII_A     10
#define CPUID_MODEL_CORE2      15
#define CPUID_MODEL_EXTENDED   15

#define CPUID_MODEL(_eax)    (((_eax) >> 4) & 0xf)
#define CPUID_EXTENDED_MODEL(_eax)    (((_eax) >> 16) & 0xf)
#define CPUID_EFFECTIVE_MODEL(_eax)                        \
	   (CPUID_MODEL(_eax) |                            \
	    (CPUID_FAMILY(_eax) == CPUID_FAMILY_EXTENDED   \
		? (CPUID_EXTENDED_MODEL(_eax) << 4) : 0))

#define CPUID_TYPE(_eax)    (((_eax) >> 12) & 0x3)
#define CPUID_TYPE_PRIMARY     0
#define CPUID_TYPE_OVERDRIVE   1
#define CPUID_TYPE_SECONDARY   2

#define CPUID_ID1EBX_LCPU_COUNT_MASK  0x00FF0000
#define CPUID_ID1EBX_LCPU_COUNT_SHIFT 16

#define CPUID_ID1EBX_APICID_MASK      0xFF000000
#define CPUID_ID1EBX_APICID_SHIFT     24
#define CPUID_APICID(_ebx)      (((_ebx) & CPUID_ID1EBX_APICID_MASK) >> \
				    CPUID_ID1EBX_APICID_SHIFT)

#define CPUID_ID4EAX_INTEL_CACHE_TYPE_MASK      0x0000001F
#define CPUID_ID4EAX_INTEL_CACHE_TYPE_SHIFT     0

#define CPUID_ID4EAX_INTEL_CACHE_TYPE_NULL      0
#define CPUID_ID4EAX_INTEL_CACHE_TYPE_DATA      1
#define CPUID_ID4EAX_INTEL_CACHE_TYPE_INST      2
#define CPUID_ID4EAX_INTEL_CACHE_TYPE_UNIF      3

#define CPUID_ID4EAX_INTEL_CACHE_LEVEL_MASK     0x000000E0
#define CPUID_ID4EAX_INTEL_CACHE_LEVEL_SHIFT    5

#define CPUID_ID4EAX_INTEL_CACHE_SELF_INIT      0x00000100
#define CPUID_ID4EAX_INTEL_CACHE_FULLY_ASSOC    0x00000200

#define CPUID_ID4EAX_INTEL_CACHE_NUMHT_SHARING_MASK     0x03FFC000
#define CPUID_ID4EAX_INTEL_CACHE_NUMHT_SHARING_SHIFT    14

#define CPUID_ID4EAX_INTEL_CORE_COUNT_MASK      0xFC000000
#define CPUID_ID4EAX_INTEL_CORE_COUNT_SHIFT     26

#define CPUID_ID4EBX_INTEL_CACHE_LINE_MASK      0x00000FFF
#define CPUID_ID4EBX_INTEL_CACHE_LINE_SHIFT     0

#define CPUID_ID4EBX_INTEL_CACHE_PART_MASK      0x003FF000
#define CPUID_ID4EBX_INTEL_CACHE_PART_SHIFT     12

#define CPUID_ID4EBX_INTEL_CACHE_WAYS_MASK      0xFFC00000
#define CPUID_ID4EBX_INTEL_CACHE_WAYS_SHIFT     22

#define CPUID_ID86ECX_AMD_CACHE_LINE_MASK       0x000000FF
#define CPUID_ID86ECX_AMD_CACHE_LINE_SHIFT      0

#define CPUID_ID86ECX_AMD_CACHE_LINE_PER_TAG_MASK       0x00000F00
#define CPUID_ID86ECX_AMD_CACHE_LINE_PER_TAG_SHIFT      8

#define CPUID_ID86ECX_AMD_CACHE_WAYS_MASK       0x0000F000
#define CPUID_ID86ECX_AMD_CACHE_WAYS_SHIFT      12

#define CPUID_ID86ECX_AMD_CACHE_SIZE_MASK       0xFFFF0000
#define CPUID_ID86ECX_AMD_CACHE_SIZE_SHIFT      16

#define CPUID_ID88EAX_PHYSBITS_MASK   0x000000FF
#define CPUID_ID88EAX_PHYSBITS_SHIFT  0
#define CPUID_ID88EAX_VIRTBITS_MASK   0x0000FF00
#define CPUID_ID88EAX_VIRTBITS_SHIFT  8

#define CPUID_ID88ECX_AMD_CORE_COUNT_MASK       0x000000FF
#define CPUID_ID88ECX_AMD_CORE_COUNT_SHIFT      0

#endif
