#
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#

package VMware::VmPerl::Server;

use strict;
use Carp;

require DynaLoader;

use vars qw($VERSION @ISA);

BEGIN{$VERSION = '1.01'};

@ISA = qw(DynaLoader);

use VMware::VmPerl $VERSION;

# Preloaded methods go here.

sub new {
   return New();
}

sub get_last_error {
    my $server = shift;
    my ($ret, $string) = $server->GetLastError();
    if (wantarray) {
        return ($ret, $string);
    } else {
        return $ret;
    }
}

sub connect {
    my $server = shift;
    my $connect_params = undef;
    $connect_params = shift if $#_ >= 0;

    my $ret = $server->Connect($connect_params);

    if ($ret) {
        return $ret;
    } else {
        return undef;
    }
}

sub is_connected {
    my $server = shift;
    return $server->IsConnected();
}

sub registered_vm_names {
    my $server = shift;
    my %hash;
    my @values = $server->Enumerate();

    # Hack to remove duplicates XXX
    my @unique_values;
    foreach my $value (@values) {
        if (!defined($hash{$value})) {
            $hash{$value} = 1;
            push @unique_values, $value;
        }
    }
    return @unique_values;
}

sub disconnect {
    my $server = shift;
    my $ret = $server->Disconnect();

    return 1;
}

sub register_vm {
    my $server = shift;
    my $cfg = shift;
    my $ret = $server->Register($cfg);
    if ($ret) {
        return $ret;
    } else {
        return undef;
    }
}

sub unregister_vm {
    my $server = shift;
    my $cfg = shift;
    my $ret = $server->Unregister($cfg);
    if ($ret) {
        return $ret;
    } else {
        return undef;
    }
}

sub is_registered {
    my $server = shift;
    my $cfg = shift;
    my ($ret, $isRegistered) = $server->IsRegistered($cfg);
    if ($ret) {
	if (!defined($isRegistered)) {
	    croak("VMware::VmPerl::Server::is_registered query succeeded but isRegistered not defined");
	}
        return $isRegistered;
    } else {
        return undef;
    }
}

sub get_resource {
   my $server = shift;
   my $key = shift;
   my $ret = $server->GetResource($key);
   return $ret;
}

sub set_resource {
   my $server = shift;
   my $key = shift;
   my $value = shift;
   my $ret = $server->SetResource($key, $value);

   if (!$ret) {
      return undef;
   } else {
      return $ret;
   }
}


#Initialize the API library and return 1 if OK
VMware::VmPerl::Server::Init();
__END__

=head1 NAME

VMware::VmPerl::Server - Programmatically manipulate a server running
VMware virtual machines.

=head1 SYNOPSIS

  use VMware::VmPerl::Server;

  # Connect to a remote machine
  $params = VMware::VmPerl::ConnectParams::new($hostname, $port, $username, $password);
  $server = VMware::VmPerl::Server::new();
  $err = $server->connect($params);

  # OR
  # Connect to the local machine as the current user
  $params = VMware::VmPerl::ConnectParams::new(undef, 0, undef, undef);
  $server = VMware::VmPerl::Server::new();
  $err = $server->connect($params);

  if (!defined($err)) {
    ($errorNumber, $errorString) = $server->get_last_error();
  }

  @vm_identifiers = $server->registered_vm_names();

  $server->disconnect();

=head1 DESCRIPTION

This package provides an interface for connecting to a VMware server
product and listing the virtual machines available on the server.

=head1 VMware::VmPerl::Server

=head2 $server->registered_vm_names()

Returns an array of strings specifying the configuration file names of the virtual machines currently registered on the server. The array is indexed beginning at zero. The object must be connected before this method can be read, otherwise an error is thrown.


=head2 $server->connect()

Connects the object to a VMware GSX or ESX server using the supplied connectparams. 

B<Syntax>

  $server->connect($connectparams) 

B<Parameters>

  $connectparams [in] A VMware::VmPerl::ConnectParams object specifying the host and user information. 

B<Remarks>

This should be the first method invoked on a C<VMware::VmPerl::Server> object. There is no method to disconnect from a server. To reconnect to a different server, destroy the C<VMware::VmPerl::Server> object, create a new one, then call its Connect() method. 

=head2 $server->register_vm()

Registers a virtual machine on a server.

B<Syntax>

  $server->register_vm($vm_name) 

B<Parameters>

  $vm_name [in] A string specifying the virtual machine's configuration file name.  

=head2 $server->unregister_vm()

Unregisters a virtual machine from a server.
<
B<Syntax>

  $server->unregister_vm($vm_name) 

B<Parameters>

  $vmName [in] A string specifying the virtual machine's configuration file name.  

=head2 $server->get_last_error()

Obtain details about the last error that occurred.

B<Returns>

An array of ($error_num, $error_string).

=head1 AUTHORS

Jeff Hu, Vui-Chiap Lam, Bich Le

=head1 COPYRIGHT

    (c) 2002 VMware Incorporated.  All rights reserved.

=head1 VERSION

Version 1.01	   29 Mar 2002

C<% pod2html --header --title="VMware::VmPerl::Server" E<lt> Server.pm E<gt> Server.html>

=head1 SEE ALSO

perl(1).

=cut
