/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

#ifndef _VMCONTROL_H_
#define _VMCONTROL_H_

#define INCLUDE_ALLOW_USERLEVEL
#include "includeCheck.h"
#include "vmcontrol_constants.h"

/*
 * vmcontrol.h --
 *
 *	External (public) C VMControl header file.
 *
 *      This file will be distributed outside VMware with the binary
 *      libraries that implement the VMware Server Control Interface.
 *      
 *      This file can also be run through h2xs to create stubs for the
 *      Perl API.  These stubs have been edited to customize them and
 *      are check in under ../perl.  (That is, h2xs can be run
 *      manually, but is not run during the build process)
 *      
 *      The syntax for running h2xs to generate stubs for our three
 *      packages is as follows:
        
        h2xs -p VMControl_Server -v $VERSION -n VMware::Control::Server \
          -M '^VMControl_Server' -Axf `pwd`/../public/vmcontrol.h
        
        h2xs -O -p VMControl_VM -v $VERSION -n VMware::Control::VM \
          -M '^VMControl_VM' -Axf `pwd`/../public/vmcontrol.h

        h2xs -O -p VMControl_MKS -v $VERSION -n VMware::Control::VM \
          -M '^VMControl_MKS' -Axf `pwd`/../public/vmcontrol.h
                
        h2xs -O -p VMControl_ -v $VERSION -n VMware::Control \
          -M '^VMControl_(?!VM)(?!Server)' -Axf `pwd`/../public/vmcontrol.h

 *      The syntax for running h2xs to generate stubs for the #defined
 *      constants is as follows:

        h2xs -p VMControl_ -v $VERSION -n VMware::Control \
          -M '^VM_E_' -xf `pwd`/../public/vmcontrol.h
        
 *      Then, manually merge new or changed functions into the .pm and
 *      .xs files in apps/perl.  The other generated files (Changes,
 *      MANIFEST, typemap, Makefile.PL, test.pl) will not contain
 *      anything useful.
 */

// This version number should correspond to the version in the .pm files
#define VMCONTROL_VERSION "1.01"

#define VMAUTHD_DEFAULT_PORT 902

#define MAX_CONFIG_FILES 256

// Timeout for responses from vmauthd and VMX, in microseconds:
#define VMCONTROL_DEFAULT_TIMEOUT (120*1000*1000)  // 120 seconds
#define VMCONTROL_POWEROP_MULT (2)                 // 4 minutes

#define VMCONTROL_MAX_TIMEOUT 0x7FFFFFFF

// Default location if authd.fullpath isn't set
#if linux
#  define VMAUTHD_LOCATION "/usr/sbin/vmware-authd"
#else
#  ifndef VMX86_DEVEL
#    define VMAUTHD_LOCATION "vmware-authd.exe"
#  else
#    if defined(VMX86_DEBUG)
#      define VMAUTHD_LOCATION_PREFIX "N:\\BinDbg\\"
#    else
#      define VMAUTHD_LOCATION_PREFIX "N:\\BinOpt\\"
#    endif
#    define VMAUTHD_LOCATION  VMAUTHD_LOCATION_PREFIX "vmware-authd.exe"
#  endif
#endif

typedef struct VMControlServer VMControlServer;
typedef struct VMControlVM VMControlVM;

//XXX #include "vm_basic_types.h" //for Bool, TRUE, FALSE
#ifndef _VM_BASIC_TYPES_H_
typedef char           Bool;

#ifndef FALSE
#define FALSE          0
#endif

#ifndef TRUE
#define TRUE           1
#endif
#endif

/* 
 * VMControl -- top level functions
 * --------------------------------
 */

typedef struct {
   void * (*mallocFunction) (size_t size);
   void * (*callocFunction) (size_t n, size_t size);
   void (*freeFunction) (void *buf);
} VMControlInitParameters;

Bool VMControl_Init(void);
Bool VMControl_SetMemoryRoutines(VMControlInitParameters *initParams);
void VMControl_Free(void *p);
const char *VMControl_Version(void);
const char *VMControl_GetErrorMessage(int err);
int VMControl_GetDefaultTCPPort(void);


/*
 * VMControl Server -- functions that act on a VMControlServer object
 * -----------------------------------------------------------
 */

char VMControl_ServerInit(void);

VMControlServer *
VMControl_ServerNew(const char *hostname, int port, const char *username,
                    const char *password);
int VMControl_ServerGetLastError(VMControlServer *s, char **cptr);
Bool VMControl_ServerConnect(VMControlServer *s);
void VMControl_ServerDisconnect(VMControlServer *s);
void VMControl_ServerDestroy(VMControlServer *s);
Bool VMControl_ServerIsConnected(VMControlServer *s);
char **VMControl_ServerEnumerate(VMControlServer *s);
Bool VMControl_ServerIsRegistered(VMControlServer *s, const char *config, Bool *isRegistered);
Bool VMControl_ServerKill(VMControlServer *s, const char *vm); // Not implemented
char *VMControl_ServerExec(VMControlServer *s, const char *xmlargs);
Bool VMControl_ServerShutdown(VMControlServer *s);
Bool VMControl_ServerSingleVMQuery(VMControlServer *s, char *config, char *query, char **result);
Bool VMControl_ServerMultipleVMQuery(VMControlServer *s, char **vmList, int numVmList,
				     char *query, char ***result);
Bool VMControl_ServerVMQuery(VMControlServer *s, char **vmList, int numVmList, 
			     char **queries, int numQueries, char ***results, int *numResults);
Bool VMControl_ServerTest(VMControlServer *s, const char *query);
Bool VMControl_ServerSetAsString(VMControlServer *s, const char *query, const char *value);

char *VMControl_ServerGetResource(VMControlServer *server, char *key);
Bool VMControl_ServerSetResource(VMControlServer *server, char *key, char *newVal);

/*
 * Not sure whether these will be in the final API:
 */
char *VMControl_ServerGetAsString(VMControlServer *s, const char *property);  
Bool VMControl_ServerGetProductInfo(VMControlServer *vm, int infoType, int *result);
Bool VMControl_ServerRegister(VMControlServer *s, const char *configpath);
Bool VMControl_ServerUnregister(VMControlServer *s, const char *configpath);

/* 
 * VMControl VM -- functions that act on a VMControlVM object
 * -------------------------------------------------
 */

char VMControl_VMInit(void);

VMControlVM *VMControl_VMNew(VMControlServer *s, const char *config);
int VMControl_VMGetLastError(VMControlVM *vm, char **cptr);
Bool VMControl_VMConnect(VMControlVM *vm, Bool mks);
void VMControl_VMDisconnect(VMControlVM *vm);
void VMControl_VMDestroy(VMControlVM *vm);
char *VMControl_VMGetConfigFileName(VMControlVM *vm);
int VMControl_VMGetTimeout(VMControlVM *vm);
int VMControl_VMSetTimeout(VMControlVM *vm, int timeout);

Bool VMControl_VMIsConnected(VMControlVM *vm);
Bool VMControl_VMAnswerQuestion(VMControlVM *vm, int sequence, int choicenum);
Bool VMControl_VMStart(VMControlVM *vm, int mode);
Bool VMControl_VMRequestStop(VMControlVM *vm, Bool force);
Bool VMControl_VMRequestReset(VMControlVM *vm, Bool force);
Bool VMControl_VMStop(VMControlVM *vm, Bool force);
Bool VMControl_VMReset(VMControlVM *vm, Bool force);
Bool VMControl_VMSuspendToDisk(VMControlVM *vm, int mode);
char ** VMControl_VMInfo(VMControlVM *vm);
Bool VMControl_VMSaveAs(VMControlVM *vm, const char *config);  // Not implemented
Bool VMControl_VMExit(VMControlVM *vm);  // Not implemented
Bool VMControl_VMCommit(VMControlVM *vm, const char *disk, int level,
			Bool freeze, Bool wait);
Bool VMControl_VMAddRedo(VMControlVM *vm, const char *disk);

Bool VMControl_VMSetConfigChanges(VMControlVM * vm,
                                  const char * const * assignments);
Bool VMControl_VMSaveConfig(VMControlVM *vm);

Bool VMControl_VMDeviceConnect(VMControlVM *vm, const char *device);
Bool VMControl_VMDeviceDisconnect(VMControlVM *vm, const char *device);
Bool VMControl_VMToolsInstallBegin(VMControlVM *vm);
Bool VMControl_VMToolsInstallEnd(VMControlVM *vm);
Bool VMControl_VMToolsLastActive(VMControlVM *vm, int *seconds);


/*
 * Not sure whether these will be in the final API:
 */
char * VMControl_VMGetAsString(VMControlVM *vm, const char *var);
char * VMControl_VMGetAsStringServerd(VMControlVM *vm, const char *var);
Bool VMControl_VMSetAsString(VMControlVM *vm, const char *var, const char *value);  
char ** VMControl_VMWho(VMControlVM *vm);
Bool VMControl_VMReloadConfig(VMControlVM *vm);
void VMControl_VMSetError(VMControlVM *vm, int err, const char *str);

/*
 * MKS
 */

Bool VMControl_MKSInsertKeyEvent(VMControlVM *vm, int scancode, Bool down);
Bool VMControl_MKSSaveSreenshot(VMControlVM *vm, const char *filename,
                                const char *fileType);
Bool VMControl_MKSTargetScreenshot(VMControlVM *vm,
                                   const char *filename,
                                   const char *maskFilename,
                                   const char *fileType,
                                   const char *maskFileType,
                                   int tolerance);
int VMControl_MKSMatchScreenshot(VMControlVM *vm, int timeout);
Bool VMControl_MKSSaveScreenshot(VMControlVM *vm, const char *filename,
				 const char *fileType);

/*
 * Callbacks
 */

typedef void (*VMControlVMCallback) (VMControlVM *vm, void* args);

typedef enum VMControlVMCallbackType {
   VMC_CALLBACK_POWER,
   VMC_CALLBACK_DIALOG,
   VMC_CALLBACK_ANSWER,
   VMC_CALLBACK_DEVICE
} VMControlVMCallbackType;

Bool VMControl_VMRegisterCallback(VMControlVMCallbackType type, VMControlVMCallback c);
Bool VMControl_VMUnregisterCallback(VMControlVMCallbackType type, VMControlVMCallback callback);

typedef struct VMControlPowerTransition {
   const char *username;
   const char *oldState;
   const char *newState;
} VMControlPowerTransition;

typedef struct VMControlQuestion {
   int sequence;
   const char * type; // info, warning, error, question
   const char * text;
   const char ** choices;
} VMControlQuestion;

typedef struct VMControlAnswer {
   VMControlQuestion *q;
   const char *username;
   int choice;
} VMControlAnswer;

typedef struct VMControlConfigDevice {
   char *name;
   char *connection;
   Bool connected;
} VMControlConfigDevice;

/* 
 * VMControl Profiler -- functions that act on a Profiler object
 * -------------------------------------------------
 */

unsigned long VMControl_ProfilerGetTime(void);

/*
 *------------------------------------------------------------------------
 *  New interfaces for scripting bindings (VmComScript and VmPerlScript)
 *------------------------------------------------------------------------
 */

typedef struct {
   const char *hostname;
   int port;              // Port number where vmauthd is listening
   const char *username;
   char *password;
} VMControlConnectParams;

typedef struct {
   long  id;
   char *text;
   char **choices;
} VMControlExternalQuestion;

char *VMControl_VMGetGuestInfo(VMControlVM *vm, char *key);
Bool VMControl_VMSetGuestInfo(VMControlVM *vm, char *key, char *newVal);

char *VMControl_VMGetConfig(VMControlVM *vm, char *key);
Bool VMControl_VMSetConfig(VMControlVM *vm, char *key, char *newVal);

char *VMControl_VMGetResource(VMControlVM *vm, char *key);
Bool VMControl_VMSetResource(VMControlVM *vm, char *key, char *newVal);

Bool VMControl_VMGetId(VMControlVM *vm, unsigned int *id);
Bool VMControl_VMGetPid(VMControlVM *vm, unsigned int *pid); 

Bool VMControl_VMGetUptime(VMControlVM *vm, unsigned int *result);
Bool VMControl_VMGetHeartbeat(VMControlVM *vm, unsigned int *result);
Bool VMControl_VMGetExecutionState(VMControlVM *vm, int *retVal);

Bool VMControl_VMGetCapabilities(VMControlVM *vm, unsigned int *capabilities);
Bool VMControl_VMGetRemoteConnections(VMControlVM *vm, unsigned int *numConnections);

VMControlExternalQuestion *VMControl_VMGetExternalQuestion(VMControlVM *vm);
void VMControl_VMDestroyExternalQuestion(VMControlExternalQuestion *q);

Bool VMControl_VMGetProductInfo(VMControlVM *vm, int infoType, int *result);
Bool VMControl_VMDeviceIsConnected(VMControlVM *vm, const char *devName, Bool *result);

Bool VMControl_VMStopOrReset(VMControlVM *vm, Bool stop, int mode);

Bool VMControl_VMCreateSnapshot(VMControlVM *vm);
Bool VMControl_VMHasSnapshot(VMControlVM *vm, Bool *hasSnapshot);
Bool VMControl_VMRevertSnapshot(VMControlVM *vm);
Bool VMControl_VMDeleteSnapshot(VMControlVM *vm);

Bool VMControl_VMGetRunAsUser(VMControlVM *vm, char **username);
Bool VMControl_VMSetRunAsUser(VMControlVM *vm, const char *username,
                              const char *password, Bool checkValid);


VMControlServer *VMControl_ServerNewEx(void);
VMControlVM *VMControl_VMNewEx(void);

VMControlConnectParams *
VMControl_ConnectParamsNew(const char *hostname,
                           int port,         
                           const char *username,
                           const char *password);
void VMControl_ConnectParamsDestroy(VMControlConnectParams *params);

Bool VMControl_ServerConnectEx(VMControlServer *s, VMControlConnectParams *cp);
Bool VMControl_VMConnectEx(VMControlVM *s,VMControlConnectParams *cp,
                           const char *config, Bool mks);

Bool VMControl_VMAccessTimeout(VMControlVM *vm,  int timeoutId,
                               const int *timeoutIn, int *timeoutOut);   

#endif

