#!/usr/bin/perl -w 
#
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#
# answer_question.pl
#
# You can use this script to check if the virtual machine specified by
# config is stuck. If it's stuck, you can answer any question posed by this
# virtual machine to allow it to continue.
#
# usage:
#   answer_question.pl <config-file>

BEGIN {
   if ($^O eq "MSWin32") {
      @INC = (
         # Set the path to your VmPerl Scripting directory if different
         'C:\Program Files\VMware\VMware VmPerl Scripting API\perl5\site_perl\5.005',
         'C:\Program Files\VMware\VMware VmPerl Scripting API\perl5\site_perl\5.005\MSWin32-x86');
   }
}

# Import the required VMware Perl modules and version.
use VMware::VmPerl;
use VMware::VmPerl::VM;
use VMware::VmPerl::ConnectParams;
use VMware::VmPerl::Question;
use strict;

# Read in command line options.
my $cfg = shift or die "Usage: $0 <config-file>\n";


# Connect to the local host on the default port as yourself.
my $connect_params = VMware::VmPerl::ConnectParams::new();

# Initialize the object for the virtual machine we want to check.
my $vm = VMware::VmPerl::VM::new();
my $vm_ok = $vm->connect($connect_params, $cfg);
unless ($vm_ok) {
   my ($err, $errstr) = $vm->get_last_error();
   undef $vm;
   die "Could not connect to vm; error $err: $errstr\n";
}

# Check the power state of the virtual machine.  If it's stuck, get the
# question and list the possible responses.
my $state = $vm->get_execution_state();
if (!defined($state)) {
   my ($err, $errstr) = $vm->get_last_error();
   # Destroys the virtual machine object, thus disconnecting from the virtual machine
   undef $vm;
   die "Could not get execution state of vm; error $err: $errstr\n";
}

if ($state ne VM_EXECUTION_STATE_STUCK) {
   print "There is no question to answer.\n";
} else {
   my $q = $vm->get_pending_question();
   unless (defined($q)) {
      $vm->disconnect();
      die "Could not get the pending question.\n";
   }
   my $text = $q->get_text();
   unless (defined($text)) {
      undef $vm;
      die "Could not get the text of the pending question.\n";
   }
   my @choices = $q->get_choices();
   unless (defined($choices[0])) {
      undef $vm;
      die "Could not get the choices to answer the pending question.\n";
   }
   # Print question and choices for user:   
   print "\n" . $q->get_text() . "\n";
    
   my $answer;
   do {
      prompt(@choices);
      $answer = get_answer();
   }
   until (valid_answer($answer,@choices));

   my $op_ok;
   $op_ok = $vm->answer_question($q, $answer-1);
   unless ($op_ok) {
      my ($err, $errstr) = $vm->get_last_error();
      undef $vm;
      die "Could not answer pending question; error $err: $errstr\n";
   }
}

# Destroys the virtual machine object, thus disconnecting from the virtual machine.
undef $vm;

#------------------------------------------------
# Prints answer choices, prompts user for an answer number.
sub prompt {
   my @choices = shift;
   print "To answer the question, type the number that corresponds to\n";
   print "one of the answers below:\n";
   for (my $i = 0; $i <= $#choices; $i++) {
      print "\t" . ($i + 1) . ". $choices[$i]\n";
   }
   print "Final answer? ";
}

# Reads user's answer number.
sub get_answer {
   my $answer;
   chop($answer = <STDIN>);
   print "\n";

   # Remove unintentional whitespace.
   $answer =~ s/^(\s*)(.*?)(\s*)$/$2/;
   return $answer;
}

# Checks if an answer number is within the valid range of choices.
sub valid_answer {
   my $answer = shift;
   my @choices = shift;
   $answer--;  # convert to 0-based indexing.
   if ($answer < 0 || $answer > $#choices) {
      my $num = scalar(@choices);
      print "Valid answer numbers are from 1 to $num; please try again.\n";
      return 0;
   }
   else {
      return 1;
   }
}



