/*
 * Copyright 2004-2006.  VMware, Inc.  The use of this code is subject to
 * your agreement to the terms of an applicable VMware license.
 */

/*
 * This is the C interface to the VIX API.
 * This is platform-independent.
 */

#ifndef _VIX_H_
#define _VIX_H_

#ifdef __cplusplus
extern "C" {
#endif


/*
 *-----------------------------------------------------------------------------
 *
 * Basic Types --
 *
 *-----------------------------------------------------------------------------
 */

#include "vm_basic_types.h"

enum {
   VIX_API_VERSION      = 1
};

typedef int VixHandle;
enum {
   VIX_INVALID_HANDLE   = 0,
};

/*
 * These are the types of handles.
 */
typedef int VixHandleType;
enum {
   VIX_HANDLETYPE_NONE          = 0,
   VIX_HANDLETYPE_HOST          = 2,
   VIX_HANDLETYPE_VM            = 3,
   VIX_HANDLETYPE_NETWORK       = 5,
   VIX_HANDLETYPE_JOB           = 6,
   VIX_HANDLETYPE_SNAPSHOT      = 7,
};



/*
 * An error is a 64-bit value. If there is no error, then the value is
 * set to VIX_OK. If there is an error, then the least significant bits
 * will be set to one of the integer error codes defined below. The more
 * significant bits may or may not be set to various values, depending on 
 * the errors.
 */
typedef uint64 VixError;
#define VIX_ERROR_CODE(err)   ((err) & 0xFFFF)
#define VIX_SUCCEEDED(err)    (VIX_OK == (err))
#define VIX_FAILED(err)       (VIX_OK != (err))

/*
 * The error codes are returned by all public VIX routines.
 */
enum {
   VIX_OK                                       = 0,

   /* General errors */
   VIX_E_FAIL                                   = 1,
   VIX_E_OUT_OF_MEMORY                          = 2,
   VIX_E_INVALID_ARG                            = 3,
   VIX_E_FILE_NOT_FOUND                         = 4,
   VIX_E_OBJECT_IS_BUSY                         = 5,
   VIX_E_NOT_SUPPORTED                          = 6,
   VIX_E_FILE_ERROR                             = 7,
   VIX_E_DISK_FULL                              = 8,
   VIX_E_INCORRECT_FILE_TYPE                    = 9,
   VIX_E_CANCELLED                              = 10,
   VIX_E_FILE_READ_ONLY                         = 11,
   VIX_E_FILE_ALREADY_EXISTS                    = 12,
   VIX_E_FILE_ACCESS_ERROR                      = 13,
   VIX_E_REQUIRES_LARGE_FILES                   = 14,
   VIX_E_FILE_ALREADY_LOCKED                    = 15,

   /* Handle Errors */
   VIX_E_INVALID_HANDLE                         = 1000,
   VIX_E_NOT_SUPPORTED_ON_HANDLE_TYPE           = 1001,
   VIX_E_TOO_MANY_HANDLES                       = 1002,

   /* XML errors */
   VIX_E_NOT_FOUND                              = 2000,
   VIX_E_TYPE_MISMATCH                          = 2001,
   VIX_E_INVALID_XML                            = 2002,

   /* VM Control Errors */
   VIX_E_TIMEOUT_WAITING_FOR_TOOLS              = 3000,
   VIX_E_UNRECOGNIZED_COMMAND                   = 3001,
   VIX_E_OP_NOT_SUPPORTED_ON_GUEST              = 3003,
   VIX_E_PROGRAM_NOT_STARTED                    = 3004,
   VIX_E_VM_NOT_RUNNING                         = 3006,
   VIX_E_VM_IS_RUNNING                          = 3007,
   VIX_E_CANNOT_CONNECT_TO_VM                   = 3008,
   VIX_E_POWEROP_SCRIPTS_NOT_AVAILABLE          = 3009,
   VIX_E_NO_GUEST_OS_INSTALLED                  = 3010,
   VIX_E_VM_INSUFFICIENT_HOST_MEMORY            = 3011,
   VIX_E_SUSPEND_ERROR                          = 3012,
   VIX_E_VM_NOT_ENOUGH_CPUS                     = 3013,
   VIX_E_HOST_USER_PERMISSIONS                  = 3014,
   VIX_E_GUEST_USER_PERMISSIONS                 = 3015,
   VIX_E_TOOLS_NOT_RUNNING                      = 3016,
   VIX_E_GUEST_OPERATIONS_PROHIBITED            = 3017,
   VIX_E_ANON_GUEST_OPERATIONS_PROHIBITED       = 3019,
   VIX_E_ROOT_GUEST_OPERATIONS_PROHIBITED       = 3021,
   VIX_E_MISSING_ANON_GUEST_ACCOUNT             = 3023,
   VIX_E_CANNOT_AUTHENTICATE_WITH_GUEST         = 3024,
   VIX_E_UNRECOGNIZED_COMMAND_IN_GUEST          = 3025,


   /* VM Errors */ 
   VIX_E_VM_NOT_FOUND                           = 4000,
   VIX_E_NOT_SUPPORTED_FOR_VM_VERSION           = 4001,
   VIX_E_CANNOT_READ_VM_CONFIG                  = 4002,
   VIX_E_TEMPLATE_VM                            = 4003,
   VIX_E_VM_ALREADY_LOADED                      = 4004,
 
   /* Property Errors */
   VIX_E_UNRECOGNIZED_PROPERTY                  = 6000,
   VIX_E_INVALID_PROPERTY_VALUE                 = 6001,
   VIX_E_READ_ONLY_PROPERTY                     = 6002,
   VIX_E_MISSING_REQUIRED_PROPERTY              = 6003,

   /* Completion Errors */
   VIX_E_BAD_VM_INDEX                           = 8000,
};

const char *Vix_GetErrorText(VixError err, const char *locale);


/*
 *-----------------------------------------------------------------------------
 *
 * VIX Handles --
 *
 * These are common functions that apply to handles of several types. 
 *-----------------------------------------------------------------------------
 */

/* 
 * VIX Property Type
 */

typedef int VixPropertyType;
enum {
   VIX_PROPERTYTYPE_ANY             = 0,
   VIX_PROPERTYTYPE_INTEGER         = 1,
   VIX_PROPERTYTYPE_STRING          = 2,
   VIX_PROPERTYTYPE_BOOL            = 3,
   VIX_PROPERTYTYPE_HANDLE          = 4,
   VIX_PROPERTYTYPE_INT64           = 5,
};

/*
 * VIX Property ID's
 */

typedef int VixPropertyID;
enum {
   VIX_PROPERTY_NONE                        = 0,

   /* VIX_HANDLETYPE_VM properties */
   VIX_PROPERTY_VM_NUM_VCPUS                = 101,
   VIX_PROPERTY_VM_VMX_PATHNAME             = 103, 
   VIX_PROPERTY_VM_VMTEAM_PATHNAME          = 105, 
   VIX_PROPERTY_VM_MEMORY_SIZE              = 106, 
   VIX_PROPERTY_VM_IN_VMTEAM                = 128,
   VIX_PROPERTY_VM_POWER_STATE              = 129,
   VIX_PROPERTY_VM_TOOLS_STATE              = 152,
   VIX_PROPERTY_VM_IS_RUNNING               = 196,

   /* Result properties; these are returned by various procedures */
   VIX_PROPERTY_JOB_RESULT_ERROR_CODE                = 3000,
   VIX_PROPERTY_JOB_RESULT_VM_IN_GROUP               = 3001,
   VIX_PROPERTY_JOB_RESULT_USER_MESSAGE              = 3002,
   VIX_PROPERTY_JOB_RESULT_LINE_NUM                  = 3003,
   VIX_PROPERTY_JOB_RESULT_EXIT_CODE                 = 3004,
   VIX_PROPERTY_JOB_RESULT_COMMAND_OUTPUT            = 3005,
   VIX_PROPERTY_JOB_RESULT_HANDLE                    = 3010,
   VIX_PROPERTY_JOB_RESULT_FOUND_ITEM_NAME           = 3035,
   VIX_PROPERTY_JOB_RESULT_FOUND_ITEM_DESCRIPTION    = 3036,

   /* Event properties; these are sent in the moreEventInfo for some events. */
   VIX_PROPERTY_FOUND_ITEM_LOCATION         = 4010,
};

/*
 * These are events that may be signalled by calling a procedure
 * of type VixEventProc.
 */

typedef int VixEventType;
enum {
   VIX_EVENTTYPE_JOB_COMPLETED          = 2,
   VIX_EVENTTYPE_JOB_PROGRESS           = 3,
   VIX_EVENTTYPE_FIND_ITEM              = 8,
   VIX_EVENTTYPE_CALLBACK_SIGNALLED     = 2,  // Deprecated - Use VIX_EVENTTYPE_JOB_COMPLETED instead.
};

/*
 * Procedures of this type are called when an event happens on a handle.
 */

typedef void VixEventProc(VixHandle handle,
                          VixEventType eventType,
                          VixHandle moreEventInfo,
                          void *clientData);

/*
 * Handle Property functions
 */

void Vix_ReleaseHandle(VixHandle handle);

VixHandleType Vix_GetHandleType(VixHandle handle);

VixError Vix_GetProperties(VixHandle handle, 
                           VixPropertyID firstPropertyID, ...);

VixError Vix_GetPropertyType(VixHandle handle, VixPropertyID propertyID, 
                             VixPropertyType *propertyType);

void Vix_FreeBuffer(void *p);


/*
 *-----------------------------------------------------------------------------
 *
 * VIX Host --
 *
 *-----------------------------------------------------------------------------
 */

typedef int VixHostOptions;
enum {
   VIX_HOSTOPTION_USE_EVENT_PUMP        = 0x0008,
};

typedef int VixServiceProvider;
enum {
   VIX_SERVICEPROVIDER_VMWARE_SERVER    = 2,
};

VixHandle VixHost_Connect(int apiVersion,
                          VixServiceProvider hostType,
                          const char *hostName,
                          int hostPort,
                          const char *userName,
                          const char *password,
                          VixHostOptions options,
                          VixHandle propertyListHandle,
                          VixEventProc *callbackProc,
                          void *clientData);
 
void VixHost_Disconnect(VixHandle hostHandle);

/*
 * VM Registration
 */

VixHandle VixHost_RegisterVM(VixHandle hostHandle,
                             const char *vmxFilePath,
                             VixEventProc *callbackProc,
                             void *clientData);

VixHandle VixHost_UnregisterVM(VixHandle hostHandle,
                               const char *vmxFilePath,
                               VixEventProc *callbackProc,
                               void *clientData);

/*
 * VM Search
 */

typedef int VixFindItemType;
enum {
    VIX_FIND_RUNNING_VMS      = 1,
};

VixHandle VixHost_FindItems(VixHandle hostHandle,
                            VixFindItemType searchType,
                            VixHandle searchCriteria,
                            int32 timeout,
                            VixEventProc *callbackProc,
                            void *clientData);

/*
 * Event pump
 */

typedef int VixPumpEventsOptions;
enum {
   VIX_PUMPEVENTOPTION_NONE = 0,
};

void Vix_PumpEvents(VixHandle hostHandle, VixPumpEventsOptions options);


/*
 *-----------------------------------------------------------------------------
 *
 * VIX VM --
 *
 * This describes the persistent configuration state of a single VM. The 
 * VM may or may not be running.
 *
 *-----------------------------------------------------------------------------
 */

VixHandle VixVM_Open(VixHandle hostHandle,
                     const char *vmxFilePathName,
                     VixEventProc *callbackProc,
                     void *clientData);

typedef int VixVMPowerOpOptions;
enum {
   VIX_VMPOWEROP_NORMAL    = 0,
};

/*
 * Power operations 
 */

VixHandle VixVM_PowerOn(VixHandle vmHandle,
                        VixVMPowerOpOptions powerOnOptions,
                        VixHandle propertyListHandle,
                        VixEventProc *callbackProc,
                        void *clientData);

VixHandle VixVM_PowerOff(VixHandle vmHandle,
                         VixVMPowerOpOptions powerOffOptions,
                         VixEventProc *callbackProc,
                         void *clientData);

VixHandle VixVM_Reset(VixHandle vmHandle,
                      VixVMPowerOpOptions powerOnOptions,
                      VixEventProc *callbackProc,
                      void *clientData);

VixHandle VixVM_Suspend(VixHandle vmHandle,
                        VixVMPowerOpOptions powerOffOptions,
                        VixEventProc *callbackProc,
                        void *clientData);

typedef int VixVMDeleteOptions;

VixHandle VixVM_Delete(VixHandle vmHandle,
                       VixVMDeleteOptions deleteOptions,
                       VixEventProc *callbackProc,
                       void *clientData);

/*
 * This is the state of an individual VM.
 */

typedef int VixPowerState;
enum {
      VIX_POWERSTATE_POWERING_OFF    = 0x0001,
      VIX_POWERSTATE_POWERED_OFF     = 0x0002,
      VIX_POWERSTATE_POWERING_ON     = 0x0004,
      VIX_POWERSTATE_POWERED_ON      = 0x0008,
      VIX_POWERSTATE_SUSPENDING      = 0x0010,
      VIX_POWERSTATE_SUSPENDED       = 0x0020,
      VIX_POWERSTATE_TOOLS_RUNNING   = 0x0040,
      VIX_POWERSTATE_RESETTING       = 0x0080,
      VIX_POWERSTATE_BLOCKED_ON_MSG  = 0x0100,
};

typedef int VixToolsState;
enum {
      VIX_TOOLSSTATE_UNKNOWN           = 0x0001,
      VIX_TOOLSSTATE_RUNNING           = 0x0002,
      VIX_TOOLSSTATE_NOT_INSTALLED     = 0x0004,
};


/*
 * These are special names for an anonymous user and the system administrator.
 * The password is ignored if you specify these.
 */

#define VIX_ANONYMOUS_USER_NAME        "__VMware_Vix_Guest_User_Anonymous__"
#define VIX_ADMINISTRATOR_USER_NAME    "__VMware_Vix_Guest_User_Admin__"


/*
 * Guest operations
 */

VixHandle VixVM_WaitForToolsInGuest(VixHandle vmHandle,
                                    int timeoutInSeconds,
                                    VixEventProc *callbackProc,
                                    void *clientData);

VixHandle VixVM_LoginInGuest(VixHandle vmHandle,
                             const char *userName,
                             const char *password,
                             int options,
                             VixEventProc *callbackProc,
                             void *clientData);


typedef int VixRunProgramOptions;
enum {
   VIX_RUNPROGRAM_RETURN_IMMEDIATELY   = 0x0001,
};

VixHandle VixVM_RunProgramInGuest(VixHandle vmHandle,
                                  const char *guestProgramName,
                                  const char *commandLineArgs,
                                  VixRunProgramOptions options,
                                  VixHandle propertyListHandle,
                                  VixEventProc *callbackProc,
                                  void *clientData);

VixHandle VixVM_CopyFileFromHostToGuest(VixHandle vmHandle,
                                        const char *hostPathName,
                                        const char *guestPathName,
                                        int options,
                                        VixHandle propertyListHandle,
                                        VixEventProc *callbackProc,
                                        void *clientData);

VixHandle VixVM_CopyFileFromGuestToHost(VixHandle vmHandle,
                                        const char *guestPathName,
                                        const char *hostPathName,
                                        int options,
                                        VixHandle propertyListHandle,
                                        VixEventProc *callbackProc,
                                        void *clientData);

/* 
 * Snapshot functions 
 */

VixError VixVM_GetNumRootSnapshots(VixHandle vmHandle,
                                   int *result);

VixError VixVM_GetRootSnapshot(VixHandle vmHandle,
                               int index,
                               VixHandle *snapshotHandle);

VixHandle VixVM_RemoveSnapshot(VixHandle vmHandle, 
                               VixHandle snapshotHandle,
                               int options,
                               VixEventProc *callbackProc,
                               void *clientData);

VixHandle VixVM_RevertToSnapshot(VixHandle vmHandle,
                                 VixHandle snapshotHandle,
                                 int options,
                                 VixHandle propertyListHandle,
                                 VixEventProc *callbackProc,
                                 void *clientData);

VixHandle VixVM_CreateSnapshot(VixHandle vmHandle,
                               const char *name,
                               const char *description,
                               int options,
                               VixHandle propertyListHandle,
                               VixEventProc *callbackProc,
                               void *clientData);

/*
 * Misc Functions
 */

VixHandle VixVM_UpgradeVirtualHardware(VixHandle vmHandle,
                                       int options,
                                       VixEventProc *callbackProc,
                                       void *clientData);

VixHandle VixVM_InstallTools(VixHandle vmHandle,
                             int options,
                             const char *commandLineArgs,
                             VixEventProc *callbackProc,
                             void *clientData);


/*
 *-----------------------------------------------------------------------------
 *
 * VIX Job --
 *
 *-----------------------------------------------------------------------------
 */

/* 
 * Synchronization functions 
 * (used to detect when an asynch operation completes). 
 */

VixError VixJob_Wait(VixHandle jobHandle, VixPropertyID firstPropertyID, ...);

VixError VixJob_CheckCompletion(VixHandle jobHandle, 
                                Bool *complete);

VixError VixJob_GetError(VixHandle jobHandle);


#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* _VIX_H_ */
