#!/usr/bin/perl -w 
#
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#
# hbcheck.pl
#
# You can use this script to check the virtual machine specified by
# ConfigToCheck for a heartbeat within a certain interval in seconds.
# If no heartbeat is received within the specified Interval, then this 
# script will forcefully shutdown ConfigToCheck, and start ConfigToStart.
#
# usage:
#   hbcheck.pl <ConfigToCheck> <ConfigToStart> [Interval]
#

BEGIN {
   if ($^O eq "MSWin32") {
      @INC = (
         # Set the path to your VmPerl Scripting directory if different
         'C:\Program Files\VMware\VMware VmPerl Scripting API\perl5\site_perl\5.005',
         'C:\Program Files\VMware\VMware VmPerl Scripting API\perl5\site_perl\5.005\MSWin32-x86');
   }
}

# Import required VMware Perl modules and version.
use VMware::VmPerl;
use VMware::VmPerl::VM;
use VMware::VmPerl::ConnectParams;
use strict;

# Display the script usage.
sub usage() {
   print STDERR "Usage: hbcheck.pl <config_to_check> <config_to_start> [interval_in_secs]\n";
   exit(1);
}

# Retrieves a pre-defined constant value.
sub vm_constant {
   my $constant_str = shift;
   return VMware::VmPerl::constant($constant_str, 0);
}

# Read in command line options.
usage() unless (scalar(@ARGV) == 3 || scalar(@ARGV) == 2); 
my $cfg_to_check = shift;
my $cfg_to_start = shift;
my $interval = shift;

# Set the interval to 30 seconds if it is not specified.
$interval ||= 30;

# Connect to the local host on the default port as the current user.
# Change the port number if it is different.
my $connect_params = VMware::VmPerl::ConnectParams::new(undef, 902, undef, undef);

# Initialize the object for the virtual machine we want to check.
my $vm = VMware::VmPerl::VM::new();
if (!$vm->connect($connect_params, $cfg_to_check)) {
   my ($error_number, $error_string) = $vm->get_last_error();
   die "Could not connect to virtual machine at $cfg_to_check:\n" . 
       "Error $error_number: $error_string\n";
}

# Check to see if the virtual machine is powered on; if not, end.
my $vm_state = $vm->get_execution_state();
if (!($vm_state eq vm_constant("VM_EXECUTION_STATE_ON"))) {
   # Destroys the virtual machine object, thus disconnecting from the virtual machine
   undef $vm;
   die "The virtual machine $cfg_to_check\nis not powered on. Exiting.\n";
}

# Maintain the last read heartbeat value for comparison.
# The heartbeat count begins at zero, so a value of -1 ensures
# at least one comparison.
my $last_hb = -1;

while ($vm->is_connected()) {

   # Get the current heartbeat count.  This should steadily increase
   # as long as VMware tools is running inside the virtual machine.
   my $hb = $vm->get_heartbeat();
   unless (defined $hb) {
      my ($error_number, $error_string) = $vm->get_last_error();
      die "Could not get virtual machine heartbeat:\n" . 
          "Error $error_number: $error_string\n";
   }

   if ($hb == $last_hb) {
      # Since we don't have a heartbeat, we need to do something
      # about it. Let's shut this virtual machine down, and then start 
      # the backup virtual machine (specified by vm_to_start).
      # Use the "TRYSOFT" mode to shutdown gracefully if possible.
      $vm->stop(vm_constant("VM_POWEROP_MODE_TRYSOFT"));
      undef $vm;

      # Initialize the new virtual machine object.
      my $vm_to_start = VMware::VmPerl::VM::new();
      if (!$vm_to_start->connect($connect_params, $cfg_to_start)) {
	 my ($error_number, $error_string) = $vm_to_start->get_last_error();
         die "Could not connect to virtual machine at $cfg_to_start:\n" . 
             "Error $error_number: $error_string\n";
      }

      # Start the new virtual machine and clean up.
      my $start_ok = $vm_to_start->start();
      unless ($start_ok) {
	 my ($error_number, $error_string) = $vm_to_start->get_last_error();
         undef $vm_to_start;
         die "Could not start virtual machine $cfg_to_start:n" .
             "Error $error_number: $error_string\n";
      }
      undef $vm_to_start;
      die "Lost heartbeat of $cfg_to_check,\npowered on $cfg_to_start.\n";
   } else {
      # Wait $interval seconds before checking for the virtual machine's heartbeat.
      print "Got heartbeat count $hb\n";
      sleep ($interval);
   }
   $last_hb = $hb;
}
