/* *********************************************************************
 * Copyright (c) 2007 VMware, Inc.
 * All rights not expressly granted to you by VMware, Inc. are reserved.
 * *********************************************************************/

/*
 * This demonstrates how to integrate Vix with a poll loop. This
 * is useful if you write an application that has a single thread, and
 * you do not want Vix to create its own threads. For example, you want
 * to receive asynchronous callbacks, but you can only be called on your
 * main UI thread.
 *
 * This uses the VixJob_Wait function to block after starting each
 * asynchronous function. This effectively makes the asynchronous
 * functions synchronous, because VixJob_Wait will not return until the
 * asynchronous function has completed.
 */

#ifdef _WIN32
#include "windows.h"
#else
#include <pthread.h>
#endif

#include <stdlib.h>
#include <stdio.h>

#include "vix.h"

/*
 * Certain arguments differ between VMware Server 2.0 and Workstation.
 * Comment out this definition to use this code with VMware Server.
 */
#define USE_WORKSTATION

#ifdef USE_WORKSTATION

#define  CONNTYPE    VIX_SERVICEPROVIDER_VMWARE_WORKSTATION

#define  HOSTNAME ""
#define  HOSTPORT 0
#define  USERNAME ""
#define  PASSWORD ""

#define  VMPOWEROPTIONS   VIX_VMPOWEROP_LAUNCH_GUI   // Launches the VMware Workstaion UI
                                                     // when powering on the virtual machine.

#define VMXPATH_INFO "where vmxpath is an absolute path to the .vmx file " \
                     "for the virtual machine."
                     
#else    // USE_WORKSTATION

/*
 * For VMware Server 2.0
 */

#define CONNTYPE VIX_SERVICEPROVIDER_VMWARE_VI_SERVER

#define HOSTNAME "https://192.2.3.4:8333/sdk"
/*
 * NOTE: HOSTPORT is ignored, so the port should be specified as part
 * of the URL.
 */
#define HOSTPORT 0
#define USERNAME "root"
#define PASSWORD "hideme"

#define  VMPOWEROPTIONS VIX_VMPOWEROP_NORMAL

#define VMXPATH_INFO "where vmxpath is a datastore-relative path to the " \
                     ".vmx file for the virtual machine, such as "        \
                     "\"[standard] ubuntu/ubuntu.vmx\"."

#endif    // USE_WORKSTATION


/*
 * Global variables.
 */

static char *progName;
VixHandle hostHandle = VIX_INVALID_HANDLE;

/*
 * Local functions.
 */

#ifdef _WIN32
DWORD WINAPI TestThread(LPVOID threadParameter);
#else
void *TestThread(void *threadParameter);
#endif

static void
usage()
{
   fprintf(stderr, "Usage: %s <vmxpath>\n", progName);
   fprintf(stderr, "%s\n", VMXPATH_INFO);
}


////////////////////////////////////////////////////////////////////////////////
int
main(int argc, char **argv)
{
    VixError err;
    char *vmxPath;
    VixHandle jobHandle = VIX_INVALID_HANDLE;
    VixHandle vmHandle = VIX_INVALID_HANDLE;
#ifdef _WIN32
    DWORD threadId;
    HANDLE threadHandle;
#else
   pthread_t thread;
   int result;
#endif

    progName = argv[0];
    if (argc > 1) {
        vmxPath = argv[1];
    } else {
        usage();
        exit(EXIT_FAILURE);
    }

#ifdef _WIN32
    threadHandle = CreateThread(NULL, 0, TestThread, NULL, 0, &threadId);
    if (NULL == threadHandle) {
        goto abort;
    }
#else
   result = pthread_create(&thread, NULL, TestThread, NULL);
   if (0 != result) {
      goto abort;
   }
#endif

    jobHandle = VixHost_Connect(VIX_API_VERSION,
                                CONNTYPE,
                                HOSTNAME, // *hostName,
                                HOSTPORT, // hostPort,
                                USERNAME, // *userName,
                                PASSWORD, // *password,
                                VIX_HOSTOPTION_USE_EVENT_PUMP, // options,
                                VIX_INVALID_HANDLE, // propertyListHandle,
                                NULL, // *callbackProc,
                                NULL); // *clientData);
    err = VixJob_Wait(jobHandle, 
                      VIX_PROPERTY_JOB_RESULT_HANDLE, 
                      &hostHandle,
                      VIX_PROPERTY_NONE);
    if (VIX_FAILED(err)) {
        goto abort;
    }

    Vix_ReleaseHandle(jobHandle);
    jobHandle = VixVM_Open(hostHandle,
                           vmxPath,
                           NULL, // VixEventProc *callbackProc,
                           NULL); // void *clientData);
    err = VixJob_Wait(jobHandle, 
                      VIX_PROPERTY_JOB_RESULT_HANDLE, 
                      &vmHandle,
                      VIX_PROPERTY_NONE);
    if (VIX_FAILED(err)) {
        goto abort;
    }

    Vix_ReleaseHandle(jobHandle);
    jobHandle = VixVM_PowerOn(vmHandle,
                              VMPOWEROPTIONS,
                              VIX_INVALID_HANDLE,
                              NULL, // *callbackProc,
                              NULL); // *clientData);
    err = VixJob_Wait(jobHandle, VIX_PROPERTY_NONE);
    if (VIX_FAILED(err)) {
        goto abort;
    }

    Vix_ReleaseHandle(jobHandle);
    jobHandle = VixVM_PowerOff(vmHandle,
                               VIX_VMPOWEROP_NORMAL,
                               NULL, // *callbackProc,
                               NULL); // *clientData);
    err = VixJob_Wait(jobHandle, VIX_PROPERTY_NONE);
    if (VIX_FAILED(err)) {
        goto abort;
    }

abort:

    Vix_ReleaseHandle(jobHandle);
    Vix_ReleaseHandle(vmHandle);

    VixHost_Disconnect(hostHandle);

    return 0;
}




////////////////////////////////////////////////////////////////////////////////
#ifdef _WIN32

DWORD WINAPI 
TestThread(LPVOID threadParameter)
{
   while (1) {
      Vix_PumpEvents(hostHandle, VIX_PUMPEVENTOPTION_NONE);
   }

   return(0);
} // TestThread

#else

void *
TestThread(void *threadParameter)
{
   while (1) {
      Vix_PumpEvents(hostHandle, VIX_PUMPEVENTOPTION_NONE);
   }

   pthread_exit(NULL);
} // TestThread

#endif /* _WIN32 */
