#!/usr/bin/perl

#
# perl implementation of vmrun
#
# Copyright 2006.  VMware, Inc.  The use of this code is subject to
# your agreement to the terms of an applicable VMware license.
#

# for testing w/o installation
use lib qw{ blib/lib blib/auto blib/arch blib/arch/auto/VMware blib/arch/auto };

use strict;
use VMware::Vix::Simple;

# all constants are exported
use VMware::Vix::API::Constants;

my $hostname = undef;
my $port = 123;
my $username = undef;
my $password = undef;
my $hosthandle = VIX_INVALID_HANDLE;
my $vmpath = undef;
my $vmhandle = VIX_INVALID_HANDLE;
my $name;
my $description;

my $command = undef;
my @arglist;

sub usage() {
   print "Usage: $0  [authentication flags] COMMAND [PARAMETERS]\n";
   print "authentication flags:\n";
   print "\t-host <hostname>\n";
   print "\t-Port <hostport>\n";
   print "\t-user <username>\n";
   print "\t-password <password>\n";
   print "COMMAND               PARAMETERS              DESCRIPTION\n";
   print "start                 path to vmx             Start a VM\n";
   print "stop                  path to vmx             Stop a VM\n";
   print "reset                 path to vmx             Reset a VM\n";
   print "suspend               path to vmx             Suspend a VM\n";
   print "upgrade               path to vmx             Upgrade VM file format and virtual hardware\n";
   print "installtools          path to vmx             Install Tools in Guest OS\n";
   print "listSnapshots         path to vmx             List snapshots in a VM\n";
   print "snapshot              path to vmx             Takes a snapshot of a VM\n";
   print "                      snapshot name\n";
   print "                      snapshot description\n";
   print "deleteSnapshot        path to vmx             Remove a snapshot from a VM\n";
   print "                      snapshot name\n";
   print "revertToSnapshot      path to vmx             Revert to a snapshot from a VM\n";
   print "                      snapshot name\n";
   print "list                                          List all running VMs\n";
   exit 0;
}

sub doindent($) {
   my ($indent) = @_;
   my $i;

   foreach $i (1..$indent) {
      print "\t";
   }
};

sub mydie($$) {
   my ($errstring, $err) = @_;

   die "$errstring: ", GetErrorText($err), "($err)\n";
}

sub listhosts() {
}

sub listvms($) {
   my ($hh) = @_;
   my @vmlist = FindRunningVMs($hh, 300);
   my $err = shift @vmlist;
   foreach (@vmlist) {
      print "VM: $_\n";
   }
}

sub startvm($) {
   my ($vh) = @_;
   my $err = VMPowerOn($vh, 0, VIX_INVALID_HANDLE);
   mydie("Failed to power on VM", $err) if $err != VIX_OK;
}

sub stopvm($) {
   my ($vh) = @_;
   my $err = VMPowerOff($vh, 0);
   mydie("Failed to power off VM", $err) if $err != VIX_OK;
}

sub resetvm($) {
   my ($vh) = @_;
   my $err = VMReset($vh, 0);
   mydie("Failed to reset VM", $err) if $err != VIX_OK;
}

sub suspendvm($) {
   my ($vh) = @_;
   my $err = VMSuspend($vh, 0);
   mydie("Failed to suspend VM", $err) if $err != VIX_OK;
}

sub upgradevm($) {
   my ($vh) = @_;
   my $err = VMUpgradeVirtualHardware($vh, 0);
   mydie("Failed to upgrade virtual hardware", $err) if $err != VIX_OK;
}

sub installtools($) {
   my ($vh) = @_;
   my $err = VMInstallTools($vh, 0, undef);
   mydie("Failed to install tools", $err) if $err != VIX_OK;
}

sub printsnapshot($$) {
   my ($ss, $indent) = @_;
   my $name = "";
   my $desc = "";
   my $err;

   ($err, $name, $desc) = GetProperties($ss,
                                        VIX_PROPERTY_SNAPSHOT_DISPLAYNAME,
                                        VIX_PROPERTY_SNAPSHOT_DESCRIPTION);
   doindent($indent);
   print "Snapshot: '$name', '$desc'\n";
}

sub printchildsnapshots($$) {
   my ($ss, $indent) = @_;
   my $num;
   my $err;
   my $i;
   my $kid;

   ($err, $num) = SnapshotGetNumChildren($ss);

   foreach $i (0..($num-1)) {
      ($err, $kid) = SnapshotGetChild($ss, $i);
      printsnapshot($kid, $indent);
      \printchildsnapshots($kid, $indent+1);
   }
}

sub listsnapshots($) {
   my ($vm) = @_;
   my $err;
   my $numRootSnapShots;
   my $i;
   my $rootSS;

   ($err, $numRootSnapShots) = VMGetNumRootSnapshots($vm);
   mydie("Getting snapshots", $err) if $err != VIX_OK;
   print "$numRootSnapShots Root snapshots found\n";
   foreach $i (0..($numRootSnapShots - 1)) {
      ($err, $rootSS) = VMGetRootSnapshot($vm, $i);
      printsnapshot($rootSS, 0);
      printchildsnapshots($rootSS, 1);
   }
}

sub createsnapshot($$$) {
   my ($vm, $name, $description) = @_;
   my $err;
   my $ss;

   ($err, $ss) = VMCreateSnapshot($vm, $name, $description,
                                  0, VIX_INVALID_HANDLE);
   mydie("Creating snapshot", $err) if $err != VIX_OK;
   ReleaseHandle($ss);
}

sub deletesnapshot($$) {
   my ($vm, $name) = @_;
   my $err;
   my $ss;

   ($err, $ss) = VMGetNamedSnapshot($vm, $name);
   mydie("Getting snapshot handle", $err) if $err != VIX_OK;
   $err = VMRemoveSnapshot($vm, $ss, 0);
   mydie("Removing snapshot", $err) if $err != VIX_OK;
   ReleaseHandle($ss);
}

sub revertsnapshot($$) {
   my ($vm, $name) = @_;
   my $err;
   my $ss;

   ($err, $ss) = VMGetNamedSnapshot($vm, $name);
   mydie("Getting snapshot handle", $err) if $err != VIX_OK;
   $err = VMRevertToSnapshot($vm, $ss, 0, VIX_INVALID_HANDLE);
   mydie("Reverting to snapshot", $err) if $err != VIX_OK;
   ReleaseHandle($ss);
}

sub openvm($$) {
   my ($hh, $vmpath) = @_;
   my $err;
   my $handle;

   ($err, $handle) = VMOpen($hh, $vmpath);
   mydie("Failed to open VM $vmpath", $err) if $err != VIX_OK;

   return $handle;
}

sub host_connect($$$$) {
   my ($hostname, $port, $username, $password) = @_;
   my $err;
   my $handle;

   ($err, $handle) = HostConnect(VIX_API_VERSION,
                                 VIX_SERVICEPROVIDER_DEFAULT,
                                 $hostname, $port, $username, $password,
                                 0, VIX_INVALID_HANDLE);
   mydie("Connect Failed", $err) if $err != VIX_OK;

   return $handle;
}

sub docommand($@) {
   my ($command, @arglist) = @_;

#   print "command: $command arglist @arglist\n";

   if (lc($command) eq "start") {
      $vmpath = shift(@arglist);
      $vmhandle = openvm($hosthandle, $vmpath);
      startvm($vmhandle);
   } elsif (lc($command) eq "stop") {
      $vmpath = shift(@arglist);
      $vmhandle = openvm($hosthandle, $vmpath);
      stopvm($vmhandle);
   } elsif (lc($command) eq "reset") {
      $vmpath = shift(@arglist);
      $vmhandle = openvm($hosthandle, $vmpath);
      resetvm($vmhandle);
   } elsif (lc($command) eq "suspend") {
      $vmpath = shift(@arglist);
      $vmhandle = openvm($hosthandle, $vmpath);
      suspendvm($vmhandle);
   } elsif (lc($command) eq "upgrade") {
      $vmpath = shift(@arglist);
      $vmhandle = openvm($hosthandle, $vmpath);
      upgradevm($vmhandle);
   } elsif (lc($command) eq "installtools") {
      $vmpath = shift(@arglist);
      $vmhandle = openvm($hosthandle, $vmpath);
      installtools($vmhandle);
   } elsif (lc($command) eq "listsnapshots") {
      $vmpath = shift(@arglist);
      $vmhandle = openvm($hosthandle, $vmpath);
      listsnapshots($vmhandle);
   } elsif (lc($command) eq "snapshot") {
      $vmpath = shift(@arglist);
      $name = shift(@arglist);
      $description = shift(@arglist);
      $vmhandle = openvm($hosthandle, $vmpath);
      createsnapshot($vmhandle, $name, $description);
   } elsif (lc($command) eq "deletesnapshot") {
      $vmpath = shift(@arglist);
      $name = shift(@arglist);
      $vmhandle = openvm($hosthandle, $vmpath);
      deletesnapshot($vmhandle, $name);
   } elsif (lc($command) eq "reverttosnapshot") {
      $vmpath = shift(@arglist);
      $name = shift(@arglist);
      $vmhandle = openvm($hosthandle, $vmpath);
      revertsnapshot($vmhandle, $name);
   } elsif (lc($command) eq "list") {
      listvms($hosthandle);
   } else {
      print "Unrecognized command \"$command\"\n";
      usage();
   }
}

my $arg;
@arglist = @ARGV;
while ($#arglist != -1) {

   $arg = shift(@arglist);

   if (!($arg =~ /^-/)) {
      $command = $arg;
      last;
   }

   if ($arg =~ /^-h(ost)?$/) {
      $hostname = shift(@arglist);
   } elsif ($arg =~ /^-P(ort)?$/) {
      $port = shift(@arglist);
   } elsif ($arg =~ /^-u(sername)?$/) {
      $username = shift(@arglist);
   } elsif ($arg =~ /^-p(assword)?$/) {
      $password = shift(@arglist);
   } else {
      print "Unrecognized switch \"$arg\"\n";
      usage();
   }
}


#print "hostname: $hostname, port $port, user $username, password $password, command $command, args @arglist\n";

$hosthandle = host_connect($hostname, $port, $username, $password);

# if no command is on the command line, read from stdin, which
# allows a script file to be read in
#
if (defined($command)) {
   docommand($command, @arglist);
} else {
   print "reading commands from stdin\n";
   while (<STDIN>) {
      chomp;
      @arglist = split /\s+/, $_;
      $command = shift(@arglist);
      docommand($command, @arglist);
   }
}
