#!/usr/bin/perl -w
# If your copy of perl is not in /usr/bin, please adjust the line above.
#
# Copyright 1998-2007 VMware, Inc.  All rights reserved.
#
# Tar package manager for VMware

use strict;

# BEGINNING_OF_UTIL_DOT_PL
#!/usr/bin/perl

use strict;

# Moved out of config.pl to support $gOption in spacechk_answer
my %gOption;
# Moved from various scripts that include util.pl
my %gHelper;

# Needed to access $Config{...}, the Perl system configuration information.
use Config;

# Use the Perl system configuration information to make a good guess about
# the bit-itude of our platform.  If we're running on Solaris we don't have
# to guess and can just ask isainfo(1) how many bits userland is directly.
sub is64BitUserLand {
  if (vmware_product() eq 'tools-for-solaris') {
    if (direct_command(shell_string($gHelper{'isainfo'}) . ' -b') =~ /64/) {
      return 1;
    } else {
      return 0;
    }
  }
  if ($Config{archname} =~ /^(x86_64|amd64)-/) {
    return 1;
  } else {
    return 0;
  }
}

sub isFreeBSDLibc6 {
   if (vmware_product() eq 'tools-for-freebsd') {
      if (-f '/lib/libc.so.6') {
         return 1;
      }
   }
   return 0;
}

# Print an error message and exit
sub error {
  my $msg = shift;

  print STDERR wrap($msg . 'Execution aborted.' . "\n\n", 0);
  exit 1;
}

# Convert a string to its equivalent shell representation
sub shell_string {
  my $single_quoted = shift;

  $single_quoted =~ s/'/'"'"'/g;
  # This comment is a fix for emacs's broken syntax-highlighting code --hpreg
  return '\'' . $single_quoted . '\'';
}

# Send an arbitrary RPC command to the VMX
sub send_rpc {
  my $command = shift;
  my $guestdSuffix;
  my $guestdBinary;
  my $libDir;
  if (vmware_product() eq 'tools-for-solaris') {
     $guestdSuffix = is64BitUserLand() ? '/sbin/amd64' : '/sbin/i86';
  } else {
     $guestdSuffix = is64BitUserLand() ? '/sbin64' : '/sbin32';
  }
  if (isFreeBSDLibc6()) {
    $guestdSuffix .= '-6';
  }
  $guestdSuffix .= '/vmware-guestd';

  # We don't yet know if vmware-guestd was copied into place. 
  # Let's first try getting the location from the DB.
  $libDir = db_get_answer_if_exists('LIBDIR');
  if (defined($libDir)) {
    $guestdBinary = $libDir . $guestdSuffix;
    if (not (-e $guestdBinary)) {
      
      # The DB didn't help, let's pretend we're in the context
      # of an open tarball and look for vmware-guestd.
      $guestdBinary = "./lib" .  $guestdSuffix;
      if (not (-e $guestdBinary)) {
	$guestdBinary = undef;
      }
    }
  }
  
  # If we found the binary, send the RPC.
  if (defined($guestdBinary)) {
    system(shell_string($guestdBinary)
	   . ' --cmd ' . shell_string($command)
	   . ' 2>&1 > /dev/null');
  }
}

# Create a temporary directory
#
# They are a lot of small utility programs to create temporary files in a
# secure way, but none of them is standard. So I wrote this --hpreg
sub make_tmp_dir {
  my $prefix = shift;
  my $tmp;
  my $serial;
  my $loop;

  $tmp = defined($ENV{'TMPDIR'}) ? $ENV{'TMPDIR'} : '/tmp';

  # Don't overwrite existing user data
  # -> Create a directory with a name that didn't exist before
  #
  # This may never succeed (if we are racing with a malicious process), but at
  # least it is secure
  $serial = 0;
  for (;;) {
    # Check the validity of the temporary directory. We do this in the loop
    # because it can change over time
    if (not (-d $tmp)) {
      error('"' . $tmp . '" is not a directory.' . "\n\n");
    }
    if (not ((-w $tmp) && (-x $tmp))) {
      error('"' . $tmp . '" should be writable and executable.' . "\n\n");
    }

    # Be secure
    # -> Don't give write access to other users (so that they can not use this
    # directory to launch a symlink attack)
    if (mkdir($tmp . '/' . $prefix . $serial, 0755)) {
      last;
    }

    $serial++;
    if ($serial % 200 == 0) {
      print STDERR 'Warning: The "' . $tmp . '" directory may be under attack.' . "\n\n";
    }
  }

  return $tmp . '/' . $prefix . $serial;
}

# Check available space when asking the user for destination directory.
sub spacechk_answer {
  my $msg = shift;
  my $type = shift;
  my $default = shift;
  my $srcDir = shift;
  my $id = shift;
  my $answer;
  my $space = -1;

  while ($space < 0) {

    if (!defined($id)) {
      $answer = get_answer($msg, $type, $default);
    } else {
      $answer = get_persistent_answer($msg, $id, $type, $default);
    }

    # XXX check $answer for a null value which can happen with the get_answer
    # in config.pl but not with the get_answer in pkg_mgr.pl.  Moving these
    # (get_answer, get_persistent_answer) routines into util.pl eventually.
    if ($answer && ($space = check_disk_space($srcDir, $answer)) < 0) {
      my $lmsg;
      $lmsg = 'There is insufficient disk space available in ' . $answer
              . '.  Please make at least an additional ' . -$space
              . 'k available';
      if ($gOption{'default'} == 1) {
        error($lmsg . ".\n");
      }
      print wrap($lmsg . "or choose another directory.\n", 0);
    }
  }
  return $answer;
}

sub check_tools_on_ubuntu {
      my $kvers = direct_command(shell_string($gHelper{'uname'}) . ' -r');
      chomp($kvers);
      return (-d "/lib/modules/$kvers/vmware-tools");
}
# END_OF_UTIL_DOT_PL

# Needed for WIFSIGNALED and WTERMSIG
use POSIX;

# Constants
my $cInstallerFileName = 'vmware-install.pl';
my $cModuleUpdaterFileName = 'install.pl';
my $cInstallerDir = './installer';
my $cStartupFileName = $cInstallerDir . '/services.sh';
my $cRegistryDir = '/etc/vmware';
my $cInstallerMainDB = $cRegistryDir . '/locations';
my $cInstallerObject = $cRegistryDir . '/installer.sh';
my $cConfFlag = $cRegistryDir . '/not_configured';
my $gDefaultAuthdPort = 902;
my $cServices = '/etc/services';
my $cMarkerBegin = "# Beginning of the block added by the VMware software\n";
my $cMarkerEnd = "# End of the block added by the VMware software\n";
my $cVixMarkerBegin = "# Beginning of the block added by the VIX VMware software\n";
my $cVixMarkerEnd = "# End of the block added by the VIX VMware software\n";
# Constant defined as the smallest vmnet that is allowed
my $gMinVmnet = '0';
# Linux doesn't allow more than 7 characters in the names of network
# interfaces. We prefix host only interfaces with 'vmnet' leaving us only 2
# characters.
# Constant defined as the largest vmnet that is allowed
my $gMaxVmnet = '99';

# MANIFEST file and hash for installing ACE VMs
my $cManifestFilename = 'MANIFEST';
my %gManifest;
my $gACEVMUpdate = 0;
my $gHostVmplDir = "/etc/vmware/vmware-ace/";
my $gPlayerTar = '';

# Has the uninstaller been installed?
my $gIsUninstallerInstalled;

# Hash of multi architecture supporting products
my %multi_arch_products;

# BEGINNING OF THE SECOND LIBRARY FUNCTIONS
# Global variables
my $gRegistryDir;
my $gFirstCreatedDir = undef;
my $gStateDir;
my $gInstallerMainDB;
my $gInstallerObject;
my $gConfFlag;
my $gUninstallerFileName;
my $gConfigurator;

my %gDBAnswer;
my %gDBFile;
my %gDBDir;
my %gDBLink;
my %gDBMove;

# Load the installer database
sub db_load {
  undef %gDBAnswer;
  undef %gDBFile;
  undef %gDBDir;
  undef %gDBLink;
  undef %gDBMove;

  open(INSTALLDB, '<' . $gInstallerMainDB)
    or error('Unable to open the installer database '
             . $gInstallerMainDB . ' in read-mode.' . "\n\n");
  while (<INSTALLDB>) {
    chomp;
    if (/^answer (\S+) (.+)$/) {
      $gDBAnswer{$1} = $2;
    } elsif (/^answer (\S+)/) {
      $gDBAnswer{$1} = '';
    } elsif (/^remove_answer (\S+)/) {
      delete $gDBAnswer{$1};
    } elsif (/^file (.+) (\d+)$/) {
      $gDBFile{$1} = $2;
    } elsif (/^file (.+)$/) {
      $gDBFile{$1} = 0;
    } elsif (/^remove_file (.+)$/) {
      delete $gDBFile{$1};
    } elsif (/^directory (.+)$/) {
      $gDBDir{$1} = '';
    } elsif (/^remove_directory (.+)$/) {
      delete $gDBDir{$1};
    } elsif (/^link (\S+) (\S+)/) {
      $gDBLink{$2} = $1;
    } elsif (/^move (\S+) (\S+)/) {
      $gDBMove{$2} = $1;
    }
  }
  close(INSTALLDB);
}

# Open the database on disk in append mode
sub db_append {
  if (not open(INSTALLDB, '>>' . $gInstallerMainDB)) {
    error('Unable to open the installer database ' . $gInstallerMainDB . ' in append-mode.' . "\n\n");
  }
  # Force a flush after every write operation.
  # See 'Programming Perl', p. 110
  select((select(INSTALLDB), $| = 1)[0]);
}

# Add a file to the tar installer database
# flags:
#  0x1 write time stamp
sub db_add_file {
  my $file = shift;
  my $flags = shift;

  if ($flags & 0x1) {
    my @statbuf;

    @statbuf = stat($file);
    if (not (defined($statbuf[9]))) {
      error('Unable to get the last modification timestamp of the destination file ' . $file . '.' . "\n\n");
    }

    $gDBFile{$file} = $statbuf[9];
    print INSTALLDB 'file ' . $file . ' ' . $statbuf[9] . "\n";
  } else {
    $gDBFile{$file} = 0;
    print INSTALLDB 'file ' . $file . "\n";
  }
}

# Remove a file from the tar installer database
sub db_remove_file {
  my $file = shift;

  print INSTALLDB 'remove_file ' . $file . "\n";
  delete $gDBFile{$file};
}

# Remove a directory from the tar installer database
sub db_remove_dir {
  my $dir = shift;

  print INSTALLDB 'remove_directory ' . $dir . "\n";
  delete $gDBDir{$dir};
}

# Determine if a file belongs to the tar installer database
sub db_file_in {
  my $file = shift;

  return defined($gDBFile{$file});
}

# Determine if a directory belongs to the tar installer database
sub db_dir_in {
  my $dir = shift;

  return defined($gDBDir{$dir});
}

# Return the timestamp of an installed file
sub db_file_ts {
  my $file = shift;

  return $gDBFile{$file};
}

# Add a directory to the tar installer database
sub db_add_dir {
  my $dir = shift;

  $gDBDir{$dir} = '';
  print INSTALLDB 'directory ' . $dir . "\n";
}

# Remove an answer from the tar installer database
sub db_remove_answer {
  my $id = shift;

  if (defined($gDBAnswer{$id})) {
    print INSTALLDB 'remove_answer ' . $id . "\n";
    delete $gDBAnswer{$id};
  }
}

# Add an answer to the tar installer database
sub db_add_answer {
  my $id = shift;
  my $value = shift;

  db_remove_answer($id);
  $gDBAnswer{$id} = $value;
  print INSTALLDB 'answer ' . $id . ' ' . $value . "\n";
}

# Retrieve an answer that must be present in the database
sub db_get_answer {
  my $id = shift;

  if (not defined($gDBAnswer{$id})) {
    error('Unable to find the answer ' . $id . ' in the installer database ('
          . $gInstallerMainDB . '). You may want to re-install '
          . vmware_product_name() . "." .  "\n\n");
  }

  return $gDBAnswer{$id};
}

# Retrieves an answer if it exists in the database, else returns undef;
sub db_get_answer_if_exists {
  my $id = shift;
  if (not defined($gDBAnswer{$id})) {
    return undef;
  }
  if ($gDBAnswer{$id} eq '') {
    return undef;
  }
  return $gDBAnswer{$id};
}

# Save the tar installer database
sub db_save {
  close(INSTALLDB);
}
# END OF THE SECOND LIBRARY FUNCTIONS

# BEGINNING OF THE LIBRARY FUNCTIONS
# Constants
my $cTerminalLineSize = 79;

# Global variables
my %gAnswerSize;
my %gCheckAnswerFct;

# Tell if the user is the super user
sub is_root {
  return $> == 0;
}

# Wordwrap system: append some content to the output
sub append_output {
  my $output = shift;
  my $pos = shift;
  my $append = shift;

  $output .= $append;
  $pos += length($append);
  if ($pos >= $cTerminalLineSize) {
    $output .= "\n";
    $pos = 0;
  }

  return ($output, $pos);
}

# Wordwrap system: deal with the next character
sub wrap_one_char {
  my $output = shift;
  my $pos = shift;
  my $word = shift;
  my $char = shift;
  my $reserved = shift;
  my $length;

  if (not (($char eq "\n") || ($char eq ' ') || ($char eq ''))) {
    $word .= $char;

    return ($output, $pos, $word);
  }

  # We found a separator. Process the last word

  $length = length($word) + $reserved;
  if (($pos + $length) > $cTerminalLineSize) {
    # The last word doesn't fit in the end of the line. Break the line before it
    $output .= "\n";
    $pos = 0;
  }
  ($output, $pos) = append_output($output, $pos, $word);
  $word = '';

  if ($char eq "\n") {
    $output .= "\n";
    $pos = 0;
  } elsif ($char eq ' ') {
    if ($pos) {
      ($output, $pos) = append_output($output, $pos, ' ');
    }
  }

  return ($output, $pos, $word);
}

# Wordwrap system: word-wrap a string plus some reserved trailing space
sub wrap {
  my $input = shift;
  my $reserved = shift;
  my $output;
  my $pos;
  my $word;
  my $i;

  $output = '';
  $pos = 0;
  $word = '';
  for ($i = 0; $i < length($input); $i++) {
    ($output, $pos, $word) = wrap_one_char($output, $pos, $word, substr($input, $i, 1), 0);
  }
  # Use an artifical last '' separator to process the last word
  ($output, $pos, $word) = wrap_one_char($output, $pos, $word, '', $reserved);

  return $output;
}

# Contrary to a popular belief, 'which' is not always a shell builtin command.
# So we can not trust it to determine the location of other binaries.
# Moreover, SuSE 6.1's 'which' is unable to handle program names beginning with
# a '/'...
#
# Return value is the complete path if found, or '' if not found
sub internal_which {
  my $bin = shift;

  if (substr($bin, 0, 1) eq '/') {
    # Absolute name
    if ((-f $bin) && (-x $bin)) {
      return $bin;
    }
  } else {
    # Relative name
    my @paths;
    my $path;

    if (index($bin, '/') == -1) {
      # There is no other '/' in the name
      @paths = split(':', $ENV{'PATH'});
      foreach $path (@paths) {
   my $fullbin;

   $fullbin = $path . '/' . $bin;
   if ((-f $fullbin) && (-x $fullbin)) {
     return $fullbin;
   }
      }
    }
  }

  return '';
}

# Remove leading and trailing whitespaces
sub remove_whitespaces {
  my $string = shift;

  $string =~ s/^\s*//;
  $string =~ s/\s*$//;
  return $string;
}

# Ask a question to the user and propose an optional default value
# Use this when you don't care about the validity of the answer
sub query {
    my $message = shift;
    my $defaultreply = shift;
    my $reserved = shift;
    my $reply;

    # Reserve some room for the reply
    print wrap($message . (($defaultreply eq '') ? '' : (' [' . $defaultreply . ']')), 1 + $reserved);
    # This is what the 1 is for
    print ' ';

    if ($gOption{'default'} == 1) {
      # Simulate the enter key
      print "\n";
      $reply = '';
    } else {
      $reply = <STDIN>;
      $reply = '' unless defined($reply);
      chomp($reply);
    }

    print "\n";
    $reply = remove_whitespaces($reply);
    if ($reply eq '') {
      $reply = $defaultreply;
    }
    return $reply;
}

# Check the validity of an answer whose type is yesno
# Return a clean answer if valid, or ''
sub check_answer_binpath {
  my $answer = shift;
  my $source = shift;

  if (not (internal_which($answer) eq '')) {
    return $answer;
  }

  if ($source eq 'user') {
    print wrap('The answer "' . $answer . '" is invalid. It must be the complete name of a binary file.' . "\n\n", 0);
  }
  return '';
}
$gAnswerSize{'binpath'} = 20;
$gCheckAnswerFct{'binpath'} = \&check_answer_binpath;

# Prompts the user if a binary is not found
# Return value is:
#  '': the binary has not been found
#  the binary name if it has been found
sub DoesBinaryExist_Prompt {
  my $bin = shift;
  my $answer;

  $answer = check_answer_binpath($bin, 'default');
  if (not ($answer eq '')) {
    return $answer;
  }

  if (get_answer('Setup is unable to find the "' . $bin . '" program on your machine. Please make sure it is installed. Do you want to specify the location of this program by hand?', 'yesno', 'yes') eq 'no') {
    return '';
  }

  return get_answer('What is the location of the "' . $bin . '" program on your machine?', 'binpath', '');
}

# Execute the command passed as an argument
# _without_ interpolating variables (Perl does it by default)
sub direct_command {
  return `$_[0]`;
}

# chmod() that reports errors
sub safe_chmod {
  my $mode = shift;
  my $file = shift;

  if (chmod($mode, $file) != 1) {
    error('Unable to change the access rights of the file ' . $file . '.' . "\n\n");
  }
}

# Emulate a simplified ls program for directories
sub internal_ls {
  my $dir = shift;
  my @fn;

  opendir(LS, $dir);
  @fn = grep(!/^\.\.?$/, readdir(LS));
  closedir(LS);

  return @fn;
}

# Install a file permission
sub install_permission {
  my $src = shift;
  my $dst = shift;
  my @statbuf;
  my $mode;
  @statbuf = stat($src);
  if (not (defined($statbuf[2]))) {
    error('Unable to get the access rights of source file "' . $src . '".' . "\n\n");
  }

  # ACE packages may be installed from CD/DVDs, which don't have the same file
  # permissions of the original package (no write permission). Since these
  # packages are installed by a user under a single directory, it's safe to do
  # 'u+w' on everything.
  $mode = $statbuf[2] & 07777;
  if (vmware_product() eq 'acevm') {
    $mode |= 0200;
  }
  safe_chmod($mode, $dst);
}

# Emulate a simplified sed program
# Return 1 if success, 0 if failure
# XXX as a side effect, if the string being replaced is '', remove
# the entire line.  Remove this, once we have better "block handling" of
# our config data in config files.
sub internal_sed {
  my $src = shift;
  my $dst = shift;
  my $append = shift;
  my $patchRef = shift;
  my @patchKeys;

  if (not open(SRC, '<' . $src)) {
    return 0;
  }
  if (not open(DST, (($append == 1) ? '>>' : '>') . $dst)) {
    return 0;
  }

  @patchKeys = keys(%$patchRef);
  if ($#patchKeys == -1) {
    while(defined($_ = <SRC>)) {
      print DST $_;
    }
  } else {
    while(defined($_ = <SRC>)) {
      my $patchKey;
      my $del = 0;

      foreach $patchKey (@patchKeys) {
        if (s/$patchKey/$$patchRef{$patchKey}/g) {
          if ($_ eq "\n") {
            $del = 1;
          }
        }
      }
      next if ($del);
      print DST $_;
    }
  }

  close(SRC);
  close(DST);
  return 1;
}

# Check if a file name exists
sub file_name_exist {
  my $file = shift;

  # Note: We must test for -l before, because if an existing symlink points to
  #       a non-existing file, -e will be false
  return ((-l $file) || (-e $file))
}

# Check if a file name already exists and prompt the user
# Return 0 if the file can be written safely, 1 otherwise
sub file_check_exist {
  my $file = shift;

  if (not file_name_exist($file)) {
    return 0;
  }

  # The default must make sure that the product will be correctly installed
  # We give the user the choice so that a sysadmin can perform a normal
  # install on a NFS server and then answer 'no' NFS clients
  return (get_answer('The file ' . $file . ' that this program was about to '
                     . 'install already exists. Overwrite?',
                     'yesno', 'yes') eq 'yes') ? 0 : 1;
}

# Install one file
# flags are forwarded to db_add_file()
sub install_file {
  my $src = shift;
  my $dst = shift;
  my $patchRef = shift;
  my $flags = shift;

  uninstall_file($dst);
  if (file_check_exist($dst)) {
    return;
  }
  # The file could be a symlink to another location. Remove it
  unlink($dst);
  if (not internal_sed($src, $dst, 0, $patchRef)) {
    error('Unable to copy the source file ' . $src . ' to the destination file ' . $dst . '.' . "\n\n");
  }
  db_add_file($dst, $flags);
  install_permission($src, $dst);
}

# mkdir() that reports errors
sub safe_mkdir {
  my $file = shift;

  if (mkdir($file, 0000) == 0) {
    error('Unable to create the directory ' . $file . '.' . "\n\n");
  }
}

# Remove trailing slashes in a dir path
sub dir_remove_trailing_slashes {
  my $path = shift;

  for(;;) {
    my $len;
    my $pos;

    $len = length($path);
    if ($len < 2) {
      # Could be '/' or any other character. Ok.
      return $path;
    }

    $pos = rindex($path, '/');
    if ($pos != $len - 1) {
      # No trailing slash
      return $path;
    }

    # Remove the trailing slash
    $path = substr($path, 0, $len - 1)
  }
}

# Emulate a simplified dirname program
sub internal_dirname {
  my $path = shift;
  my $pos;

  $path = dir_remove_trailing_slashes($path);

  $pos = rindex($path, '/');
  if ($pos == -1) {
    # No slash
    return '.';
  }

  if ($pos == 0) {
    # The only slash is at the beginning
    return '/';
  }

  return substr($path, 0, $pos);
}

# Create a hierarchy of directories with permission 0755
# flags:
#  0x1 write this directory creation in the installer database
# Return 1 if the directory existed before
sub create_dir {
  my $dir = shift;
  my $flags = shift;

  if (-d $dir) {
    return 1;
  }

  if (index($dir, '/') != -1) {
    create_dir(internal_dirname($dir), $flags);
  }
  safe_mkdir($dir);
  if ($flags & 0x1) {
    db_add_dir($dir);
  }
  safe_chmod(0755, $dir);
  return 0;
}

# Get a valid non-persistent answer to a question
# Use this when the answer shouldn't be stored in the database
sub get_answer {
  my $msg = shift;
  my $type = shift;
  my $default = shift;
  my $answer;

  if (not defined($gAnswerSize{$type})) {
    die 'get_answer(): type ' . $type . ' not implemented :(' . "\n\n";
  }
  for (;;) {
    $answer = check_answer(query($msg, $default, $gAnswerSize{$type}), $type, 'user');
    if (not ($answer eq '')) {
      return $answer;
    }
    if ($gOption{'default'} == 1) {
      error('Invalid default answer!' . "\n");
    }
  }
}

# Get a valid persistent answer to a question
# Use this when you want an answer to be stored in the database
sub get_persistent_answer {
  my $msg = shift;
  my $id = shift;
  my $type = shift;
  my $default = shift;
  my $answer;

  if (defined($gDBAnswer{$id})) {
    # There is a previous answer in the database
    $answer = check_answer($gDBAnswer{$id}, $type, 'db');
    if (not ($answer eq '')) {
      # The previous answer is valid. Make it the default value
      $default = $answer;
    }
  }

  $answer = get_answer($msg, $type, $default);
  db_add_answer($id, $answer);
  return $answer;
}

# Find a suitable backup name and backup a file
sub backup_file {
  my $file = shift;
  my $i;

  for ($i = 0; $i < 100; $i++) {
    if (not file_name_exist($file . '.old.' . $i)) {
      my %patch;

      undef %patch;
      if (internal_sed($file, $file . '.old.' . $i, 0, \%patch)) {
         print wrap('File ' . $file . ' is backed up to ' . $file .
         '.old.' . $i . '.' . "\n\n", 0);
      } else {
         print STDERR wrap('Unable to backup the file ' . $file .
         ' to ' . $file . '.old.' . $i .'.' . "\n\n", 0);
      }
      return;
    }
  }

   print STDERR wrap('Unable to backup the file ' . $file .
   '. You have too many backups files. They are files of the form ' .
   $file . '.old.N, where N is a number. Please delete some of them.' . "\n\n", 0);
}

# Uninstall a file previously installed by us
sub uninstall_file {
  my $file = shift;

  if (not db_file_in($file)) {
    # Not installed by this program
    return;
  }

  if (file_name_exist($file)) {
    if (db_file_ts($file)) {
      my @statbuf;

      @statbuf = stat($file);
      if (defined($statbuf[9])) {
        if (db_file_ts($file) != $statbuf[9]) {
          # Modified since this program installed it
          backup_file($file);
        }
      } else {
        print STDERR wrap('Unable to get the last modification timestamp of '
                          . 'the file ' . $file . '.' . "\n\n", 0);
      }
    }

    if (not unlink($file)) {
      error('Unable to remove the file "' . $file . '".' . "\n");
    } else {
      db_remove_file($file);
    }

  } elsif (vmware_product() ne 'acevm') {
    print wrap('This program previously created the file ' . $file . ', and '
               . 'was about to remove it.  Somebody else apparently did it '
               . 'already.' . "\n\n", 0);
    db_remove_file($file);
  }
}

# Uninstall a directory previously installed by us
sub uninstall_dir {
  my $dir = shift;
  my $force = shift;

  if (not db_dir_in($dir)) {
    # Not installed by this program
    return;
  }

  if (-d $dir) {
    if ($force eq '1') {
      system(shell_string($gHelper{'rm'}) . ' -rf ' . shell_string($dir));
    } elsif (not rmdir($dir)) {
      print wrap('This program previously created the directory ' . $dir
                 . ', and was about to remove it. Since there are files in '
                 . 'that directory that this program did not create, it will '
                 . 'not be removed.' . "\n\n", 0);
      if (   defined($ENV{'VMWARE_DEBUG'})
          && ($ENV{'VMWARE_DEBUG'} eq 'yes')) {
        system('ls -AlR ' . shell_string($dir));
      }
    }
  } elsif (vmware_product() ne 'acevm') {
    print wrap('This program previously created the directory ' . $dir
               . ', and was about to remove it. Somebody else apparently did '
               . 'it already.' . "\n\n", 0);
  }

  db_remove_dir($dir);
}

# Return the version of VMware
sub vmware_version {
  my $buildNr;

  $buildNr = '6.0.0 build-45731';
  return remove_whitespaces($buildNr);
}

# Check the validity of an answer whose type is yesno
# Return a clean answer if valid, or ''
sub check_answer_yesno {
  my $answer = shift;
  my $source = shift;

  if (lc($answer) =~ /^y(es)?$/) {
    return 'yes';
  }

  if (lc($answer) =~ /^n(o)?$/) {
    return 'no';
  }

  if ($source eq 'user') {
    print wrap('The answer "' . $answer . '" is invalid. It must be one of "y" or "n".' . "\n\n", 0);
  }
  return '';
}
$gAnswerSize{'yesno'} = 3;
$gCheckAnswerFct{'yesno'} = \&check_answer_yesno;

# Check the validity of an answer based on its type
# Return a clean answer if valid, or ''
sub check_answer {
  my $answer = shift;
  my $type = shift;
  my $source = shift;

  if (not defined($gCheckAnswerFct{$type})) {
    die 'check_answer(): type ' . $type . ' not implemented :(' . "\n\n";
  }
  return &{$gCheckAnswerFct{$type}}($answer, $source);
}

# END OF THE LIBRARY FUNCTIONS

# Append a clearly delimited block to an unstructured text file --hpreg
# Result:
#  1 on success
#  -1 on failure
sub block_append {
   my $file = shift;
   my $begin = shift;
   my $block = shift;
   my $end = shift;

   if (not open(BLOCK, '>>' . $file)) {
      return -1;
   }

   print BLOCK $begin . $block . $end;

   if (not close(BLOCK)) {
      return -1;
   }

   return 1;
}


# Test if specified file contains line matching regular expression
# Result:
#  false on failure
#  true on a match
sub block_match {
   my $file = shift;
   my $block = shift;
   my $line = undef;

   if (open(BLOCK, '<' . $file)) {
      while (defined($line = <BLOCK>)) {
         chomp $line;
         last if ($line =~ /$block/);
      }
      close(BLOCK);
   }
   return defined($line);
}

# Remove all clearly delimited blocks from an unstructured text file --hpreg
# Result:
#  >= 0 number of blocks removed on success
#  -1 on failure
sub block_remove {
   my $src = shift;
   my $dst = shift;
   my $begin = shift;
   my $end = shift;
   my $count;
   my $state;

   if (not open(SRC, '<' . $src)) {
      return -1;
   }

   if (not open(DST, '>' . $dst)) {
      close(SRC);
      return -1;
   }

   $count = 0;
   $state = 'outside';
   while (<SRC>) {
      if      ($state eq 'outside') {
         if ($_ eq $begin) {
            $state = 'inside';
            $count++;
         } else {
            print DST $_;
         }
      } elsif ($state eq 'inside') {
         if ($_ eq $end) {
            $state = 'outside';
         }
      }
   }

   if (not close(DST)) {
      close(SRC);
      return -1;
   }

   if (not close(SRC)) {
      return -1;
   }

   return $count;
}


# Emulate a simplified basename program
sub internal_basename {
  return substr($_[0], rindex($_[0], '/') + 1);
}

# Set the name of the main /etc/vmware* directory.
sub initialize_globals {
  my $dirname = shift;

  if (vmware_product() eq 'console') {
    $gRegistryDir = '/etc/vmware-console';
    $gUninstallerFileName = 'vmware-uninstall-console.pl';
    $gConfigurator = 'vmware-config-console.pl';
  } elsif (vmware_product() eq 'api') {
    $gRegistryDir = '/etc/vmware-api';
    $gUninstallerFileName = 'vmware-uninstall-api.pl';
    $gConfigurator = 'vmware-config-api.pl';
  } elsif (vmware_product() eq 'mui') {
    $gRegistryDir = '/etc/vmware-mui';
    $gUninstallerFileName = 'vmware-uninstall-mui.pl';
    $gConfigurator = 'vmware-config-mui.pl';
  } elsif (vmware_product() eq 'acevm') {
    %gManifest = acevm_parse_manifest($dirname . '/' . $cManifestFilename);
    $gRegistryDir = $dirname;
    $gUninstallerFileName = 'vmware-uninstall-ace.pl';
    $gACEVMUpdate = (defined $gManifest{'update'}) && ($gManifest{'update'} == 1);
    $gPlayerTar = 'VMware-player.' . (is64BitUserLand() ? 'x86_64' : 'i386') .
                     '.tar.gz';
  } elsif (vmware_product() eq 'tools-for-linux' ||
           vmware_product() eq 'tools-for-freebsd' ||
           vmware_product() eq 'tools-for-solaris') {
    $gRegistryDir = '/etc/vmware-tools';
    $gUninstallerFileName = 'vmware-uninstall-tools.pl';
    $gConfigurator = 'vmware-config-tools.pl';
  } elsif (vmware_product() eq 'vix') {
    $gRegistryDir = '/etc/vmware-vix';
    $gUninstallerFileName = 'vmware-uninstall-vix.pl';
    $gConfigurator = 'vmware-config-vix.pl';
  } else {
    $gRegistryDir = '/etc/vmware';
    $gUninstallerFileName = 'vmware-uninstall.pl';
    $gConfigurator = 'vmware-config.pl';
  }
  $gStateDir = $gRegistryDir . '/state';
  $gInstallerMainDB = $gRegistryDir . '/locations';
  $gInstallerObject = $gRegistryDir . '/installer.sh';
  $gConfFlag = $gRegistryDir . '/not_configured';

  $gOption{'default'} = 0;
  $gOption{'upgrade'} = 0;
  $gOption{'eula_agreed'} = 0;
  $gOption{'create_shortcuts'} = 1;
}

# Set up the location of external helpers
sub initialize_external_helpers {
  my $program;
  my @programList;

  if (not defined($gHelper{'more'})) {
    $gHelper{'more'} = '';
    if (defined($ENV{'PAGER'})) {
      my @tokens;

      # The environment variable sometimes contains the pager name _followed by
      # a few command line options_.
      #
      # Isolate the program name (we are certain it does not contain a
      # whitespace) before dealing with it.
      @tokens = split(' ', $ENV{'PAGER'});
      $tokens[0] = DoesBinaryExist_Prompt($tokens[0]);
      if (not ($tokens[0] eq '')) {
        $gHelper{'more'} = join(' ', @tokens); # This is _already_ a shell string
      }
    }
    if ($gHelper{'more'} eq '') {
      $gHelper{'more'} = DoesBinaryExist_Prompt('more');
      if ($gHelper{'more'} eq '') {
        error('Unable to continue.' . "\n\n");
      }
      $gHelper{'more'} = shell_string($gHelper{'more'}); # Save it as a shell string
    }
  }

  if (vmware_product() eq 'tools-for-freebsd') {
    @programList = ('tar', 'sed', 'rm', 'killall', 'kldstat', 'umount',
                    'mv', 'uname', 'mount', 'du', 'df');
  } elsif (vmware_product() eq 'tools-for-solaris') {
    @programList = ('tar', 'sed', 'rm', 'killall', 'add_drv', 'rem_drv',
                    'modload', 'modunload', 'umount', 'mv', 'uname',
                    'mount', 'cat', 'update_drv', 'grep', 'gunzip',
                    'gzip', 'du', 'df', 'isainfo');
  } elsif (vmware_product() eq 'acevm') {
    @programList = ('tar', 'sed', 'rm', 'rmdir', 'mv', 'ps', 'du', 'df', 'mkdir',
		    'cp', 'chown');
  } elsif (vmware_product() eq 'vix') {
    @programList = ('tar', 'sed', 'rm', 'mv', 'ps', 'du', 'df');
  } else {
    @programList = ('tar', 'sed', 'rm', 'killall', 'lsmod', 'umount', 'mv',
                    'uname', 'mount', 'du', 'df');
  }

  foreach $program (@programList) {
    if (not defined($gHelper{$program})) {
      $gHelper{$program} = DoesBinaryExist_Prompt($program);
      if ($gHelper{$program} eq '') {
        error('Unable to continue.' . "\n\n");
      }
    }
  }

  # Used for removing links that were not added as files to the database.
  $gHelper{'insserv'} = internal_which('insserv');
  $gHelper{'chkconfig'} = internal_which('chkconfig');
}

# Check the validity of an answer whose type is dirpath
# Return a clean answer if valid, or ''
sub check_answer_dirpath {
  my $answer = shift;
  my $source = shift;

  $answer = dir_remove_trailing_slashes($answer);

  if (substr($answer, 0, 1) ne '/') {
      print wrap('The path "' . $answer . '" is a relative path. Please enter '
		 . 'an absolute path.' . "\n\n", 0);
      return '';
  }

  if (-d $answer) {
    # The path is an existing directory
    return $answer;
  }

  # The path is not a directory
  if (file_name_exist($answer)) {
    if ($source eq 'user') {
      print wrap('The path "' . $answer . '" exists, but is not a directory.'
                 . "\n\n", 0);
    }
    return '';
  }

  # The path does not exist
  if ($source eq 'user') {
    return (get_answer('The path "' . $answer . '" does not exist currently. '
                       . 'This program is going to create it, including needed '
                       . 'parent directories. Is this what you want?',
                       'yesno', 'yes') eq 'yes') ? $answer : '';
  } else {
    return $answer;
  }
}
$gAnswerSize{'dirpath'} = 20;
$gCheckAnswerFct{'dirpath'} = \&check_answer_dirpath;

# Check the validity of an answer whose type is existdirpath
# Return a clean answer if valid, or ''
sub check_answer_existdirpath {
  my $answer = shift;
  my $source = shift;

  $answer = dir_remove_trailing_slashes($answer);

  if (substr($answer, 0, 1) ne '/') {
      print wrap('The path "' . $answer . '" is a relative path. Please enter '
		 . 'an absolute path.' . "\n\n", 0);
      return '';
  }

  if (-d $answer) {
    # The path is an existing directory
    return $answer;
  }

  # The path is not a directory
  if (file_name_exist($answer)) {
    if ($source eq 'user') {
      print wrap('The path "' . $answer . '" exists, but is not a directory.'
		 . "\n\n", 0);
    }
  } else {
    if ($source eq 'user') {
      print wrap('The path "' . $answer . '" is not an existing directory.'
		 . "\n\n", 0);
    }
  }
  return '';
}
$gAnswerSize{'existdirpath'} = 20;
$gCheckAnswerFct{'existdirpath'} = \&check_answer_existdirpath;

# Check the validity of an answer whose type is initdirpath
# Return a clean answer if valid, or ''
sub check_answer_initdirpath {
  my $answer = shift;
  my $source = shift;
  my $testdir;
  my @rcDirList;

  $answer = dir_remove_trailing_slashes($answer);

  if (not (-d $answer)) {
    if ($source eq 'user') {
      print wrap('The path "' . $answer . '" is not an existing directory.' . "\n\n", 0);
    }
    return '';
  }

  if (vmware_product() eq 'tools-for-solaris') {
    @rcDirList = ('rc0.d', 'rc1.d', 'rc2.d', 'rc3.d');
  } else {
    @rcDirList = ('rc0.d', 'rc1.d', 'rc2.d', 'rc3.d', 'rc4.d', 'rc5.d', 'rc6.d');
  }

  foreach $testdir (@rcDirList) {
    if (not (-d $answer . '/' . $testdir)) {
      if ($source eq 'user') {
         print wrap('The path "' . $answer . '" is a directory which does not contain a ' .
         $testdir . ' directory.' . "\n\n", 0);
      }
      return '';
    }
  }

  return $answer;
}
$gAnswerSize{'initdirpath'} = 15;
$gCheckAnswerFct{'initdirpath'} = \&check_answer_initdirpath;

# Check the validity of an answer whose type is initscriptsdirpath
# Return a clean answer if valid, or ''
sub check_answer_initscriptsdirpath {
  my $answer = shift;
  my $source = shift;

  $answer = dir_remove_trailing_slashes($answer);

  if (not (-d $answer)) {
    if ($source eq 'user') {
      print wrap('The path "' . $answer . '" is not an existing directory.' . "\n\n", 0);
    }
    return '';
  }

  return $answer;
}
$gAnswerSize{'initscriptsdirpath'} = 15;
$gCheckAnswerFct{'initscriptsdirpath'} = \&check_answer_initscriptsdirpath;

# Check the validity of an answer whose type is authdport
# Return a clean answer if valid, or ''
sub check_answer_authdport {
  my $answer = shift;
  my $source = shift;

  if (($answer =~ /^\d+$/) && ($answer > 0) && ($answer < 65536)) {
    return $answer;
  }
  if ($source eq 'user') {
    print wrap('The answer '. $answer . ' is invalid. Please enter a valid '
               . 'port number in the range 1 to 65535.' . "\n\n", 0);
  }
  return '';
}

$gAnswerSize{'authdport'} = 5;
$gCheckAnswerFct{'authdport'} = \&check_answer_authdport;

# Check the validity of an answer whose type is username
# Return a clean answer if valid, or ''
sub check_answer_username {
  my $answer = shift;
  my $source = shift;

  my ($name, $passwd, $uid, $gid) = getpwnam($answer);
  if (!defined $name) {
    print wrap('The answer '. $answer . ' is invalid. Please enter a valid '
	       . 'user on this system.' . "\n\n", 0);
    return '';
  }
  return $answer;
}

$gAnswerSize{'username'} = 8;
$gCheckAnswerFct{'username'} = \&check_answer_username;

# Install one symbolic link
sub install_symlink {
  my $to = shift;
  my $name = shift;

  uninstall_file($name);
  if (file_check_exist($name)) {
    return;
  }
  # The file could be a symlink to another location.  Remove it
  unlink($name);
  if (not symlink($to, $name)) {
    error('Unable to create symbolic link "' . $name . '" pointing to file "'
          . $to . '".' . "\n\n");
  }
  db_add_file($name, 0);
}

# Install one directory (recursively)
# flags are forwarded to install_file calls and recursive install_dir calls
sub install_dir {
  my $src_dir = shift;
  my $dst_dir = shift;
  my $patchRef = shift;
  my $flags = shift;
  my $file;

  if (create_dir($dst_dir, 0x1)) {
    my @statbuf;

    @statbuf = stat($dst_dir);
    if (not (defined($statbuf[2]))) {
      error('Unable to get the access rights of destination directory "' . $dst_dir . '".' . "\n\n");
    }

    # Was bug 15880 --hpreg
    if (   ($statbuf[2] & 0555) != 0555
        && get_answer('Current access permissions on directory "' . $dst_dir
                      . '" will prevent some users from using '
                      . vmware_product_name()
                      . '. Do you want to set those permissions properly?',
                      'yesno', 'yes') eq 'yes') {
      safe_chmod(($statbuf[2] & 07777) | 0555, $dst_dir);
    }
  } else {
    install_permission($src_dir, $dst_dir);
  }
  foreach $file (internal_ls($src_dir)) {
    my $src_loc = $src_dir . '/' . $file;
    my $dst_loc = $dst_dir . '/' . $file;

    if (-l $src_loc) {
      install_symlink(readlink($src_loc), $dst_loc);
    } elsif (-d $src_loc) {
      install_dir($src_loc, $dst_loc, $patchRef, $flags);
    } else {
      install_file($src_loc, $dst_loc, $patchRef, $flags);
    }
  }
}

# Display the end-user license agreement
sub show_EULA {
  if ((not defined($gDBAnswer{'EULA_AGREED'}))
      || (db_get_answer('EULA_AGREED') eq 'no')) {
    query('You must read and accept the ' . vmware_product_name()
          . ' End User License Agreement to continue.'
          .  "\n" . 'Press enter to display it.', '', 0);

    open(EULA, './doc/EULA') ||
      error("$0: can't open EULA file: $!\n");

    my $origRecordSeparator = $/;
    undef $/;

    my $eula = <EULA>;
    close(EULA);
    
    $/ = $origRecordSeparator;
    
    $eula =~ s/(.{50,76})\s/$1\n/g;
   
    # Trap the PIPE signal to avoid broken pipe errors on RHEL4 U4.
    local $SIG{PIPE} = sub {};
       
    open(PAGER, '| ' . $gHelper{'more'}) ||
      error("$0: can't open $gHelper{'more'}: $!\n");
    print PAGER $eula . "\n";
    close(PAGER);

    print "\n";
    
    # Make sure there is no default answer here
    if (get_answer('Do you accept? (yes/no)', 'yesno', '') eq 'no') {
      print wrap('Please try again when you are ready to accept.' . "\n\n", 0);
      uninstall_file($gInstallerMainDB);
      exit 0;
    }
    print wrap('Thank you.' . "\n\n", 0);
  }
}

# XXX This code is mostly duplicated from the main server installer. -jhu
sub build_perl_api {
  my $control;
  my $build_dir;
  my $program;
  my $cTmpDirPrefix = 'api-config';

  foreach $program ('tar', 'perl', 'make', 'touch') {
    if (not defined($gHelper{$program})) {
      $gHelper{$program} = DoesBinaryExist_Prompt($program);
      if ($gHelper{$program} eq '') {
        error('Unable to continue.' . "\n\n");
      }
    }
  }

  print wrap('Installing the VMware VmPerl Scripting API.' . "\n", 0);

  $control = './control.tar';
  if (not (file_name_exist($control))) {
    error('Unable to find the VMware VmPerl Scripting API. '
          . 'You may want to re-install ' . vmware_product_name()
          . '.' .  "\n\n");
  }

  $build_dir = make_tmp_dir($cTmpDirPrefix);

  if (system(shell_string($gHelper{'tar'}) . ' -C ' . shell_string($build_dir) . ' -xopf ' .
             shell_string($control))) {
    print wrap('Unable to untar the "' . $control . '" file in the "' . $build_dir .
               '" directory.' . "\n\n", 0);
    error('');
  }

  if (system('cd ' . shell_string($build_dir . '/control-only') . ' && ' .
             shell_string($gHelper{'perl'}) . ' Makefile.PL > make.log 2>&1')) {
    print wrap('Unable to create the VMware VmPerl Scripting API makefile.' . "\n\n", 0);

    # Look for the header files needed to build the Perl module.  If we don't
    # find them, suggest to the user how they can install the files. -jhu
    if (open(PERLINC, shell_string($gHelper{'perl'}) . ' -MExtUtils::Embed ' .
             '-e perl_inc |')) {
      my $inc = <PERLINC>;
      close(PERLINC);
      $inc =~ s/\s*-I//;
      if (not file_name_exist($inc . '/perl.h')) {
        print wrap('Could not find necessary components to build the '
                   . 'VMware VmPerl Scripting API.  Look in your Linux '
                   . 'distribution to see if there is a perl-devel package.  '
                   . 'Install that package if it exists and then re-run this '
                   . 'installation program.' . "\n\n", 0);
      }
    }
    return(perl_config_fail($build_dir));
  }

  print wrap("\n", 0);
  print wrap('Building the VMware VmPerl Scripting API.' . "\n\n", 0);

  # Make sure we have a compiler available
  if (get_cc() eq '') {
    print wrap('Unable to install the VMware VmPerl Scripting API.', 0);
    print wrap('A C compiler is required to install the API.' . "\n\n",  0);
    remove_tmp_dir($build_dir);
    return;
  }

  # We touch all our files in case the system clock is set to the past.  Make will get confused and
  # delete our shipped .o file(s).
  # More code duplication from pkg_mgr.pl (really, really bad)
  system(shell_string($gHelper{'touch'}) . ' '
         . shell_string($build_dir . '/control-only') . '/* >>'
         . shell_string($build_dir . '/control-only') . '/make.log 2>&1');

  if (system(shell_string($gHelper{'make'}) . ' -C '
             . shell_string($build_dir . '/control-only') . ' '
             . shell_string('CC=' . $gHelper{'gcc'}) . ' '
             . ' >>' . shell_string($build_dir . '/control-only') . '/make.log 2>&1')) {
    print wrap('Unable to compile the VMware VmPerl Scripting API.' . "\n\n", 0);
    return(perl_config_fail($build_dir));
  }

  print wrap("Installing the VMware VmPerl Scripting API.\n\n", 0);


  # XXX This is deeply broken: we let a third party tool install a file without
  #     adding it to our installer database.  This file will never get
  #     uninstalled by our uninstaller --hpreg
  if (system(shell_string($gHelper{'make'}) . ' -C '
             . shell_string($build_dir . '/control-only') . ' '
             . shell_string('CC=' . $gHelper{'gcc'}) . ' '
             . ' install >>' . shell_string($build_dir . '/control-only')
             . '/make.log 2>&1')) {
    print wrap('Unable to install the VMware VmPerl Scripting API.' . "\n\n", 0);
    return(perl_config_fail($build_dir));
  }

  print wrap('The installation of the VMware VmPerl Scripting API succeeded.' . "\n\n", 0);
  remove_tmp_dir($build_dir);
}

# XXX Mostly duplicated from the main server installer. -jhu
# Common error message when we can't compile or install our perl modules
sub perl_config_fail {
  my $dir = shift;

  print wrap('********' . "\n". 'The VMware VmPerl Scripting API was not '
             . 'installed.  Errors encountered during compilation and '
             . 'installation of the module can be found here: ' . $dir
             . "\n\n" . 'You will not be able to use the "vmware-cmd" '
             . 'program.' . "\n\n" . 'Errors can be found in the log file: '
             . shell_string($dir . '/control-only/make.log')
             . "\n" . '********' . "\n\n", 0);
  error('');
}

# Handle the installation and configuration of vmware's perl module
sub install_perl_api {
  my $rootdir;
  my $answer;
  my $mandir;
  my $docdir;
  my %patch;

  undef %patch;
  install_dir('./etc', $gRegistryDir, \%patch, 0x1);

  $rootdir = '/usr';

  $answer = spacechk_answer('In which directory do you want to install '
                            . 'the executable files?', 'dirpath',
                            $rootdir . '/bin', './bin', 'BINDIR');
  undef %patch;
  install_dir('./bin', $answer, \%patch, 0x1);
  $gIsUninstallerInstalled = 1;

  $rootdir = internal_dirname($answer);
  # Don't display a double slash (was bug 14109) --hpreg
  if ($rootdir eq '/') {
    $rootdir = '';
  }

  # We don't use default answers here because once the user has
  # selected the root directory, we can give him better default answers than
  # his/her previous answers but we do want to make sure the directory
  # chosen has enough space to hold the data.

  $answer = spacechk_answer('In which directory do you want to install '
                            . 'the library files?', 'dirpath',
                            $rootdir . '/lib/vmware-api', './lib');
  db_add_answer('LIBDIR', $answer);
  undef %patch;
  install_dir('./lib', $answer, \%patch, 0x1);

  $docdir = $rootdir . '/share/doc';
  if (not (-d $docdir)) {
    $docdir = $rootdir . '/doc';
  }
  $answer = spacechk_answer('In which directory do you want to install the '
                            . 'documentation files?', 'dirpath',
                            $docdir . '/vmware-api', './doc');
  db_add_answer('DOCDIR', $answer);
  undef %patch;
  install_dir('./doc', $answer, \%patch, 0x1);

  build_perl_api();
}

# Look for the location of the vmware-acetool binary. We search /etc/vmware/locations
# for it and return the path or '' if we can't find it.
sub acevm_find_acetool  {
  my $path = '';

  if (open(PLAYERINSTALLDB, '</etc/vmware/locations')) {
    while (<PLAYERINSTALLDB>) {
      chomp;
      if (/^answer BINDIR (.+)$/) {
	$path = $1;
      } elsif (/^remove_answer BINDIR$/) {
	$path = '';
      }
    }
    close(PLAYERINSTALLDB);
  }
  if ($path ne '') {
    $path .= '/vmware-acetool';
    if (!file_name_exist($path)) {
      $path = '';
    }
  }
  return $path
}


# Some packages specify nat policies. Theses setting specify a subnet and whether or not to use
# dhcp for the vmnet NAT network. We have to apply these to the /etc/vmware/locations file
# then restart player's network services.
sub acevm_setup_nat  {
  my $hostvmpl = $gHostVmplDir . 'host.vmpl';
  my $subnet;
  my $dhcp;
  my $result;
  my $cmd = shell_string($gHelper{'vmware-acetool'}) . ' configureNATSubnet '
            . shell_string($hostvmpl)
	    . ' 2> /dev/null';
  open(CMD, $cmd . ' |');
  ($subnet, $dhcp) = <CMD>;
  $result = close(CMD);
  if (!defined($subnet) || !defined($dhcp) || !$result) {
    return;
  }

  open(LOCATIONS, '>>/etc/vmware/locations');
  print (LOCATIONS 'remove_answer VNET_8_HOSTONLY_SUBNET' . "\n");
  print (LOCATIONS 'answer VNET_8_HOSTONLY_SUBNET ' . $subnet . "\n");
  print (LOCATIONS 'remove_answer VNET_8_HOSTONLY_HOSTADDR' . "\n");
  print (LOCATIONS 'answer VNET_8_HOSTONLY_HOSTADDR ' . $subnet . "\n");
  print (LOCATIONS 'remove_answer VNET_8_DHCP' . "\n");
  print (LOCATIONS 'answer VNET_8_DHCP ' . ($dhcp ? 'yes' : 'no') . "\n");
  close(LOCATIONS);
  system(shell_string($gHelper{'vmware-config.pl'}) . ' --make-all-net');
  acevm_restart_vmnet8();
}

# Returns the build number of the current Player if
# installed, else ''
sub acevm_get_current_player_build {
  my $build = '';

  if (open(CONFIG, '/etc/vmware/config')) {
    while (<CONFIG>) {
       chomp;
       if (/product\.buildnumber = "(\d+)"/) {
          $build = $1;
       }
    }
    close CONFIG;
  }

  return $build;
}

# Returns build number from the included Player package
# if possible, else ''
sub acevm_get_included_player_build {
  my ($tmp, $cmd, $build);
  $build = '';

  if (-e $gPlayerTar) {
    print wrap("Checking VMware Player versions...\n\n", 0);
    $tmp = make_tmp_dir('vmware-installer-tmp');
    $cmd = $gHelper{'tar'} . ' -C ' . $tmp . ' -xzf ' . shell_string($gPlayerTar) .
           ' vmware-player-distrib/bin/vmware-uninstall.pl';
    system($cmd);

    if (open(FILE, $tmp . '/vmware-player-distrib/bin/vmware-uninstall.pl')) {
       while (<FILE>) {
          chomp;
          if (/'\S+\.\S+\.\S+.build-(\d+)'/) {
             $build = $1;
          }
       }
       close FILE;
    }

    remove_tmp_dir($tmp);
  }
  return $build;
}

# returns true if the included build player is newer
sub acevm_included_player_newer {
  return acevm_get_included_player_build() gt acevm_get_current_player_build();
}

# untar and install vmware-player.
sub acevm_install_vmplayer {
  my $ret;

  if (!-e $gPlayerTar) {
    error('This ACE package does not contain a VMware Player installer. You must ' .
	  'install VMware Player before installing this package.' . "\n\n");
  } elsif (!is_root()) {
    error('This program can install VMware Player installer for you, but you must ' .
	  'be a system administrator. Try running this setup program with sudo or ' .
	  'contact your system administrator for help.' . "\n\n");
  } else {
    my $cmd;
    my $tmpDir;

    print wrap("Installing VMware Player\n\n", 0);
    $tmpDir = make_tmp_dir('vmware-installer');
    $cmd .= $gHelper{'tar'} . ' -C ' . $tmpDir . ' -xzf ' . shell_string($gPlayerTar) .
            ' && ';
    $cmd .= shell_string($tmpDir . '/vmware-player-distrib/vmware-install.pl') .
            ' --default --no-create-shortcuts';
    $ret = system($cmd . ' && exit 64') >> 8;
    remove_tmp_dir($tmpDir);
    return ($ret == 64);
  }
}

# Open and parse the manifest file. This file contains little bits of information
# we need to install and ACE.
sub acevm_parse_manifest {
  my $path = shift;
  my $line;
  my %ret;

  if (not open(MANIFEST, '<' . $path)) {
    error('Unable to open the MANIFEST file in read-mode.' . "\n\n");
  }
  while ($line = <MANIFEST>) {
    # Strip carriage returns, if they exist.
    $line =~ s/\r//g;
    $line =~ m/^(.+)=(.+)$/;
    $ret{$1} = $2;
  }
  close MANIFEST;
  return %ret;
}

# Get the install (non-update) directory for this ACE package. If the directory
# contains a VMX file, make sure this package does not. Finally, there must be enough
# space to hold the package.
sub acevm_get_dir {
  my $rootdir = shift;
  my $answer = undef;

  while (!defined($answer)) {
    $answer = get_answer("In which directory do you want to install this ACE?",
			 'dirpath', $rootdir, 0);
    if (acevm_find_vmx($answer) ne '' && $gManifest{'hasVM'}) {
      print wrap("There is another ACE already installed in '" . $answer .
		 "'. Please choose another directory.\n\n", 0);
      $answer = '';
    } else {
      if (!check_dir_writeable($answer)) {
	print wrap("No permission to write to directory '" . $answer .
		   "'. Please choose another directory.\n\n", 0);
	$answer = '';
      }
      if (check_disk_space('.', internal_dirname($answer)) < 0) {
	print wrap("There is not enough space to install this ACE in '" .
		   $answer . "'. Please choose another directory.\n\n", 0);
	$answer = '';
      }
    }
    if ($answer eq '' && !$gOption{'default'}) {
      undef $answer;
    }
  }
  return $answer;
}

# Return the config filename. Usually, the MANIFEST file has this for us. If it
# doesn't we'll try to construct it from the acename entry.
sub acevm_get_config_filename {
  return (defined $gManifest{'configFile'} ? $gManifest{'configFile'} :
                                              $gManifest{'acename'}) .
          '.vmx';
}

# Get the installed directory for an already installed ACE package that we will update.
# The directory must exist already and there must be enough space to hold the update
# content.
sub acevm_get_updatedir {
  my $rootdir = shift;
  my $answer = undef;
  my $cfgFile;

  while (!defined($answer)) {
    $answer = get_answer('Which directory contains the ACE you want to update?',
			 'existdirpath', $rootdir, 0);
    if (($cfgFile = acevm_find_vmx($answer)) eq '') {
      print wrap("There is no ACE installed in '$answer'. ", 0);
      $answer = '';
    } elsif (!acevm_checkMasterID($cfgFile)) {
      print wrap("The ACE in '$answer' does not have the same ID as this " .
		 'ACE update package. ', 0);
      $answer = '';
    } elsif (check_disk_space('.', internal_dirname($answer)) < 0) {
      print wrap("There is not enough space to install this ACE update package in '" .
		 $answer . "'. ", 0);
      $answer = '';
    }
    if ($answer eq '' && !$gOption{'default'}) {
      undef $answer;
      print wrap('Please choose another directory.' . "\n\n", 0);
    }
  }
  return $answer;
}

# Does the package we're about to install have host.vmpl or host-update.vmpl?
sub acevm_package_has_host_policies  {
  return (file_name_exist('./VM/host.vmpl') ||
	  file_name_exist('./VM/host-update.vmpl'));
}

# Can we install host.vmpl? We will install our host.vmpl if:
# - There is no /etc/vmware-ace/host.vmpl installed
# - If there is a host.vmpl, is this an update? Specifically, the aceid of this
#   package matches the aceid of the package that installed the host.vmpl
sub acevm_can_install_host_policies {
  my $hostvmpl = $gHostVmplDir . 'host.vmpl';
  my $ret;

  if (file_name_exist($hostvmpl)) {
    system(shell_string($gHelper{'vmware-acetool'}) . ' checkMasterID '
	   . shell_string($hostvmpl) . ' ' . shell_string($gManifest{'aceid'})
	   . '> /dev/null 2>&1');
    $ret = ($? >> 8) == 0;
  } else {
    $ret = 1;
  }
  return $ret;
}

# Handle the installation and configuration of a packaged ACE vm.
# These are installed per user.
sub install_content_acevm {
  my %patch;

  # First, write BINDIR. It's the same directory as the one
  # containing $gInstallerMainDB.
  db_add_answer('BINDIR', $gRegistryDir);
  db_add_dir($gRegistryDir, 0x0);
  undef %patch;
  install_file("./vmware-install.pl", $gRegistryDir . '/' . $gUninstallerFileName,
	       \%patch, 0x0);
  undef %patch;
  install_file("./installer.sh", $gRegistryDir . '/installer.sh', \%patch, 0x0);
  undef %patch;
  install_file("./MANIFEST", $gRegistryDir . '/MANIFEST', \%patch, 0x0);
  $gIsUninstallerInstalled = 1;

  # copy VM to installPath
  install_dir("./VM", $gRegistryDir, \%patch, 0x0);
  acevm_install_host_policies();
  acevm_setup_nat();

  if ($gManifest{'hasVM'} == 1) {
    print wrap('Finalizing ACE... ', 0);
    my $ret = acevm_finalize();
    print wrap("\n", 0);
    if (!$ret) {
      error('This ACE was not finalized correctly. You should uninstall this ' .
	    'ACE with vmware-uninstall.pl. Cannot continue.' .
	    "\n\n");
    }
  }

  acevm_create_desktop_icon($gManifest{'acename'}, $gManifest{'aceid'}, $gRegistryDir .
                            '/VM/' . $gManifest{'configFile'});
}

# Handle an update to a packaged ACE vm.
# A matching ACE package must already have been installed
sub install_content_acevm_update {
  my %patch;

  # copy VM to installPath
  install_dir("./VM", $gRegistryDir, \%patch, 0x0);
  acevm_install_host_policies();
  acevm_setup_nat();
}

sub acevm_restart_vmnetdetect() {
  my $netservices;
  if (!defined db_get_answer_if_exists('NETSERVICES')) {
    db_add_answer('NETSERVICES',
		  internal_dirname(internal_dirname(internal_which(
		  $gHelper{'vmware-acetool'}))) . '/lib/vmware/net-services.sh');
  }
  system(shell_string(db_get_answer('NETSERVICES')) . ' restart 0 vmnet-detect');
}

sub acevm_restart_vmnet8() {
  my $netservices;
  if (!defined db_get_answer_if_exists('NETSERVICES')) {
    db_add_answer('NETSERVICES',
		  internal_dirname(internal_dirname($gHelper{'vmware-acetool'}))
		  . '/lib/vmware/net-services.sh');
  }
  system(shell_string(db_get_answer('NETSERVICES')) . ' restart 8');
}

# Install a host policy file, a ace.dat, maybe an an ace.crt, and restart
# vmware-netdetect
sub acevm_install_host_policies() {
  my %patch;
  my $update = '';

  if (file_name_exist($gRegistryDir . '/host.vmpl')) {
    if (not (-d $gHostVmplDir)) {
      safe_mkdir($gHostVmplDir);
      safe_chmod(0755, $gHostVmplDir);
    }
    if (file_name_exist($gHostVmplDir . 'host.vmpl')) {
      $update = '-update';
    }

    undef %patch;
    install_file($gRegistryDir . '/host.vmpl',
		 $gHostVmplDir . 'host' . $update . '.vmpl',
		 \%patch, 0x0);
    undef %patch;
    install_file($gRegistryDir . '/ace.dat',
		 $gHostVmplDir . 'ace.dat',
		 \%patch, 0x0);
    uninstall_file($gRegistryDir . '/host.vmpl');
    # If ace.crt exists, install it next to host policies. ace.crt lives either in the
    # VM directory or in the Resources directory.
    my $ace_crt_file = $gRegistryDir .
                       ($gManifest{'hasVM'} ? '/ACE Resources' : '') .
		       '/ace.crt';
    if (file_name_exist($ace_crt_file)) {
      undef %patch;
      install_file($ace_crt_file, $gHostVmplDir . 'ace.crt', \%patch, 0x0);
    }
    acevm_restart_vmnetdetect();
  }
}

# Install the content of the tools tar package
sub install_content_tools {
  my $rootdir;
  my $answer;
  my %patch;
  my $mandir;
  my $docdir;

  undef %patch;
  install_dir('./etc', $gRegistryDir, \%patch, 0x1);

  if (vmware_product() eq 'tools-for-freebsd') {
    $rootdir = '/usr/local';
  } elsif (vmware_product() eq 'tools-for-solaris') {
    $rootdir = '/usr';
  } else {
    $rootdir = '/usr';
  }
  $answer = spacechk_answer('In which directory do you want to '
                            . 'install the binary files?', 'dirpath',
                            $rootdir . '/bin', './bin', 'BINDIR');
  undef %patch;
  install_dir('./bin', $answer, \%patch, 0x1);

  $rootdir = internal_dirname($answer);
  # Don't display a double slash (was bug 14109) --hpreg
  if ($rootdir eq '/') {
    $rootdir = '';
  }

  # Finds the location of the initscripts dir
  $answer = get_initscriptsdir();

  # install the service script.
  if (vmware_product() eq 'tools-for-freebsd') {
    $answer = get_answer('In which directory do you want to install the '
                         . 'startup script?', 'dirpath', $answer);
    create_dir($answer,0);
  }
  undef %patch;
  install_file($cStartupFileName,
               $answer . (vmware_product() eq 'tools-for-freebsd' ?
                          '/vmware-tools.sh' : '/vmware-tools'), \%patch, 0x1);

  $gIsUninstallerInstalled = 1;

  # We don't use get_persistent_answer() here because once the user has
  # selected the root directory, we can give him better default answers than
  # his/her previous answers but we do want to make sure the directory
  # chosen has enough space to hold the data.

  $answer = get_answer('In which directory do you want to install '
                       . 'the daemon files?', 'dirpath', $rootdir . '/sbin');
  db_add_answer('SBINDIR', $answer);
  undef %patch;
  create_dir($answer, 0x1);

  $answer = spacechk_answer('In which directory do you want to install '
                            . 'the library files?', 'dirpath', $rootdir
                            . '/lib/vmware-tools', './lib');
  db_add_answer('LIBDIR', $answer);
  undef %patch;
  install_dir('./lib', $answer, \%patch, 0x1);

  $docdir = $rootdir . '/share/doc';
  if (not (-d $docdir)) {
    $docdir = $rootdir . '/doc';
  }
  $answer = spacechk_answer('In which directory do you want to install the '
                            . 'documentation files?', 'dirpath', $docdir
                            . '/vmware-tools', './doc');
  db_add_answer('DOCDIR', $answer);
  undef %patch;
  install_dir('./doc', $answer, \%patch, 0x1);

  if (vmware_product() eq 'tools-for-linux') {
     install_content_crosstalk_sdk($rootdir);
  }
}

sub uninstall_content_legacy_tools {
  my $OldInstallerDB = '/etc/vmware-tools/tools_log';
  my $OldInstallerDBOld = '/etc/vmware/tools_log';
  my $TmpMainDB = $gInstallerMainDB;
  my $File;
  my @Files;
  my $MovedFile;
  my $LinkedFile;
  my $answer;
  my $runlevel;

  # This is necessary for old installations of the tools
  # when /etc/vmware was one and unique dump for all the products
  if (-e $OldInstallerDBOld) {
    $OldInstallerDB = $OldInstallerDBOld;
  }
  if (!-e $OldInstallerDB) {
    # Old tools database not found, assume that the system is clean.
    return;
  }
  # Swap the db with the old one temporarely.
  $gInstallerMainDB = $OldInstallerDB;

  db_load();
  if (not open(INSTALLDB, '>>' . $gInstallerMainDB)) {
    error('Unable to open the tar installer database ' . $gInstallerMainDB
          . ' in write-mode.' . "\n\n");
  }

  $answer = get_answer('An old installation of the tools is detected. '
                       . 'Should this installation be removed ?',
                       'yesno', 'yes');
  if ($answer eq 'no') {
    error('');
  }

  # Stop the services
  foreach $File (keys %gDBFile) {
    if ($File =~ /\S+\/dualconf(\.sh)?$/) {
      system(shell_string($File) . ' stop');
      print "\n";
      last;
    }
  }
  # Remove the files
  foreach $File (keys %gDBFile) {
    if ($File !~ /\/tmp\S+/) {
      uninstall_file($File);
    }
  }
  # Remove the links
  foreach $LinkedFile (keys %gDBLink) {
    unlink $LinkedFile;
  }
  # At last, replace the original files.
  foreach $MovedFile (keys %gDBMove) {
    # XXX we do not have a timestamp for those files so we can't
    # know if the user changed it, so I back it up.
    if (-e $gDBMove{$MovedFile}) {
      backup_file($gDBMove{$MovedFile});
      unlink $gDBMove{$MovedFile};
    }
    if (-e $MovedFile) {
      if ($MovedFile =~ /\S+\.org/) {
        rename $MovedFile, $gDBMove{$MovedFile};
      } elsif ($gDBMove{$MovedFile} =~ /\.new$/) {
        # Nothing to do for /etc/rc and /etc/rc.shutdown
      } else {
        backup_file($MovedFile);
        unlink $MovedFile;
      }
    }
  }

  # Clean up the broken links.
  foreach $File (qw(/etc/modules.conf /etc/conf.modules /etc/XF86Config
                    /etc/X11/XF86Config /etc/X11/XF86Config-4)) {
    if ((-l $File) && (-e ($File . '.org'))) {
      unlink $File;
      rename $File . '.org', $File;
    }
  }

  get_initscriptsdir();
  $Files[0] = db_get_answer('INITSCRIPTSDIR') . '/vmmemctl';
  foreach $runlevel ('0', '1', '2', '3', '4', '5', '6', 'S', 's') {
    push @Files, db_get_answer('INITDIR') . '/rc' . $runlevel
                 . '.d/S99vmmemctl';
  }
  # Cleanup the files that aren't mentionned in the install database.
  foreach $File (@Files) {
    if (file_name_exist($File)) {
      unlink $File;
    }
  }

  db_save();
  unlink $gInstallerMainDB;

  if (direct_command('LANG=C ' .
                     shell_string(vmware_product() eq 'tools-for-freebsd' ?
                                  $gHelper{'kldstat'} : $gHelper{'lsmod'})) =~
                     /vmmemctl/) {
    print wrap('The uninstallation of legacy tools completed. '
             . 'Please restart this virtual machine to ensure that '
             . 'all the loaded components are removed from the memory and '
             . 'run this installer again to continue with the upgrade.'
             . "\n\n", 0);
    exit 0;
  }
  # Restore the original database file name in case we don't have
  # to reboot because of the loaded vmmemctl.
  $gInstallerMainDB = $TmpMainDB;
}

# Install the content of the WGS client tar package
sub install_content_console {
  my $rootdir;
  my $answer;
  my %patch;
  my $mandir;
  my $docdir;

  undef %patch;
  install_dir('./etc', $gRegistryDir, \%patch, 0x1);

  $rootdir = '/usr';

  $answer = spacechk_answer('In which directory do you want to '
                            . 'install the binary files?', 'dirpath',
                            $rootdir . '/bin', './bin', 'BINDIR');
  undef %patch;
  install_dir('./bin', $answer, \%patch, 0x1);
  $gIsUninstallerInstalled = 1;

  $rootdir = internal_dirname($answer);
  # Don't display a double slash (was bug 14109) --hpreg
  if ($rootdir eq '/') {
    $rootdir = '';
  }

  # We don't use a get_persistent_answer() call here because once the user has
  # selected the root directory, we can give him better default answers than
  # his/her previous answers. But we do want to make sure the directory
  # chosen has enough space to hold the data.

  $answer = spacechk_answer('In which directory do you want to install '
                            . 'the library files?', 'dirpath',
                            $rootdir . '/lib/vmware-console', './lib');
  db_add_answer('LIBDIR', $answer);
  undef %patch;
  install_dir('./lib', $answer, \%patch, 0x1);

  $mandir = $rootdir . '/share/man';
  if (not (-d $mandir)) {
    $mandir = $rootdir . '/man';
  }
  $answer = spacechk_answer('In which directory do you want to install '
                            . 'the manual files?', 'dirpath', $mandir, './man');
  db_add_answer('MANDIR', $answer);
  undef %patch;
  install_dir('./man', $answer, \%patch, 0x1);

  $docdir = $rootdir . '/share/doc';
  if (not (-d $docdir)) {
    $docdir = $rootdir . '/doc';
  }
  $answer = spacechk_answer('In which directory do you want to install '
                            . 'the documentation files?', 'dirpath',
                            $docdir . '/vmware-console', './doc');
  db_add_answer('DOCDIR', $answer);
  undef %patch;
  install_dir('./doc', $answer, \%patch, 0x1);
}

# Return GSX or ESX for server products, Workstation for ws
sub installed_vmware_version {
  my $vmware_version;
  my $vmware_version_string;

  if (not defined($gHelper{"vmware"})) {
    $gHelper{"vmware"} = DoesBinaryExist_Prompt("vmware");
    if ($gHelper{"vmware"} eq '') {
      error('Unable to continue.' . "\n\n");
    }
  }

  $vmware_version_string = direct_command(shell_string($gHelper{"vmware"})
                                          . ' -v 2>&1 < /dev/null');
  if ($vmware_version_string =~ /.*VMware\s*(\S+)\s*Server.*/) {
    $vmware_version = $1;
  } else {
    $vmware_version = "Workstation";
  }
  return $vmware_version;
}

# Install the mui package
sub install_content_mui {
  my $rootdir;
  my $answer;
  my %patch;
  my $program;
  my $vmware_version;
  my $mui_dir = "./mui";
  my $docdir;

  # Find the programs we need to go ahead with the install
  foreach $program ('vmware', 'hostname', 'tar', 'rm', 'cp', 'rmdir') {
    if (not defined($gHelper{$program})) {
      $gHelper{$program} = DoesBinaryExist_Prompt($program);
      if ($gHelper{$program} eq '') {
	uninstall_file($gInstallerMainDB);
        error('Unable to continue.' . "\n\n");
      }
    }
  }

  $vmware_version = installed_vmware_version();

  # Force use of the RPM if this is ESX
  if ($vmware_version eq "ESX") {
    error('VMware ESX Server requires that the ' . vmware_product_name() .
     ' be installed via RPM' . "\n\n");
  }

  # Make sure GSX is installed
  if ($vmware_version ne "GSX") {
    error('VMware GSX Server must be installed on this machine for the ' .
     vmware_product_name() . ' to work'
    . "\n\n");
  }

  # remove install made by old installer
  removeExistingInstall();

  # Install the uninstaller
  undef %patch;
  install_dir('./etc', $gRegistryDir, \%patch, 0x1);

  $rootdir = '/usr';
  $answer = spacechk_answer('In which directory do you want to '
                            . 'install the binary files?', 'dirpath',
                            $rootdir . './bin', './bin', 'BINDIR');
  undef %patch;

  install_dir( './bin' , $answer, \%patch, 0x1);

  # Finds the location of the initscripts dir
  get_initscriptsdir();

  $rootdir = internal_dirname($answer) . '/lib/vmware-mui';
  db_add_answer('INSTALLDIR', $rootdir);

  $gIsUninstallerInstalled = 1;

  # Find whether this is an install of gsx or esx
  $vmware_version = installed_vmware_version();

  $rootdir = '/home/vmware/mui';
  $answer = spacechk_answer('In which directory do you want to '
                            . 'install the ' . vmware_product_name(). ' files?',
                            'dirpath', $rootdir, $mui_dir, 'INSTALLDIR');
  $rootdir = $answer;

  # Install the mui package to the right place
  undef %patch;
  install_dir($mui_dir, $rootdir, \%patch, 0x1);

  # Install the console distrib package
  undef %patch;
  install_dir('./console-distrib', $rootdir . '/apache/htdocs/vmware/bin'
              , \%patch, 0x1);

  # Install the documentation
  $docdir = $rootdir . '/share/doc';
  if (not (-d $docdir)) {
    $docdir = $rootdir . '/doc';
  }
  $answer = spacechk_answer('In which directory would you like to install '
                            . 'the ' . 'documentation files?', 'dirpath',
                            $docdir, './doc');
  db_add_answer('DOCDIR', $answer);
  undef %patch;
  install_dir('./doc', $answer, \%patch, 0x1);
}

sub install_content_crosstalk_sdk {
  my %patch;
  my $libdir;
  my $pcfile;
  my $rootdir = shift;

  # If the sdk is not there return
  if (! -d './lib/vmware-vmci') {
     return;
  }

  $libdir = is64BitUserLand() ? '/lib64' : '/lib32';
  # Put libs in arch specific lib dir if it exists, otherwise in 'lib'
  if (! -d $rootdir . $libdir) {
    $libdir = '/lib';
  }

  # Install the console distrib package
  undef %patch;
  install_dir( './lib/vmware-vmci/include', $rootdir . '/include', \%patch, 0x1);
  undef %patch;
  if (is64BitUserLand()) {
    install_dir( './lib/vmware-vmci/lib64', $rootdir . $libdir, \%patch, 0x1);
  } else {
    install_dir( './lib/vmware-vmci/lib', $rootdir . $libdir, \%patch, 0x1);
  }

  create_dir($rootdir . $libdir . '/pkgconfig', 1);
  $pcfile =  $rootdir . $libdir . '/pkgconfig/vmci.pc';
  if (open(PKGCONFIG, '>' . $pcfile)) {
    db_add_file($pcfile, 0);
    print PKGCONFIG 'prefix=' . $rootdir . "\n";
    print PKGCONFIG 'exec_prefix=${prefix}' . "\n";
    print PKGCONFIG 'libdir=${exec_prefix}' . $libdir . "\n";
    print PKGCONFIG 'includedir=${prefix}/include/vmci' . "\n\n";
    print PKGCONFIG 'Name: vmci' . "\n";
    print PKGCONFIG 'Description: VMware Virtual Machine Communication Interface' . "\n";
    print PKGCONFIG 'Version: ' . vmware_version() . "\n";
    print PKGCONFIG 'Libs: -L${libdir} -lvmci' . "\n";
    print PKGCONFIG 'Cflags: -I${includedir}' . "\n";
    close PKGCONFIG;
  }
}

#BEGIN UNINSTALLER SECTION
# Uninstaller section for old style MUI installer: Most of this code is
# directly copied over from the old installer
my %gConfData;

# Read the config vars to our internal array
sub readConfig {
  my $registryFile = shift;
  if (open(OLDCONFIG, $registryFile)) {
    # Populate our array with everthing from the conf file.
    while (<OLDCONFIG>) {
      m/^\s*(\S*)\s*=\s*(\S*)/;
      $gConfData{$1} = $2;
    }
    close(OLDCONFIG);
    return(1);
  }
  return(0);
}

# Remove our files at a very high level
sub uninstallAllFiles {
  my $key;
  my @killFiles = (
                    "EULA.txt",
                    "VMware",
                    "apache",
                    "bin",
                    "include",
                    "lib",
                    "man",
                    "ssl",
                   );
  my %rCLinks = (
                  "rc3.d/S91httpd.vmware", "httpd.vmware",
                  "rc4.d/S91httpd.vmware", "httpd.vmware",
                  "rc5.d/S91httpd.vmware", "httpd.vmware",
                  "rc3.d/K07httpd.vmware", "httpd.vmware",
                  "rc4.d/K07httpd.vmware", "httpd.vmware",
                  "rc5.d/K07httpd.vmware", "httpd.vmware",
                 );

  print wrap('Removing the old installation....' . "\n" , 0);
  # Remove our init.d symlinks and files
  foreach $key (keys %rCLinks ) {
    system(shell_string($gHelper{'rm'}). ' -f '
           . shell_string("$gConfData{'mui.initd.fullpath'}/../$key"));
    system(shell_string($gHelper{'rm'}). ' -f '
           . shell_string("$gConfData{'mui.initd.fullpath'}/$key"));
    system(shell_string($gHelper{'rm'}). ' -f '
           . shell_string("$gConfData{'mui.initd.fullpath'}/$rCLinks{$key}"));
  }

  # Remove our MUI directories and files
  foreach $key (@killFiles) {
    system(shell_string($gHelper{'rm'}) . ' -rf '
           . shell_string("$gConfData{'mui.fullpath'}/$key"));
  }

  system(shell_string($gHelper{'rmdir'}) . ' '
         . shell_string("$gConfData{'mui.fullpath'}") .' > /dev/null 2>&1');

  # Remove secuity-config.pl
  system(shell_string($gHelper{'rm'}) . ' -f '
         . shell_string("$gConfData{'security.fullpath'}"));

  return(1);
}

#
# check to see if this is a certificate that was generated
# by this program (or the mui).  If so, then it is safe to delete
# it. (don't want to accidently blow away an expensive certificate).
#
sub certSafeToDelete {
  my $instDir = shift;
  my $certLoc = "/etc/vmware-mui/ssl";
  my $certUniqIdent = "(564d7761726520496e632e)";
  local *F;
  if (not open(F, "$instDir/bin/openssl x509 -in $certLoc/mui.crt " .
               " -noout -subject" . '|')) {
    return 1;
  }

  while (<F>) {
    if (m/$certUniqIdent/) {
      return 1;
    }
  }
  close (F);

  #Certificate didn't have our uniq identifier, so don't delete it.
  return 0;
}

# See if we've installed already and nuke it if found.
sub removeExistingInstall {
  my $instDir;
  my $key;
  my $registryDir  = "/etc/vmware-mui";
  my $registryFile = "$registryDir/config";
  my $readSuccess;
  if (-e "$registryDir") {
    # Populate our array with the data from the conf file.
    $readSuccess = readConfig($registryFile);
    if ($readSuccess) {
      $instDir = $gConfData{'mui.fullpath'};
      # Warn them if we can't figure out where thier old install is.
      if (not -e $instDir) {
        print wrap('You seem to have a previous installation of '
                   . vmware_product_name() . ' but the installer '
                   . 'cant find where it is. You might want to '
                   . 'remove your old installation by hand.' . "\n", 0);
        return;
      }
      stop_mui($instDir);
      if (!certSafeToDelete($instDir)) {
        print wrap('The SSL certificate in ' . "$registryDir " . 'doesn\'t appear'
                   . ' to have been generated by ' . vmware_product_name()
                   . '. Not deleting files in ' . "$registryDir"
                   . ' as a precaution.' . "\n" , 0);
      } else {
        # We cant just delete the entire $registryDir directory since it
        # will now contain our locations file that we created to hold the
        # new install database
        system(shell_string($gHelper{'rm'}) . ' -rf '
               . shell_string($registryDir . '/ssl'));
        system(shell_string($gHelper{'rm'}) . ' -rf '
               . shell_string($registryDir . '/config'));
        # Since this directory existed due to the old install, we need
        # to add it to our database to make it be deleted properly on
        # uninstall
        db_add_dir($registryDir, 0x1);
      }
      uninstallAllFiles();
      return(1);
    }
  }
  return(0);
}

# END UNINSTALLER SECTION
# Install the content of the tar package
sub install_content {
  my $rootdir;
  my $answer;
  my %patch;
  my $mandir;
  my $docdir;
  my $initdir;
  my $initscriptsdir;

  undef %patch;
  install_dir('./etc', $gRegistryDir, \%patch, 0x1);

  $rootdir = '/usr';

  $answer = spacechk_answer('In which directory do you want '
                            . 'to install the binary files?', 'dirpath',
                            $rootdir . '/bin', './bin', 'BINDIR');
  undef %patch;
  install_dir('./bin', $answer, \%patch, 0x1);

  #
  # Install the startup script (and make the old installer aware of this one)
  #
  undef %patch;
  install_file($cStartupFileName,
               get_initscriptsdir() . '/vmware', \%patch, 0x1);

  $gIsUninstallerInstalled = 1;

  # Setuid root
  safe_chmod(04555, $answer . '/vmware-ping');

  $rootdir = internal_dirname($answer);
  # Don't display a double slash (was bug 14109) --hpreg
  if ($rootdir eq '/') {
    $rootdir = '';
  }

  # We don't use get_persistent_answer() here because once the user has
  # selected the root directory, we can give him better default answers than
  # his/her previous answers.  Even though this is asking for a directory,
  # the actual source of the files is within the source ./lib so the
  # spacechk_answer() below handles it.
  if (vmware_product() eq 'wgs' || vmware_product() eq 'ws') {
    $answer = get_answer('In which directory do you want to install '
                         . 'the daemon files?', 'dirpath', $rootdir . '/sbin');
    db_add_answer('SBINDIR', $answer);
    undef %patch;
    install_dir('./sbin', $answer, \%patch, 0x1);
    # Setuid root
    safe_chmod(04555, $answer . '/vmware-authd');
  }

  $answer = spacechk_answer('In which directory do you want to install '
                            . 'the library files?', 'dirpath',
                            $rootdir . '/lib/vmware', './lib');
  db_add_answer('LIBDIR', $answer);
  undef %patch;
  install_dir('./lib', $answer, \%patch, 0x1);
  # Setuid root
  safe_chmod(04555, $answer . '/bin/vmware-vmx');
  safe_chmod(04555, $answer . '/bin-debug/vmware-vmx');
  safe_chmod(04555, $answer . '/bin-stats/vmware-vmx');

  # If the product has man pages ask for the man pages location. */
  if (-d './man') {
    $mandir = $rootdir . '/share/man';
    if (not (-d $mandir)) {
      $mandir = $rootdir . '/man';
    }
    $answer = spacechk_answer('In which directory do you want to install '
                              . 'the manual files?', 'dirpath',
                              $mandir, './man');
    db_add_answer('MANDIR', $answer);
    undef %patch;
    install_dir('./man', $answer, \%patch, 0x1);
  }

  $docdir = $rootdir . '/share/doc';
  if (not (-d $docdir)) {
    $docdir = $rootdir . '/doc';
  }
  $answer = spacechk_answer('In which directory do you want to install '
                            . 'the documentation files?', 'dirpath',
                            $docdir . '/vmware', './doc');
  db_add_answer('DOCDIR', $answer);
  undef %patch;
  install_dir('./doc', $answer, \%patch, 0x1);
   install_symlink(db_get_answer('DOCDIR') . '/EULA',
                   db_get_answer('LIBDIR') . '/share/EULA.txt');

  # Don't forget the vix perl tar ball..
  if (-d 'vmware-vix/api' ) {
    undef %patch;
    install_dir( './vmware-vix/api', db_get_answer('LIBDIR')
                  . '/vmware-vix/api', \%patch, 0x1);
  }

  find_vix_tar();
  install_content_crosstalk_sdk($rootdir);

  if (vmware_product() eq 'ws') {
    install_content_player();
  }
}

sub install_content_player {
  my %patch;
  install_dir('./system_etc', '/etc', \%patch, 1);
  undef %patch;
  install_dir('./usr', '/usr', \%patch, 1);
}

sub get_initscriptsdir {
  my $initdir;
  my $initscriptsdir;
  my $answer;

  if (vmware_product() eq 'tools-for-freebsd') {
    $initdir = '/usr/local/etc/rc.d';
    $initscriptsdir = '/usr/local/etc/rc.d';
    db_add_answer('INITDIR', $initdir);
    db_add_answer('INITSCRIPTSDIR', $initscriptsdir);
    return $initscriptsdir;
  }

  # The "SuSE version >= 7.1" way
  $initdir = '/etc/init.d';
  if (check_answer_initdirpath($initdir, 'default') eq '') {
    # The "SuSE version < 7.1" way
    $initdir = '/sbin/init.d';
    if (check_answer_initdirpath($initdir, 'default') eq '') {
      # The "RedHat" way
      $initdir = '/etc/rc.d';
      if (check_answer_initdirpath($initdir, 'default') eq '') {
        # The "Debian" way
        $initdir = '/etc';
        if (check_answer_initdirpath($initdir, 'default') eq '') {
          $initdir = '';
        }
      }
    }
  }
  $answer = get_persistent_answer('What is the directory that contains the init'
                                  .' directories (rc0.d/ to rc6.d/)?'
                                  , 'INITDIR', 'initdirpath', $initdir);

  # The usual way
  $initscriptsdir = $answer . '/init.d';
  if (check_answer_initscriptsdirpath($initscriptsdir, 'default') eq '') {
    # The "SuSE version >= 7.1" way
    $initscriptsdir = $answer;
    if (check_answer_initscriptsdirpath($initscriptsdir, 'default') eq '') {
      $initscriptsdir = '';
    }
  }
  $answer = get_persistent_answer('What is the directory that contains the init'
                                  .' scripts?', 'INITSCRIPTSDIR'
                                  , 'initscriptsdirpath', $initscriptsdir);
  return $answer;
}

sub get_home_dir {
   return (getpwnam(get_user()))[7] || (getpwuid($<))[7];
}

sub get_user {
   if (defined $ENV{'SUDO_USER'}) {
      return $ENV{'SUDO_USER'};
   }
   else {
      return $ENV{'USER'};
   }
}

# Install a tar package or upgrade an already installed tar package
sub install_or_upgrade {
  print wrap('Installing ' . vmware_product_name() . '.  This may take '
             . 'from several minutes to over an hour depending upon its size.'
             . "\n\n", 0);

  if (vmware_product() eq 'console') {
    install_content_console();
  } elsif (vmware_product() eq 'api') {
    install_perl_api();
  } elsif (vmware_product() eq 'acevm') {
    if (acevm_is_instance_running()) {
     error ('Cannot install this package because this ACE appears to be ' .
	     'running. Please close the running ACE and run this setup ' .
	     'program again.' . "\n\n");
    }
    if ($gACEVMUpdate) {
      install_content_acevm_update();
    } else {
      install_content_acevm();
    }
    if (is_root()) {
      print wrap('You have installed this ACE package as the root user. This setup ' .
		 'package can change the ownership (chown) of the package files so ' .
		 'that another user may run this package.'. "\n\n", 0);
      my $user = get_answer('Enter a username to change package ownership',
			 'username', get_user());
      my ($name,$passwd,$uid,$gid,$quota,$comment,$gcos,$home) = getpwnam($user);

      if ($gACEVMUpdate) {
         system("chown -R $uid:$gid " . shell_string(db_get_answer('BINDIR')));
      } else {
         system("chown -R $uid:$gid " . shell_string($gFirstCreatedDir));
         system("chown -R $uid:$gid " . shell_string($home . '/.local'));
      }
    }
  } elsif (vmware_product() eq 'mui') {
    install_content_mui();
  } elsif (vmware_product() eq 'tools-for-linux' ||
           vmware_product() eq 'tools-for-freebsd' ||
           vmware_product() eq 'tools-for-solaris') {
    install_content_tools();
  } elsif (vmware_product() eq 'vix') {
    install_content_vix();
  } else {
    install_content();
  }

  if (vmware_product() eq 'acevm' && (my $vmx = acevm_find_vmx($gRegistryDir)) ne '') {
    print wrap('The installation of this VMware ACE package completed successfully. '
	       . 'You can decide to remove this software from your system at any '
	       . 'time by invoking the following command: "'
	       . db_get_answer('BINDIR') . '/' . $gUninstallerFileName . '".'
	       . "\n\n", 0);

    print wrap('You can now run this ACE by invoking the following ' .
	       ' command: "vmplayer ' . $vmx . '"' . "\n\n", 0);
    print wrap('A shortcut to your ACE package has been added to your ' .
               'application menu under the "VMware ACE" folder.  You may have to ' .
               "logout and log back in for it to appear.\n\n", 0);
  } else {
    print wrap('The installation of ' . vmware_longname()
               . ' completed successfully. '
               . 'You can decide to remove this software from your system at any '
               . 'time by invoking the following command: "'
               . db_get_answer('BINDIR') . '/' . $gUninstallerFileName . '".'
               . "\n\n", 0);
  }

}

# Uninstall files and directories beginning with a given prefix
sub uninstall_prefix {
  my $prefix = shift;
  my $prefix_len;
  my $file;
  my $dir;

  $prefix_len = length($prefix);

  # Remove all files beginning with $prefix
  foreach $file (keys %gDBFile) {
    if (substr($file, 0, $prefix_len) eq $prefix) {
      uninstall_file($file);
    }
  }

  # Remove all directories beginning with $prefix
  # We sort them by decreasing order of their length, to ensure that we will
  # remove the inner ones before the outer ones
  foreach $dir (sort {length($b) <=> length($a)} keys %gDBDir) {
    if (substr($dir, 0, $prefix_len) eq $prefix) {
      uninstall_dir($dir, vmware_product() eq 'acevm' ? '1' : '0');
    }
  }
}

# Uninstall a tar package
sub uninstall {
  my $service_name = shift;
  my $hostVmplPresent = db_file_in($gHostVmplDir . 'host.vmpl');

  if ($hostVmplPresent) {
    if (!is_root()) {
      error ('This package installed host policies and must be uninstalled by a ' .
	     'system administrator. Try running this setup program with sudo ' .
	     'or contact your system administrator for help.' . "\n\n");
    } else {
      uninstall_file($gHostVmplDir . 'host.vmpl');
      uninstall_file($gHostVmplDir . 'ace.dat');
      acevm_restart_vmnetdetect();
    }
  }

  if (defined($gDBAnswer{'INITSCRIPTSDIR'})
      && db_file_in(db_get_answer('INITSCRIPTSDIR') . $service_name)) {
    # The installation process ran far enough to create the startup script
    my $status;

    # In case service links were created the LSB way, remove them
    if ($gHelper{'insserv'} ne '') {
      system(shell_string($gHelper{'insserv'}) . ' -r '
             . shell_string(db_get_answer('INITSCRIPTSDIR') . $service_name));
    } elsif ($gHelper{'chkconfig'} ne '') {

      # We need to trim the leading '/' off of the service name.
      my $trim_service_name = (substr($service_name, 0, 1) eq '/')
	? substr($service_name, 1) : $service_name;
      system(shell_string($gHelper{'chkconfig'}) . ' --del ' . $trim_service_name);
    }

    # Stop the services
    $status = system(shell_string(db_get_answer('INITSCRIPTSDIR')
                                  . $service_name) . ' stop') >> 8;
    if ($status) {
      if ($status == 2) {
        # At least one instance of VMware is still running. We must refuse to
        # uninstall
        error('Unable to stop ' . vmware_product_name()
              . '\'s services. Aborting the uninstallation.' . "\n\n");
      }

      # Oh well, at worst the user will have to reboot the machine... The
      # uninstallation process should go as far as possible
      print STDERR wrap('Unable to stop ' . vmware_product_name()
                        . '\'s services.' . "\n\n", 0);
    } else {
      print "\n";
    }
  }

  my $eclipse_dir = db_get_answer_if_exists('ECLIPSEDIR');
  if (defined $eclipse_dir) {
     system($gHelper{'rm'} . ' -rf ' . $eclipse_dir . '/../configuration/com.vmware.bfg*');
  }

  # Let the VMX know that we're uninstalling the Tools. We need to
  # do this before we remove the files, because we use guestd to
  # send the RPC. But we'd like to do it as late as possible in the
  # uninstall process so that we won't accidentally tell the VMX that the
  # Tools are gone when they're still there.
  if (vmware_product() eq 'tools-for-linux' ||
      vmware_product() eq 'tools-for-freebsd' ||
      vmware_product() eq 'tools-for-solaris') {
    send_rpc('tools.set.version 0');
  }

  uninstall_prefix('');
}

# Ask the user for file locations for libs, bins, etc.  Check
# to see if this build is an official one or not, and show the
# EULA if it's not an official build.
sub install_content_vix {
  my $rootdir;
  my $answer;
  my %patch;
  my $mandir;
  my $docdir;
  my $initdir;
  my $initscriptsdir;
  my $libdir;

  undef %patch;
  install_dir('./etc', $gRegistryDir, \%patch, 0x1);

  if ('45731' != 0) {
    # suspend any '--default' option to force user interaction here.  The user
    # must answer the EULA question before continuing.
    my $tmp = $gOption{'default'};
    $gOption{'default'} = 0;
    show_EULA();
    $gOption{'default'} = $tmp;
  }

  $rootdir = '/usr';
  $answer = spacechk_answer('In which directory do you want '
                            . 'to install the ' . vmware_product_name()
                            . ' binary files?', 'dirpath',
                            $rootdir . '/bin', './bin', 'BINDIR');
  db_add_answer('BINDIR', $answer);
  undef %patch;
  install_dir('./bin', $answer, \%patch, 0x1);

  $rootdir = internal_dirname($answer);
  # Don't display a double slash (was bug 14109) --hpreg
  if ($rootdir eq '/') {
    $rootdir = '';
  }
  $gIsUninstallerInstalled = 1;

  $libdir = 'vmware-vix/lib';
  $answer = spacechk_answer('In which directory do you want '
                            . 'to install the ' . vmware_product_name()
                            . ' library files?', 'dirpath',
                            $rootdir . '/lib/'. $libdir, 'vmware-vix/lib' );

  db_add_answer('VIXLIBDIR', $answer);
  db_add_answer('LIBDIR', $answer);
  undef %patch;
  install_dir($libdir, $answer, \%patch, 0x1);

  # If the product has man pages ask for the man pages location. */
  if (-d './man') {
    $mandir = $rootdir . '/share/man';
    if (not (-d $mandir)) {
      $mandir = $rootdir . '/man';
    }
    $answer = spacechk_answer('In which directory do you want '
                              . 'to install the ' . vmware_product_name()
                              . ' man pages?', 'dirpath',
                              $rootdir . '/man', 'vmware-vix/man');
    db_add_answer('MANDIR', $answer);
    undef %patch;
    install_dir('vmware-vix/man', $answer, \%patch, 0x1);
  }

  $docdir = $rootdir . '/share/doc';
  if (not (-d $docdir)) {
    $docdir = $rootdir . '/doc';
  }
  $answer = spacechk_answer('In which directory do you want '
                            . 'to install the ' . vmware_product_name()
                            . ' document pages?', 'dirpath',
                            $docdir . '/vmware-vix', 'doc');

  undef %patch;
  install_dir('./doc', $answer, \%patch, 0x1);
  db_add_answer('DOCDIR', $answer);
  undef %patch;
  install_dir('vmware-vix/include', $rootdir . '/include', \%patch, 0x1);
  undef %patch;
  install_dir('vmware-vix/api', db_get_answer('VIXLIBDIR') . '/api', \%patch, 0x1);

  return 1;
}

# Add the variable, vix.libdir, to /etc/vmware/config
# so the Vix API can figure out where its libraries
# are.
sub finalize_vix_install {
  my $vixAltDir = '/etc/vmware';
  my $vixAltState = $vixAltDir . '/state';
  my $vixAltConfig = $vixAltDir . '/config';
  my $vixLibdir = "vix.libdir";
  my $tmpFile = $vixAltConfig . '.tmp';

  if (! -d $gRegistryDir) {
    create_dir($gRegistryDir, 0x1);
  }

  my $tmpVar = db_get_answer_if_exists('VIXLIBDIR');
  # Remove any pre-existing values for 'vix.libdir'
  if (-e $vixAltConfig) {
    block_remove($vixAltConfig, $tmpFile, $cVixMarkerBegin,
                 $vixLibdir, $cVixMarkerEnd);
    rename ($tmpFile, $vixAltConfig);
  } else {
    create_dir($vixAltState, 0x1);
    block_append($vixAltState . '/config', $cVixMarkerBegin,
                 $vixLibdir . '= "' . $tmpVar . '"' . "\n",
                 $cVixMarkerEnd);
    # Add the backup config file to the list of files to remove on uninstall.
    # Make sure that the timeStamp is recorded so the file won't be removed
    # if it's been modified since its creation by us.
    db_add_file($vixAltState . '/config', 1);
  }

  # add or update the value of vix.libdir in the workstation's config file.
  # If the config file is not there, this call will automatically create it.
  block_append($vixAltConfig, $cVixMarkerBegin, 'vix.libdir = "' . $tmpVar
                . '"' . "\n", $cVixMarkerEnd);

  exit 0;
}

# find the vix tar package within the workstation distribution tree and install
# it into db_get_answer('LIBDIR')/vmware-vix
sub find_vix_tar {
  my %patch;
  my $vixTarFile = 'vmware-vix/vmware-vix.tar.gz';
  my $vixInstalledTarFile = db_get_answer('LIBDIR') . '/' . $vixTarFile;

  # If this is a workstation tar install there will be a vix tarball:
  # vmware-distrib/vmware-vix/vmware-vix.tar.gz.  "Install" it into
  # its final spot in the tree.  If this is an rpm install, the tarball
  # will already be in its final location.  This is where the vmware-config
  # script will look for the tar file.
  if (-e $vixTarFile) {
    undef %patch;
    install_file($vixTarFile, $vixInstalledTarFile, \%patch, 0x1);
    return 1;
  }
  return 0;
}

# Remove the text we added to the core config file in
# /etc/vmware/config.  Call uninstall on the whole tree.
sub uninstall_vix {
  my $vixAltConfig = '/etc/vmware/config';
  my $tmpVixConfig = $vixAltConfig . '.tmp';
  my @statbuf;

  uninstall_prefix('');

  # check to see if $gRegistryDir is still around.  If it has no files,
  # remove it.
  if (-d $gRegistryDir) {
    if (!direct_command('find ' .  $gRegistryDir . '  -type f -o -type l')) {
      rmdir($gRegistryDir);
    }
  }

  if (! -e $vixAltConfig) {
    return undef;
  }
  if (block_remove($vixAltConfig, $tmpVixConfig,
                   $cVixMarkerBegin, $cVixMarkerEnd) != -1) {
    rename($tmpVixConfig, $vixAltConfig);
  }

  # If we just removed the last bits from the file,
  # then remove the file so it won't get left behind
  # in an uninstall.
  @statbuf = stat($vixAltConfig);
  if ( !$statbuf[7]) {
    unlink ($vixAltConfig);
  }
}

# Return the specific VMware product
sub vmware_product {
  return 'ws';
}

# this is a function instead of a macro in the off chance that product_name
# will one day contain a language-specific escape character. (ask hpreg)
sub vmware_product_name {
  return 'VMware Workstation';
}

# This function returns i386 under most circumstances, returning x86_64 when
# the product is Workstation and is the 64bit version at that.
sub vmware_product_architecture {
  return 'i386';
}

# Stop any currently running instance of the mui
sub stop_mui {
  my $installDir = shift;
  if (-e "$installDir/apache/bin") {
    system(shell_string($installDir) . '/apache/bin/apachectl stop > /dev/null 2>&1');
  } else {
    # Maybe the install didnt get this far! We will ignore it
    print wrap('Couldnt find a running instance of Apache.' . "\n\n", 0);
  }
}

# Delete the mui log files that were created once the mui started running
sub clear_mui_logs {
  my $installDir = db_get_answer('INSTALLDIR');
  if (-e "$installDir/apache/logs") {
    system(shell_string($gHelper{'rm'}) . ' -f ' . "$installDir/apache/logs/*");
  }
}

# Return product name and version
sub vmware_longname {
   my $name = vmware_product_name() . ' ' . vmware_version();

   if (not (vmware_product() eq 'server')) {
      if (vmware_product() eq 'tools-for-solaris') {
        $name .= ' for Solaris';
      } elsif (vmware_product() eq 'tools-for-freebsd') {
        $name .= ' for FreeBSD';
      } else {
        $name .= ' for Linux';
      }
   }

   return $name;
}

# If this was a WGS build, remove our inetd.conf entry for auth daemon
# and stop the vmware-serverd
sub wgs_uninstall {

  system(shell_string($gHelper{'killall'}) . ' -TERM vmware-serverd  >/dev/null 2>&1');
  uninstall_superserver();
}

# Try and figure out which "superserver" is installed and unconfigure correct
# one.
sub uninstall_superserver {
  my $inetd_conf  = "/etc/inetd.conf";
  my $xinetd_dir  = "/etc/xinetd.d";

  # check for xinetd
  # XXX Could be a problem, as they could start xinetd with '-f config_file'.
  #     We could do a ps -ax, look for xinetd, parse the line, find the config
  #     file, parse the config file to find the xinet.d directory.  Or parse if
  #     from the init.d script somewhere.  If they use init.d.
  if ( -d $xinetd_dir ) {
    uninstall_xinetd($xinetd_dir);
  }

  # check for inetd
  if ( -e $inetd_conf ) {
    uninstall_inetd($inetd_conf);
  }
}

# Restart the inetd service
sub restart_inetd {
  my $inetd_restart = db_get_answer('INITSCRIPTSDIR') . '/inetd';
  if (-e $inetd_restart) {
    if (!system(shell_string($inetd_restart) . ' restart')) {
      return;
    }
  }
  system(shell_string($gHelper{'killall'}) . ' -HUP inetd');
}

# Cleanup the inetd.conf file.
sub uninstall_inetd {
  my $inetd = shift;
  my %patch = ('^# VMware auth.*$' => '',
          '^.*stream\s+tcp\s+nowait.*vmauthd.*$' => '',
          '^.*stream\s+tcp\s+nowait.*vmware-authd.*$' => '');
  my $tmp_dir = make_tmp_dir('vmware-installer');
  # Build the temp file's path
  my $tmp = $tmp_dir . '/tmp';

  # XXX Use the block_*() API instead, like we do for $cServices. --hpreg
  internal_sed($inetd, $tmp, 0, \%patch);
  undef %patch;

  if (not internal_sed($tmp, $inetd, 0, \%patch)) {
    print STDERR wrap('Unable to copy file ' . $tmp . ' back to ' . $inetd
                      . '.' . "\n" . 'The authentication daemon was not removed from '
                      . $inetd . "\n\n", 0);
  }
  remove_tmp_dir($tmp_dir);
  restart_inetd();
}

#Restart xinetd
sub restart_xinetd {
  my $xinetd_restart = db_get_answer('INITSCRIPTSDIR') . '/xinetd';
  if (-e $xinetd_restart) {
    if (!system(shell_string($xinetd_restart) . ' restart')) {
      return;
    }
  }
  system(shell_string($gHelper{'killall'}) . ' -USR2 xinetd');
}


# Cleanup the xinetd.d directory.
sub uninstall_xinetd {
  my $conf_dir = shift;
  my $tmp_dir;
  my $tmp;

  # XXX What the heck is that? Why isn't this file registered with the
  #     installer's database, and automatically removed? --hpreg
  unlink($conf_dir . '/vmware-authd');

  # Unregister the IP service. --hpreg
  $tmp_dir = make_tmp_dir('vmware-installer');
  $tmp = $tmp_dir . '/tmp';
  if (block_remove($cServices, $tmp, $cMarkerBegin, $cMarkerEnd) >= 0) {
    system(shell_string($gHelper{'mv'}) . ' -f ' . shell_string($tmp) . ' '
           . shell_string($cServices));
  }
  remove_tmp_dir($tmp_dir);

  restart_xinetd();
}


# Display a usage error message for the install program and exit
sub install_usage {
  print STDERR wrap(vmware_longname() . ' installer' . "\n" . 'Usage: ' . $0
                    . ' [[-][-]d[efault]]' . "\n"
                    . '    default: Automatically answer questions with the '
                    . 'proposed answer.' . "\n\n", 0);
  exit 1;
}

# Remove a temporary directory
sub remove_tmp_dir {
  my $dir = shift;

  if (system(shell_string($gHelper{'rm'}) . ' -rf ' . shell_string($dir))) {
    error('Unable to remove the temporary directory ' . $dir . '.' . "\n\n");
  };
}

# ARGH! More code duplication from pkg_mgr.pl
# We really need to have some kind of include system
sub get_cc {
  $gHelper{'gcc'} = '';
  if (defined($ENV{'CC'}) && (not ($ENV{'CC'} eq ''))) {
    $gHelper{'gcc'} = internal_which($ENV{'CC'});
    if ($gHelper{'gcc'} eq '') {
      print wrap('Unable to find the compiler specified in the CC environnment variable: "'
                 . $ENV{'CC'} . '".' . "\n\n", 0);
    }
  }
  if ($gHelper{'gcc'} eq '') {
    $gHelper{'gcc'} = internal_which('gcc');
    if ($gHelper{'gcc'} eq '') {
      $gHelper{'gcc'} = internal_which('egcs');
      if ($gHelper{'gcc'} eq '') {
        $gHelper{'gcc'} = internal_which('kgcc');
        if ($gHelper{'gcc'} eq '') {
          $gHelper{'gcc'} = DoesBinaryExist_Prompt('gcc');
        }
      }
    }
  }
  print wrap('Using compiler "' . $gHelper{'gcc'}
             . '". Use environment variable CC to override.' . "\n\n", 0);
  return $gHelper{'gcc'};
}

# These quaddot functions and compute_subnet are from config.pl and are needed
# for the tar4|rpm4 upgrade
# Converts an quad-dotted IPv4 address into a integer
sub quaddot_to_int {
  my $quaddot = shift;
  my @quaddot_a;
  my $int;
  my $i;

  @quaddot_a = split(/\./, $quaddot);
  $int = 0;
  for ($i = 0; $i < 4; $i++) {
    $int <<= 8;
    $int |= $quaddot_a[$i];
  }

  return $int;
}

# Converts an integer into a quad-dotted IPv4 address
sub int_to_quaddot {
  my $int = shift;
  my @quaddot_a;
  my $i;

  for ($i = 3; $i >= 0; $i--) {
    $quaddot_a[$i] = $int & 0xFF;
    $int >>= 8;
  }

  return join('.', @quaddot_a);
}

# Compute the subnet address associated to a couple IP/netmask
sub compute_subnet {
  my $ip = shift;
  my $netmask = shift;

  return int_to_quaddot(quaddot_to_int($ip) & quaddot_to_int($netmask));
}

# Make sure we have an initial database suitable for this installer. The goal
# is to encapsulates all the compatibilty issues in this (consequently ugly)
# function
sub get_initial_database {
  my $made_dir1;
  my $made_dir2;
  my $bkp_dir;
  my $bkp;
  my $kind;
  my $version;
  my $intermediate_format;
  my $status;
  my $state_file;
  my $state_files;

  if (not (-e $gInstallerMainDB)) {
    # This is the first installation. Create the installer database from
    # scratch
    print wrap('Creating a new ' . vmware_product_name()
               . ' installer database using the tar4 format.' . "\n\n", 0);

    $made_dir1 = create_dir($gRegistryDir, 0);
    safe_chmod(0755, $gRegistryDir);

    if (not open(INSTALLDB, '>' . $gInstallerMainDB)) {
      if ($made_dir1) {
        rmdir($gRegistryDir);
      }
      error('Unable to open the tar installer database ' . $gInstallerMainDB
            . ' in write-mode.' . "\n\n");
    }
    # Force a flush after every write operation.
    # See 'Programming Perl', p. 110
    select((select(INSTALLDB), $| = 1)[0]);

    if ($made_dir1) {
      db_add_dir($gRegistryDir);
    }
    # This file is going to be modified after its creation by this program.
    # Do not timestamp it
    db_add_file($gInstallerMainDB, 0);

    return;
  }

  print wrap('A previous installation of ' . vmware_product_name()
             . ' has been detected.' . "\n\n", 0);

  #
  # Convert the previous installer database to our format and backup it
  # Uninstall the previous installation
  #

  $bkp_dir = make_tmp_dir('vmware-installer');
  $bkp = $bkp_dir . '/prev_db.tar.gz';

  if (-x $gInstallerObject) {
    $kind = direct_command(shell_string($gInstallerObject) . ' kind');
    chop($kind);
    if (system(shell_string($gInstallerObject) . ' version >/dev/null 2>&1')) {
      # No version method -> this is version 1
      $version = '1';
    } else {
      $version = direct_command(shell_string($gInstallerObject) . ' version');
      chop($version);
    }
    print wrap('The previous installation was made by the ' . $kind
               . ' installer (version ' . $version . ').' . "\n\n", 0);

    if ($version < 2) {
      # The best database format those installers know is tar. We will have to
      # upgrade the format
      $intermediate_format = 'tar';
    } elsif ($version == 2) {
      # Those installers at least know about the tar2 database format. We won't
      # have to do too much
      $intermediate_format='tar2'
    } elsif ($version == 3) {
      # Those installers at least know about the tar3 database format. We won't
      # have to do much
      $intermediate_format = 'tar3';
    } else {
      # Those installers at least know about the tar4 database format. We won't
      # have to do anything
      $intermediate_format = 'tar4';
    }
    system(shell_string($gInstallerObject) . ' convertdb '
           . shell_string($intermediate_format) . ' ' . shell_string($bkp));

    # Uninstall the previous installation
    if (vmware_product() ne 'vix') {
      db_load();
      db_append();
      db_add_answer('UPGRADE', 'yes');
      db_save();
    }
    $status = system(shell_string($gInstallerObject) . ' uninstall');
    # Beware, beyond this point, $gInstallerObject does not exist
    # anymore.
  } else {
    # No installer object -> this is the old installer, which we don't support
    # anymore.
    $status = 1;
  }
  if ($status) {
    remove_tmp_dir($bkp_dir);
    # remove the installer db so the next invocation of install can proceed.
    if (get_answer('Uninstallation of previous install failed. ' .
		   'Would you like to remove the install DB?', 'yesno', 'no') eq 'yes') {
      print wrap('Removing installer DB, please re-run the installer.' . "\n\n", 0);
      unlink $gInstallerMainDB;
    }

    error('Failure' . "\n\n");
  }

  # Create the directory structure to welcome the restored database
  $made_dir1 = 0;
  if (not (-d $gRegistryDir)) {
    safe_mkdir($gRegistryDir);
    $made_dir1 = 1;
  }
  safe_chmod(0755, $gRegistryDir);
  $made_dir2 = 0;
  if ($version >= 2) {
    if (not (-d $gStateDir)) {
      safe_mkdir($gStateDir);
      $made_dir2 = 1;
    }
    safe_chmod(0755, $gStateDir);
  }

  # Some versions of tar (1.13.17+ are ok) do not untar directory permissions
  # as described in their documentation (they overwrite permissions of
  # existing, non-empty directories with permissions stored in the archive)
  #
  # Because we didn't know about that at the beginning, the previous
  # uninstallation may have included the directory structure in their database
  # backup.
  #
  # To avoid that, we must re-package the database backup
  if (vmware_product() eq 'tools-for-solaris') {
    # Solaris' default tar(1) does not support gnu tar's -C or -z options.
    system('cd ' . shell_string($bkp_dir) . ';'
           . shell_string($gHelper{'gunzip'}) . ' -c ' . shell_string($bkp)
           . ' | ' . shell_string($gHelper{'tar'}) . ' -xopf -');
  } else {
    system(shell_string($gHelper{'tar'}) . ' -C ' . shell_string($bkp_dir)
           . ' -xzopf ' . shell_string($bkp));
  }
  $state_files = '';
  if (-d $bkp_dir . $gStateDir) {
    foreach $state_file (internal_ls($bkp_dir . $gStateDir)) {
      $state_files .= ' ' . shell_string('.' . $gStateDir . '/'. $state_file);
    }
  }
  $bkp = $bkp_dir . '/prev_db2.tar.gz';
  if (vmware_product() eq 'tools-for-solaris') {
    system('cd ' . shell_string($bkp_dir) . ';'
           . shell_string($gHelper{'tar'}) . ' -copf - '
           . shell_string('.' . $gInstallerMainDB) . $state_files
           . ' | ' . shell_string($gHelper{'gzip'}) . ' > ' . shell_string($bkp));
  } else {
    system(shell_string($gHelper{'tar'}) . ' -C ' . shell_string($bkp_dir)
           . ' -czopf ' . shell_string($bkp) . ' '
           . shell_string('.' . $gInstallerMainDB) . $state_files);
  }

  # Restore the database ready to be used by our installer
  if (vmware_product() eq 'tools-for-solaris') {
    system('cd /;'
           . shell_string($gHelper{'gunzip'}) . ' -c ' . shell_string($bkp)
           . ' | ' . shell_string($gHelper{'tar'}) . ' -xopf -');
  } else {
    system(shell_string($gHelper{'tar'}) . ' -C / -xzopf ' . shell_string($bkp));
  }
  remove_tmp_dir($bkp_dir);

  if ($version < 2) {
    print wrap('Converting the ' . $intermediate_format
               . ' installer database format to the tar4 installer database format.'
               . "\n\n", 0);
    # Upgrade the database format: keep only the 'answer' statements, and add a
    # 'file' statement for the main database file
    my $id;

    db_load();
    if (not open(INSTALLDB, '>' . $gInstallerMainDB)) {
      error('Unable to open the tar installer database ' . $gInstallerMainDB
            . ' in write-mode.' . "\n\n");
    }
    db_add_file($gInstallerMainDB, 0);
    foreach $id (keys %gDBAnswer) {
      print INSTALLDB 'answer ' . $id . ' ' . $gDBAnswer{$id} . "\n";
    }
    db_save();
  } elsif( $version == 2 ) {
    print wrap('Converting the ' . $intermediate_format
               . ' installer database format to the tar4 installer database format.'
               . "\n\n", 0);
    # Upgrade the database format: keep only the 'answer' statements, and add a
    # 'file' statement for the main database file
    my $id;

    db_load();
    if (not open(INSTALLDB, '>' . $gInstallerMainDB)) {
      error('Unable to open the tar installer database ' . $gInstallerMainDB
            . ' in write-mode.' . "\n\n");
    }
    db_add_file($gInstallerMainDB, 0);
    foreach $id (keys %gDBAnswer) {
      # For the rpm3|tar3 format, a number of keywords were removed.  In their
      # place a more flexible scheme was implemented for which each has a semantic
      # equivalent:
      #
      #   VNET_HOSTONLY          -> VNET_1_HOSTONLY
      #   VNET_HOSTONLY_HOSTADDR -> VNET_1_HOSTONLY_HOSTADDR
      #   VNET_HOSTONLY_NETMASK  -> VNET_1_HOSTONLY_NETMASK
      #   VNET_INTERFACE         -> VNET_0_INTERFACE
      #
      # Note that we no longer use the samba variables, so these entries are
      # removed (and not converted):
      #   VNET_SAMBA             -> VNET_1_SAMBA
      #   VNET_SAMBA_MACHINESID  -> VNET_1_SAMBA_MACHINESID
      #   VNET_SAMBA_SMBPASSWD   -> VNET_1_SAMBA_SMBPASSWD
      my $newid = $id;
      if ("$id" eq 'VNET_SAMBA') {
         next;
      } elsif ("$id" eq 'VNET_SAMBA_MACHINESID') {
         next;
      } elsif ("$id" eq 'VNET_SAMBA_SMBPASSWD') {
         next;
      } elsif ("$id" eq 'VNET_HOSTONLY') {
        $newid='VNET_1_HOSTONLY';
      } elsif ("$id" eq 'VNET_HOSTONLY_HOSTADDR') {
        $newid='VNET_1_HOSTONLY_HOSTADDR';
      } elsif ("$id" eq 'VNET_HOSTONLY_NETMASK') {
        $newid='VNET_1_HOSTONLY_NETMASK';
      } elsif ("$id" eq 'VNET_INTERFACE') {
        $newid='VNET_0_INTERFACE';
      }

      print INSTALLDB 'answer ' . $newid . ' ' . $gDBAnswer{$id} . "\n";
    }

    # For the rpm4|tar4 format, two keyword were added. We add them here if
    # necessary.  Note that it is only necessary to check the existence of two
    # VNET_HOSTONLY_ keywords since the rpm2|tar2 format contained only a few
    # VNET_ keywords
    my $addr = db_get_answer_if_exists('VNET_HOSTONLY_HOSTADDR');
    my $mask = db_get_answer_if_exists('VNET_HOSTONLY_NETMASK');
    if (defined($addr) and defined($mask)) {
       print INSTALLDB 'answer VNET_1_HOSTONLY_SUBNET ' .
                        compute_subnet($addr, $mask) . "\n";
       print INSTALLDB "answer VNET_1_DHCP yes\n";
    }

    db_save();
  } elsif ( $version == 3 ) {
    print wrap('Converting the ' . $intermediate_format
               . ' installer database format to the tar4 installer database format.'
               . "\n\n", 0);
    # Upgrade the database format: keep only the 'answer' statements, and add a
    # 'file' statement for the main database file
    my $id;

    db_load();
    if (not open(INSTALLDB, '>' . $gInstallerMainDB)) {
      error('Unable to open the tar installer database ' . $gInstallerMainDB
            . ' in write-mode.' . "\n\n");
    }
    db_add_file($gInstallerMainDB, 0);

    # No conversions necessary between version 3 and 4, so add all answers
    foreach $id (keys %gDBAnswer) {
      print INSTALLDB 'answer ' . $id . ' ' . $gDBAnswer{$id} . "\n";
    }

    # Check whether we need to add the two new keywords for each virtual network:
    #   VNET_n_HOSTONLY_SUBNET -> set if VNET_n_HOSTONLY_{HOSTADDR,NETMASK} are set
    #   VNET_n_DHCP            -> 'yes' iff VNET_n_INTERFACE is not defined and
    #                              VNET_n_HOSTONLY_{HOSTADDR,NETMASK} are defined
    #
    my $i;
    for ($i = $gMinVmnet; $i < $gMaxVmnet; $i++) {
      my $pre = 'VNET_' . $i . '_';
      my $interface = db_get_answer_if_exists($pre . 'INTERFACE');
      my $hostaddr  = db_get_answer_if_exists($pre . 'HOSTONLY_HOSTADDR');
      my $netmask   = db_get_answer_if_exists($pre . 'HOSTONLY_NETMASK');

      if (defined($hostaddr) && defined($netmask)) {
         my $subnet = compute_subnet($hostaddr, $netmask);
         print INSTALLDB 'answer ' . $pre . 'HOSTONLY_SUBNET ' . $subnet . "\n";

         if (not defined($interface)) {
            print INSTALLDB 'answer ' . $pre . "DHCP yes\n";
         }
      }
    }

    db_save();
  }

  db_load();
  db_append();
  if ($made_dir1) {
    db_add_dir($gRegistryDir);
  }
  if ($made_dir2) {
    db_add_dir($gStateDir);
  }
}

# SIGINT handler. We will never reset the handler to the DEFAULT one, because
# with the exception of pre-uninstaller not being installed, this one does
# the same thing as the default (kills the process) and even sends the end
# RPC for us in tools installations.
sub sigint_handler {
  if ($gIsUninstallerInstalled == 0) {
    print STDERR wrap("\n\n" . 'Ignoring attempt to kill the installer with Control-C, because the uninstaller has not been installed yet. Please use the Control-Z / fg combination instead.' . "\n\n", 0);

    return;
  }

  # If we are installing the VMware Tools and the user cancelled the
  # installation, remember to tell the VMX to unmount the Tools image.
  if (vmware_product() eq 'tools-for-linux' ||
      vmware_product() eq 'tools-for-freebsd' ||
      vmware_product() eq 'tools-for-solaris') {
    send_rpc('toolinstall.end 0');
  }
  error('');
}

#  Write the VMware host-wide configuration file - only if console
sub write_vmware_config {
  my $name;

  $name = $gRegistryDir . '/config';

  uninstall_file($name);
  if (file_check_exist($name)) {
    return;
  }
  # The file could be a symlink to another location. Remove it
  unlink($name);

  open(CONFIGFILE, '>' . $name) or error('Unable to open the configuration file '
                                         . $name . ' in write-mode.' . "\n\n");
  db_add_file($name, 0x1);
  safe_chmod(0444, $name);
  print CONFIGFILE 'libdir = "' . db_get_answer('LIBDIR') . '"' . "\n";
  close(CONFIGFILE);
}

# Get the installed version of VMware
# Return the version if found, or ''
sub get_installed_version() {
  my $backslash;
  my $dollar;
  my $pattern;
  my $version;
  my $nameTag;

  # XXX In the future, we should use a method of the installer object to
  #     retrieve the installed version --hpreg

  #
  # Try to retrieve the installed version from the configurator program. This
  # works for both the tar and the rpm installers --hpreg
  #

  if (not defined($gDBAnswer{'BINDIR'})) {
    return '';
  }

  if (not open(FILE, '<' . db_get_answer('BINDIR') . $gConfigurator)) {
    return '';
  }

  # Build the pattern without using the dollar character, so that CVS doesn't
  # modify the pattern in tagged builds (bug 9303) --hpreg
  $backslash = chr(92);
  $dollar = chr(36);
  $pattern = '^  ' . $backslash . $dollar . 'buildNr = ' .
      "'" . '(\S+) ' . "'" . ' ' . $backslash . '. q' .
      $backslash . $dollar . 'Name: (\S+)? ' . $backslash . $dollar . ';' . $dollar;

  $version = '';
  $nameTag = '';
  while (<FILE>) {
    if (/$pattern/) {
      $version = $1;
      $nameTag = defined($2) ? $2 : '';
    }
  }
  close(FILE);

  return $version;
}

# Get the installed kind of VMware
# Return the kind if found, or ''
sub get_installed_kind() {
  my $kind;

  if (not (-x $cInstallerObject)) {
    return '';
  }

  $kind = direct_command(shell_string($cInstallerObject) . ' kind');
  chop($kind);

  return $kind;
}

# Install the content of the module package
sub install_module {
  my %patch;

  print wrap('Installing the kernel modules contained in this package.' . "\n\n", 0);

  undef %patch;
  install_dir('./lib', db_get_answer('LIBDIR'), \%patch, 0x1);
}

# Uninstall modules
sub uninstall_module {
  print wrap('Uninstalling currently installed kernel modules.' . "\n\n", 0);

  uninstall_prefix(db_get_answer('LIBDIR') . '/modules');
}

# XXX Duplicated in config.pl
# format of the returned hash:
#          - key is the system file
#          - value is the backed up file.
# This function should never know about filenames. Only database
# operations.
sub db_get_files_to_restore {
  my %fileToRestore;
  undef %fileToRestore;
  my $restorePrefix = 'RESTORE_';
  my $restoreBackupSuffix = '_BAK';
  my $restoreBackList = 'RESTORE_BACK_LIST';

  if (defined db_get_answer_if_exists($restoreBackList)) {
    my $restoreStr;
    foreach $restoreStr (split(/:/, db_get_answer($restoreBackList))) {
      if (defined db_get_answer_if_exists($restorePrefix . $restoreStr)) {
        $fileToRestore{db_get_answer($restorePrefix . $restoreStr)} =
          db_get_answer($restorePrefix . $restoreStr
                        . $restoreBackupSuffix);
      }
    }
  }
  return %fileToRestore;
}

# Returns an array with the list of files that changed since we installed
# them.
sub db_is_file_changed {

  my $file = shift;
  my @statbuf;

  @statbuf = stat($file);
  if (defined $gDBFile{$file} && $gDBFile{$file} ne '0' &&
      $gDBFile{$file} ne $statbuf[9]) {
    return 'yes';
  } else {
    return 'no';
  }
}

sub filter_out_bkp_changed_files {

  my $filesToRestoreRef = shift;
  my $origFile;

  foreach $origFile (keys %$filesToRestoreRef) {
    if (db_file_in($origFile) && !-l $origFile &&
        db_is_file_changed($origFile) eq 'yes') {
      # We are in the case of bug 25444 where we are restoring a file
      # that we backed up and was changed in the mean time by someone else
      db_remove_file($origFile);
      backup_file($$filesToRestoreRef{$origFile});
      unlink $$filesToRestoreRef{$origFile};
      print wrap("\n" . 'File ' . $$filesToRestoreRef{$origFile}
                 . ' was not restored from backup because our file '
                 . $origFile
                 . ' got changed or overwritten between the time '
                 . vmware_product_name()
                 . ' installed the file and now.' . "\n\n"
                 ,0);
      delete $$filesToRestoreRef{$origFile};
    }
  }
}

sub restore_backedup_files {
  my $fileToRestore = shift;
  my $origFile;

  foreach $origFile (keys %$fileToRestore) {
    if (file_name_exist($origFile) &&
        file_name_exist($$fileToRestore{$origFile})) {
      backup_file($origFile);
      unlink $origFile;
    }
    if ((not file_name_exist($origFile)) &&
        file_name_exist($$fileToRestore{$origFile})) {
      rename $$fileToRestore{$origFile}, $origFile;
    }
  }
}

### Finalize this ACE package so it can be used.
sub acevm_finalize {
  my $vmxPath = acevm_find_vmx($gRegistryDir);
  my $ret;
  if ($vmxPath eq '') {
    error('Failed. Can\'t find VMX file for this package' . "\n\n");
  }
  $ret = system(shell_string($gHelper{'vmware-acetool'}) . ' finalizePolicies '
		. shell_string($vmxPath));
  return (($ret >> 8) == 0);
}

### Check to see if the passed in instance has the same ace id as this update package
sub acevm_checkMasterID {
  my $vmxPath = shift;
  my $exitValue;

  if ($vmxPath eq '') {
    error('Failed. Can\'t find vmx file for this package' . "\n\n");
  }
  system(shell_string($gHelper{'vmware-acetool'}) . ' checkMasterID '
	 . shell_string($vmxPath) . ' ' . shell_string($gManifest{'aceid'})
	 . '> /dev/null 2>&1');
  $exitValue = $? >> 8;
  return $exitValue == 0;
}

### Check to see if "now" falls outside this package's valid date range.
sub acevm_check_valid_date {
  my $now = time();
  my $ret = 1;
  if (!defined($gManifest{'validFrom'})) {
    $gManifest{'validFrom'} = 0;
  }
  if (!defined($gManifest{'validTo'})) {
    $gManifest{'validTo'} = 0;
  }

  # Default to returning true. Return false only if either of the
  # following is true:
  #   - 'from' is valid and we're not there yet.
  #   - 'to' is valid and we've already passed it.
  if ($gManifest{'validFrom'} > 0 && $gManifest{'validFrom'} > $now) {
    $ret = 0;
  } elsif ($gManifest{'validTo'} > 0 && $gManifest{'validTo'} < $now) {
    $ret = 0;
  }

  return $ret;
}

### Find the vmx file for this package.
sub acevm_find_vmx {
  my $dir = shift;
  my $ret = '';
  my $file;

  opendir(DIR, $dir) or return '';

  while( defined ($file = readdir(DIR)) ) {
    if ($file =~ /^.+\.vmx$/) {
      $ret = $file;
      last;
    }
  }
  close DIR;
  if ($ret ne '') {
    $ret = $dir . '/' . $ret;
  }
  return $ret;
}

### Check if this ace instance is running.
### We do this by pattern matching 'vmware-vmx' and the vmx filename in a single
### line of the 'ps -ef' output.
sub acevm_is_instance_running {
  my $line;
  my $found = 0;
  my $vmxFile = acevm_find_vmx($gRegistryDir);

  if ($vmxFile eq '') {
    return 0;
  }
  open(PSOUT, 'ps -ef|');
  while (<PSOUT>) {
    if (m/vmware-vmx.+$vmxFile/) {
      $found = 1;
      last;
    }
  }
  close(PSOUT);
  return $found;
}

# Create launcher (shortcuts) in the user's home directory.
sub acevm_create_desktop_icon {
   my $acename = shift;
   my $aceid = shift;
   my $file = shift;
   my %patch;
   my $homedir = get_home_dir();

   if (! $homedir) {
      print STDERR 'Unable to install shortcuts because the ' .
                   "home directory can't be found.\n\n";
      return;
   }

   # create base menu file structure
   create_dir($homedir . '/.local/share/applications', 0);

   # count number of packages from this ACE master
   my $count = 0;
   foreach $file (glob("$homedir/.local/share/applications/*.desktop")) {
      if ($file =~ /$gManifest{'aceid'}(.)*\.desktop$/) {
         $count++;
      }
   }
   $count = $count gt 0 ? " $count" : '';

   # create desktop entry
   my $DESKTOP;
   my $desktop_file_name = $aceid . $count . '.desktop';
   my $tmpdir = make_tmp_dir('vmware-installer');
   my $tmpfile = $tmpdir . '/' . $desktop_file_name;
   if (!open(DESKTOP, ">$tmpfile")) {
      print STDERR wrap("Unable to open file $tmpfile for writing.  You must either " .
                        "create the launcher by hand or launch the ACE VM directly.");
      remove_tmp_dir($tmpdir);
      return;
   }

   my $vmx = acevm_find_vmx($gRegistryDir);
   print DESKTOP <<EOF;
[Desktop Entry]
Encoding=UTF-8
Type=Application
Icon=vmware-acevm.png
Exec=vmplayer "$vmx"
Name=$acename$count
Categories=VirtualMachines;ACE;
MimeType=application/x-vmware-vm
EOF

   close(DESKTOP);
   my $desktop_dest = $homedir . "/.local/share/applications/$desktop_file_name";
   undef %patch;
   install_file($tmpfile, $desktop_dest, \%patch, 1);
   remove_tmp_dir($tmpdir);
}

### Does the dstDir have enough space to hold srcDir
sub check_disk_space {
  my $srcDir = shift;
  my $dstDir = shift;
  my $srcSpace;
  my $dstSpace;
  my @parser;

  # get the src usage
  open (OUTPUT, shell_string($gHelper{'du'}) . ' -sk ' . shell_string($srcDir)
	. ' 2>/dev/null|') or error("Failed to open 'du'.");
  $_ = <OUTPUT>;
  @parser = split(/\s+/);
  $srcSpace = $parser[0];
  close OUTPUT;

  # Before we can check the space, $dst must exist. Walk up the directory path
  # until we find something that exists.
  while (! -d $dstDir) {
    $dstDir = internal_dirname($dstDir);
  }
  open (OUTPUT, shell_string($gHelper{'df'}) . ' -k ' .  shell_string($dstDir)
	. ' 2>/dev/null|');
  while (<OUTPUT>) {
    @parser = split(/\s+/);
    if ($parser[0] ne 'Filesystem') {
      $dstSpace = $parser[3];
    }
  }
  close OUTPUT;

  # Return the amount of space available in kbytes.
  return ($dstSpace - $srcSpace);
}


### Does the user have permission to write to this directory?
sub check_dir_writeable {
  my $dstDir = shift;

  # Before we can check the permission, $dst must exist. Walk up the directory path
  # until we find something that exists.
  while (! -d $dstDir) {
    $dstDir = internal_dirname($dstDir);
  }

  # Return whether this directory is writeable.
  return (-w $dstDir);
}

#
#  Check to see that the product architecture is a mismatch for this os.
#  Return an error string if there is a mismatch, otherwise return undef
#
sub product_os_match {

  init_product_arch_matrix();
  if (!defined($multi_arch_products{vmware_product()})) {
    return undef;
  }

  if (is64BitUserLand() == (vmware_product_architecture() eq "x86_64")) {
    return undef;
  }
  if (is64BitUserLand() != (vmware_product_architecture() ne "x86_64")) {
    return undef;
  }

  return sprintf('This version of "%s" is incompatible with this '
		. 'operating system.  Please install the "%s" '
		. 'version of this program instead.'
		. "\n\n", vmware_product_name(),
		  is64BitUserLand() ? 'x86_64' : 'i386');
}

#
#  Create a list of products that support both a 32bit and a 64bit
#  architecture and thus should be matched to the running OS.
#
sub init_product_arch_matrix {
  $multi_arch_products{'ws'} = 1;
  $multi_arch_products{'vix'} = 1;
  $multi_arch_products{'api'} = 1;
}

# Look for the location of an answer in a different database and return the
# the value or the empty string if no answer or file is found.
sub alt_db_get_answer  {
  my $db_file = shift;
  my $key = shift;
  my $answer = '';

  if (open(PLAYERINSTALLDB, '<' . $db_file)) {
    while (<PLAYERINSTALLDB>) {
      chomp;
      if (/^answer\s+$key\s+(.+)$/) {
 	$answer = $1;
      } elsif (/^remove_answer\s+$key\s*$/) {
	  $answer = '';
      }
    }
    close(PLAYERINSTALLDB);
  }
  return $answer
}

# Program entry point
sub main {
  my (@setOption, $opt);
  my $chk_msg;
  my $progpath = $0;
  my $scriptname = internal_basename($progpath);

  $chk_msg = product_os_match();
  if (defined($chk_msg)) {
  	error($chk_msg);
  }

  if (vmware_product() ne 'acevm' && !is_root()) {
    error('Please re-run this program as the super user.' . "\n\n");
  }

  # Force the path to reduce the risk of using "modified" external helpers
  # If the user has a special system setup, he will will prompted for the
  # proper location anyway
  $ENV{'PATH'} = '/bin:/usr/bin:/sbin:/usr/sbin';

  initialize_globals(internal_dirname($0));
  initialize_external_helpers();

  # List of questions answered with command-line arguments
  @setOption = ();

  if (internal_basename($0) eq $cInstallerFileName) {
    my $answer;

    if ($#ARGV > -1) {
      if ($#ARGV > 1) {
        install_usage();
      }

      # There are only two possible arguments
      while ($#ARGV != -1) {
        my $arg;
        $arg = shift(@ARGV);

        if (lc($arg) =~ /^(-)?(-)?d(efault)?$/) {
          $gOption{'default'} = 1;
        } elsif ($arg =~ /=yes/ || $arg =~ /=no/) {
          push(@setOption, $arg);
        } elsif (lc($arg) =~ /^(-)?(-)?(no-create-shortcuts)$/) {
          $gOption{'create_shortcuts'} = 0;
        } else {
          install_usage();
        }
      }
    }

    # Other installers will be able to remove this installation cleanly only if
    # they find the uninstaller. That's why we:
    # . Install the uninstaller ASAP
    # . Prevent dumb users from playing with Control-C while doing so

    $gIsUninstallerInstalled = 0;

    # Install the SIGINT handler. Don't bother resetting it, see
    # sigint_handler for details.
    $SIG{INT} = \&sigint_handler;

    # Don't allow installation on Ubuntu when the user has installed
    # vmware-tools-kernel-modules
    if (vmware_product() eq 'tools-for-linux' && check_tools_on_ubuntu()) {
         error(wrap("The Ubuntu vmware-tools-kernel-modules package can not be " .
            "installed simultaneously with VMware Tools.  Please uninstall the " .
            "package then run the installer again.\n\n", 0));
    }

    # Don't allow installation of tools-for-solaris unless this is Solaris 9
    # or later and 32-bit (for now).  Note that we only officially support
    # Solaris 10 Update 1 and higher, but we'll allow users to install on 9.
    if (vmware_product() eq 'tools-for-solaris') {
      my $solVersion = direct_command(shell_string($gHelper{'uname'}) . ' -r');
      chomp($solVersion);
      my ($major, $minor) = split /\./, $solVersion;

      if ($major != 5 || $minor < 9) {
        error('VMware Tools for Solaris is only supported on Solaris 10 and later.'
              . "\n\n");
      }

      # Warn users that we don't support Solaris 9, but they can install
      if ($minor == 9) {
        if (get_answer('WARNING: VMware Tools for Solaris is officially supported '
                       . 'on Solaris 10 and later, but you are running Solaris 9.  '
                       . 'Would you like to continue with the installation?',
                       'yesno', 'yes') eq 'no') {
           error('You have selected to not install VMware Tools for Solaris on '
                 . 'Solaris 9.' . "\n\n");
        }
      }
    }

    # The uninstall of legacy tools must come before get_initial_database()
    if (vmware_product() eq 'tools-for-linux' ||
        vmware_product() eq 'tools-for-freebsd') {
      uninstall_content_legacy_tools();
    }

    if ( $progpath =~ /(.*?)\/$scriptname/ ) {
      chdir($1);
    }

    # The acevm install puts all files for a given install, including the installdb,
    # under a single directory.
    if (vmware_product() eq 'acevm') {
      my $answer = undef;
      my $home = get_home_dir() ||
                 error("Can't find home directory, unable to continue.\n\n");

      #
      # Check valid range for this package
      #
      if (!acevm_check_valid_date()) {
         error('This setup program cannot continue. This ACE can only be ' .
               'installed during its valid time period.' . "\n\n");
      }

      $gHelper{'vmware-acetool'} = acevm_find_acetool();
      if ($gHelper{'vmware-acetool'} eq '' || acevm_included_player_newer()) {
         print STDERR wrap('This setup program requires that the included VMware ' .
                           'Player must be installed with this ACE.' . "\n\n", 0);
         if (acevm_install_vmplayer()) {
           $gHelper{'vmware-acetool'} = acevm_find_acetool();
         }
         else {
            error('VMware Player installation failed.  Please install manually then '.
                  'run setup again.');
         }
      }

      $gHelper{'vmware-config.pl'} = internal_dirname($gHelper{'vmware-acetool'}) .
                                                     '/vmware-config.pl';
      if (acevm_package_has_host_policies()) {
         if (!is_root()) {
            error ('This package specifies host policies and must be installed by a ' .
                   'system administrator. Try running this setup program with sudo or ' .
                   'contact your system administrator for help.' . "\n\n");
      } elsif (!acevm_can_install_host_policies()) {
            error('This package contains a host policies file, but there is already ' .
                  'a host policies file from another package installed. ' .
                  'Unable to continue' . "\n\n");
         }
      }
      my $rootdir = $home . '/vmware-ace/' . $gManifest{'acename'};
      if ($gACEVMUpdate) {
        $answer = acevm_get_updatedir($rootdir);
      } else {
        $answer = acevm_get_dir($rootdir);
      }

      if ($answer eq '') {
        error ('Cannot get install diretory. Unable to contine.' . "\n\n");
      }
      $gRegistryDir= $answer;
      $gStateDir = $gRegistryDir . '/state';
      $gInstallerMainDB = $gRegistryDir . '/locations';
      $gInstallerObject = $gRegistryDir . '/installer.sh';
      $gConfFlag = $gRegistryDir . '/not_configured';
    }
    if (vmware_product() eq 'acevm' && file_name_exist($gInstallerMainDB)) {
      db_load();
      db_append();
    } else {
      my $dstDir = $gRegistryDir;
      $gFirstCreatedDir = $dstDir;
      while (!-d $dstDir) {
        $gFirstCreatedDir = $dstDir;
        $dstDir = internal_dirname($dstDir);
      }
      get_initial_database();
      # Binary wrappers can be run by any user and need to read the
      # database. --hpreg
      safe_chmod(0644, $gInstallerMainDB);
    }

    db_add_answer('INSTALL_CYCLE', 'yes');
    if (@setOption > 0) {
      $gOption{'default'} = 1;
    }

    foreach $opt (@setOption) {
      my ($key, $val);
      ($key, $val) = ($opt =~ /^([^=]*)=([^=]*)/);
      db_add_answer($key, $val);
    }

    if (vmware_product() eq 'api' || vmware_product() eq 'mui') {
      show_EULA();
    }
    install_or_upgrade();

    # Reset these answers in case we have installed new versions of these
    # documents.
    if (vmware_product() ne 'ws') {
      db_remove_answer('EULA_AGREED');
      db_remove_answer('ISC_COPYRIGHT_SEEN');
    }
    if (vmware_product() eq 'vix') {
      finalize_vix_install();
    }

    # We need to write the config file for the remote console
    if (vmware_product() eq 'console') {
      write_vmware_config();
    }

    if (vmware_product() ne 'api' && vmware_product() ne 'acevm') {
      if (file_name_exist($gConfFlag)) {
        $answer = get_persistent_answer('Before running '
                                        . vmware_product_name()
                                        . ' for the first time, you need to '
                                        . 'configure it by invoking the '
                                        . 'following command: "'
                                        . db_get_answer('BINDIR')
                                        . '/' . "$gConfigurator" . '". Do you '
                                        . 'want this program to invoke the '
                                        . 'command for you now?'
                                        , 'RUN_CONFIGURATOR', 'yesno', 'yes');
      } else {
        if (vmware_product() ne 'vix') {
          print wrap('Before running ' . vmware_product_name() . ' for the '
                     . 'first time, you need to configure it by invoking the'
                     . ' following command: "' . db_get_answer('BINDIR')
                     . '/' . "$gConfigurator" . '"' . "\n\n", 0);
        }
        $answer = 'no';
      }
    }

    db_save();

    my $configResult;

    if ((vmware_product() ne 'api') && (vmware_product() ne 'acevm') &&
	($answer eq 'yes')) {
      my $defaultOpt = ($gOption{'default'} == 1) ? ' --default' : '';

      # If we're the tools installer, forego sending the end RPC and let
      # the configurator do it.
      my $rpcOpt = (vmware_product() eq 'tools-for-linux' ||
		    vmware_product() eq 'tools-for-freebsd' ||
		    vmware_product() eq 'tools-for-solaris') ?
		    ' --rpc-on-end' : '';

      my $shortcutOpt = $gOption{'create_shortcuts'} ? '' : ' --no-create-shortcuts';

      # Catch error result to see if configurator died abnormally.
      $configResult = system(shell_string(db_get_answer('BINDIR') .
					  '/' . $gConfigurator) .
			     $defaultOpt . $rpcOpt . $shortcutOpt);
    } else {
      print wrap('Enjoy,' . "\n\n" . '    --the VMware team' . "\n\n", 0);
    }

    # If we're the tools installer, now is a good time to send an end RPC.
    if (vmware_product() eq 'tools-for-linux' ||
	vmware_product() eq 'tools-for-freebsd' ||
	vmware_product() eq 'tools-for-solaris') {

      #
      # If the configurator never ran, send success RPC. If it was interrupted
      # before signal handler could be installed or it failed(exit status 1),
      # send a failure RPC on its behalf.  On success(exit status 0), assume
      # that the configurator sent the RPC itself.
      #
      if (not defined($configResult)) {
	send_rpc('toolinstall.end 1');
      } elsif (WIFSIGNALED($configResult) &&
	       WTERMSIG($configResult) == SIGINT) {
	send_rpc('toolinstall.end 0');
      } elsif (WIFEXITED($configResult) && WEXITSTATUS($configResult) == 1) {
         send_rpc('toolinstall.end 0');
      }
    }
    exit 0;
  }

  #
  # Module updater.
  #
  # XXX This is not clean. We really need separate packages, managed
  #     by the VMware package manager --hpreg
  #

  if (internal_basename($0) eq $cModuleUpdaterFileName) {
    my $installed_version;
    my $installed_kind;
    my $answer;

    print wrap('Looking for a currently installed '
               . vmware_longname() . ' tar package.' . "\n\n", 0);

    if (not (-e $cInstallerMainDB)) {
      error('Unable to find the ' . vmware_product_name() .
      ' installer database file (' . $cInstallerMainDB . ').' .
      "\n\n" . 'You may want to re-install the ' .
      vmware_longname() . ' package, then re-run this program.' . "\n\n");
    }
    db_load();

    $installed_version = get_installed_version();
    $installed_kind = get_installed_kind();

    if (not (($installed_version eq '6.0.0') and
             ($installed_kind eq 'tar'))) {
      error('This ' . vmware_product_name()
            . ' Kernel Modules package is intended to be used in conjunction '
            . 'with the ' . vmware_longname() . ' tar package only.' . "\n\n");
    }

    # All module files are under LIBDIR --hpreg
    if (not defined($gDBAnswer{'LIBDIR'})) {
       error('Unable to determine where the ' . vmware_longname()
       . ' package installed the library files.' . "\n\n"
       . 'You may want to re-install the ' . vmware_product_name() . ' '
       . vmware_version() . ' package, then re-run this program.' . "\n\n");
    }

    db_append();
    uninstall_module();
    install_module();

    print wrap('The installation of ' . vmware_product_name()
               . ' Kernel Modules '
               . vmware_version() . ' completed successfully.' . "\n\n", 0);

    if (-e $cConfFlag) {
       $answer = get_persistent_answer('Before running the VMware software for '
                                       . 'the first time after this update, you'
                                       . ' need to configure it for your '
                                       . 'running kernel by invoking the '
                                       . 'following command: "'
                                       . db_get_answer('BINDIR')
                                       . '/' . $gConfigurator . '". Do you want this '
                                       . 'program to invoke the command for you now?',
                                       'RUN_CONFIGURATOR', 'yesno', 'yes');
    } else {
      $answer = 'no';
    }

    db_save();

    if ($answer eq 'yes') {
       system(shell_string(db_get_answer('BINDIR') . '/' . $gConfigurator));
    } else {
       print wrap('Enjoy,' . "\n\n" . '    --the VMware team' . "\n\n", 0);
    }
    exit 0;
  }

  if (internal_basename($0) eq $gUninstallerFileName) {
    if (vmware_product() eq 'acevm') {
       print wrap("Uninstalling VMware ACE package $gManifest{'acename'}...\n\n", 0);
    } else {
       print wrap('Uninstalling the tar installation of ' .
       vmware_product_name() . '.' . "\n\n", 0);
    }

    if ($#ARGV > -1) {
      # There is currently only one option:  --upgrade.
      while ($#ARGV != -1) {
        my $arg;
        $arg = shift(@ARGV);
        if (lc($arg) =~ /^(-)?(-)?u(pgrade)?$/) {
           $gOption{'upgrade'} = 1;
        }
      }
    }

    if (not (-e $gInstallerMainDB)) {
      error('Unable to find the tar installer database file (' .
      $gInstallerMainDB . ')' . "\n\n");
    }
    db_load();

    db_append();

    if (vmware_product() eq 'wgs') {
      wgs_uninstall();
    }
    if (vmware_product() eq 'mui') {
      stop_mui(db_get_answer('INSTALLDIR'));
      clear_mui_logs();
      uninstall('/httpd.vmware');
    } elsif (vmware_product() eq 'acevm') {
      # First, check to see if this instance has been copied
      # by comparing the device and inode of the BINDIR against
      # the directory this uninstaller is in.
      my ($aDev, $aIno) = stat(internal_dirname($progpath));
      my ($bDev, $bIno) = stat(db_get_answer('BINDIR'));
      if ($aIno != $bIno || $aDev != $bDev) {
	error ('Cannot uninstall because this ACE has been copied ' . "\n\n");
      } elsif (acevm_is_instance_running()) {
	error ('Cannot uninstall because this ACE appears to be running. ' .
	       'Please close the running ACE and run this uninstaller again.' .
	       "\n\n");
      } else {
	uninstall('/acevm');
      }
    } elsif (vmware_product() eq 'tools-for-linux' ||
             vmware_product() eq 'tools-for-freebsd' ||
             vmware_product() eq 'tools-for-solaris') {
      my %fileToRestore;

      # Clean up the module loader config file from vmxnet.
      if (vmware_product() eq 'tools-for-freebsd' &&
          defined db_get_answer_if_exists('VMXNET_CONFED') &&
          db_get_answer('VMXNET_CONFED') eq 'yes') {
        my $loader_conf = '/boot/loader.conf';
        my $tmp_dir;
        my $tmp;
        $tmp_dir = make_tmp_dir('vmware-installer');
        $tmp = $tmp_dir . '/loader.conf';
        if (block_remove($loader_conf, $tmp, $cMarkerBegin, $cMarkerEnd) >= 0) {
          system(shell_string($gHelper{'mv'}) . ' -f ' . shell_string($tmp)
                 . ' '
                 . shell_string($loader_conf));
        }
        remove_tmp_dir($tmp_dir);
      }
      if (defined(db_get_answer_if_exists('THINPRINT_CONFED')) &&
	  (db_get_answer('THINPRINT_CONFED') eq 'yes')) {
	my $cupsPrinters = '/etc/cups/printers.conf';
	block_remove($cupsPrinters, $cupsPrinters . '.bak',
		     $cMarkerBegin, $cMarkerEnd);
	rename($cupsPrinters . '.bak', $cupsPrinters);
      }

      # Get the file names before they disappear from the database.
      %fileToRestore = db_get_files_to_restore();

      filter_out_bkp_changed_files(\%fileToRestore);

      # Do the bulk of the file uninstallation. Pass in the (slightly)
      # different name for FreeBSD Tools so that they get stopped correctly.
      if (vmware_product() eq 'tools-for-freebsd') {
	uninstall('/vmware-tools.sh');
      } else {
	uninstall('/vmware-tools');
      }

      # Clean up drivers with rem_drv(1M) (corresponds to add_drv(1M) calls in
      # configure_module_solaris() in configure script).  This needs to happen
      # after the services are stopped in uninstall().
      if (vmware_product() eq 'tools-for-solaris' &&
          !file_name_exist($gConfFlag)) {
         if (defined db_get_answer_if_exists('VMXNET_CONFED') &&
             db_get_answer('VMXNET_CONFED') eq 'yes') {

            system(shell_string($gHelper{'rem_drv'}) . ' vmxnet');

            # Give pcn its claim on pci1022,2000 back
            if (direct_command(shell_string($gHelper{'uname'}) . ' -r') =~ 5.9) {
              # Try to add back the pcn driver we removed
              system(shell_string($gHelper{'add_drv'})
                     . ' -i \'"pci103c,104c" "pci1022,2000"\' pcn >/dev/null 2>&1');
            } else {
              system(shell_string($gHelper{'update_drv'}) . ' -a -i \'"pci1022,2000"\' '
                                  . 'pcn >/dev/null 2>&1');
            }
         }
         if (defined db_get_answer_if_exists('VMMEMCTL_CONFED') &&
             db_get_answer('VMMEMCTL_CONFED') eq 'yes') {
            system(shell_string($gHelper{'rem_drv'}) . ' vmmemctl');
         }
         if (defined db_get_answer_if_exists('VMHGFS_CONFED') &&
             db_get_answer('VMHGFS_CONFED') eq 'yes') {
            my $devLinkTable = "/etc/devlink.tab";
            my $searchString = "name=vmhgfs";

            system(shell_string($gHelper{'rem_drv'}) . ' vmhgfs');

            if (system(shell_string($gHelper{'grep'}) . ' ' . $searchString
                       . ' ' . $devLinkTable . ' > /dev/null 2>&1') == 0) {
               # XXX There has to be a better way, but I don't know Perl
               my $tmpFile = "/tmp/VMware.devlink.tab";
               system(shell_string($gHelper{'cat'}) . ' ' . $devLinkTable . ' | '
                      . shell_string($gHelper{'grep'}) . ' -v ' . $searchString
                      . ' > ' . $tmpFile);
               system(shell_string($gHelper{'mv'}) . ' ' . $tmpFile . ' '
                      . $devLinkTable);
            }
         }
      }


      restore_backedup_files(\%fileToRestore);
    } elsif (vmware_product() eq 'vix') {
      # The upgrade flag indicates that this uninstall call is part of a
      # workstation install that is trying to remove its previous installation.
      # Let Vix handle its previous installation as part of its install process
      # and not as part of workstation's process.
      if ($gOption{'upgrade'} != 1) {
        uninstall_vix();
      }
    } elsif (vmware_product() eq 'ws') {
      my $vixArgs = '';

      # UPGRADE is set only inside get_initial_database() and only
      # when uninstalling a previous install when running vmware-install.pl.
      # Let vix uninstall its previous install when its vmware-install.pl
      # script is called and not as a result of workstation's install.
      if (defined(db_get_answer_if_exists('UPGRADE')) &&
                 (db_get_answer('UPGRADE') eq 'yes')) {
        $vixArgs = '--upgrade';
      }
      uninstall('/vmware');
      my $where = alt_db_get_answer('/etc/vmware-vix/locations', 'BINDIR');
      if ($where) {
        $where .= '/vmware-uninstall-vix.pl';
        if (-f $where) {
          system(shell_string($where) . " $vixArgs");
        }
      }
    }

    db_save();
    if (vmware_product() eq 'acevm') {
      print wrap("The removal of VMware ACE package $gManifest{'acename'}" . 
                 " completed successfully.\n\n", 0);
    } else {
       print wrap('The removal of ' . vmware_longname() . ' completed '
                  . 'successfully. Thank you for having tried this software.'
                  . "\n\n", 0);
    }

    exit 0;
  }

  error('This program must be named ' . $cInstallerFileName . ' or '
        . $gUninstallerFileName . '.' . "\n\n");
}

main();
