#!/bin/sh
libs="$1"
shift
binary="$1"
shift
confs="$1"
shift

# For the sake of simplicity, we only check the first arg for the gdb/valgrind
# flag.
valgrind=0
debugger=0
if [ "$1" = '-V' ] ; then
   # Pop -V.
   shift
   valgrind=1
elif [ "$1" = '-G' ] ; then
   # Pop -G.
   shift
   debugger=1
fi

libpath=

# Apppend a directory to LD_LIBRARY_PATH
vm_append_lib() {
   local lib="$1" # IN: The library to append
   local path="$libs"'/'"$lib"

   if [ "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      path="$path"'/'"$lib"
   fi
    
   if [ "$libpath" = '' ]; then
      libpath="$path"
   else
      libpath="$libpath":"$path"
   fi
}

vm_append_cplusplus() {
   vm_append_lib 'libstdc++.so.6'
   vm_append_lib 'libgcc_s.so.1'
   vm_append_lib 'libsigc-2.0.so.0'
   vm_append_lib 'libglibmm-2.4.so.1'
   vm_append_lib 'libglibmm_generate_extra_defs-2.4.so.1'
   vm_append_lib 'libatkmm-1.6.so.1'
   vm_append_lib 'libpangomm-1.4.so.1'
   vm_append_lib 'libgdkmm-2.4.so.1'
   vm_append_lib 'libgtkmm-2.4.so.1'
   vm_append_lib 'libcairomm-1.0.so.1'
   vm_append_lib 'libgnomecanvasmm-2.6.so.1'
   vm_append_lib 'libview.so.2'
   vm_append_lib 'libsexymm.so.2'
}

# Append the GTK+ libraries to LD_LIBRARY_PATH
vm_fix_gtk() {
   # GLib
   vm_append_lib 'libglib-2.0.so.0'
   vm_append_lib 'libgmodule-2.0.so.0'
   vm_append_lib 'libgobject-2.0.so.0'
   vm_append_lib 'libgthread-2.0.so.0'

   # ATK
   vm_append_lib 'libatk-1.0.so.0'

   # Pango
   vm_append_lib 'libpango-1.0.so.0'
   vm_append_lib 'libpangoft2-1.0.so.0'
   vm_append_lib 'libpangoxft-1.0.so.0'
   vm_append_lib 'libpangox-1.0.so.0'
   vm_append_lib 'libpangocairo-1.0.so.0'

   # GTK+
   vm_append_lib 'libgdk-x11-2.0.so.0'
   vm_append_lib 'libgdk_pixbuf-2.0.so.0'
   vm_append_lib 'libgtk-x11-2.0.so.0'

   # Cairo
   vm_append_lib 'libcairo.so.2'

   # librsvg
   vm_append_lib 'librsvg-2.so.2'

   # libsexy
   vm_append_lib 'libsexy.so.2'

   # GnomeCanvas
   # We use gnomecanvas 2.6. Apparently 2.4 is symbol compatible but not
   # compatible enough to avoid crashing at times, so consider gnomecanvas to
   # be part of the base Gtk set.
   vm_append_lib 'libgnomecanvas-2.so.0'

   # libglade
   # libglade has a similar potential for trouble as GnomeCanvas
   vm_append_lib 'libglade-2.0.so.0'

   # These are not a direct dependency of GTK+, but they keep breaking
   # binary compatibility in both directions, so use ours.
   vm_append_lib 'libfreetype.so.6'
   vm_append_lib 'libXft.so.2'
   vm_append_lib 'libXrender.so.1'

   # We have observed cases of incompatibility between system gtk and our
   # libpng and system libpng and our gtk. So they should be considered
   # part of the same set.
   vm_append_lib 'libpng12.so.0'

   # This is technically not needed here but it makes things simpler than
   # having to handle 'force' separately for libxml2.
   vm_append_lib 'libxml2.so.2'

   # C++ libs better match the base gtk ones in this case...
   vm_append_cplusplus

   echo "$libpath"
}

# Append the C++ libraries to LD_LIBRARY_PATH
vm_fix_cplusplus() {
   vm_append_cplusplus
   echo "$libpath"
}

vm_fix_libxml2() {
   vm_append_lib 'libxml2.so.2'
   echo "$libpath"
}

# Append the remaining, usually safe, libraries
# This is used in the 'force' case to ensure all our libs are used
vm_force_other_libs() {
   # Expat
   vm_append_lib 'libexpat.so.0'

   # libart
   vm_append_lib 'libart_lgpl_2.so.2'

   echo "$libpath"
}

# Parse ldd's output and append all missing libraries to LD_LIBRARY_PATH
vm_append_missing() {
   local lib
   local dummy
   local status

   while read -r lib dummy status; do
      if [ "$status" = 'not found' ]; then
         vm_append_lib "$lib"
      fi
   done

   echo "$libpath"
}

# Run "$binary" with the correct environment.
vm_run() {
   # Append any libraries that are still missing.
   if [ "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export LD_PRELOAD="$LD_PRELOAD":"`LANGUAGE=C LANG=C ldd "$binary" | vm_append_missing`"
   else
      local ldBasePath
      if [ ! -z "$LD_LIBRARY_PATH" ]; then
         ldBasePath="$LD_LIBRARY_PATH:"
      fi
      export LD_LIBRARY_PATH="$ldBasePath""`LANGUAGE=C LANG=C ldd "$binary" | vm_append_missing`"
   fi

  #
  # We must check for libxml2 at the last minute because we won't know if it's needed
  # until libvmwarebase has been resolved by appending missing libraries.
  #
  if [ "$VMWARE_USE_SHIPPED_GTK" = '' ]; then
      if LANGUAGE=C LANG=C ldd -r "$binary" 2>&1 >/dev/null \
            | grep -q 'xmlReadFile'; then
         # No sufficiently new libxml2 on the system
         local ldBasePath
         if [ ! -z "$LD_LIBRARY_PATH" ]; then
            ldBasePath="$LD_LIBRARY_PATH:"
         fi
         export LD_LIBRARY_PATH="$ldBasePath""`vm_fix_libxml2`"
      fi
   fi

   if [ "$VMWARE_USE_SHIPPED_GTK" = '' ]; then
      if LANGUAGE=C LANG=C ldd -r "$binary" 2>&1 >/dev/null \
            | grep -q 'CXXABI_1.3.1'; then
         # No sufficiently new C++ runtime environment on the system.
         local ldBasePath
         if [ ! -z "$LD_LIBRARY_PATH" ]; then
            ldBasePath="$LD_LIBRARY_PATH:"
         fi
         export LD_LIBRARY_PATH="$ldBasePath""`vm_fix_cplusplus`"
      fi
   fi

   if [ $valgrind = 1 ] ; then
      valgrind --log-file-exactly=/tmp/valgrind-ui.log --leak-check=yes \
               --show-reachable=no --partial-loads-ok=yes \
               --num-callers=32 "$binary" "$@"
   elif [ $debugger = 1 ] ; then
      gdb $VMUI_DEBUG_FLAGS --args "$binary" "$@"
   else
      "$binary" "$@"
   fi

   return "$?"
}

#
# This function tests if an environment variable is actually set or not.
# As we had to dig quite a bit to work out what to do, we've put it in
# a separate function.
#
vm_is_var_set() {
   local var="$1"
   
   eval '[ "${'"$var"'+yes}" = 'yes' ]'
}
      
#
# We must save all environment variables that we manipulate because we will
# be fork/exec'ing help/web browsers. We must also explicitly indicate which
# vars were originally unset. There is a third state where we don't try and
# save a var at all and this is indicated by not setting VMWARE_<foo> at all.
#
vm_save_var() {
   local var="$1"

   if vm_is_var_set 'VMWARE_'"$var"; then
      echo 'VMWARE_'"$var"' is already set, but it will be overridden.'
   fi

   if vm_is_var_set "$var"; then
      eval 'export VMWARE_'"$var"'="1$'"$var"'"'
   else
      eval 'export VMWARE_'"$var"'="0"'
   fi
}

#
# Function to strip qtengine out of the set of GTK2_RC_FILES. SuSE evilness
# necessitates this. It has to be a function because we wipe the 'command line'
# argument space when we do the stripping.
#
vm_strip_out_qt_engine() {
   local gtk2_rc_files="$1"
   local tmp_rc_files=""

   #
   # gtk-qt-engine is evil and SuSE use an evil mechanism to force it.
   # So, we have to strip it out like this:
   #
   set -- `IFS=':'; echo $gtk2_rc_files`
   while [ "$#" -gt 0 ]; do
      case "$1" in
         *'qtengine'*)
         ;;
         *'/Qt/'*)
         ;;
         *)
         tmp_rc_files="$tmp_rc_files""$1"':'
         ;;
      esac
      shift
   done
   echo "$tmp_rc_files"
}

#
# This list must be kept in sync with the corresponding UI code that restores
# these variables.
#
vm_save_var 'LD_LIBRARY_PATH'
vm_save_var 'LD_PRELOAD'
vm_save_var 'PANGO_RC_FILE'
vm_save_var 'GDK_PIXBUF_MODULE_FILE'
vm_save_var 'GTK_IM_MODULE_FILE'
vm_save_var 'FONTCONFIG_PATH'
vm_save_var 'GTK2_RC_FILES'
vm_save_var 'GTK_PATH'

#
# When environment variable VMWARE_USE_SHIPPED_GTK is set to "force", we
# forcefully use the GTK+ runtime environment (libraries + their
# configuration files) that we ship.
#
# When environment variable VMWARE_USE_SHIPPED_GTK is set to "yes", we
# thoughtfully use the GTK+ runtime environment (libraries + their
# configuration files) that we ship.  This works for all but the most broken
# environments, so if we're guessing we try this.
#
# When environment variable VMWARE_USE_SHIPPED_GTK is set to "no", we use 
# the system's GTK+ runtime environment.
#
# When environment variable VMWARE_USE_SHIPPED_GTK is not set (the default), we
# try to make an educated guess.
#

if [ "$VMWARE_USE_SHIPPED_GTK" = '' ]; then
   # GTK 2.10
   GTK_SYMBOL='gtk_print_error_quark'

   if LANGUAGE=C LANG=C ldd -r "$binary" 2>&1 >/dev/null \
         | grep -q $GTK_SYMBOL; then
      # No sufficiently new GTK+ runtime environment on the system.
      export VMWARE_USE_SHIPPED_GTK=yes
   fi
fi

if [ "$VMWARE_USE_SHIPPED_GTK" = 'yes' -o "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
   #
   # Libraries
   #
   
   if [ "$VMWARE_USE_SHIPPED_GTK" = 'yes' ]; then
      export LD_LIBRARY_PATH="`vm_fix_gtk`"
   else
      export LD_PRELOAD="`vm_fix_gtk`":"`vm_force_other_libs`"
   fi

   #
   # Library configuration files
   #

   if [ -z "$PANGO_RC_FILE" -o "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export PANGO_RC_FILE="$confs"'/etc/pango/pangorc'
   fi

   if [ -z "$GDK_PIXBUF_MODULE_FILE" -o "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export GDK_PIXBUF_MODULE_FILE="$confs"'/etc/gtk-2.0/gdk-pixbuf.loaders'
   fi

   if [ -z "$GTK_IM_MODULE_FILE" -o "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export GTK_IM_MODULE_FILE="$confs"'/etc/gtk-2.0/gtk.immodules'
   fi

   # We always want to append our gtkrc, unless we're forcing it
   if [ "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export GTK2_RC_FILES="$confs"'/etc/gtk-2.0/gtkrc'
   else
      if [ -z "$GTK2_RC_FILES" ]; then
         GTK2_RC_FILES='/etc/gtk-2.0/gtkrc:'"$HOME"'/.gtkrc-2.0:'
      else
         GTK2_RC_FILES="`vm_strip_out_qt_engine $GTK2_RC_FILES`"
      fi
      export GTK2_RC_FILES="$GTK2_RC_FILES""$confs"'/etc/gtk-2.0/gtkrc'
   fi

   # We will try as hard as possible to use the system fonts.conf
   if [ \( ! -e "/etc/fonts/fonts.conf" -a -z "$FONTCONFIG_PATH" \) \
        -o "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export FONTCONFIG_PATH="$confs"'/etc/fonts'
   fi

   # always append our libdir to GTK_PATH
   vm_gtk_path="$confs"'/lib/gtk-2.0/2.10.0/'
   if [ -z "$GTK_PATH" -o "$VMWARE_USE_SHIPPED_GTK" = 'force' ]; then
      export GTK_PATH="$vm_gtk_path"
   else
      export GTK_PATH="$GTK_PATH"':'"$vm_gtk_path"
   fi
fi

vm_run "$@"
